//================================================================================
// MiniBot.
//================================================================================

class MiniBot extends Pawn;

var (Bowen) int BulletDamage;
var (Bowen) float sightRadius, desiredMaxRange, desiredMinRange, speed, MaxFlameRange, MaxBulletRange, AttackDelay;
var (Bowen) class <Projectile> FlameClass, RocketClass;
var Pawn target;
var DeusExPlayer PlayerOwner;
var float enemyCheckTimer;
var vector myDestination, commandDestination;
var bool gunOffsetToggle;

var(Sounds) sound SpeechTargetAcquired;
var() int EMPHitPoints;
var ParticleGenerator sparkGen;
var(Sounds) sound explosionSound;

//DeusExWeapon primary, secondary, tertiary;


function deactivate()
{
	local MBDecoration d;
	if(PlayerOwner != None)
		PlayerOwner.clientMessage("MiniBot deactivated");
	d = spawn(class'MBDecoration',,,Location,Rotation);
	if(d != None)
	{
		d.HitPoints = Health;
		destroy();
	}
}

function setTarget(pawn p)
{
	PlayerOwner.clientMessage("MiniBot new target: " $ p);
	target = p;
	GoToState('Attacking');
}

function navigateTo(vector destination)
{
	commandDestination = destination;
	PlayerOwner.clientMessage("MiniBot en route to destination");
	return;
}

function botPickTarget()
{
	if (Level.NetMode == NM_Standalone)
		SPPickTarget();
	else
		MPPickTarget();
}

function MPpickTarget()
{
	local DeusExPlayer p, bestTarget;
	local PlayerReplicationInfo PRI,oPRI;
	local int closestRange;
	
	closestRange=sightRadius;
	if (DeusExPlayer(Owner) == None) return;
	oPRI=DeusExPlayer(Owner).PlayerReplicationInfo;
	
	foreach VisibleActors (class 'DeusExPlayer', p, sightRadius)
	{
		PRI=p.PlayerReplicationInfo;
		//if potential target is on the other team, or this is not a team game
		if (PRI.Team != oPRI.Team || TeamDMGame(DeusExPlayer(Owner).DXGame) == None)
		{ // pick the closest one
			if (VSize(Location - p.Location) < closestRange)
				bestTarget = p;
		}
	}
	target=bestTarget;
	if(playerOwner == None)
		deactivate();
	if(target != None)
		PlayerOwner.ClientMessage("MiniBot: Found target:" $ p);
}

function SPpickTarget()
{
	local ScriptedPawn p, bestTarget;
	local int closestRange;
	
	closestRange=sightRadius;
	if (DeusExPlayer(Owner) == None) return;
	
	foreach VisibleActors (class 'ScriptedPawn', p, sightRadius)
	{
		if (p.isValidEnemy(PlayerOwner))
		{ // pick the closest one
			if (VSize(Location - p.Location) < closestRange)
				bestTarget = p;
		}
	}
	target=bestTarget;
	if (playerOwner == None)
		deactivate();
	if (target != None)
		broadcastMessage("MiniBot: Found target:" $ p);
}

function bool noTarget()
{
	return (target == None || VSize(target.Location - Location) > desiredMaxRange || !fastTrace(Location, target.Location));
}


auto state Seeking
{
	function tick (float deltaTime)
	{
		enemyCheckTimer += deltaTime;
		if (noTarget() && enemyCheckTimer > 0.5)
		{
			botPickTarget();
		}
		else if (!noTarget())
			GoToState('Attacking');
	}
	
	function Vector PickDestination()
	{
		if(commandDestination != vect(0,0,0))
		{
			if(vsize(commandDestination - Location) > desiredMinRange)
				return commandDestination;
			else
			{
				PlayerOwner.clientMessage("MiniBot reached destination.");
				commandDestination = vect(0,0,0);
			}
		}
		else
			return Location+(VRand()*Vect(1,1,0))*Frand()*1000;
	}
	
	begin:
		botPickTarget();
		if(!noTarget())
			GoToState('Attacking');
	moverandomly:
		MyDestination=PickDestination();
		if (vsize(Location - MyDestination) > DesiredMinRange)
			MoveTo(MyDestination);
		else
			sleep(1);
		Goto('Begin');
}

state Attacking
{
	function attackTarget()
	{
		local vector startLocation, gunOffset;
		LookAtVector(target.location, true, true, false);
		if(!facingTarget())
			return;
		
		startLocation = Location;
		gunOffset = collisionRadius*normal((target.location - startLocation) cross vect(0,0,1));
		gunOffsetToggle = !gunOffsetToggle;
		
		if(gunOffsetToggle)
			startLocation += gunOffset;
		else
			startLocation -= gunOffset;
			
		Fire(startLocation, target);
	}
	
	function bool facingTarget() ///todo
	{
		return true;
	}
	
	simulated function Fire(vector startLocation, actor target)
	{
		local int range;
		range = vsize(startLocation - Target.Location);
		
		if (range > MaxBulleTrange || target.isA('Robot'))  // Rockets for long range and robots
		{
			FireRocket(startLocation, target);
		}
		else if (range < MaxFlameRange)
			fireFlamethrower(startLocation, target);
		else if (Range < MaxBulletRange)
			fireBullets(startLocation, target);
	}
	
	simulated function fireFlamethrower(vector startLocation, actor target)
	{
		local vector intercept;
		
		intercept = class'ProjectileTurret'.static.CalcIntercept(startLocation,target.location,target.velocity,FlameClass.default.speed, MaxFlameRange);
		Spawn(FlameClass,PlayerOwner,,startLocation,rotator(intercept));
		PlayerOwner.ClientMessage("MiniBot: Firing flamethrower");
	}
	
	simulated function FireBullets (Vector startLocation, actor target)
	{
		local Vector EndTrace;
		local Vector TargetLocation;
		local Vector TraceNormal;
		local Vector HitLocation;
		local Rotator Direction;
		local Actor Other;
	
		PlayerOwner.ClientMessage("MiniBot: Firing guns");
		
		TargetLocation = Target.Location;
		Direction = rotator(normal(TargetLocation-Location));
		EndTrace = Location + normal(TargetLocation-Location) * MaxBulletRange;
		Other = Trace(HitLocation,TraceNormal,EndTrace,Location,True);
		if ( Other != Level )
		{
			Spawn(Class'Tracer',,,startLocation,Direction);
			DoDamage(Other);
			DoSound(Sound'RobotFireGun');
		}
	}
	
	simulated function fireRocket(vector startLocation, actor target)
	{
		local vector intercept;

		intercept = class'ProjectileTurret'.static.CalcIntercept(startLocation,target.location,target.velocity,RocketClass.default.speed);
		Spawn(RocketClass,PlayerOwner,,startLocation,rotator(intercept));
		PlayerOwner.ClientMessage("MiniBot: Firing rocket");
	}
	
	function DoDamage (Actor Other)
	{
		Other.TakeDamage(BulletDamage,PlayerOwner,Location,vect(0.00,0.00,0.00),'shot');
	}
	
	simulated function DoSound (Sound SoundToPlay)
	{
		PlaySound(SoundToPlay,SLOT_None,,,8192.0);
	}
	
	begin:
		if (vsize(Location - Target.Location) > DesiredMinRange)
			MoveToward(Target);
		AttackTarget();
		Sleep(AttackDelay);
		if(!noTarget())
			Goto('Begin');
		else
			GoToState('Seeking');
}

function PostBeginPlay()
{
	local DeusExPlayer p;
	if (Level.NetMode==NM_StandAlone)
		foreach allactors (class'DeusExPlayer', p)
			SetOwner(p);
}











//////////////////////////
/// Rest from Robot.uc ///
//////////////////////////

function TakeDamage( int Damage, Pawn instigatedBy, Vector hitlocation, 
						Vector momentum, name damageType)
{
	local float actualDamage;
	local int oldEMPHitPoints;
	
	// enough EMP damage shuts down the robot
	if (damageType == 'EMP')
	{
		oldEMPHitPoints = EMPHitPoints;
		EMPHitPoints   -= Damage;

		// make smoke!
		if (EMPHitPoints <= 0)
		{
			EMPHitPoints = 0;
			if (oldEMPHitPoints > 0)
			{
				PlaySound(sound'EMPZap', SLOT_None,,, (CollisionRadius+CollisionHeight)*8, 2.0);
				InitGenerator();
				if (sparkGen != None)
				{
					sparkGen.LifeSpan = 6;
					sparkGen.particleTexture = Texture'Effects.Smoke.SmokePuff1';
					sparkGen.particleDrawScale = 0.3;
					sparkGen.bRandomEject = False;
					sparkGen.ejectSpeed = 10.0;
					sparkGen.bGravity = False;
					sparkGen.bParticlesUnlit = True;
					sparkGen.frequency = 0.3;
					sparkGen.riseRate = 3;
					sparkGen.spawnSound = Sound'Spark2';
				}
			}
			AmbientSound = None;
			Deactivate();
		}

		// make sparks!
		else if (sparkGen == None)
		{
			InitGenerator();
			if (sparkGen != None)
			{
				sparkGen.particleTexture = Texture'Effects.Fire.SparkFX1';
				sparkGen.particleDrawScale = 0.2;
				sparkGen.bRandomEject = True;
				sparkGen.ejectSpeed = 100.0;
				sparkGen.bGravity = True;
				sparkGen.bParticlesUnlit = True;
				sparkGen.frequency = 0.2;
				sparkGen.riseRate = 10;
				sparkGen.spawnSound = Sound'Spark2';
			}
		}

		return;
	}
	
	// play a hit sound
	PlayTakeHitSound(Damage, damageType, 1);

	// increase the pitch of the ambient sound when damaged
	if (SoundPitch == Default.SoundPitch)
		SoundPitch += 16;

	actualDamage = Level.Game.ReduceDamage(Damage, DamageType, self, instigatedBy);

	// robots don't have soft, squishy bodies like humans do, so they're less
	// susceptible to gunshots...
	if (damageType == 'Shot')
		actualDamage *= 0.25;  // quarter strength

	// hitting robots with a prod won't stun them, and will only do a limited
	// amount of damage...
	else if ((damageType == 'Stunned') || (damageType == 'KnockedOut'))
		actualDamage *= 0.5;  // half strength

	// flame attacks don't really hurt robots much, either
	else if ((damageType == 'Flamed') || (damageType == 'Burned'))
		actualDamage *= 0.25;  // quarter strength

	actualDamage = int(actualDamage+0.5);
	if (actualDamage < 5)
		return;

	Health -= int(actualDamage);

	if (Health <= 0)
	{
		if ( actualDamage > mass )
			Health = -1 * actualDamage;
		Explode(hitLocation);
	}
	MakeNoise(1.0);

// 	ReactToInjury(instigatedBy, damageType, HITLOC_None);
	
// 	Super.takeDamage(Damage, instigatedBy, hitLocation, momentum, damageType);
}

function bool IgnoreDamageType(Name damageType)
{
	if ((damageType == 'TearGas') || (damageType == 'HalonGas') || (damageType == 'PoisonGas') || (damageType == 'Radiation'))
		return True;
	else if ((damageType == 'Poison') || (damageType == 'PoisonEffect'))
		return True;
	else if (damageType == 'KnockedOut')
		return True;
	else
		return False;
}

function Explode(vector HitLocation)
{
	local int i, num;
	local float explosionRadius;
	local Vector loc;
	local DeusExFragment s;
	local ExplosionLight light;

	explosionRadius = (CollisionRadius + CollisionHeight) / 2;
	PlaySound(explosionSound, SLOT_None, 2.0,, explosionRadius*32);

	if (explosionRadius < 48.0)
		PlaySound(sound'LargeExplosion1', SLOT_None,,, explosionRadius*32);
	else
		PlaySound(sound'LargeExplosion2', SLOT_None,,, explosionRadius*32);

	AISendEvent('LoudNoise', EAITYPE_Audio, , explosionRadius*16);

	// draw a pretty explosion
	light = Spawn(class'ExplosionLight',,, HitLocation);
	for (i=0; i<explosionRadius/20+1; i++)
	{
		loc = Location + VRand() * CollisionRadius;
		if (explosionRadius < 16)
		{
			Spawn(class'ExplosionSmall',,, loc);
			light.size = 2;
		}
		else if (explosionRadius < 32)
		{
			Spawn(class'ExplosionMedium',,, loc);
			light.size = 4;
		}
		else
		{
			Spawn(class'ExplosionLarge',,, loc);
			light.size = 8;
		}
	}

	// spawn some metal fragments
	num = FMax(3, explosionRadius/6);
	for (i=0; i<num; i++)
	{
		s = Spawn(class'MetalFragment', Owner);
		if (s != None)
		{
			s.Instigator = Instigator;
			s.CalcVelocity(Velocity, explosionRadius);
			s.DrawScale = explosionRadius*0.075*FRand();
			s.Skin = GetMeshTexture();
			if (FRand() < 0.75)
				s.bSmoking = True;
		}
	}

	// cause the damage
	HurtRadius(0.5*explosionRadius, 8*explosionRadius, 'Exploded', 100*explosionRadius, Location);
	
	Destroy();
}

simulated event Destroyed()
{
	Super.Destroyed();

	DestroyGenerator();
}


function PlayTargetAcquiredSound()
{
	PlaySound(SpeechTargetAcquired, SLOT_None,,, 2048);
}

function InitGenerator()
{
	local Vector loc;

	if ((sparkGen == None) || (sparkGen.bDeleteMe))
	{
		loc = Location;
		loc.z += CollisionHeight/2;
		sparkGen = Spawn(class'ParticleGenerator', Self,, loc, rot(16384,0,0));
		if (sparkGen != None)
			sparkGen.SetBase(Self);
	}
}

function DestroyGenerator()
{
	if (sparkGen != None)
	{
		sparkGen.DelayedDestroy();
		sparkGen = None;
	}
}

defaultproperties
{
    BulletDamage=20
    SightRadius=65536.00
    desiredMaxRange=20000.00
    desiredMinRange=400.00
    speed=400.00
    MaxFlameRange=450.00
    MaxBulletRange=5000.00
    AttackDelay=0.10
    FlameClass=Class'DeusEx.Fireball'
    RocketClass=Class'DeusEx.RocketMini'
    SpeechTargetAcquired=Sound'DeusExSounds.Robot.SecurityBot3TargetAcquired'
    EMPHitPoints=50
    GroundSpeed=95.00
    WaterSpeed=50.00
    AirSpeed=144.00
    Health=5000
    Physics=2
    DrawType=2
    Mesh=LodMesh'DeusExCharacters.SecurityBot4'
    SoundRadius=16
    SoundVolume=128
    AmbientSound=Sound'DeusExSounds.Robot.SecurityBot4Move'
    CollisionRadius=27.50
    CollisionHeight=28.50
    Mass=1000.00
    Buoyancy=100.00
    BindName="SecurityBot4"
    FamiliarName="Security Bot"
    UnfamiliarName="Security Bot"
}
