/*=============================================================================
	ClusterSubsystem.h: Unreal audio interface object.
	Copyright 1999 Epic Games, Inc. All Rights Reserved.
	Copyright 2015 Sebastian Kaufel. All Rights Reserved.

Revision history:
	* Created by Joseph I. Valenzuela
=============================================================================*/

//#include "AudioTypes.h"
#include "ALAudioTypes.h"

/*------------------------------------------------------------------------------------
	Helpers
------------------------------------------------------------------------------------*/

// Constants.
#define MAX_EFFECTS_CHANNELS 64
#define MUSIC_CHANNELS       32
#define EFFECT_FACTOR        1.0

// Utility Macros.
#define safecall(f) \
{ \
	guard(f); \
	INT Error=f; \
	if( Error==0 ) \
		debugf( NAME_Warning, TEXT("%s failed: %i"), TEXT(#f), Error ); \
	unguard; \
}
#define silentcall(f) \
{ \
	guard(f); \
	f; \
	unguard; \
}

#define CLUSTER_START_PROPERTY_BITFIELD \
	union \
	{ \
		BITFIELD _Bits; \
		struct \
		{ \

#define CLUSTER_FINISH_PROPERTY_BITFIELD \
		};\
	};

#define CPP_CLUSTER_PROPERTY_BITFIELD CPP_PROPERTY(_Bits)

/*------------------------------------------------------------------------------------
	UClusterAudioSubsystem.
------------------------------------------------------------------------------------*/

//
// The Generic implementation of UAudioSubsystem.
//
class CLUSTER_API UClusterAudioSubsystem : public UAudioSubsystem
{
	//DECLARE_CLASS(UClusterAudioSubsystem,UAudioSubsystem,CLASS_Config,Cluster)
	DECLARE_CLASS(UClusterAudioSubsystem,UAudioSubsystem,CLASS_Config)

	// Configuration.
	CLUSTER_START_PROPERTY_BITFIELD;
	BITFIELD UseFilter:1;
	BITFIELD UseSurround:1;
	BITFIELD UseStereo:1;
	BITFIELD UseCDMusic:1;
	BITFIELD UseDigitalMusic:1;
	BITFIELD ReverseStereo:1;
	CLUSTER_FINISH_PROPERTY_BITFIELD;
	FLOAT    AmbientFactor;
	FLOAT    DopplerSpeed;
	INT      Latency;
	INT      EffectsChannels;
	BYTE     OutputRate;
	BYTE     MusicVolume;
	BYTE     SoundVolume;
	BYTE     SpeechVolume;
	FStringNoInit DeviceName;

	// Variables.
	UBOOL      Initialized;
	UBOOL      InitializedMikMod;
	UViewport* Viewport;
	DOUBLE     LastTime;
	UMusic*    CurrentMusic;
	BYTE       CurrentCDTrack;
	FLOAT      MusicFade;
	INT        FreeSlot;
	UBOOL      AudioStats;
	UBOOL      DetailStats;

	// Currently used volumes. Not to be confused with the config values.
	BYTE CurrentMusicVolume;
	BYTE CurrentSoundVolume;
	BYTE CurrentSpeechVolume;

	// Marker to update Sound/Speech/Music volume on next Update() call.
	UBOOL UpdateSoundVolume;
	UBOOL UpdateSpeechVolume;
	UBOOL UpdateMusicVolume;

	// foo
	//void      *context_id;
	FClusterSoundMap *IdMapping;
	UMusic    *MusicUSound;
	MODULE    *MikModModule;

	// MusicBufferSize is the size of music data we buffer, in bytes
	//INT MusicBufferSize;

	// Constructor.
	UClusterAudioSubsystem();
	void StaticConstructor();

	// UObject interface.
	void Destroy();
	void PostEditChange();
	void ShutdownAfterError();

	// UAudioSubsystem interface.
	UBOOL Init();
	void SetViewport( UViewport* Viewport );
	UBOOL Exec( const TCHAR* Cmd, FOutputDevice& Ar=*GLog );
	void Update( FPointRegion Region, FCoords& Coords );
	void UnregisterSound( USound* Sound );
	void UnregisterMusic( UMusic* Music );
	UBOOL PlaySound( AActor* Actor, INT Id, USound* Sound, FVector Location, FLOAT Volume, FLOAT Radius, FLOAT Pitch );
	void NoteDestroy( AActor* Actor );
	void RegisterSound( USound* Sound );
	void RegisterMusic( UMusic* Music ) {};
	UBOOL GetLowQualitySetting() {return 0;}
	UViewport* GetViewport();
	void RenderAudioGeometry( FSceneNode* Frame );
	void PostRender( FSceneNode* Frame );

	// UAudioSubsystem interface additions for Deus Ex.
#if ENGINE_VERSION==1100
	void SetInstantSoundVolume( BYTE InstantSoundVolume );
	void SetInstantSpeechVolume( BYTE InstantSpeechVolume );
	void SetInstantMusicVolume( BYTE InstantMusicVolume );
	void StopSoundId( INT Id );
#endif

	// CD Audio control.
	UBOOL StartCDAudio( INT Track );
	UBOOL StopCDAudio();

	// Inlines.
	Sample* GetSound( USound* Sound )
	{
		check(Sound);
		if( !Sound->Handle )
			RegisterSound( Sound );
		return (Sample*)Sound->Handle;
	}
	FLOAT SoundPriority( UViewport* Viewport, FVector Location, FLOAT Volume, FLOAT Radius )
	{
		return Volume * (1.0 - (Location - (Viewport->Actor->ViewTarget?Viewport->Actor->ViewTarget:Viewport->Actor)->Location).Size()/Radius);
	}

private:
	// Context.
	FClusterDeviceContext DeviceContext;

};
