
//#include <mikmod.h>
//#include <stdlib.h>

#include "Cluster.h"

#define MINSIZE  16

#define UNIQUEKEY(x) ((int) (x->Handle))
#define CLEARKEY(us) (us->Handle = (void *) 0)

// https://mackron.github.io/dr_wav
#define DR_WAV_IMPLEMENTATION
//#define DR_WAV_NO_CONVERSION_API // !! BROKEN
#define DR_WAV_NO_STDIO
#include "dr_wav.h"

FClusterSoundMap::FClusterSoundMap()
{
	guard(FClusterSoundMap::FClusterSoundMap);
	Pool = new HashMM<poolNode_t>(MINSIZE);
	unguard;
}

FClusterSoundMap::~FClusterSoundMap()
{
	guard(FClusterSoundMap::~FClusterSoundMap);
	delete Pool;
	unguard;
}

// Sorf of a drop in replacement for alutBufferAndConvertData_LOKI. --han
UBOOL BufferAndConvertData( ALuint BufferId, BYTE* Data, DWORD Size )
{
	guard(BufferAndConvertData);
	ALenum Error;

	// Sanity checks.
	if ( !Data )
	{
		debugSoundf( NAME_DevAudio, TEXT("BufferAndConvertData failed. Data is NULL.") );
		return 0;
	}
	if ( !Size )
	{
		debugSoundf( NAME_DevAudio, TEXT("BufferAndConvertData failed. Size is 0.") );
		return 0;
	}

	// Try open with drwav.
	guard(WAV);
	drwav WAV;
	if ( drwav_init_memory(&WAV,Data,Size) )
	{
		debugSoundf( NAME_DevAudio, TEXT("Detected WAV file.") );

		SIZE_T BufferSize = 0;
		BYTE*  Buffer     = NULL;
		ALenum Format     = 0;

		// Require mono.
		if ( WAV.fmt.channels!=1 )
		{
			debugSoundf( NAME_DevAudio, TEXT("BufferAndConvertData failed. Sound is not mono.") );
			goto WavFailed;
		}
		// Require PCM.
		if ( WAV.fmt.formatTag!=DR_WAVE_FORMAT_PCM )
		{
			debugSoundf( NAME_DevAudio, TEXT("BufferAndConvertData failed. Sound is not PCM.") );
			goto WavFailed;
		}
		// Determine format. FIX-ME: What about signed/unsigned?
		switch ( WAV.fmt.bitsPerSample )
		{
			case 8:  Format = AL_FORMAT_MONO8;  break;
			case 16: Format = AL_FORMAT_MONO16; break;

			default:
				debugSoundf( NAME_DevAudio, TEXT("BufferAndConvertData failed. Sound is neither 8 nor 16 bits per sample.") );
				goto WavFailed;
				break;
		}

		BufferSize = WAV.totalSampleCount*WAV.bytesPerSample;
		check(BufferSize>0);

		Buffer = new BYTE[BufferSize];

		verify(drwav_read(&WAV,WAV.totalSampleCount,Buffer)==WAV.totalSampleCount);

		// Clear error code.
		alGetError();

		// Buffer into OpenAL.
		alBufferData( BufferId, Format, Buffer, BufferSize, WAV.sampleRate );
		if ( (Error=alGetError())!=AL_NO_ERROR )
		{
			debugf( NAME_Warning, TEXT("Failed to buffer (%i)"), Error );
			goto WavFailed;
		}

		// Success.
		drwav_uninit(&WAV);
		if ( Buffer )
			delete Buffer;
		return 1;

		// Failed.
		WavFailed:
		drwav_uninit(&WAV);
		if ( Buffer )
			delete Buffer;
		return 0;
	}
	unguard;

	debugSoundf( NAME_DevAudio, TEXT("BufferAndConvertData failed. Unrecognices format.") );
	return 0;
	unguard;
}

INT FClusterSoundMap::Add( USound* Sound )
{
	guard(FClusterSoundMap::Add);
	checkSlow(Sound&&Sound!=(USound*)INDEX_NONE);

	ALenum Error;
	ALuint BufferId;

	// Clear error code.
	alGetError();

	// Generate Buffer.
	alGenBuffers( 1, &BufferId );
	if ( (Error=alGetError())!=AL_NO_ERROR )
	{
		debugf( NAME_Warning, TEXT("Failed to generate Buffer for %s (%i)"), Sound->GetFullName(), Error );
		return 0;
	}

	// Try to Buffer.
	if ( !BufferAndConvertData(BufferId,&Sound->Data(0),Sound->Data.Num()) )
	{
		debugSoundf( NAME_DevAudio, TEXT("Could not buffer and convert %s"), Sound->GetFullName() );

		alDeleteBuffers( 1, &BufferId );
		return 0;
	}

	poolNode_t* entry = new poolNode_t;

	entry->bid = BufferId;
	entry->us  = Sound;

	Pool->add( entry->bid, entry );

	return entry->bid;
	unguard;
}

void FClusterSoundMap::Remove( USound* Sound )
{
	guard(FClusterSoundMap::Remove);

	const poolNode_t *entry = Pool->get(UNIQUEKEY(Sound));
	ALuint bid;

	if(entry == NULL) {
		return;
	}

	bid = entry->bid;

	alDeleteBuffers(1, &bid);

	Pool->remove(UNIQUEKEY(Sound));

	CLEARKEY(Sound);

	delete (void*)entry;

	unguard;
}

ALuint FClusterSoundMap::Find( USound* Sound )
{
	guard(FClusterSoundMap::Find);

	const poolNode_t *entry = Pool->get(UNIQUEKEY(Sound));

	if(entry == NULL) {
		return 0;
	}

	return entry->bid;
	unguard;
}
