#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "hash.h"

#include "Cluster.h"

/* FIXME: replace with good mojo */
#define keytoindex(p, c) (((size_t) p) % (size_t ) c)

/* static function prototypes */
static int nextprime(size_t number);
static long get_first_position_key(hash *h);

const unsigned int hash_primes[] = {
	   23,    47,    83,     107,
	  139,   167,   199,     211,
	  239,   307,   397,     503,
	  809,  1601,  3203,    6421,
	 8009, 10007, 11003,   20983,
	40993, 80989, 160997, 320953,
	0 };

static int nextprime(size_t number) {
	int i = 0;

	for(i = 0; hash_primes[i] > 0; i++) {
	    if(hash_primes[i] >= number)
	        return hash_primes[i];
	}
	
	debugf( NAME_DevAudio, TEXT("no stored prime > %d"), (int)number );
	return -1;
}

hash *hash_alloc (unsigned int size) {
	hash *retval = NULL;
	unsigned int i = 0;

	if(size < DEFAULT_HASHSIZE) {
	    size = DEFAULT_HASHSIZE;
	}

	size = nextprime(size);
	
	retval = (hash *) malloc(sizeof *retval);
	if(retval == NULL) {
	    perror("hash_alloc malloc");
	    return NULL;
	}

	retval->table = (h_entry **) malloc(size * sizeof *retval->table);
	if(retval->table == NULL) {
		perror("hash_alloc malloc");
		free(retval);
		return NULL;
	}

	for(i = 0; i < size; i++) {
		retval->table[i] = NULL;
	}

	retval->items    = 0;
	retval->capacity = size;

	return retval;
}

h_entry *h_entry_alloc(int key, void *newdata) {
	h_entry *retval;

	retval = (h_entry *) malloc(sizeof *retval);
	if(retval == NULL) {
		return NULL;
	}

	retval->key  = key;
	retval->data = newdata;

	return retval;
}

/*
 *  add hash item he to hash h.  We actually just store the pointer to 
 *  he so freeing he will result in bad mojo:  once put in h, the only 
 *  method used to free should be hash_free
 */
hash *hash_add(hash *h, h_entry *he) {
	hash *newhash; /* if needed */
	int index = -1;

	if(h == NULL) {
		h = hash_alloc(1);
	    	if(h == NULL) {
			perror("malloc");
			return NULL;
		}
	}
	
	if(h->capacity <= h->items)
	{
	  /*
	   *  We should allocate more space.
	   */
	  newhash = hash_resize(h, h->capacity * 2);

	  if(newhash == NULL)
		{
			debugf( NAME_DevAudio, TEXT("couldn't realloc hash table: %s %d\n"), appFromAnsi(__FILE__), __LINE__ );
	    return h;
	  }

	  h = newhash;
	}

	/*
	 *  Add the key
	 */

	/* get index */
	if((index = keytoindex(he->key, h->capacity)) < 0)
	{
		debugf( NAME_DevAudio, TEXT("%s: Couldn't get good index: %s %d\n"), TEXT("jlib"), appFromAnsi(__FILE__), __LINE__ );
		return h;
	}

	/* avoid collision */
	while(h->table[index])
	{
		if(h->table[index]->key == he->key)
		{
			/* duplicate entry */
			debugf( NAME_DevAudio, TEXT("duplicate key [%p]\n"), he);
			exit(-1);
		}

		index = (index+1) % (h->capacity);
	}

	h->table[index] = he;
	h->items++;

	return h;
}

/*
 *  add hash item he to hash h.  We ignore the key and put in the first
 *  free position, setting he's key in the process.
 */
hash *hash_add_first_free(hash *h, h_entry *he)
{
	hash *newhash; /* if needed */
	int index = -1;

	if(h == NULL)
	{
		h = hash_alloc(1);
	  if(h == NULL)
		{
			perror("malloc");
			return NULL;
		}
	}
	
	if(h->capacity <= h->items) {
		/*
		 *  We should allocate more space.
		 */
		newhash = hash_resize(h, h->capacity * 2);

		if(newhash == NULL)
		{
			debugf( NAME_DevAudio, TEXT("couldn't realloc hash table: %s %d"), appFromAnsi(__FILE__), __LINE__ );
			return h;
		}

		h = newhash;
	}

	/*
	 *  Add the key
	 */
	he->key = get_first_position_key(h);

	/* get index */
	if((index = keytoindex(he->key, h->capacity)) < 0)
	{
		debugf( NAME_DevAudio, TEXT("%s: Couldn't get good index: %s %d"), TEXT("jlib"), appFromAnsi(__FILE__), __LINE__ );
		return h;
	}

	/* avoid collision */
	while(h->table[index])
	{
		if(h->table[index]->key == he->key)
		{
	    /* duplicate entry */
			debugf( NAME_DevAudio, TEXT("duplicate key [%p]"), he );
	    exit(-1);
		}

		index = (index+1) % (h->capacity);
	}

	h->table[index] = he;
	h->items++;

	return h;
}

hash *hash_remove(hash *h, int key)
{
	int index = -1;
	unsigned int total_itr = 0;

	if(h == NULL)
	{
		return NULL;
	}
	
	index = keytoindex(key, h->capacity);

	/*
	 *  Iterate over keys until we find the right one.  Is it right
	 *  to have to skip NULL ones?  Why am I getting NULL values?
	 */
	while((total_itr <= h->capacity) && (!h->table[index] || ((h->table[index]->key != key))))
	{
		index = (index+1) % (h->capacity);
		total_itr++;
	}

	/*
	 *  We've gone over the entire hash and not found it, so someone
	 *  has requested an illegal remove.
	 */
	if(total_itr > h->capacity) {
	    return h; 
	}

	free(h->table[index]);
	h->table[index] = NULL;

	h->items--;

	return h;
}

h_entry *hash_get(hash *h, int key) {
	unsigned int total_itr = 0;
	int index;

	if(h == NULL) {
		return NULL;
	}

	index = keytoindex(key, h->capacity);

	while(((h->table[index]) && (h->table[index]->key != key))
		&& (total_itr < h->capacity)) {
	    index = (index+1)%(h->capacity);
	    total_itr++;
	}

	if(h->table[index] == NULL) {
		return NULL;
	}

	if(h->table[index]->key == key) {
	    return h->table[index];
	}

	return NULL;
}


/*
 */
hash *hash_resize(hash *h, size_t capacityrequest) {
	h_entry **newtable;
	int newcapacity = nextprime(capacityrequest);
	int i;

	if(newcapacity == -1) {
		/* ran out of prime numbers */
		return NULL;
	}

	if(h == NULL) {
	    return hash_alloc(newcapacity);
	}

	newtable = (h_entry **) malloc(newcapacity * sizeof(h_entry *));
	if(newtable == NULL) {
		perror("malloc");
		return NULL;
	}

	for(i = 0; (unsigned) i < h->capacity; i++) {
		newtable[i] = h->table[i];
	}
	for(; i < newcapacity; i++) {
		newtable[i] = NULL;
	}

	free(h->table);

	h->table    = newtable;
	h->capacity = newcapacity;

	return h;
}

hash *hash_delete(hash *h, int key, void (*freer_func)(void *)) {
	unsigned int index;
	unsigned int total_itr = 0;

	if(h == NULL) {
		return NULL;
	}

	index = keytoindex(key, h->capacity);

	while(((h->table[index]) && (h->table[index]->key != key))
		&& (total_itr < h->capacity)) {
	    index = (index+1)%(h->capacity);
	    total_itr++;
	}

	/* already missing */
	if(h->table[index] == NULL) {
		return h;
	}

	if(h->table[index]->key == key) {
		(*freer_func)(h->table[index]->data);
	}

	free(h->table[index]);
	h->table[index] = NULL;

	h->items--;
	return h;
}

/*
 * free all of hash h, calling the passed freer_func on the
 * data of each h_entry in h 
 */
void hash_free(hash *h, void (*freer_func)(void *)) {
	unsigned int i;

	if(h == NULL) {
		return;
	}

	if(freer_func == NULL) {
		return;
	}

	for(i = 0; i < h->capacity; i++) {
		if(h->table[i] != NULL) {
			if(h->table[i]->data != NULL) {
				(*freer_func)(h->table[i]->data);
			}

			free(h->table[i]);
			h->table[i] = NULL;
		}
	}

	free(h->table);
	free(h);
}

/* returns first unoccupied position in hash h, or -1 if h is NULL */
static long get_first_position_key(hash *h) {
	long i;

	if(h == NULL) {
		return -1;
	}

	for(i = 0; (unsigned int) i < h->capacity; i++) {
		if(h->table[i] == NULL) {
			return i;
		}
	}

	return -1;
}
