/*=============================================================================
	ClusterDeviceContext.cpp: Cluster device context handling implementation.
	Copyright 2015 Sebastian Kaufel. All Rights Reserved.

Revision history:
	* Created by Sebastian Kaufel.
=============================================================================*/

/*-----------------------------------------------------------------------------
	Includes.
------------------------------------------------------------------------------*/

#include "Cluster.h"

/*-----------------------------------------------------------------------------
	FClusterDeviceContext.
-----------------------------------------------------------------------------*/

UBOOL FClusterDeviceContext::Init()
{
	guard(FClusterDeviceContext::Init);

	// Don't initialize again.
	if ( Initialized )
		return 0;

	// Open the device.
	alcGetError( NULL );
	Device = alcOpenDevice( DeviceName.Num() ? (ALCchar*)DeviceName.GetData() : NULL );
	if ( !Device )
	{
		debugf( NAME_DevAudio, TEXT("Failed to open device (%s)."), *ALCCurrentErrorString(NULL) );
		return 0;
	}

	// Build AttributeList for context creation.
	TArray<ALCint> AttributeList;
	for ( INT i=0; i<ContextAttributes.Num(); i++ )
	{
		AttributeList.AddItem( ContextAttributes(i).Attribute );
		AttributeList.AddItem( ContextAttributes(i).Value );
	}
	AttributeList.AddItem( (ALCint)0 );

	// Create a context.
	alcGetError( Device );
	Context = alcCreateContext( Device, (ALCint*)AttributeList.GetData() );
	if ( !Context )
	{
		debugf( NAME_DevAudio, TEXT("Failed to create context (%s)."), *ALCCurrentErrorString(Device) );

		// Close device.
		if ( alcCloseDevice(Device)==ALC_FALSE  )
			debugf( NAME_DevAudio, TEXT("Failed to close device (%s)."), *ALCCurrentErrorString(Device) );
		Device = NULL;
		return 0;
	}

	// Make context current.
	alcGetError( NULL );
	if ( alcMakeContextCurrent(Context)==ALC_FALSE )
	{
		debugf( NAME_DevAudio, TEXT("Failed to make context current (%s)."), *ALCCurrentErrorString(NULL) );

		// Destroy context.
		alcDestroyContext( Context );
		Context = NULL;

		// Close device.
		if ( alcCloseDevice(Device)==ALC_FALSE  )
			debugf( NAME_DevAudio, TEXT("Failed to close device (%s)."), *ALCCurrentErrorString(Device) );
		Device = NULL;
		return 0;
	}

	// Mark as initialized.
	Initialized = 1;

	// Add some log output.
	debugf( NAME_DevAudio, TEXT("Device context initialized.") );

	// Print general information.
	debugf( NAME_Init, TEXT("AL_VENDOR    : %s"), appFromAnsi((ANSICHAR*)alGetString(AL_VENDOR)) );
	debugf( NAME_Init, TEXT("AL_RENDERER  : %s"), appFromAnsi((ANSICHAR*)alGetString(AL_RENDERER)) );
	debugf( NAME_Init, TEXT("AL_VERSION   : %s"), appFromAnsi((ANSICHAR*)alGetString(AL_VERSION)) );
	SafePrint( NAME_Init, TEXT("AL_EXTENSIONS: "), (ANSICHAR*)alGetString(AL_EXTENSIONS) );

	return 1;
	unguard;
}

UBOOL FClusterDeviceContext::Shutdown()
{
	guard(FClusterDeviceContext::Shutdown);

	// No valid context, so don't bother.
	if ( !Initialized )
		return 1;

	// Destroy context.
	if ( Context )
	{
		// Destroy context.
		alcDestroyContext( Context );
		Context = NULL;
	}

	// Close device.
	if ( Device )
	{
		if ( alcCloseDevice(Device)==ALC_FALSE  )
			debugf( NAME_DevAudio, TEXT("Failed to close device (%s)."), *ALCCurrentErrorString(Device) );
		Device = NULL;
	}

	// Mark as uninitialized.
	Initialized = 0;

	// Add some log output.
	debugf( NAME_DevAudio, TEXT("Device context shut down.") );

	return 1;
	unguard;
}

void FClusterDeviceContext::SetContextAttribute( ALCint Attribute, ALCint Value )
{
	guard(FClusterDeviceContext::SetContextAttribute);
	ContextAttributes.AddUniqueItem( FContextAttribute(Attribute,Value) );
	unguard;
}

void FClusterDeviceContext::ClearContextAttributes()
{
	guard(FClusterDeviceContext::ClearContextAttributes);
	ContextAttributes.Empty();
	unguard;
}

void FClusterDeviceContext::SetDeviceName( const ALCchar *InDeviceName )
{
	guard(FClusterDeviceContext::SetDeviceName);

	// If no device name is supplied reset to default device.
	if ( !InDeviceName || !InDeviceName[0] )
	{
		ClearDeviceName();
		return;
	}

	size_t Len = strlen( InDeviceName );
	DeviceName.Empty();
	DeviceName.InsertZeroed( 0, Len+1 );
	appMemcpy( DeviceName.GetData(), InDeviceName, Len );

	unguard;
}

void FClusterDeviceContext::ClearDeviceName()
{
	guard(FClusterDeviceContext::ClearDeviceName);
	DeviceName.Empty();
	unguard;
}

/*-----------------------------------------------------------------------------
	The End.
-----------------------------------------------------------------------------*/
