/*=============================================================================
	ClusterAudioSubsystem.cpp: Cluster OpenAL Audio interface object.
	Copyright 1999 Epic Games, Inc. All Rights Reserved.
	Copyright 2015-2018 Sebastian Kaufel. All Rights Reserved.

	Revision history:
   * Created by Joseph I. Valenzuela (based on AudioSubsystem.cpp)
   * Taken over by Sebastian Kaufel
=============================================================================*/

/*-----------------------------------------------------------------------------
	Audio includes.
-----------------------------------------------------------------------------*/

#include "Cluster.h"
//#include "MikModInterface.h"

//#include <AL/al.h>
//#include <AL/alc.h>

//#include <mikmod.h>
//#include <stdio.h>

#define PROCADDRESS(x) (alGetProcAddress((const ALchar *) x ## "_LOKI"))

/*#ifndef AL_GAIN_LINEAR
#error Update your openal
#endif*/

#define DEFMUSICBUFFERSIZE 16384

IMPLEMENT_CLASS(UClusterAudioSubsystem);

//
// Information about a playing sound.
//
struct FClusterPlayingSound
{
	// Variables.
	ALuint  Source;
	AActor* Actor;
	INT     Id;
	UBOOL		Is3D;
	USound* Sound;
	FVector Location;
	FLOAT		Volume;
	FLOAT		Radius;
	FLOAT		Pitch;
	FLOAT		Priority;

	// Constructors.
	FClusterPlayingSound()
	:	Source  (0u)
	,	Actor	  (NULL)
	,	Id		  (0)
	,	Is3D	  (0)
	,	Sound	  (0)
	, Location(FVector(0.f,0.f,0.f))
	,	Volume	(0)
	,	Radius	(0)
	,	Pitch	  (0)
	,	Priority(0)
	{}
	FClusterPlayingSound( AActor* InActor, INT InId, USound* InSound, FVector InLocation, FLOAT InVolume, FLOAT InRadius, FLOAT InPitch, FLOAT InPriority )
	:	Source  (0u)
	,	Actor	  (InActor)
	,	Id		  (InId)
	,	Is3D    (0)
	,	Sound	  (InSound)
	,	Location(InLocation)
	,	Volume	(InVolume)
	,	Radius	(InRadius)
	,	Pitch	  (InPitch)
	,	Priority(InPriority)
	{}
};

FClusterPlayingSound PlayingSounds[MAX_EFFECTS_CHANNELS];

void StopSound( INT Index ) ;
bool sourceIsPlaying(ALuint sid);
static void SetVolumes(UClusterAudioSubsystem *uas);
//static int SmallestPowerOfTwoThatExceedesN(INT N);

/*-----------------------------------------------------------------------------
	UClusterAudioSubsystem.
-----------------------------------------------------------------------------*/

UClusterAudioSubsystem::UClusterAudioSubsystem()
{
	guard(UClusterAudioSubsystem::UClusterAudioSubsystem);

	MusicFade      = 1.0;
	CurrentCDTrack = 255;
	LastTime       = appSeconds();

	IdMapping      = new FClusterSoundMap; // !! FIX-ME: Might not always get deleted.
	unguard;
}

void UClusterAudioSubsystem::StaticConstructor()
{
	guard(UClusterAudioSubsystem::StaticConstructor);

	// Enumerations.
	UEnum* OutputRates = new( GetClass(), TEXT("OutputRates") )UEnum( NULL );
	new( OutputRates->Names )FName( TEXT("Default") );
	new( OutputRates->Names )FName( TEXT("44100Hz") );
	new( OutputRates->Names )FName( TEXT("48000Hz") );
	new( OutputRates->Names )FName( TEXT("96000Hz") );
	new( OutputRates->Names )FName( TEXT("192000Hz") );

	// Properties. !! Need to be added in reverse order to not break bool properties. !!
	new(GetClass(),TEXT("DeviceName"     ), RF_Public) UStrProperty  (CPP_PROPERTY(DeviceName     ), TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("SpeechVolume"   ), RF_Public) UByteProperty (CPP_PROPERTY(SpeechVolume   ), TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("SoundVolume"    ), RF_Public) UByteProperty (CPP_PROPERTY(SoundVolume    ), TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("MusicVolume"    ), RF_Public) UByteProperty (CPP_PROPERTY(MusicVolume    ), TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("OutputRate"     ), RF_Public) UByteProperty (CPP_PROPERTY(OutputRate     ), TEXT("Audio"), CPF_Config, OutputRates );
	new(GetClass(),TEXT("EffectsChannels"), RF_Public) UIntProperty  (CPP_PROPERTY(EffectsChannels), TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("Latency"        ), RF_Public) UIntProperty  (CPP_PROPERTY(Latency        ), TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("DopplerSpeed"   ), RF_Public) UFloatProperty(CPP_PROPERTY(DopplerSpeed   ), TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("AmbientFactor"  ), RF_Public) UFloatProperty(CPP_PROPERTY(AmbientFactor  ), TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("ReverseStereo"  ), RF_Public) UBoolProperty (CPP_CLUSTER_PROPERTY_BITFIELD, TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("UseDigitalMusic"), RF_Public) UBoolProperty (CPP_CLUSTER_PROPERTY_BITFIELD, TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("UseCDMusic"     ), RF_Public) UBoolProperty (CPP_CLUSTER_PROPERTY_BITFIELD, TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("UseStereo"      ), RF_Public) UBoolProperty (CPP_CLUSTER_PROPERTY_BITFIELD, TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("UseSurround"    ), RF_Public) UBoolProperty (CPP_CLUSTER_PROPERTY_BITFIELD, TEXT("Audio"), CPF_Config );
	new(GetClass(),TEXT("UseFilter"      ), RF_Public) UBoolProperty (CPP_CLUSTER_PROPERTY_BITFIELD, TEXT("Audio"), CPF_Config );

	// Defaults.
	UseFilter       = 1;
	UseSurround     = 0;
	UseStereo       = 1;
	UseCDMusic      = 0;
	UseDigitalMusic = 1;
	ReverseStereo   = 0;
	AmbientFactor   = 0.7;
	DopplerSpeed    = 6500.0;
	Latency         = 40;
	EffectsChannels = 32;
	OutputRate      = 0;
	MusicVolume     = 153;
	SoundVolume     = 204;
	SpeechVolume    = 255;

	//MusicBufferSize = DEFMUSICBUFFERSIZE;
	unguard;
}

/*-----------------------------------------------------------------------------
	UObject Interface.
-----------------------------------------------------------------------------*/

void UClusterAudioSubsystem::PostEditChange()
{
	guard(UClusterAudioSubsystem::PostEditChange);

	// Validate configurable variables.
	Latency         = Clamp(Latency,10,250);
	EffectsChannels = Clamp(EffectsChannels,0,MAX_EFFECTS_CHANNELS);
	DopplerSpeed    = Clamp(DopplerSpeed,1.f,100000.f);
	AmbientFactor   = Clamp(AmbientFactor,0.f,10.f);


	// Copy config to current volumes and raise update marker.
	CurrentMusicVolume  = MusicVolume;
	CurrentSoundVolume  = SoundVolume;
	CurrentSpeechVolume = SpeechVolume;

	UpdateSoundVolume   = 1;
	UpdateSpeechVolume  = 1;
	UpdateMusicVolume   = 1;



	SetVolumes(this);

	unguard;
}

void UClusterAudioSubsystem::Destroy()
{
	guard(UClusterAudioSubsystem::Destroy);

	if( Initialized )
	{
		// Unhook.
		USound::Audio = NULL;
		UMusic::Audio = NULL;

		// Shut down viewport.
		SetViewport( NULL );

		// Stop CD.
		if( UseCDMusic && CurrentCDTrack != 255 )
			StopCDAudio();

		if ( InitializedMikMod )
			MikMod_Exit();

		// Shutdown device context.
		DeviceContext.Shutdown();

		delete IdMapping;

		debugf( NAME_Exit, TEXT("Cluster shut down.") );
	}

	Super::Destroy();
	unguard;
}

void UClusterAudioSubsystem::ShutdownAfterError()
{
	guard(UClusterAudioSubsystem::ShutdownAfterError);

	// Unhook.
	USound::Audio = NULL;
	UMusic::Audio = NULL;

	// Safely shut down.
	debugf( NAME_Exit, TEXT("UClusterAudioSubsystem::ShutdownAfterError") );
	//safecall(AudioStopOutput());
	if( Viewport )
	{
		//safecall(AudioShutdown());
	}

	// Shutdown device context.
	DeviceContext.Shutdown();

	Super::ShutdownAfterError();
	unguard;
}

/*-----------------------------------------------------------------------------
	UAudioSubsystem Interface.
-----------------------------------------------------------------------------*/

/*
	void (*attScale)(ALfloat );
	attScale    = (void (*)(ALfloat)) PROCADDRESS("alAttenuationScale");
	// set attenuation
	// FIXME: is this right?
	attScale(32000.0 / 60.0);
*/

UBOOL UClusterAudioSubsystem::Init()
{
	guard(UClusterAudioSubsystem::Init);

	// Initialize OpenAL.
	if ( !DeviceContext.IsInitialized() )
	{
		// Reset attributes.
		DeviceContext.ClearContextAttributes();

		// Set sample rate.
		switch ( OutputRate )
		{
			// No sample rate requested, e.g. implementation default.
			case 0:
				break;

			// Valid config options.
			case 1: DeviceContext.SetContextAttribute( ALC_FREQUENCY,  44100 ); break;
			case 2: DeviceContext.SetContextAttribute( ALC_FREQUENCY,  48000 ); break;
			case 3: DeviceContext.SetContextAttribute( ALC_FREQUENCY,  96000 ); break;
			case 4: DeviceContext.SetContextAttribute( ALC_FREQUENCY, 192000 ); break;

			// Yell if option was invalid (should actually not happen).
			default:
				debugf( NAME_Warning, TEXT("Invalid OutputRate option selected (%i)."), OutputRate );
				break;
		}

		// Try to init configured device first.
		if ( DeviceName.Len() )
		{
			// Set device name based on config option.
			DeviceContext.SetDeviceName( (ALCchar*)appToAnsi(*DeviceName) );

			// Try to init device context.
			if ( !DeviceContext.Init() )
				debugf( NAME_Warning, TEXT("Failed to create device context for configured OpenAL device. Retrying default device.") );
		}

		// Try to init default device in case no device was configured or init went wrong.
		if ( !DeviceContext.IsInitialized() )
		{
			// Set default device.
			DeviceContext.ClearDeviceName();

			// Try to init device context.
			if ( !DeviceContext.Init() )
			{
				debugf( NAME_Warning, TEXT("Failed to create device context for default OpenAL device. Aborting.") );
				return 0;
			}
		}
	}
	else
	{
		debugf( NAME_Warning, TEXT("OpenAL device context is already initialized.") );
	}

	// Copy configuration to current volumes.
	CurrentMusicVolume  = MusicVolume;
	CurrentSoundVolume  = SoundVolume;
	CurrentSpeechVolume = SpeechVolume;

	// Set global OpenAL parameters.
	guard(InitGlobal);
	alDistanceModel( AL_LINEAR_DISTANCE );
	alSpeedOfSound( DopplerSpeed ); // Default ZoneInfo.SpeedOfSound value.

	// !! Move towards this.
	//alSpeedOfSound( 8000.f ); // Default ZoneInfo.SpeedOfSound value.
	//AL_API void AL_APIENTRY alDopplerFactor( ALfloat value );
	unguard;

	// Set Listeners initial values.
	guard(InitListener);
	ALfloat DefaultGain           =   1.f;
	ALfloat DefaultPosition[3]    = { 0.f, 0.f, 0.f };
	ALfloat DefaultVelocity[3]    = { 0.f, 0.f, 0.f };
	ALfloat DefaultOrientation[6] = { 0.f,-1.f, 0.f, 0.f, 0.f, -1.f };

	alListenerf ( AL_GAIN,        DefaultGain        );
	alListenerfv( AL_POSITION,    DefaultPosition    );
	alListenerfv( AL_VELOCITY,    DefaultVelocity    );
	alListenerfv( AL_ORIENTATION, DefaultOrientation );
	unguard;

	// set MusicUSound to NULL
	MusicUSound = NULL;

	guard(InitDigitalMusic);
	UseDigitalMusic = 0;
	if ( UseDigitalMusic )
	{
		char MikModCmdLine[128];

		MikMod_RegisterAllDrivers();
		MikMod_RegisterAllLoaders();

		md_mode |= DMODE_SOFT_MUSIC;
		md_device = MikMod_DriverFromAlias("openal"); // FIX-ME: This will cause mikmod to open it's own context!

		if( md_device==0 )
		{
			UseDigitalMusic = 0;

#ifdef DEBUG_SOUND
			debugf( NAME_DevAudio, TEXT("Could not use Digital Music") );
#endif
		}

		//MusicBufferSize = Clamp(MusicBufferSize, 2048, 65536);
		//sprintf( MikModCmdLine, "openal-bufsize=%d\n", MusicBufferSize);
		sprintf( MikModCmdLine, "" );

		
		if ( MikMod_Init(MikModCmdLine) )
		{
			InitializedMikMod = UseDigitalMusic = 0; // !! Should not toggle a config variable.

#ifdef DEBUG_SOUND
			debugf( NAME_DevAudio, TEXT("Could not initialize sound, reason: %s"), appFromAnsi(MikMod_strerror(MikMod_errno)) );
#endif
			//return 1;
		}
		else
		{
			InitializedMikMod = 1;
		}
	}
	unguard;

	// Initialized!
	USound::Audio = this;
	UMusic::Audio = this;
	Initialized = 1;

	debugf( NAME_Init, TEXT("Cluster initialized.") );

	SetVolumes(this);

	return true;
	unguard;
}

void UClusterAudioSubsystem::SetViewport( UViewport* InViewport )
{
	guard(UClusterAudioSubsystem::SetViewport);

	// Stop playing sounds.
	for( INT i=0; i<EffectsChannels; i++ )
		StopSound( i );

	// Remember the viewport.
	if( Viewport != InViewport )
	{
		if( Viewport )
		{
			//unreal_stop_song(0);

			// Unregister everything.
			for( TObjectIterator<UMusic> MusicIt; MusicIt; ++MusicIt )
				if( MusicIt->Handle )
					UnregisterMusic( *MusicIt );

			// Shut down.
			//safecall(AudioStopOutput());
		}
		Viewport = InViewport;
		if( Viewport )
		{
			// Determine startup parameters.
			if( Viewport->Actor->Song && Viewport->Actor->Transition==MTRAN_None )
				Viewport->Actor->Transition = MTRAN_Instant;
		}
	}
	unguard;
}

UViewport* UClusterAudioSubsystem::GetViewport()
{
	guard(UClusterAudioSubsystem::GetViewport);
	return Viewport;
	unguard;
}

void UClusterAudioSubsystem::RegisterSound( USound* Sound )
{
	guard(UClusterAudioSubsystem::RegisterSound);
	checkSlow(Sound);

	if( !Sound->Handle )
	{
		// Set the handle to avoid reentrance.
		//Sound->Handle = (void*)-1; // Not needed anymore, as I don't call FSoundData::Load() anymore.

		// Load the data.
		//Sound->Data.Load();
		Sound->Data.TLazyArray<BYTE>::Load();
		Sound->OriginalSize = Sound->Data.Num();

		debugf( NAME_DevSound, TEXT("Register sound: %s (Size=%i)"), Sound->GetPathName(), Sound->Data.Num() );
		check(Sound->Data.Num()>0);

		// tsao: add sound.  the push should generate bid
		int id = IdMapping->Add(Sound);

		if ( id )
		{
			Sound->Handle = (void*)id;
			debugSoundf( NAME_DevSound, TEXT("Registered sound: %s (Handle=%d)"), Sound->GetPathName(), (INT)Sound->Handle );
		}
		else
		{
			//appErrorf( TEXT("Invalid sound format in %s"), Sound->GetFullName() );
			debugf( NAME_Warning, TEXT("Invalid sound format in %s"), Sound->GetFullName() );

			//Sound->Handle = (void*)0;
			debugSoundf( NAME_DevSound, TEXT("Failed to register sound: %s"), Sound->GetPathName() );
		}

		// Unload the data.
		Sound->Data.Unload();
	}
	else
	{
		debugSoundf( NAME_DevAudio, TEXT("skip REGISTERed (name %s handle %d):\n"), Sound->GetPathName(), (int) Sound->Handle);
	}
	unguard;
}

void UClusterAudioSubsystem::UnregisterSound( USound* Sound )
{
	guard(UClusterAudioSubsystem::UnregisterSound);
	check(Sound);
	if( Sound->Handle )
	{
		debugf( NAME_DevSound, TEXT("Unregister sound: %s"), Sound->GetFullName() );

		// Stop this sound.
		for( INT i=0; i<EffectsChannels; i++ )
			if ( PlayingSounds[i].Sound == Sound )
				StopSound( i );

		// Unload this sound.
		//safecall( UnloadSample( (Sample*) Sound->Handle ) );

		IdMapping->Remove(Sound);
	}
	unguard;
}

void UClusterAudioSubsystem::UnregisterMusic( UMusic* Music )
{
	guard(UClusterAudioSubsystem::UnregisterMusic);

	stopMusic(Music);
	MusicUSound = NULL;

	unguard;
}

UBOOL UClusterAudioSubsystem::Exec( const TCHAR* Cmd, FOutputDevice& Ar )
{
	guard(UClusterAudioSubsystem::Exec);

	const TCHAR* Str = Cmd;
	if ( ParseCommand(&Str,TEXT("ASTAT")) )
	{
		if ( ParseCommand(&Str,TEXT("Audio")) )
		{
			AudioStats ^= 1;
			return 1;
		}
		if ( ParseCommand(&Str,TEXT("Detail")) )
		{
			DetailStats ^= 1;
			return 1;
		}
		return 0;
	}
	if ( ParseCommand(&Str,TEXT("CLUSTER")) )
	{
		if ( ParseCommand(&Cmd, TEXT("BUILD")) )
		{
			Ar.Logf( TEXT("Cluster built: %s"), appFromAnsi(__DATE__" "__TIME__) );
			return 1;
		}
		return 0;
	}
	if ( ParseCommand(&Str, TEXT("MAXEFFECTSCHANNELS")) )
	{
		Ar.Logf( TEXT("%i"), MAX_EFFECTS_CHANNELS );
		return 1;
	}
	return 0;
	
	unguard;
}

UBOOL UClusterAudioSubsystem::PlaySound( AActor*	Actor, INT	Id, USound*	Sound, FVector Location, FLOAT Volume, FLOAT Radius, FLOAT Pitch )
{
	guard(UClusterAudioSubsystem::PlaySound);
	check(Radius);
	if( !Viewport || !Sound )
		return 0;

	debugSoundf( NAME_DevSound, TEXT("Play %s"), Sound==(USound*)INDEX_NONE ? TEXT("INDEX_NONE") : Sound->GetPathName() );

	// Allocate a new slot if requested.
	if( (Id&14)==2*SLOT_None )
		Id = 16 * --FreeSlot;

	// Compute priority.
	FLOAT Priority = SoundPriority( Viewport, Location, Volume, Radius );

	// If already playing, stop it.
	INT   Index        = -1;
	FLOAT BestPriority = Priority;
	for( INT i=0; i<EffectsChannels; i++ )
	{
		FClusterPlayingSound& Playing = PlayingSounds[i];

		if( (Playing.Id&~1)==(Id&~1) )
		{
			// Skip if not interruptable.
			if( Id&1 )
				return 0;

			// Stop the sound.
			Index = i;
			break;
		}
		else if ( Playing.Priority<=BestPriority )
		{
			Index = i;
			BestPriority = Playing.Priority;
		}
	}

	// If no sound, or its priority is overruled, stop it.
	if( Index==-1 )
	{
		return 0;
	}

	// Put the sound on the play-list.
	StopSound( Index );

	// Editor does this for AUDIO PLAY command.
	if ( Sound==(USound*)INDEX_NONE )
	{
		// Return value doesn't matter here, but stick with return 1 as everything worked fine.
		return 1;
	}

	//bid = IdMapping->Find(Sound);
	//if(bid == 0)
	//{
		//bid = IdMapping->Add(Sound);
		//if ( !bid )
			//return 0;
		//Sound->Handle = (void *) bid;
	//}

	if ( !Sound->Handle )
		RegisterSound( Sound );
	if ( !Sound->Handle )
		return 0;

	ALuint bid = (ALuint)Sound->Handle;
	ALuint sid;
	alGenSources(1, &sid);

	alSourcei(sid, AL_BUFFER, bid);

	/* set volume */
	ALfloat fvol = Volume / 2.0f;

	fvol *= ((float) SoundVolume / AUDIO_MAXVOLUME);

	//if ( Actor!=NULL )
		//Location = Actor->Location;

	// Init Source.
	guard(InitSource);
	FVector Velocity = (Actor && ((Id&14)==SLOT_Ambient*2)) ? Actor->Velocity : FVector(0.f,0.f,0.f); // Just do velocity for Ambient.

	//alSourcef( sid, AL_PITCH, Pitch );
	//alSourcef( sid, AL_GAIN_LINEAR, fvol );
	alSourcef( sid, AL_GAIN, fvol ); // FIXME!
	alSourcef( sid, AL_MAX_DISTANCE, Radius );
	alSourcef( sid, AL_ROLLOFF_FACTOR, 1.f );
	alSourcef( sid, AL_REFERENCE_DISTANCE, 1.f );
	alSource3f( sid, AL_POSITION, Location.X, Location.Y, -Location.Z );
	alSource3f( sid, AL_VELOCITY, Velocity.X, Velocity.Y, -Velocity.Z );
	alSourcei( sid, AL_SOURCE_RELATIVE, AL_FALSE );

	//AL_SOURCE_TYPE?

	// FIX-ME: Add proper check for AL_EXT_SOURCE_RADIUS extension.
	//alSourcef( sid, AL_SOURCE_RADIUS, Radius/4.f );
	alSourcef( sid, AL_SOURCE_RADIUS, Radius/10.f );
	//alSourcef( sid, AL_SOURCE_RADIUS, 0.f );
	unguard;

	alSourcePlay(sid);

	PlayingSounds[Index] = FClusterPlayingSound( Actor, Id, Sound, Location, fvol, Radius, Pitch, Priority );
	PlayingSounds[Index].Source = sid;

	return 1;
	unguard;
}

void UClusterAudioSubsystem::NoteDestroy( AActor* Actor )
{
	guard(UClusterAudioSubsystem::NoteDestroy);
	check(Actor);
	check(Actor->IsValid());

	// Stop referencing actor.
	for( INT i=0; i<EffectsChannels; i++ )
	{
		if( PlayingSounds[i].Actor==Actor )
		{
			if( (PlayingSounds[i].Id&14)==SLOT_Ambient*2 )
			{
				// Stop ambient sound when actor dies.
				StopSound( i );
			}
			else
			{
				// Unbind regular sounds from actors.
				PlayingSounds[i].Actor = NULL; 
			}
		}
	}
	unguard;
}

void UClusterAudioSubsystem::RenderAudioGeometry( FSceneNode* Frame )
{
	guard(UClusterAudioSubsystem::RenderAudioGeometry);
	unguard;
}

void UClusterAudioSubsystem::Update( FPointRegion Region, FCoords& Coords )
{
	guard(UClusterAudioSubsystem::Update);
	if ( !Viewport ) 
		return;
	if ( !Viewport->Actor ) // Shutdown everything? --han
		return;
	
	// Lock to sync sound.
	//ALock;

	// Time passes...
	DOUBLE DeltaTime = appSeconds() - LastTime; // !! FIX-ME: Use appCycles() and calculate differences as integers!
	LastTime += DeltaTime;
	DeltaTime = Clamp( DeltaTime, 0.0, 1.0 );




	// Figure out ViewActor.
	AActor* ViewActor = Viewport->Actor->ViewTarget ? Viewport->Actor->ViewTarget : Viewport->Actor;

	//
	// Update Listener.
	//
	// OpenAL uses a right handed coordinate system, while Unreal uses a left handed one.
	// So we need to account for that by inverting the sign on the z component of vectors.
	//
	// In addition the YAxis is facing down [Verify].
	//
	// We can't reliably figure out what the listeners velocity is as PlayerCalcView can do anything.
	//
	// Even worse, we can't call PlayerCalcView here a second time, for like Deus Ex 
	// as it executes code it shouldn't there, so we can't even figure out what the ViewActor
	// really is supposed to be.
	//
	// Best idea would be to check whether location passed in by Coords matches the ViewActors
	// position and use it's velocity and otherwise use zero velocity.
	//
	// Use ViewActor's Velocity for now as a best first guess.
	//
	guard(UpdateListener);
	ALfloat ListenerOrientation[6];
	ListenerOrientation[0] =  Coords.ZAxis.X; // At.
	ListenerOrientation[1] =  Coords.ZAxis.Y;
	ListenerOrientation[2] =  Coords.ZAxis.Z;
	ListenerOrientation[3] = -Coords.YAxis.X; // Up.
	ListenerOrientation[4] = -Coords.YAxis.X;
	ListenerOrientation[5] =  Coords.YAxis.X;

	alListenerfv( AL_ORIENTATION, ListenerOrientation );
	alListener3f( AL_POSITION, Coords.Origin.X, Coords.Origin.Y, -Coords.Origin.Z );
	//alListener3f( AL_VELOCITY, ViewActor->Velocity.X, ViewActor->Velocity.Y, -ViewActor->Velocity.Z );
	alListener3f( AL_VELOCITY, 0.f, 0.f, 0.f ); // Galaxy doesn't account for Listeners velocity.
	unguard;



	for(INT i = 0 ; i < EffectsChannels; i++ )
	{
		FClusterPlayingSound& Playing = PlayingSounds[i];
		ALuint sid              = Playing.Source;

		if ( alIsSource(Playing.Source)==AL_TRUE )
		{
			if ( sourceIsPlaying(Playing.Source)==false )
			{
				alDeleteSources( 1, &Playing.Source );

				Playing = FClusterPlayingSound();

				/*Playing.Sound  = NULL;
				Playing.Actor  = NULL;
				Playing.Source = 0;
				Playing.Id     = 0;
				Playing.Priority = 0.0F;*/
			}
			else
			{
				/* Update position, gain etc */
				if ( Playing.Actor==NULL )
					continue;

				// Update position from actor.
				FVector Location = Playing.Actor->Location;

				// Update velocity from actor if ambient.
				FVector Velocity = ((Playing.Id&14)==SLOT_Ambient*2) ? Playing.Actor->Velocity : FVector(0.f,0.f,0.f);

				alSource3f( Playing.Source, AL_POSITION, Location.X, Location.Y, -Location.Z );
				alSource3f( Playing.Source, AL_VELOCITY, Velocity.X, Velocity.Y, -Velocity.Z );
			}
		}
	}

	// See if any new ambient sounds need to be started.
	UBOOL Realtime = Viewport->IsRealtime() && Viewport->Actor->Level->Pauser==TEXT("");
	if ( Realtime )
	{
		guard(StartAmbience);
		for( INT i=0; i<Viewport->Actor->GetLevel()->Actors.Num(); i++ )
		{
			AActor* Actor = Viewport->Actor->GetLevel()->Actors(i);
			if
			(	Actor
			&&	Actor->AmbientSound
			&&	FDistSquared(ViewActor->Location,Actor->Location)<=Square(Actor->WorldSoundRadius()) )
			{
				INT Id = Actor->GetIndex()*16+SLOT_Ambient*2;
				for( INT j=0; j<EffectsChannels; j++ )
					if( PlayingSounds[j].Id==Id )
						break;
				if( j==EffectsChannels )
					PlaySound( Actor, Id, Actor->AmbientSound, Actor->Location, AmbientFactor*Actor->SoundVolume/255.0, Actor->WorldSoundRadius(), Actor->SoundPitch/64.0 );
			}
		}
		unguard;
	}


	// Nicholas Vinen - now is the time to change music.
	// I didn't implement transitions. Does anybody miss them?

	// Handle music transitions.
	guard(UpdateMusic);
	if ( InitializedMikMod )
	{
		// vogel: I guess most of this code is crap...
		if ( Viewport->Actor->Transition != MTRAN_None && UseDigitalMusic )
		{
			if (Viewport->Actor->Song)
			{
				int transitiontime;

				/* Work out what music to play and start playing it */
				Viewport->Actor->Song->Data.Load();
				switch(Viewport->Actor->Transition)
				{
				case MTRAN_Fade:
					transitiontime = 1000;
					break;
				case MTRAN_SlowFade:
					transitiontime = 5000;
					break;
				case MTRAN_FastFade:
					transitiontime = 333;
					break;
				default:
					transitiontime = 0;
					break;
				}

				/* stop music if playing */
				if(MusicUSound != NULL) {
					stopMusic(MusicUSound);
				}

				MusicUSound = Viewport->Actor->Song;

				int err = playMusic(MusicUSound);
				if(err == 0) {
					MusicUSound = NULL;
				}
			} else {
				stopMusic(MusicUSound);

				MusicUSound = NULL;
			}

			Viewport->Actor->Transition = MTRAN_None;
		}

		/* music, get correct volume */
		md_musicvolume = (MusicVolume/2);

		if(Player_Active() == 0)
		{
			/* Music must be over */
			//MikMod_Update();

			if(MusicUSound != NULL) {
				/* loop the music */
				int err = playMusic(MusicUSound);
				if(err == 0) {
					MusicUSound = NULL;
				}
			}
		}
		else
		{
			if(MusicUSound != NULL)
			{
				//MikMod_Update();
			}
		}

		// Always call MikMod_Update() no matter what.
		MikMod_Update();
	}
	unguard; // UpdateMusic.

	// tsao: STUB STUB STUB
	unguard;
}

void UClusterAudioSubsystem::PostRender( FSceneNode* Frame )
{
	guard(UClusterAudioSubsystem::PostRender);
	if( AudioStats )
	{
		INT DY = DetailStats ? 16 : 8;

		// Print Header.
		Frame->Viewport->Canvas->Color = FColor(255,255,255);
		Frame->Viewport->Canvas->CurX  = 0;
		Frame->Viewport->Canvas->CurY  = 16;
		Frame->Viewport->Canvas->WrappedPrintf( Frame->Viewport->Canvas->SmallFont, 0, TEXT("AUDIO:") );

		for ( INT i=0; i<EffectsChannels; i++ )
		{
			FClusterPlayingSound& PlayingSound = PlayingSounds[i];
			checkSlow(!PlayingSound.Source||PlayingSound.Sound);

			Frame->Viewport->Canvas->CurX = 10;
			Frame->Viewport->Canvas->CurY = 24 + i*DY;
			Frame->Viewport->Canvas->WrappedPrintf( Frame->Viewport->Canvas->SmallFont, 0, TEXT("Channel %2i: %s"), i, PlayingSound.Source ? PlayingSound.Sound->GetFullName() : TEXT("None") );

			if ( DetailStats )
			{
				Frame->Viewport->Canvas->CurX = 10;
				Frame->Viewport->Canvas->CurY = 32 + i*DY;

				if ( PlayingSound.Source )
					Frame->Viewport->Canvas->WrappedPrintf( Frame->Viewport->Canvas->SmallFont, 0, TEXT("  Vol: %05.2f Pitch: %05.2f Radius: %07.2f Priority: %05.2f"), PlayingSounds[i].Volume, PlayingSounds[i].Pitch, PlayingSounds[i].Radius, PlayingSounds[i].Priority );
				else
					Frame->Viewport->Canvas->WrappedPrintf( Frame->Viewport->Canvas->SmallFont, 0, TEXT("  ...") );
			}
		}
	}
	unguard;
}

/*-----------------------------------------------------------------------------
	DeusEx Interface.
-----------------------------------------------------------------------------*/
#if ENGINE_VERSION==1100

// Instantly sets the sound volume.
void UClusterAudioSubsystem::SetInstantSoundVolume( BYTE InstantSoundVolume )
{
	guard(UClusterAudioSubsystem::SetInstantSoundVolume);
	CurrentSoundVolume = InstantSoundVolume;
	UpdateSoundVolume  = 1;
	unguard;
}

// Instantly sets the speech volume.
void UClusterAudioSubsystem::SetInstantSpeechVolume( BYTE InstantSpeechVolume )
{
	guard(UClusterAudioSubsystem::SetInstantSpeechVolume);
	CurrentSpeechVolume = InstantSpeechVolume;
	UpdateSpeechVolume  = 1;
	unguard;
}

// Instantly sets the music volume.
void UClusterAudioSubsystem::SetInstantMusicVolume( BYTE InstantMusicVolume )
{
	guard(UClusterAudioSubsystem::SetInstantMusicVolume);
	CurrentMusicVolume = InstantMusicVolume;
	UpdateMusicVolume  = 1;
	unguard;
}

// Instantly stops a playing sound.
void UClusterAudioSubsystem::StopSoundId( INT Id )
{
	guard(UClusterAudioSubsystem::StopSoundId);

	// If already playing, stop it.
	INT Index = -1;

	for( INT i=0; i<EffectsChannels; i++ )
	{
		FClusterPlayingSound& Playing = PlayingSounds[i];

		// Comparison of Id&~1 is done to ignore bNoOverride.
		if( (Playing.Id&~1)==(Id&~1) )
		{
			// Stop the sound.
			Index = i;
			break;
		}
	}

	// Stop it.
	if( Index!=-1 )
		StopSound( Index );

	unguard;
}
#endif

/*-----------------------------------------------------------------------------
	CD Audio control.
-----------------------------------------------------------------------------*/

UBOOL UClusterAudioSubsystem::StartCDAudio( INT Track )
{
	guard(UClusterAudioSubsystem::StartCDAudio);

	// STUB STUB STUB

	return 1;
	unguard;
}

UBOOL UClusterAudioSubsystem::StopCDAudio()
{
	guard(UClusterAudioSubsystem::StopCDAudio);

	// STUB STUB STUB

	return 1;
	unguard;
}

/*-----------------------------------------------------------------------------
	Internals.
-----------------------------------------------------------------------------*/

void StopSound( INT Index )
{
	guard(UClusterAudioSubsystem::StopSound);

	ALuint sid;

	sid = PlayingSounds[Index].Source;

	if(alIsSource(sid) == AL_TRUE) {
		if(sourceIsPlaying(sid) == true) {
			alSourceStop(sid);
		}

		alDeleteSources(1, &sid);
	}


	PlayingSounds[Index].Sound    = NULL;
	PlayingSounds[Index].Actor    = NULL;
	PlayingSounds[Index].Id       = 0;
	PlayingSounds[Index].Source   = 0;
	PlayingSounds[Index].Priority = 0.0F;

	return;

	unguard;
}

bool sourceIsPlaying(ALuint sid) {
	ALint state;

	alGetSourcei(sid, AL_SOURCE_STATE, &state);

	if(state == AL_PLAYING) {
		return true;
	}

	return false;
}

static void SetVolumes(UClusterAudioSubsystem *uas)
{
	// This sucks, as SoundVolume affects all volume and acts as sort of a master switch, which it wasn't for Galaxy.
	//alListenerf(AL_GAIN_LINEAR, static_cast<ALfloat>(uas->SoundVolume) / AUDIO_MAXVOLUME);
	alListenerf(AL_GAIN, static_cast<ALfloat>(uas->SoundVolume) / AUDIO_MAXVOLUME); // FIXME!

	md_musicvolume = (uas->CurrentMusicVolume/2);

	MikMod_Update(); /* allow music changes to take effect */

	return;
}

/*static int SmallestPowerOfTwoThatExceedesN(INT N)
{
	int retval = 1;

	while(retval < N) {
		retval <<= 1;
	}

	return retval;
}*/

/*-----------------------------------------------------------------------------
	The End.
-----------------------------------------------------------------------------*/
