/**
Main shader class, offers housekeeping and helper functions.
*/

#include "shader.h"

Shader::Shader(ID3D10Device *device): geometryBuffer(0), renderTargetView(0), depthStencilView(0), shaderResourceView(0), effect(0), device(device),topology(D3D10_PRIMITIVE_TOPOLOGY_TRIANGLELIST)
{

}

Shader::~Shader()
{
	delete geometryBuffer;	
	SAFE_RELEASE(effect);
	SAFE_RELEASE(vertexLayout);
}


/**
Check shader compile result and show errors if applicable.
*/
bool Shader::checkCompileResult(HRESULT hr) const
{
	if(blob) //Show compile errors if present
			UD3D10RenderDevice::debugs((char*) blob->GetBufferPointer());		
	if(FAILED(hr))
	{
		UD3D10RenderDevice::debugs("Error compiling effects file. Please make sure it resides in the \"\\system\\d3d10drv\" directory.");		
		return false;
	}
	return true;
}

/**
Create render target, depth and resource views.

\param samples Desired multisample amount for these buffers (can be different than what the game is using)

\note If a format is null the buffer/view will be skipped.
*/
bool Shader::createRenderTargetViews(DXGI_FORMAT format, DXGI_FORMAT depthFormat, float scaleX, float scaleY, int samples, const DXGI_SWAP_CHAIN_DESC &swapChainDesc)
{
	
	ID3D10Texture2D *tex=NULL;
	if(format!=DXGI_FORMAT_UNKNOWN)
	{
		
		//Create texture
		D3D10_TEXTURE2D_DESC texDesc;
		texDesc.ArraySize=1;
		texDesc.BindFlags=D3D10_BIND_RENDER_TARGET | D3D10_BIND_SHADER_RESOURCE;
		texDesc.CPUAccessFlags=0;
		texDesc.Format= format;
		texDesc.Height = swapChainDesc.BufferDesc.Height*scaleY;
		texDesc.Width = swapChainDesc.BufferDesc.Width*scaleX;
		texDesc.MipLevels = 1;
		texDesc.MiscFlags = 0;
		texDesc.SampleDesc.Count=samples;
		texDesc.SampleDesc.Quality = 0;
		texDesc.Usage=D3D10_USAGE_DEFAULT;

		if(FAILED(device->CreateTexture2D(&texDesc,0,&tex)))
		{
			UD3D10RenderDevice::debugs("Error creating buffer texture.");
			return 0;
		}

		//Render target view
		hr = device->CreateRenderTargetView(tex,NULL,&renderTargetView);	
		if(FAILED(hr))
		{
			UD3D10RenderDevice::debugs("Error creating render target view.");
			return 0;
		}

		//Shader resource view
		if(FAILED(device->CreateShaderResourceView(tex, NULL,&shaderResourceView)))
		{
			UD3D10RenderDevice::debugs("Error creating shader resource view.");
			return 0;
		}

		SAFE_RELEASE(tex);
		
	}

	if(depthFormat!=DXGI_FORMAT_UNKNOWN)
	{
		//Depth stencil
		D3D10_TEXTURE2D_DESC descDepth;
		
		//Internal texture
		ID3D10Texture2D *depthTexInternal;
		descDepth.Width = swapChainDesc.BufferDesc.Width*scaleX;
		descDepth.Height = swapChainDesc.BufferDesc.Height*scaleY;
		descDepth.MipLevels = 1;
		descDepth.ArraySize = 1;
		descDepth.Format =  depthFormat;
		descDepth.SampleDesc.Count = samples;
		descDepth.SampleDesc.Quality = 0;
		descDepth.Usage = D3D10_USAGE_DEFAULT;
		descDepth.BindFlags = D3D10_BIND_DEPTH_STENCIL;
		descDepth.CPUAccessFlags = 0;
		descDepth.MiscFlags = 0;
		if(FAILED(device->CreateTexture2D( &descDepth, NULL,&depthTexInternal )))
		{
			UD3D10RenderDevice::debugs("Depth texture creation failed.");
			return 0;
		}

		//Depth Stencil view
		if(FAILED(device->CreateDepthStencilView(depthTexInternal,NULL,&depthStencilView )))
		{
			UD3D10RenderDevice::debugs("Error creating render target view (depth).");
			return 0;
		}
		SAFE_RELEASE(depthTexInternal);
	}
	return 1;
}

void Shader::releaseRenderTargetViews()
{	
	SAFE_RELEASE(renderTargetView);
	SAFE_RELEASE(depthStencilView);
	SAFE_RELEASE(shaderResourceView);
}



void Shader::bind()
{
	UINT offset=0;	
	geometryBuffer->bind();
	device->IASetPrimitiveTopology(topology);	
	device->IASetInputLayout(vertexLayout);
	if(renderTargetView!=0) //If not set, reuse existing one
		device->OMSetRenderTargets(1,&renderTargetView,depthStencilView);
}

/**
Draw the shader's buffer contents.
*/
void Shader::apply()
{
	effect->GetTechniqueByIndex(0)->GetPassByIndex(0)->Apply(0);
	geometryBuffer->draw();
}

GeometryBuffer *Shader::getGeometryBuffer() const
{
	return geometryBuffer;
}

ID3D10ShaderResourceView *Shader::getResourceView() const
{
	return shaderResourceView;
}