class DXRActorsBase extends DXRBase;

var globalconfig string skipactor_types[6];
var class<Actor> _skipactor_types[6];

struct LocationNormal {
    var vector loc;
    var vector norm;
};

struct FMinMax {
    var float min;
    var float max;
};

struct safe_rule {
    var string map;
    var name item_name;
    var vector min_pos;
    var vector max_pos;
    var bool allow;
};

function CheckConfig()
{
    local class<Actor> temp_skipactor_types[6];
    local int i, t;
    if( ConfigOlderThan(1,5,9,8) ) {
        for(i=0; i < ArrayCount(skipactor_types); i++) {
            skipactor_types[i] = "";
        }
        i=0;
        skipactor_types[i++] = "BarrelAmbrosia";
        skipactor_types[i++] = "NanoKey";
    }
    Super.CheckConfig();

    //sort skipactor_types so that we only need to check until the first None
    t=0;
    for(i=0; i < ArrayCount(skipactor_types); i++) {
        if( skipactor_types[i] != "" )
            _skipactor_types[t++] = GetClassFromString(skipactor_types[i], class'Actor');
    }
}

function SwapAll(name classname, float percent_chance)
{
    local Actor temp[4096];
    local Actor a, b;
    local int num, i, slot;

    SetSeed( "SwapAll " $ classname );
    num=0;
    foreach AllActors(class'Actor', a )
    {
        if( SkipActor(a, classname) ) continue;
        temp[num++] = a;
    }

    for(i=0; i<num; i++) {
        if( percent_chance<100 && !chance_single(percent_chance) ) continue;
        slot=rng(num-1);// -1 because we skip ourself
        if(slot >= i) slot++;
        Swap(temp[i], temp[slot]);
    }
}

static function vector AbsEach(vector v)
{// not a real thing in math? but it's convenient
    v.X = abs(v.X);
    v.Y = abs(v.Y);
    v.Z = abs(v.Z);
    return v;
}

static function bool AnyGreater(vector a, vector b)
{
    return a.X > b.X || a.Y > b.Y || a.Z > b.Z;
}

function bool CarriedItem(Actor a)
{// I need to check Engine.Inventory.bCarriedItem
    if( PlayerPawn(a.Base) != None )
        return true;
    return a.Owner != None && a.Owner.IsA('Pawn');
}

static function bool IsHuman(Actor a)
{
    return #var prefix HumanMilitary(a) != None || #var prefix HumanThug(a) != None || #var prefix HumanCivilian(a) != None;
}

static function bool IsCritter(Actor a)
{
    if( #var prefix CleanerBot(a) != None ) return true;
    if( #var prefix Animal(a) == None ) return false;
    return #var prefix Doberman(a) == None && #var prefix Gray(a) == None && #var prefix Greasel(a) == None && #var prefix Karkian(a) == None;
}

static function bool RemoveItem(Pawn p, class c)
{
    local ScriptedPawn sp;
    local Inventory Inv, next;
    local int i;
    local bool found;
    sp = ScriptedPawn(p);
    found = false;

    if( sp != None ) {
        for (i=0; i<ArrayCount(sp.InitialInventory); i++)
        {
            if ((sp.InitialInventory[i].Inventory != None) && (sp.InitialInventory[i].Count > 0))
            {
                if( ClassIsChildOf(sp.InitialInventory[i].Inventory, c) ) {
                    sp.InitialInventory[i].Count = 0;
                    found = True;
                }
            }
        }
    }

    for( Inv=p.Inventory; Inv!=None; Inv=next ) {
        next = Inv.Inventory;
        if ( ClassIsChildOf(Inv.class, c) ) {
            Inv.Destroy();
            found = true;
        }
    }

    return found;
}

static function bool HasItem(Pawn p, class c)
{
    local ScriptedPawn sp;
    local int i;
    sp = ScriptedPawn(p);

    if( sp != None ) {
        for (i=0; i<ArrayCount(sp.InitialInventory); i++)
        {
            if ((sp.InitialInventory[i].Inventory != None) && (sp.InitialInventory[i].Count > 0))
            {
                if( sp.InitialInventory[i].Inventory == c ) return True;
            }
        }
    }
    return p.FindInventoryType(c) != None;
}

static function bool HasItemSubclass(Pawn p, class<Inventory> c)
{
    local Inventory Inv;
    local ScriptedPawn sp;
    local int i;
    sp = ScriptedPawn(p);

    if( sp != None ) {
        for (i=0; i<ArrayCount(sp.InitialInventory); i++)
        {
            if ((sp.InitialInventory[i].Inventory != None) && (sp.InitialInventory[i].Count > 0))
            {
                if( ClassIsChildOf(sp.InitialInventory[i].Inventory, c) ) return True;
            }
        }
    }

    for( Inv=p.Inventory; Inv!=None; Inv=Inv.Inventory )
        if ( ClassIsChildOf(Inv.class, c) )
            return true;
    return false;
}

static function bool HasMeleeWeapon(Pawn p)
{
    return HasItem(p, class'#var prefix WeaponBaton')
        || HasItem(p, class'#var prefix WeaponCombatKnife')
        || HasItem(p, class'#var prefix WeaponCrowbar')
        || HasItem(p, class'#var prefix WeaponSword')
        || HasItem(p, class'#var prefix WeaponNanoSword');
}

static function bool IsMeleeWeapon(Inventory item)
{
    return item.IsA('#var prefix WeaponBaton')
        || item.IsA('#var prefix WeaponCombatKnife')
        || item.IsA('#var prefix WeaponCrowbar')
        || item.IsA('#var prefix WeaponSword')
        || item.IsA('#var prefix WeaponNanoSword');
}

static function Ammo GiveAmmoForWeapon(Pawn p, DeusExWeapon w, int add_ammo)
{
    local int i;

    if( w == None || add_ammo <= 0 )
        return None;

    if ( w.AmmoName == None || w.AmmoName == Class'AmmoNone' )
        return None;

    for(i=0; i<add_ammo; i++)
        w.AmmoType = Ammo(GiveItem(p, w.AmmoName));
    return w.AmmoType;
}

static function Inventory GiveExistingItem(Pawn p, Inventory item, optional int add_ammo)
{
    local DeusExWeapon w;
    local Ammo a;
    local DeusExPickup pickup;
    local DeusExPlayer player;
    local bool PlayerTraveling;

    if( item == None ) return None;

    player = #var PlayerPawn (p);
    if( Ammo(item) != None ) {
        a = Ammo(p.FindInventoryType(item.class));
        if( a != None ) {
            a.AmmoAmount += a.default.AmmoAmount + add_ammo;
            if( player != None )
                player.UpdateAmmoBeltText(a);
            item.Destroy();
            return a;
        }
    }

    if( DeusExWeapon(item) != None ) {
        w = DeusExWeapon(p.FindInventoryType(item.class));
        if( w != None ) {
            GiveAmmoForWeapon(p, w, 1 + add_ammo);
            item.Destroy();
            return w;
        }
    }

    if( DeusExPickup(item) != None ) {
        pickup = DeusExPickup(p.FindInventoryType(item.class));
        if( pickup != None ) {
            if( pickup.bCanHaveMultipleCopies && pickup.NumCopies < pickup.MaxCopies ) {
                pickup.NumCopies++;
                item.Destroy();
                return pickup;
            }
        }
    }

    item.InitialState='Idle2';
    item.SetLocation(p.Location);

    if( (#defined gmdx || #defined revision ) && player != None ) {
        PlayerTraveling = player.FlagBase.GetBool('PlayerTraveling');
        if(PlayerTraveling) {
            player.FlagBase.SetBool('PlayerTraveling', false);
        }
    }

    if( player != None ) {
        player.FrobTarget = item;
        player.ParseRightClick();
    } else {
        item.GiveTo(p);
        item.SetBase(p);
    }

    if(PlayerTraveling) {
        player.FlagBase.SetBool('PlayerTraveling', true);
    }

    GiveAmmoForWeapon(p, DeusExWeapon(item), add_ammo);

    if( player != None )
        player.UpdateBeltText(item);

    return item;
}

static function inventory GiveItem(Pawn p, class<Inventory> iclass, optional int add_ammo)
{
    local inventory item;

    item = p.Spawn(iclass, p);
    if( item == None ) return None;
    return GiveExistingItem(p, item, add_ammo);
}

static function ThrowItem(Inventory item, float VelocityMult)
{
    local Actor a;
    local vector loc, rot;

    a = item.Owner;
    if( Pawn(a) != None )
        Pawn(a).DeleteInventory(item);
    if( a != None ) {
        loc = a.Location;
        rot = vector(a.Rotation);
    } else {
        loc = item.Location;
        rot = vector(item.Rotation);
    }
    item.DropFrom(loc + (VRand()*vect(32,32,16)) + vect(0,0,16) );
    // kinda copied from DeusExPlayer DropItem function
    item.Velocity = rot * 300 + vect(0,0,220) + VRand()*32;
    item.Velocity *= VelocityMult;
}

function bool SkipActorBase(Actor a)
{
    if( (a.Owner != None) || a.bStatic || a.bHidden || a.bMovable==False )
        return true;
    return false;
}

function bool SkipActor(Actor a, name classname)
{
    local int i;
    if( SkipActorBase(a) || ( ! a.IsA(classname) ) ) {
        return true;
    }
    for(i=0; i < ArrayCount(_skipactor_types); i++) {
        if(_skipactor_types[i] == None) break;
        if( a.IsA(_skipactor_types[i].name) ) return true;
    }
    return false;
}

function bool SetActorLocation(Actor a, vector newloc, optional bool retainOrders)
{
    local ScriptedPawn p;

    if( ! a.SetLocation(newloc) ) return false;

    p = ScriptedPawn(a);
    if( p != None && p.Orders == 'Patrolling' && !retainOrders ) {
        p.SetOrders('Wandering');
        p.HomeTag = 'Start';
        p.HomeLoc = p.Location;
    }

    return true;
}

function RemoveFears(ScriptedPawn p)
{
    if( p == None ) {
        err("RemoveFears "$p);
        return;
    }
    p.bFearHacking = false;
    p.bFearWeapon = false;
    p.bFearShot = false;
    p.bFearInjury = false;
    p.bFearIndirectInjury = false;
    p.bFearCarcass = false;
    p.bFearDistress = false;
    p.bFearAlarm = false;
    p.bFearProjectiles = false;
}

function RemoveReactions(ScriptedPawn p)
{
    if( p == None ) {
        err("RemoveReactions "$p);
        return;
    }
    RemoveFears(p);
    p.bHateHacking = false;
    p.bHateWeapon = false;
    p.bHateShot = false;
    p.bHateInjury = false;
    p.bHateIndirectInjury = false;
    p.bHateCarcass = false;
    p.bHateDistress = false;
    p.bReactFutz = false;
    p.bReactPresence = false;
    p.bReactLoudNoise = false;
    p.bReactAlarm = false;
    p.bReactShot = false;
    p.bReactCarcass = false;
    p.bReactDistress = false;
    p.bReactProjectiles = false;
}

function bool Swap(Actor a, Actor b, optional bool retainOrders)
{
    local vector newloc, oldloc;
    local rotator newrot;
    local bool asuccess, bsuccess;
    local Actor abase, bbase;
    local bool AbCollideActors, AbBlockActors, AbBlockPlayers;
    local EPhysics aphysics, bphysics;

    if( a == b ) return true;

    l("swapping "$ActorToString(a)$" and "$ActorToString(b)$" distance == " $ VSize(a.Location - b.Location) );

    AbCollideActors = a.bCollideActors;
    AbBlockActors = a.bBlockActors;
    AbBlockPlayers = a.bBlockPlayers;
    a.SetCollision(false, false, false);

    oldloc = a.Location;
    newloc = b.Location;

    bsuccess = SetActorLocation(b, oldloc + (b.CollisionHeight - a.CollisionHeight) * vect(0,0,1), retainOrders );
    a.SetCollision(AbCollideActors, AbBlockActors, AbBlockPlayers);
    if( bsuccess == false ) {
        warning("bsuccess failed to move " $ ActorToString(b) $ " into location of " $ ActorToString(a) );
        return false;
    }

    asuccess = SetActorLocation(a, newloc + (a.CollisionHeight - b.CollisionHeight) * vect(0,0,1), retainOrders);
    if( asuccess == false ) {
        warning("asuccess failed to move " $ ActorToString(a) $ " into location of " $ ActorToString(b) );
        SetActorLocation(b, newloc, retainOrders);
        return false;
    }

    newrot = b.Rotation;
    b.SetRotation(a.Rotation);
    a.SetRotation(newrot);

    aphysics = a.Physics;
    bphysics = b.Physics;
    abase = a.Base;
    bbase = b.Base;

    a.SetPhysics(bphysics);
    if(abase != bbase) a.SetBase(bbase);
    b.SetPhysics(aphysics);
    if(abase != bbase) b.SetBase(abase);

    return true;
}

function SwapNames(out Name a, out Name b) {
    local Name t;
    t = a;
    a = b;
    b = t;
}

function SwapVector(out vector a, out vector b) {
    local vector t;
    t = a;
    a = b;
    b = t;
}

function SwapProperty(Actor a, Actor b, string propname) {
    local string t;
    t = a.GetPropertyText(propname);
    a.SetPropertyText(propname, b.GetPropertyText(propname));
    b.SetPropertyText(propname, t);
}

function ResetOrders(ScriptedPawn p) {
    p.OrderActor = None;
    p.NextState = '';
    p.NextLabel = '';

    if(p.Orders == 'Idle') {
        p.Orders = 'Standing';
        p.OrderTag = '';
    }
    p.SetOrders(p.Orders, p.OrderTag, false);
    p.FollowOrders();
}

function bool HasConversation(Actor a) {
    local ConListItem i;

    for(i=ConListItem(a.ConListItems); i!=None; i=i.next) {
        //warning(a$" HasConversation "$i.con.conName@i.con.bFirstPerson@i.con.bNonInteractive@i.con.bDataLinkCon@i.con.conOwnerName@i.con.bCanBeInterrupted@i.con.bCannotBeInterrupted);
        return true;
    }
    return false;
}

function bool HasBased(Actor a) {
    local Actor b;
    foreach a.BasedActors(class'Actor', b)
        return true;
    return false;
}

function bool DestroyActor( Actor d )
{
    // If this item is in an inventory chain, unlink it.
    local Decoration downer;

    if( d.IsA('Inventory') && d.Owner != None && d.Owner.IsA('Pawn') )
    {
        Pawn(d.Owner).DeleteInventory( Inventory(d) );
    }
    return d.Destroy();
}

function Actor ReplaceActor(Actor oldactor, string newclassstring)
{
    local Actor a;
    local class<Actor> newclass;
    local vector loc;
    local float scalefactor;
    local float largestDim;

    loc = oldactor.Location;
    newclass = class<Actor>(DynamicLoadObject(newclassstring, class'class'));
    if( newclass.default.bStatic ) warning(newclassstring $ " defaults to bStatic, Spawn probably won't work");
    a = Spawn(newclass,,,loc);

    if( a == None ) {
        warning("ReplaceActor("$oldactor$", "$newclassstring$"), failed to spawn in location "$oldactor.Location);
        return None;
    }

    //Get the scaling to match
    if (a.CollisionRadius > a.CollisionHeight) {
        largestDim = a.CollisionRadius;
    } else {
        largestDim = a.CollisionHeight;
    }
    scalefactor = oldactor.CollisionHeight/largestDim;

    //DrawScale doesn't work right for Inventory objects
    a.DrawScale = scalefactor;
    if (a.IsA('Inventory')) {
        Inventory(a).PickupViewScale = scalefactor;
    }

    //Floating decorations don't rotate
    if (a.IsA('DeusExDecoration')) {
        DeusExDecoration(a).bFloating = False;
    }

    //Get it at the right height
    a.move(a.PrePivot);
    oldactor.bHidden = true;
    oldactor.Destroy();

    return a;
}

function Conversation GetConversation(Name conName)
{
    local Conversation c;
    foreach AllObjects(class'Conversation', c) {
        if( c.conName == conName ) return c;
    }
    return None;
}

static function DeusExDecoration _AddSwitch(Actor a, vector loc, rotator rotate, name Event)
{
    local DeusExDecoration d;
    d = DeusExDecoration( _AddActor(a, class'Switch2', loc, rotate) );
    d.Event = Event;
    return d;
}

function DeusExDecoration AddSwitch(vector loc, rotator rotate, name Event)
{
    return _AddSwitch(Self, loc, rotate, Event);
}

static function Actor _AddActor(Actor a, class<Actor> c, vector loc, rotator rotate, optional Actor owner, optional Name tag)
{
    local Actor d;
    local bool oldCollideWorld;

    oldCollideWorld = c.default.bCollideWorld;
    c.default.bCollideWorld = false;
    d = a.Spawn(c, owner, tag, loc, rotate );
    if(d == None) {
        return None;
    }

    d.bCollideWorld = false;
    d.SetLocation(loc);
    d.SetRotation(rotate);
    d.bCollideWorld = oldCollideWorld;
    c.default.bCollideWorld = oldCollideWorld;
    return d;
}

function Containers AddBox(class<Containers> c, vector loc, optional rotator rotate)
{
    local Containers box;
    box = Containers(_AddActor(Self, c, loc, rotate));
    box.bInvincible = true;
    return box;
}

function Actor SpawnReplacement(Actor a, class<Actor> newclass)
{
    local Actor newactor;
    local bool bCollideActors, bBlockActors, bBlockPlayers;

    bCollideActors = a.bCollideActors;
    bBlockActors = a.bBlockActors;
    bBlockPlayers = a.bBlockPlayers;
    a.SetCollision(false, false, false);

    newactor = _AddActor(a, newclass, a.Location, a.Rotation, a.Owner, a.Tag);
    if(newactor == None) {
        err("SpawnReplacement("$a$", "$newclass$") failed");
        a.SetCollision(bCollideActors, bBlockActors, bBlockPlayers);
        return None;
    }

    l("SpawnReplacement("$a$", "$newclass$") " $ newactor);

    newactor.SetCollision(bCollideActors, bBlockActors, bBlockPlayers);
    newactor.SetPhysics(a.Physics);
    newactor.SetCollisionSize(a.CollisionRadius, a.CollisionHeight);
    newactor.SetBase(a.Base);
    newactor.Texture = a.Texture;
    newactor.Mesh = a.Mesh;
    newactor.Mass = a.Mass;
    newactor.Buoyancy = a.Buoyancy;
    newactor.Event = a.Event;
    return newactor;
}

static function SetActorScale(Actor a, float scale)
{
    local Vector newloc;

    newloc = a.Location + ( (a.CollisionHeight*scale - a.CollisionHeight*a.DrawScale) * vect(0,0,1) );
    a.SetLocation(newloc);
    a.SetCollisionSize(a.CollisionRadius, a.CollisionHeight / a.DrawScale * scale);
    a.DrawScale = scale;
}

function vector GetRandomPosition(optional vector target, optional float mindist, optional float maxdist, optional bool allowWater, optional bool allowPain)
{
    local PathNode temp[4096];
    local PathNode p;
    local int i, num, slot;
    local float dist;

    if( maxdist <= mindist )
        maxdist = 9999999;

    foreach AllActors(class'PathNode', p) {
        if( (!allowWater) && p.Region.Zone.bWaterZone ) continue;
        if( (!allowPain) && (p.Region.Zone.bKillZone || p.Region.Zone.bPainZone ) ) continue;
        dist = VSize(p.Location-target);
        if( dist < mindist ) continue;
        if( dist > maxdist ) continue;
        temp[num++] = p;
    }
    if( num == 0 ) return target;
    slot = rng(num);
    return temp[slot].Location;
}

function vector JitterPosition(vector loc)
{
    loc.X += rngfn() * 160.0;//10 feet in any direction
    loc.Y += rngfn() * 160.0;
    return loc;
}

function vector GetRandomPositionFine(optional vector target, optional float mindist, optional float maxdist, optional bool allowWater, optional bool allowPain)
{
    local vector loc;
    loc = GetRandomPosition(target, mindist, maxdist, allowWater, allowPain);
    loc = JitterPosition(loc);
    return loc;
}

function vector GetCloserPosition(vector target, vector current, optional float maxdist)
{
    local PathNode p;
    local float dist, farthest_dist, dist_move;
    local vector farthest;

    if( maxdist == 0.0 || VSize(target-current) < maxdist )
        maxdist = VSize(target-current);
    farthest = current;
    foreach AllActors(class'PathNode', p) {
        dist = VSize(target-p.Location);
        dist_move = VSize(p.Location-current);//make sure the distance that we're moving is shorter than the distance to the target (aka move forwards, not to the opposite side)
        if( dist > farthest_dist && dist < maxdist && dist > maxdist/2 && dist > dist_move ) {
            farthest_dist = dist;
            farthest = p.Location;
        }
    }
    return farthest;
}

//I could have fuzzy logic and allow these Is___Normal functions to have overlap? or make them more strict where some normals don't classify as any of these?
function bool IsWallNormal(vector n)
{
    return n.Z > -0.5 && n.Z < 0.5;
}

function bool IsFloorNormal(vector n)
{
    return n.Z > 0.5;
}

function bool IsCeilingNormal(vector n)
{
    return n.Z < -0.5;
}

function bool NearestSurface(vector StartTrace, vector EndTrace, out LocationNormal ret)
{
    local Actor HitActor;
    local vector HitLocation, HitNormal;

    HitActor = Trace(HitLocation, HitNormal, EndTrace, StartTrace, false);
    if( StartTrace == HitLocation ) {
        return false;
    }
    if ( HitActor == Level ) {
        ret.loc = HitLocation;
        ret.norm = HitNormal;
        return true;
    }
    return false;
}

function float GetDistanceFromSurface(vector StartTrace, vector EndTrace)
{
    local LocationNormal locnorm;
    locnorm.loc = EndTrace;
    NearestSurface(StartTrace, EndTrace, locnorm);
    return VSize( StartTrace - locnorm.loc );
}

function bool NearestCeiling(out LocationNormal out, FMinMax distrange, optional float away_from_wall)
{
    local vector EndTrace, MoveOffWall;
    EndTrace = out.loc;
    EndTrace.Z += distrange.max;
    if( NearestSurface(out.loc + (vect(0,0,1)*distrange.min), EndTrace, out) == false ) {
        return false;
    }
    if( ! IsCeilingNormal(out.norm) ) {
        return false;
    }
    MoveOffWall.X = away_from_wall;
    MoveOffWall.Y = away_from_wall;
    MoveOffWall.Z = away_from_wall;
    MoveOffWall *= out.norm;
    out.loc += MoveOffWall;
    return true;
}

function bool NearestFloor(out LocationNormal out, FMinMax distrange, optional float away_from_wall)
{
    local vector EndTrace, MoveOffWall;
    EndTrace = out.loc;
    EndTrace.Z -= distrange.max;
    if( NearestSurface(out.loc - (vect(0,0,1)*distrange.min), EndTrace, out) == false ) {
        return false;
    }
    if( ! IsFloorNormal(out.norm) ) {
        return false;
    }
    MoveOffWall.X = away_from_wall;
    MoveOffWall.Y = away_from_wall;
    MoveOffWall.Z = away_from_wall;
    MoveOffWall *= out.norm;
    out.loc += MoveOffWall;
    return true;
}

function bool _FindWallAlongNormal(out LocationNormal out, vector against, FMinMax distrange)
{
    local LocationNormal t, ret;
    local vector end, along;
    local float closest_dist, dist;
    local int i;

    closest_dist = distrange.max;
    along = Normal(against cross vect(0, 0, 1));

    for(i=0; i<2; i++) {
        end = out.loc;
        along *= -1;//negative first and then positive
        end += along * closest_dist;
        t = out;
        t.loc += along * distrange.min;
        if( NearestSurface(t.loc, end, t) ) {
            if( IsWallNormal(t.norm) ) {
                ret = t;
                dist = VSize(t.loc - out.loc);
                closest_dist = dist;
            }
        }
    }

    if( closest_dist >= distrange.max ) {
        return false;
    }
    out = ret;
    return true;
}

function bool _NearestWall(out LocationNormal out, FMinMax distrange)
{
    local LocationNormal t, ret;
    local vector along;
    local float closest_dist, dist;
    local int i;

    closest_dist = distrange.max;
    along = vect(1,0,0);//first we do the X axis

    for(i=0; i<2; i++) {
        t = out;
        if( _FindWallAlongNormal(t, along, distrange) ) {
            dist = VSize(t.loc - out.loc);
            if( dist < closest_dist ) {
                ret = t;
                closest_dist = dist;
            }
        }
        along = vect(0,1,0);//next we do the Y axis
    }

    if( closest_dist >= distrange.max ) {
        return false;
    }
    out = ret;
    return true;
}

function bool NearestWall(out LocationNormal out, FMinMax distrange, optional float away_from_wall)
{
    local vector MoveOffWall, normal;

    if( _NearestWall(out, distrange) == false ) {
        return false;
    }
    MoveOffWall.X = away_from_wall;
    MoveOffWall.Y = away_from_wall;
    MoveOffWall.Z = away_from_wall;
    MoveOffWall *= out.norm;
    out.loc += MoveOffWall;
    return true;
}

function bool NearestWallSearchZ(out LocationNormal out, FMinMax distrange, float z_range, vector target, optional float away_from_wall)
{
    local LocationNormal wall, ret;
    local vector MoveOffWall;
    local float closest_dist, dist;
    local int i;
    local float len;

    closest_dist = distrange.max;
    len = 4.0;// 4 checks plus the center
    for(i=0; i<5; i++) {
        wall = out;
        wall.loc.Z += z_range * (Float(i) / len * 2.0 - 1.0);
        if( _NearestWall(wall, distrange) ) {
            dist = VSize(wall.loc - target);
            if( dist < closest_dist && dist >= distrange.min ) {
                closest_dist = dist;
                ret = wall;
            }
        }
    }

    if( closest_dist >= distrange.max ) {
        return false;
    }
    MoveOffWall.X = away_from_wall;
    MoveOffWall.Y = away_from_wall;
    MoveOffWall.Z = away_from_wall;
    MoveOffWall *= ret.norm;
    ret.loc += MoveOffWall;
    out = ret;
    return true;
}

function bool NearestCornerSearchZ(out LocationNormal out, FMinMax distrange, vector against, float z_range, vector target, optional float away_from_wall)
{
    local LocationNormal wall, ret;
    local vector MoveOffWall;
    local float closest_dist, dist;
    local int i;
    local float len;

    closest_dist = distrange.max;
    len = 4.0;// 4 checks plus the center
    for(i=0; i<5; i++) {
        wall = out;
        wall.loc.Z += z_range * (Float(i) / len * 2.0 - 1.0);
        if( _FindWallAlongNormal(wall, against, distrange) ) {
            dist = VSize(wall.loc - target);
            if( dist < closest_dist && dist >= distrange.min ) {
                closest_dist = dist;
                ret = wall;
            }
        }
    }

    if( closest_dist >= distrange.max ) {
        return false;
    }
    MoveOffWall.X = away_from_wall;
    MoveOffWall.Y = away_from_wall;
    MoveOffWall.Z = away_from_wall;
    MoveOffWall *= ret.norm;
    ret.loc += MoveOffWall;
    out = ret;
    return true;
}

function Vector GetCenter(Actor test)
{
    local Vector MinVect, MaxVect;

    test.GetBoundingBox(MinVect, MaxVect);
    return (MinVect+MaxVect)/2;
}

function int GetSafeRule(safe_rule rules[32], name item_name, vector newpos)
{
    local int i;

    for(i=0; i<ArrayCount(rules); i++) {
        if( item_name != rules[i].item_name ) continue;
        if( dxr.localURL != rules[i].map ) continue;
        if( AnyGreater( rules[i].min_pos, newpos ) ) continue;
        if( AnyGreater( newpos, rules[i].max_pos ) ) continue;
        return i;
    }
    return -1;
}

function bool _PositionIsSafeOctant(Vector oldloc, Vector TestPoint, Vector newloc)
{
    local Vector distsold, diststest, distsoldtest;
    //l("results += testbool( _PositionIsSafeOctant(vect("$oldloc$"), vect("$ TestPoint $"), vect("$newloc$")), truefalse, \"test\");");
    distsoldtest = AbsEach(oldloc - TestPoint);
    distsold = AbsEach(newloc - oldloc) - (distsoldtest*0.999);
    diststest = AbsEach(newloc - TestPoint);
    if ( AnyGreater( distsold, diststest ) ) return False;
    return True;
}

function bool PositionIsSafe(Vector oldloc, Actor test, Vector newloc)
{// https://github.com/Die4Ever/deus-ex-randomizer/wiki#smarter-key-randomization
    local Vector TestPoint;
    local float distold, disttest;

    TestPoint = GetCenter(test);

    distold = VSize(newloc - oldloc);
    disttest = VSize(newloc - TestPoint);

    return _PositionIsSafeOctant(oldloc, TestPoint, newloc);
}

function bool PositionIsSafeLenient(Vector oldloc, Actor test, Vector newloc)
{// https://github.com/Die4Ever/deus-ex-randomizer/wiki#smarter-key-randomization
    return _PositionIsSafeOctant(oldloc, GetCenter(test), newloc);
}

function DebugMarkKeyPosition(Actor a, coerce string id)
{
    if( ! #defined debug ) err("don't call DebugMarkKeyPosition without debug mode!");

    if(DeusExDecoration(a) != None) {
        DeusExDecoration(a).ItemName = id @ DeusExDecoration(a).ItemName;
    } else if(Inventory(a) != None) {
        Inventory(a).ItemName = id @ Inventory(a).ItemName;
    }
    a.LightType=LT_Steady;
    a.LightEffect=LE_WateryShimmer;
    a.LightBrightness=255;
    a.LightHue=155;
    a.LightRadius=20;
    debug("DebugMarkKeyPosition "$a$ " ("$a.Location$") " $ id);
}

defaultproperties
{
}
