//=============================================================================
// FlameTurretBase.
//=============================================================================
class FlameTurretBase extends DeusExDecoration;

var FlameTurret Fgun;

enum EAreaType
{
	AOE_Point,
	AOE_Cone,
	AOE_Sphere
};

var() localized String titleString;		// So we can name specific turrets in multiplayer
var() bool bTrackPawnsOnly;
var() bool bTrackPlayersOnly;
var(WeaponAI) bool	  bWarnTarget;		 // When firing projectile, warn the target
var() bool bActive;
var() int maxRange;
var() float fireRate;
var() float FgunAccuracy;
var() int FgunDamage;
var() int ammoAmount;
var Actor curTarget;
var Actor prevTarget;         // target we had last tick.
var Pawn safeTarget;          // in multiplayer, this actor is strictly off-limits
                               // Usually for the player who activated the turret.
var() class<Projectile> ProjectileNames[3];		// projectile classes for different ammo
var() EAreaType			AreaOfEffect;			// area of effect of the weapon#

var()   class<projectile> ProjectileClass;

var float fireTimer;
var bool bConfused;				// used when hit by EMP
var float confusionTimer;		// how long until turret resumes normal operation
var float confusionDuration;	// how long does an EMP hit last?
var Actor LastTarget;			// what was our last target?
var float pitchLimit;			// what's the maximum pitch?
var Rotator origRot;			// original rotation
var bool bPreAlarmActiveState;	// was I previously awake or not?
var bool bDisabled;				// have I been hacked or shut down by computers?
var float TargetRefreshTime;    // used for multiplayer to reduce rate of checking for targets.
var float currentAccuracy;		// what the currently calculated accuracy is (updated every tick)
var float MinSpreadAcc;         // Minimum accuracy for multiple slug weapons (shotgun).  Affects only multiplayer,
                                // keeps shots from all going in same place (ruining shotgun effect)
var float MinProjSpreadAcc;
var float MinWeaponAcc;         // Minimum accuracy for a weapon at all.  Affects only multiplayer.
var	travel float	ProjectileSpeed;

var int team;						// Keep track of team the turrets on

var int BurnTime, BurnDamage;

var int mpTurretDamage;			// Settings for multiplayer
var int mpTurretRange;

var bool bComputerReset;			// Keep track of if computer has been reset so we avoid all actors checks

var bool bSwitching;
var float SwitchTime, beepTime;
var Pawn savedTarget;



// networking replication
replication
{
   //server to client
   reliable if (Role == ROLE_Authority)
      safeTarget, bDisabled, bActive, team, titleString;
}

// if we are triggered, turn us on
function Trigger(Actor Other, Pawn Instigator)
{
	if (bConfused || bDisabled)
		return;

	if (!bActive)
	{
		bActive = True;
		AmbientSound = Default.AmbientSound;
	}

	Super.Trigger(Other, Instigator);
}

// if we are untriggered, turn us off
function UnTrigger(Actor Other, Pawn Instigator)
{
	if (bConfused || bDisabled)
		return;

	if (bActive)
	{
		bActive = False;
		AmbientSound = None;
	}

	Super.UnTrigger(Other, Instigator);
}

function Destroyed()
{
	if (Fgun != None)
	{
		Fgun.Destroy();
		Fgun = None;
	}

	Super.Destroyed();
}

function UpdateSwitch()
{
	if ( Level.Timeseconds > SwitchTime )
	{
		bSwitching = False;
		//safeTarget = savedTarget;
		SwitchTime = 0;
		beepTime = 0;
	}
	else
	{
		if ( Level.Timeseconds > beepTime )
		{
			PlaySound(Sound'TurretSwitch', SLOT_Interact, 1.0,, maxRange );
			beepTime = Level.Timeseconds + 0.75;
		}
	}
}

function SetSafeTarget( Pawn newSafeTarget )
{
	local DeusExPlayer aplayer;

	bSwitching = True;
	SwitchTime = Level.Timeseconds + 2.5;
	beepTime = 0.0;
   safeTarget = newSafeTarget;
   //savedTarget = newSafeTarget;
}

function Actor AcquireMultiplayerTarget()
{
   local Pawn apawn;
	local DeusExPlayer aplayer;
	local Vector dist;
	local Actor noActor;

	if ( bSwitching )
	{
		noActor = None;
		return noActor;
	}

   //DEUS_EX AMSD See if our old target is still valid.
   if ((prevtarget != None) && (prevtarget != safetarget) && (Pawn(prevtarget) != None))
   {
      if (Pawn(prevtarget).AICanSee(self, 1.0, false, false, false, true) > 0)
      {
         if ((DeusExPlayer(prevtarget) == None) && !DeusExPlayer(prevtarget).bHidden )
         {
				dist = DeusExPlayer(prevtarget).Location - Fgun.Location;
				if (VSize(dist) < maxRange )
				{
					curtarget = prevtarget;
					return curtarget;
				}
         }
         else
         {
            if ((DeusExPlayer(prevtarget).AugmentationSystem.GetAugLevelValue(class'AugRadarTrans') == -1.0) && !DeusExPlayer(prevtarget).bHidden )
            {
					dist = DeusExPlayer(prevtarget).Location - Fgun.Location;
					if (VSize(dist) < maxRange )
					{
						curtarget = prevtarget;
						return curtarget;
					}
            }
         }
      }
   }
	// MB Optimized to use pawn list, previous way used foreach VisibleActors
	apawn = Fgun.Level.PawnList;
	while ( apawn != None )
	{
      if (apawn.bDetectable && !apawn.bIgnore && apawn.IsA('DeusExPlayer'))
      {
			aplayer = DeusExPlayer(apawn);

			dist = aplayer.Location - Fgun.Location;

			if ( VSize(dist) < maxRange )
			{
				// Only players we can see
				if ( aplayer.FastTrace( aplayer.Location, Fgun.Location ))
				{
					//only shoot at players who aren't the safetarget.
					//we alreayd know prevtarget not valid.
					if ((aplayer != safeTarget) && (aplayer != prevTarget))
					{
						if (! ( (TeamDMGame(aplayer.DXGame) != None) &&	(safeTarget != None) &&	(TeamDMGame(aplayer.DXGame).ArePlayersAllied( DeusExPlayer(safeTarget),aplayer)) ) )
						{
							// If the player's RadarTrans aug is off, the turret can see him
							if ((aplayer.AugmentationSystem.GetAugLevelValue(class'AugRadarTrans') == -1.0) && !aplayer.bHidden )
							{
								curTarget = apawn;
								PlaySound(Sound'TurretLocked', SLOT_Interact, 1.0,, maxRange );
								break;
							}
						}
					}
				}
			}
      }
		apawn = apawn.nextPawn;
	}
   return curtarget;
}

function Tick(float deltaTime)
{
	local Pawn pawn;
	local ScriptedPawn sp;
	local DeusExDecoration deco;
	local float near;
	local Rotator destRot;
	local bool bSwitched;

	Super.Tick(deltaTime);

	bSwitched = False;

	if ( bSwitching )
	{
		UpdateSwitch();
		return;
	}

	// Make sure everything is valid and account for when players leave or switch teams
	if ( !bDisabled && (Level.NetMode != NM_Standalone) )
	{
		if ( safeTarget == None )
		{
			bDisabled = True;
			bComputerReset = False;
		}
		else
		{
			if ( DeusExPlayer(safeTarget) != None )
			{
				if ((TeamDMGame(DeusExPlayer(safeTarget).DXGame) != None) && (DeusExPlayer(safeTarget).PlayerReplicationInfo.team != team))
					bSwitched = True;
				else if ((DeathMatchGame(DeusExPlayer(safeTarget).DXGame) != None ) && (DeusExPlayer(safeTarget).PlayerReplicationInfo.PlayerID != team))
					bSwitched = True;

				if ( bSwitched )
				{
					bDisabled = True;
					safeTarget = None;
					bComputerReset = False;
				}
			}
		}
	}
	if ( bDisabled && (Level.NetMode != NM_Standalone) )
	{
		team = -1;
		safeTarget = None;
		if ( !bComputerReset )
		{
			Fgun.ResetComputerAlignment();
			bComputerReset = True;
		}
	}

	if (bConfused)
	{
		confusionTimer += deltaTime;

		// pick a random facing
		if (confusionTimer % 0.25 > 0.2)
		{
			Fgun.DesiredRotation.Pitch = origRot.Pitch + (pitchLimit / 2 - Rand(pitchLimit));
			Fgun.DesiredRotation.Yaw = Rand(65535);
		}
		if (confusionTimer > confusionDuration)
		{
			bConfused = False;
			confusionTimer = 0;
			confusionDuration = Default.confusionDuration;
		}
	}

	if (bActive && !bDisabled)
	{
		curTarget = None;

		if ( !bConfused )
		{
			// if we've been EMP'ed, act confused
			if ((Level.NetMode != NM_Standalone) && (Role == ROLE_Authority))
			{
				// DEUS_EX AMSD If in multiplayer, get the multiplayer target.

				if (TargetRefreshTime < 0)
					TargetRefreshTime = 0;

				TargetRefreshTime = TargetRefreshTime + deltaTime;

				if (TargetRefreshTime >= 0.3)
				{
					TargetRefreshTime = 0;
					curTarget = AcquireMultiplayerTarget();
					if (( curTarget != prevTarget ) && ( curTarget == None ))
							PlaySound(Sound'TurretUnlocked', SLOT_Interact, 1.0,, maxRange );
					prevtarget = curtarget;
				}
				else
				{
					curTarget = prevtarget;
				}
			}
			else
			{
				//
				// Logic table for turrets
				//
				// bTrackPlayersOnly		bTrackPawnsOnly		Should Attack
				// 			T						X				Allies
				//			F						T				Enemies
				//			F						F				Everything
				//

				// Attack allies and neutrals
				if (bTrackPlayersOnly || (!bTrackPlayersOnly && !bTrackPawnsOnly))
				{
					foreach Fgun.VisibleActors(class'Pawn', pawn, maxRange, Fgun.Location)
					{
						if (pawn.bDetectable && !pawn.bIgnore)
						{
							if (pawn.IsA('DeusExPlayer'))
							{
								// If the player's RadarTrans aug is off, the turret can see him
								if (DeusExPlayer(pawn).AugmentationSystem.GetAugLevelValue(class'AugRadarTrans') == -1.0)
								{
									curTarget = pawn;
									break;
								}
							}
							else if (pawn.IsA('ScriptedPawn') && (ScriptedPawn(pawn).GetPawnAllianceType(GetPlayerPawn()) != ALLIANCE_Hostile))
							{
								curTarget = pawn;
								break;
							}
						}
					}
				}

				if (!bTrackPlayersOnly)
				{
					// Attack everything
					if (!bTrackPawnsOnly)
					{
						foreach Fgun.VisibleActors(class'DeusExDecoration', deco, maxRange, Fgun.Location)
						{
							if (!deco.IsA('ElectronicDevices') && !deco.IsA('FlameTurretBase') &&
								!deco.bInvincible && deco.bDetectable && !deco.bIgnore)
							{
								curTarget = deco;
								break;
							}
						}
					}

					// Attack enemies
					foreach Fgun.VisibleActors(class'ScriptedPawn', sp, maxRange, Fgun.Location)
					{
						if (sp.bDetectable && !sp.bIgnore && (sp.GetPawnAllianceType(GetPlayerPawn()) == ALLIANCE_Hostile))
						{
							curTarget = sp;
							break;
						}
					}
				}
			}

			// if we have a target, rotate to face it
			if (curTarget != None)
			{
				destRot = Rotator(curTarget.Location - Fgun.Location);
				Fgun.DesiredRotation = destRot;
				near = pitchLimit / 2;
				Fgun.DesiredRotation.Pitch = FClamp(Fgun.DesiredRotation.Pitch, origRot.Pitch - near, origRot.Pitch + near);
			}
			else
				Fgun.DesiredRotation = origRot;
		}
	}
	else
	{
		if ( !bConfused )
			Fgun.DesiredRotation = origRot;
	}

	near = (Abs(Fgun.Rotation.Pitch - Fgun.DesiredRotation.Pitch)) % 65536;
	near += (Abs(Fgun.Rotation.Yaw - Fgun.DesiredRotation.Yaw)) % 65536;

	if (bActive && !bDisabled)
	{
		// play an alert sound and light up
		if ((curTarget != None) && (curTarget != LastTarget))
			PlaySound(Sound'Beep6',,,, 1280);

		// if we're aiming close enough to our target
		if (curTarget != None)
		{
			if ((near < 4096) && (((Abs(Fgun.Rotation.Pitch - destRot.Pitch)) % 65536) < 8192))
			{
				if (fireTimer > fireRate)
				{
					Fire();
					fireTimer = 0;
				}
			}
		}
		else
		{
			if (Fgun.IsAnimating())
				Fgun.PlayAnim('Still', 10.0, 0.001);
		}

		fireTimer += deltaTime;
		LastTarget = curTarget;
	}
	else
	{
		if (Fgun.IsAnimating())
			Fgun.PlayAnim('Still', 10.0, 0.001);
	}

	// make noise if we're still moving
	if (near > 64)
	{
		Fgun.AmbientSound = Sound'AutoTurretMove';
		if (bConfused)
			Fgun.SoundPitch = 128;
		else
			Fgun.SoundPitch = 64;
	}
	else
		Fgun.AmbientSound = None;
}

auto state Active
{
	function TakeDamage(int Damage, Pawn EventInstigator, vector HitLocation, vector Momentum, name DamageType)
	{
		local float mindmg;

		if (DamageType == 'EMP')
		{
			// duration is based on daamge
			// 10 seconds min to 30 seconds max
			mindmg = Max(Damage - 15.0, 0.0);
			confusionDuration += mindmg / 5.0;
         confusionDuration = FClamp(confusionDuration,10.0,30.0);
			confusionTimer = 0;
			if (!bConfused)
			{
				bConfused = True;
				PlaySound(sound'EMPZap', SLOT_None,,, 1280);
			}
			return;
		}
		if (( Level.NetMode != NM_Standalone ) && (EventInstigator.IsA('DeusExPlayer')))
			DeusExPlayer(EventInstigator).ServerConditionalNotifyMsg( DeusExPlayer(EventInstigator).MPMSG_TurretInv );

		Super.TakeDamage(Damage, EventInstigator, HitLocation, Momentum, DamageType);
	}
}

function Fire()
{
	local Vector HitLocation, HitNormal, StartTrace, EndTrace, X, Y, Z;
	local Rotator rot;
	local Actor hit;
	local ShellCasing shell;
	local Spark spark;
	local Pawn attacker;

	if (!Fgun.IsAnimating())
		Fgun.LoopAnim('Fire');

	// Ammo will now run out.
	if (ammoAmount > 0)
	{
		ammoAmount--;
        ProjectileFire(ProjectileClass, ProjectileSpeed, bWarnTarget);
        PlaySound(sound'FlamethrowerFire', SLOT_None);


	}
	else
	{
		PlaySound(sound'DryFire', SLOT_None);
	}
}

simulated function SpawnEffects(Vector HitLocation, Vector HitNormal, Actor Other)
{
	local SmokeTrail puff;
	local int i;
	local BurnMark burn;
	local Rotator rot;

   if ((DeusExMPGame(Level.Game) != None) && (!DeusExMPGame(Level.Game).bSpawnEffects))
      return;

   if (FRand() < 0.5)
	{
		puff = spawn(class'SmokeTrail',,,HitLocation+HitNormal, Rotator(HitNormal));
		if (puff != None)
		{
			puff.DrawScale *= 0.3;
			puff.OrigScale = puff.DrawScale;
			puff.LifeSpan = 0.25;
			puff.OrigLifeSpan = puff.LifeSpan;
		}
	}

	if (!Other.IsA('BreakableGlass'))
		for (i=0; i<2; i++)
			if (FRand() < 0.8)
				spawn(class'Rockchip',,,HitLocation+HitNormal);

	burn = spawn(class'BurnMark', Other,, HitLocation, Rotator(HitNormal));

	// should we crack glass?
	if (GetWallMaterial(HitLocation, HitNormal) == 'Glass')
	{
		if (FRand() < 0.5)
			burn.Texture = Texture'FlatFXTex38';
		else
			burn.Texture = Texture'FlatFXTex39';

		burn.DrawScale = 0.5;
		burn.ReattachDecal();
	}
}

function name GetWallMaterial(vector HitLocation, vector HitNormal)
{
	local vector EndTrace, StartTrace;
	local actor newtarget;
	local int texFlags;
	local name texName, texGroup;

	StartTrace = HitLocation + HitNormal*16;		// make sure we start far enough out
	EndTrace = HitLocation - HitNormal;

	foreach TraceTexture(class'Actor', newtarget, texName, texGroup, texFlags, StartTrace, HitNormal, EndTrace)
		if ((newtarget == Level) || newtarget.IsA('Mover'))
			break;

	return texGroup;
}

/*function PlayHitSound(actor destActor, Actor hitActor)
{
	local float rnd;
	local sound snd;

	rnd = FRand();

	if (rnd < 0.25)
		snd = sound'Ricochet1';
	else if (rnd < 0.5)
		snd = sound'Ricochet2';
	else if (rnd < 0.75)
		snd = sound'Ricochet3';
	else
		snd = sound'Ricochet4';

	// play a different ricochet sound if the object isn't damaged by normal bullets
	if (hitActor != None)
	{
		if (hitActor.IsA('DeusExDecoration') && (DeusExDecoration(hitActor).minDamageThreshold > 10))
			snd = sound'ArmorRicochet';
		else if (hitActor.IsA('Robot'))
			snd = sound'ArmorRicochet';
	}

	if (destActor != None)
		destActor.PlaySound(snd, SLOT_None,,, 1024, 1.1 - 0.2*FRand());
}*/

// turn off the muzzle flash
simulated function Timer()
{
	Fgun.LightType = LT_None;
}

function AlarmHeard(Name event, EAIEventState state, XAIParams params)
{
	if (state == EAISTATE_Begin)
	{
		if (!bActive)
		{
			bPreAlarmActiveState = bActive;
			bActive = True;
		}
	}
	else if (state == EAISTATE_End)
	{
		if (bActive)
			bActive = bPreAlarmActiveState;
	}
}

function PostBeginPlay()
{
   safeTarget = None;
   prevTarget = None;
   TargetRefreshTime = 0;
   Super.PostBeginPlay();
}

function PreBeginPlay()
{
	local Vector v1, v2;
	local class<FlameTurret> FgunClass;
	local Rotator rot;

	Super.PreBeginPlay();

	//if (IsA('FlameTurretBase'))
		FgunClass = class'FlameTurret';
	//else
	//	gunClass = class'FlameTurret';

	rot = Rotation;
	rot.Pitch = 0;
	rot.Roll = 0;
	origRot = rot;
	Fgun = Spawn(FgunClass, Self,, Location, rot);
	if (Fgun != None)
	{
//		v1.X = 12;
		v1.X = 0;
		v1.Y = 0;
		v1.Z = CollisionHeight + Fgun.Default.CollisionHeight;
		v2 = v1 >> Rotation;
		v2 += Location;
		Fgun.SetLocation(v2);
		Fgun.SetBase(Self);
	}

	// set up the alarm listeners
	AISetEventCallback('Alarm', 'AlarmHeard');

	if ( Level.NetMode != NM_Standalone )
	{
		maxRange = mpTurretRange;
		FgunDamage = mpTurretDamage;
		bInvincible = True;
        bDisabled = !bActive;
	}
}

simulated function Projectile ProjectileFire(class<projectile> ProjClass, float ProjSpeed, bool bWarn)
{
	local Vector Vec1, X, Y, Z;
	local Rotator Vec2;
	local DeusExProjectile proj;
	local float mult;
	local float volume, radius;
	local int i, numProj;


	// should we shoot multiple projectiles in a spread?
	if (AreaOfEffect == AOE_Cone)
		numProj = 3;
	else
		numProj = 1;

	GetAxes(Fgun.Rotation,X,Y,Z);
	Vec1 = Fgun.Location;
	Vec2 = Fgun.Rotation;


	for (i=0; i<numProj; i++)
	{
      // If we have multiple slugs, then lower our accuracy a bit after the first slug so the slugs DON'T all go to the same place
      if ((i > 0) && (Level.NetMode != NM_Standalone))
         if (currentAccuracy < MinProjSpreadAcc)
            currentAccuracy = MinProjSpreadAcc;;


		if (( Level.NetMode == NM_Standalone ) || ( Owner.IsA('DeusExPlayer') && DeusExPlayer(Owner).PlayerIsListenClient()) )
		{
			proj = DeusExProjectile(Spawn(ProjClass, Owner,, Vec1, Vec2));

		}
     }
	return proj;
}

#exec MESH IMPORT MESH=FlameTurretBase ANIVFILE=MODELS\FlameTurretBase_a.3d DATAFILE=MODELS\FlameTurretBase_d.3d X=0 Y=0 Z=0
#exec mesh ORIGIN MESH=FlameTurretBase X=0 Y=0 Z=0 PITCH=128

#exec MESH SEQUENCE MESH=FlameTurretBase SEQ=All             STARTFRAME=0 NUMFRAMES=1
#exec MESH SEQUENCE MESH=FlameTurretBase SEQ=FlameTurretBase STARTFRAME=0 NUMFRAMES=1
#exec MESH SEQUENCE MESH=FlameTurretBase SEQ=Fire            STARTFRAME=0 NUMFRAMES=1

#exec TEXTURE IMPORT NAME=JFlameTurretBase1 FILE=Textures\FTBaseTex1.PCX GROUP=Skins // 01 - default

#exec MESHMAP NEW   MESHMAP=FlameTurretBase MESH=FlameTurretBase
#exec MESHMAP SCALE MESHMAP=FlameTurretBase X=0.00390625 Y=0.00390625 Z=0.00390625

#exec MESHMAP SETTEXTURE MESHMAP=FlameTurretBase NUM=1 TEXTURE=JFlameTurretBase1

defaultproperties
{
     titleString="FlameTurret"
     bTrackPlayersOnly=True
     bActive=True
     maxRange=512
     fireRate=0.100000
     FgunAccuracy=0.200000
     FgunDamage=5
     AmmoAmount=100
     ProjectileClass=Class'TNM.TNMFireball'
     confusionDuration=20.000000
     pitchLimit=11000.000000
     Team=-1
     burnTime=10
     BurnDamage=5
     mpTurretDamage=20
     mpTurretRange=1024
     HitPoints=50
     minDamageThreshold=50
     bHighlight=False
     ItemName="Flame Turret Base"
     bPushable=False
     Physics=PHYS_None
     Mesh=LodMesh'TNMDeco.FlameTurretBase'
     SoundRadius=48
     SoundVolume=192
     AmbientSound=Sound'DeusExSounds.Generic.AutoTurretHum'
     CollisionRadius=14.000000
     CollisionHeight=20.200001
     Mass=50.000000
     Buoyancy=10.000000
     bVisionImportant=True
}
