﻿Imports CsGL.OpenGL
Imports UPLib

Public Class Form1
    Dim WithEvents myGLView As GLViewControl

    Private _fastLoading As Boolean
    Public Property FastLoading() As Boolean
        Get
            Return _fastLoading
        End Get
        Set(value As Boolean)
            _fastLoading = value
            FastLoadToolStripMenuItem.Checked = _fastLoading
            SplitContainer1.Visible = Not _fastLoading
        End Set
    End Property

#Region "Mouse handling"
    'Camera movement
    Dim InitialMouseX, InitialMouseY As Integer ' When pressed down
    Dim PrevMouseX, PrevMouseY As Integer ' In previous frame
    Dim CameraMoveMode As CameraMovement = CameraMovement.None
    Dim RotationSpeed As Double = 0.18
    Dim MoveSpeed As Double = 4
    Dim GridAccumulator As Single

    Enum CameraMovement
        None
        Left
        Right
        LeftRight
    End Enum

    Sub SetCamModeFromMousePress(btn As MouseButtons)
        If btn = Windows.Forms.MouseButtons.Left Then
            If CameraMoveMode = CameraMovement.Right Then
                CameraMoveMode = CameraMovement.LeftRight
            Else
                CameraMoveMode = CameraMovement.Left
            End If
        ElseIf btn = Windows.Forms.MouseButtons.Right Then
            If CameraMoveMode = CameraMovement.Left Then
                CameraMoveMode = CameraMovement.LeftRight
            Else
                CameraMoveMode = CameraMovement.Right
            End If
        ElseIf btn = (Windows.Forms.MouseButtons.Left Or Windows.Forms.MouseButtons.Right) Then
            CameraMoveMode = CameraMovement.LeftRight
        ElseIf btn = Windows.Forms.MouseButtons.None Then
            CameraMoveMode = CameraMovement.None
        End If
    End Sub

    Sub SetCamModeFromMouseRelease(btn As MouseButtons)
        If CameraMoveMode = CameraMovement.LeftRight Then
            If btn = Windows.Forms.MouseButtons.Left Then
                CameraMoveMode = CameraMovement.Right
            ElseIf btn = Windows.Forms.MouseButtons.Right Then
                CameraMoveMode = CameraMovement.Left
            End If
        Else
            CameraMoveMode = CameraMovement.None
        End If
    End Sub

    Sub WrapMouse()
        Dim pos = MousePosInPreview()
        Dim newpos = pos

        If pos.X >= myGLView.Width Then
            newpos.X -= myGLView.Width
        ElseIf pos.X < 0 Then
            newpos.X += myGLView.Width
        End If

        If pos.Y >= myGLView.Height Then
            newpos.Y -= myGLView.Height
        ElseIf pos.Y < 0 Then
            newpos.Y += myGLView.Height
        End If

        PrevMouseX += newpos.X - pos.X
        PrevMouseY += newpos.Y - pos.Y

        SetMousePosition(newpos)
    End Sub

    Sub SetMousePosition(pos As Point)
        Cursor.Position = myGLView.PointToScreen(pos)
    End Sub

    Function MousePosInPreview() As Point
        Return myGLView.PointToClient(Cursor.Position)
    End Function

    Function DegToRad(angle As Double) As Double
        Return angle / 360 * 2 * Math.PI
    End Function

    Private Sub myGLView_MouseDown(ByVal sender As Object, ByVal e As MouseEventArgs) Handles myGLView.MouseDown
        InitialMouseX = e.X
        InitialMouseY = e.Y
        PrevMouseX = e.X
        PrevMouseY = e.Y
        GridAccumulator = 0
        SetCamModeFromMousePress(e.Button)

        UpdateView()
    End Sub

    Private Sub myGLView_MouseUp(ByVal sender As Object, ByVal e As MouseEventArgs) Handles myGLView.MouseUp
        SetCamModeFromMouseRelease(e.Button)
    End Sub

    Private Sub myGLView_MouseMove(ByVal sender As Object, ByVal e As MouseEventArgs) Handles myGLView.MouseMove
        Dim MouseFrameDeltaX, MouseFrameDeltaY As Integer

        'in most cases it is unnecessary
        If e.Button = Windows.Forms.MouseButtons.None Then
            CameraMoveMode = CameraMovement.None
        End If

        If CameraMoveMode <> CameraMovement.None Then
            WrapMouse()
        End If

        MouseFrameDeltaX = e.X - PrevMouseX
        MouseFrameDeltaY = e.Y - PrevMouseY

        ' Moving the camera
        Select Case CameraMoveMode
            Case CameraMovement.Right
                myGLView.CameraRotationH += MouseFrameDeltaX * RotationSpeed
                myGLView.CameraRotationV += MouseFrameDeltaY * RotationSpeed
                myGLView.CameraRotationV = Math.Max(-90, Math.Min(90, myGLView.CameraRotationV))

                UpdateView()
            Case CameraMovement.Left
                myGLView.CameraRotationH += MouseFrameDeltaX * RotationSpeed

                Dim moveDelta As Vector
                moveDelta.X = Math.Cos(DegToRad(myGLView.CameraRotationH - 90))
                moveDelta.Y = Math.Sin(DegToRad(myGLView.CameraRotationH - 90))
                moveDelta *= -MouseFrameDeltaY
                moveDelta *= MoveSpeed

                myGLView.CameraPosition += moveDelta
                UpdateYoureHere()
                UpdateView()
            Case CameraMovement.LeftRight
                myGLView.CameraPosition.Z -= MouseFrameDeltaY * MoveSpeed

                Dim moveDelta As Vector
                moveDelta.X = Math.Cos(DegToRad(myGLView.CameraRotationH))
                moveDelta.Y = Math.Sin(DegToRad(myGLView.CameraRotationH))
                moveDelta *= MouseFrameDeltaX
                moveDelta *= MoveSpeed

                myGLView.CameraPosition += moveDelta
                UpdateYoureHere()
                UpdateView()
        End Select


        PrevMouseX = e.X
        PrevMouseY = e.Y
    End Sub
#End Region

    Private Sub UpdateView()
        myGLView.GLDraw()
        myGLView.Refresh()
    End Sub

    Private Sub Form1_Shown(sender As System.Object, e As System.EventArgs) Handles MyBase.Shown
        myGLView = New GLViewControl
        myGLView.Dock = DockStyle.Fill
        myGLView.Parent = ViewportPanel
        myGLView.AllowDrop = True

        Try
            If IO.File.Exists("BSPTool.ini") Then
                Using r As New IO.StreamReader("BSPTool.ini")
                    While Not r.EndOfStream
                        Dim line = r.ReadLine
                        If line.StartsWith("FastLoading=", StringComparison.OrdinalIgnoreCase) Then
                            FastLoading = line.Substring(line.IndexOf("=") + 1)
                        End If
                    End While
                End Using
            End If
        Catch
        End Try

    End Sub

    Private Sub OpenToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles OpenToolStripMenuItem.Click
        If OpenDlg.ShowDialog = Windows.Forms.DialogResult.OK Then
            SaveDlg.InitialDirectory = IO.Path.GetDirectoryName(OpenDlg.FileName)
            SaveDlg.FileName = IO.Path.GetFileName(OpenDlg.FileName)
            Text = IO.Path.GetFileNameWithoutExtension(OpenDlg.FileName) & " - BSP Viewer"
            Dim model = GetModel(FindLevelModel(New Package(OpenDlg.FileName)))

            myGLView.Model = model

            If Not FastLoading Then
                tvBSP.BeginUpdate()
                tvBSP.Nodes.Clear()
                Dim root = tvBSP.Nodes.Add("Root node 0")
                root.Tag = 0 'Tag is iNode

                AddNodesUnder(root, model)
                tvBSP.SelectedNode = tvBSP.Nodes(0)
                tvBSP.EndUpdate()

                UpdateYoureHere()

                cmbChangeZone.BeginUpdate()
                cmbChangeZone.Items.Clear()
                For i = 0 To model.Zones.Count - 1
                    Dim name = If(i = 0, "(void)", model.Level.GetShortObjectName(model.Zones(i).ZoneActor))
                    cmbChangeZone.Items.Add(i & ". " & name)
                Next
                cmbChangeZone.Items.Add("63. LevelInfo")
                cmbChangeZone.SelectedIndex = cmbChangeZone.Items.Count - 1
                cmbChangeZone.EndUpdate()
            End If
        End If
    End Sub

    Private Sub AddNodesUnder(root As TreeNode, model As Model)
        Dim iRootNode As Integer = root.Tag

        Dim node = model.Nodes(iRootNode)
        If node.iChild0 <> -1 Then
            Dim childNode = root.Nodes.Add("Back node " & node.iChild0)
            childNode.Tag = node.iChild0
            AddNodesUnder(childNode, model)
        End If
        If node.iChild1 <> -1 Then
            Dim childNode = root.Nodes.Add("Front node " & node.iChild1)
            childNode.Tag = node.iChild1
            AddNodesUnder(childNode, model)
        End If
        If node.iChild2 <> -1 Then
            Dim childNode = root.Nodes.Add("Coplanar " & node.iChild2)
            childNode.Tag = node.iChild2
            childNode.ImageIndex = 2
            childNode.SelectedImageIndex = 2
            AddNodesUnder(childNode, model)
        End If
        If node.iLeaf0 <> -1 Then
            Dim childNode = root.Nodes.Add("Back leaf " & node.iLeaf0 & ", zone " & node.iZone0)
            childNode.Tag = -1
            childNode.ImageIndex = 1
            childNode.SelectedImageIndex = 1
        End If
        If node.iLeaf1 <> -1 Then
            Dim childNode = root.Nodes.Add("Front leaf " & node.iLeaf1 & ", zone " & node.iZone1)
            childNode.Tag = -1
            childNode.ImageIndex = 1
            childNode.SelectedImageIndex = 1
        End If
    End Sub

    Private Sub tvBSP_AfterSelect(sender As System.Object, e As System.Windows.Forms.TreeViewEventArgs) Handles tvBSP.AfterSelect
        If e.Node IsNot Nothing Then
            Dim iNode = e.Node.Tag
            If iNode <> -1 Then
                Dim node = myGLView.Model.Nodes(iNode)
                Dim s As New System.Text.StringBuilder

                Dim source = "None"
                If node.iSurf <> -1 Then
                    Dim surf = myGLView.Model.Surfs(node.iSurf)
                    If surf.Actor <> 0 Then source = myGLView.Model.Level.GetShortObjectName(surf.Actor) & ", poly " & surf.iBrushPoly
                End If
                s.AppendLine("Source: " & source)
                s.AppendLine("Plane: (" & NumToStr(node.Plane.X) & ", " & NumToStr(node.Plane.Y) & ", " & NumToStr(node.Plane.Z) & "), " & NumToStr(node.Plane.W))
                s.AppendLine("NodeFlags: " & node.NodeFlags)
                s.AppendLine("iSurf: " & node.iSurf)
                s.AppendLine("iVertPool: " & node.iVertPool)
                s.AppendLine("NumVertices: " & node.NumVertices)
                s.AppendLine("iCollisionBound: " & node.iCollisionBound)
                s.AppendLine("iRenderBound: " & node.iRenderBound)
                s.AppendLine("iBackChild: " & node.iChild0)
                s.AppendLine("iFrontChild: " & node.iChild1)
                s.AppendLine("iPlanarChild: " & node.iChild2)
                s.AppendLine("iBackLeaf: " & node.iLeaf0)
                s.AppendLine("iFrontLeaf: " & node.iLeaf1)
                s.AppendLine("iBackZone: " & node.iZone0)
                s.AppendLine("iFrontZone: " & node.iZone1)
                s.Append("ZoneMask: 0x" & Hex(node.ZoneMask))

                tbInfo.Text = s.ToString
            Else
                tbInfo.Text = ""
            End If
            myGLView.iSelectedNode = iNode
        Else
            tbInfo.Text = ""
            myGLView.iSelectedNode = -1
        End If
        UpdateView()
    End Sub

    Private Sub AboutToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles AboutToolStripMenuItem.Click
        MessageBox.Show(My.Resources.AboutText, "About")
    End Sub

    Sub UpdateYoureHere()
        If myGLView.Model IsNot Nothing Then
            Dim model = myGLView.Model
            Dim msg = ""

            Dim pr = model.GetLocZone(myGLView.CameraPosition)

            msg &= "iLeaf: " & pr.iLeaf & ", ZoneNumber: " & pr.ZoneNumber & ", ZoneActor: "

            If pr.ZoneNumber = 0 Then
                msg &= "(void)"
            ElseIf pr.Zone = 0 Then
                msg &= "LevelInfo"
            Else
                msg &= model.Level.GetShortObjectName(pr.Zone)
            End If

            lbYoureHere.Text = msg
            lbYoureHere.Refresh()
        End If
    End Sub

    Private Sub SaveToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles SaveToolStripMenuItem.Click
        If myGLView.Model Is Nothing Then Return
        If SaveDlg.ShowDialog = Windows.Forms.DialogResult.OK Then
            OpenDlg.InitialDirectory = IO.Path.GetDirectoryName(SaveDlg.FileName)
            OpenDlg.FileName = IO.Path.GetFileName(SaveDlg.FileName)
            SetModel(FindLevelModel(myGLView.Model.Level), myGLView.Model)
            myGLView.Model.Level.Save(SaveDlg.FileName)
            Text = IO.Path.GetFileNameWithoutExtension(SaveDlg.FileName) & " - BSP Viewer"
        End If
    End Sub

    Private Sub ImportOBJToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles ImportOBJToolStripMenuItem.Click
        If OpenOBJDlg.ShowDialog = Windows.Forms.DialogResult.OK Then

            Dim obj As New OBJ(OpenOBJDlg.FileName)

            Dim bsp As New BSPBuilder
            For Each poly In obj.Polys
                Dim verts As New List(Of Vector)
                Dim uvs As New List(Of UV)
                For i = poly.Verts.Count - 1 To 0 Step -1 'flip normals
                    Dim vert = poly.Verts(i)
                    Dim pos = obj.Verts(vert.iVert)
                    Swap(pos.Y, pos.Z)
                    verts.Add(pos)
                    uvs.Add(obj.UVs(vert.iUV))
                Next
                bsp.AddPolygon(verts, uvs, poly.Material)
            Next
            MessageBox.Show("Building BSP tree")
            bsp.Build()

            Dim model = myGLView.Model

            model.Verts.Clear()
            model.Points.Clear()
            model.Vectors.Clear()
            model.Surfs.Clear()
            model.NumSharedSides = 0
            model.Nodes.Clear()
            myGLView.Model.LeafHulls.Clear()

            For Each surf In bsp.Surfs
                Dim newSurf As BspSurf
                newSurf.Actor = -1
                newSurf.iBrushPoly = -1
                newSurf.iLightMap = -1
                newSurf.Texture = -1
                model.Points.Add(surf.Origin)
                newSurf.pBase = model.Points.Count - 1
                model.Vectors.Add(surf.Normal)
                newSurf.vNormal = model.Vectors.Count - 1
                model.Vectors.Add(surf.TextureU)
                newSurf.vTextureU = model.Vectors.Count - 1
                model.Vectors.Add(surf.TextureV)
                newSurf.vTextureV = model.Vectors.Count - 1
                model.Surfs.Add(newSurf)
            Next

            Dim nodeList As New List(Of BuilderPoly)
            bsp.Root.ToList(nodeList)
            For Each node In nodeList
                Dim newNode As BspNode
                newNode.iChild0 = -1
                newNode.iChild1 = -1
                newNode.iChild2 = -1
                If node.iCoplanar = 0 Then
                    If node.Node.Back IsNot Nothing Then newNode.iChild0 = node.Node.Back.Coplanars(0).iNode
                    If node.Node.Front IsNot Nothing Then newNode.iChild1 = node.Node.Front.Coplanars(0).iNode
                End If
                ' Has next coplanar?
                If node.Node.Coplanars.Count - 1 > node.iCoplanar Then newNode.iChild2 = node.Node.Coplanars(node.iCoplanar + 1).iNode
                newNode.iZone0 = 63
                newNode.iZone1 = 63
                newNode.iLeaf0 = -1
                newNode.iLeaf1 = -1
                newNode.iSurf = node.Surf.iSurf
                newNode.iRenderBound = -1
                newNode.Plane = node.Node.Plane
                newNode.ZoneMask = ULong.MaxValue
                newNode.iCollisionBound = -1
                newNode.NumVertices = node.Verts.Count
                newNode.iVertPool = model.Verts.Count
                For Each v In node.Verts
                    model.Points.Add(v.Position)
                    model.Verts.Add(New Vert(model.Points.Count - 1, -1))
                Next
                model.Nodes.Add(newNode)
            Next

            model.RecalcNodeParents()

            For i = 0 To model.Nodes.Count - 1
                model.RecalcNodeCollision(i)
            Next

            MessageBox.Show("Done")
        End If
    End Sub

    Private Sub RemoveVoidToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles RemoveVoidToolStripMenuItem.Click
        If myGLView.Model Is Nothing Then Return
        For i = 0 To myGLView.Model.Nodes.Count - 1
            Dim node = myGLView.Model.Nodes(i)
            If node.iZone0 = 0 Then node.iZone0 = 63
            If node.iZone1 = 0 Then node.iZone1 = 63
            myGLView.Model.Nodes(i) = node
        Next
    End Sub

    Private Sub RemoveAllZonesToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles RemoveAllZonesToolStripMenuItem.Click
        If myGLView.Model Is Nothing Then Return
        For i = 0 To myGLView.Model.Nodes.Count - 1
            Dim node = myGLView.Model.Nodes(i)
            node.iZone0 = 63
            node.iZone1 = 63
            myGLView.Model.Nodes(i) = node
        Next
    End Sub

    Private Sub RecalculateLeafHullsToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs)
        If myGLView.Model Is Nothing Then Return

        myGLView.Model.LeafHulls.Clear()

        For i = 0 To myGLView.Model.Nodes.Count - 1
            myGLView.Model.RecalcNodeCollision(i)
        Next
    End Sub

    Private Sub WigglePointsToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles WigglePointsToolStripMenuItem.Click
        If myGLView.Model Is Nothing Then Return
        For i = 0 To myGLView.Model.Points.Count - 1
            myGLView.Model.Points(i) += New Vector(Rnd() * 64 - 32, Rnd() * 64 - 32, Rnd() * 64 - 32)
        Next
    End Sub

    Private Sub UpscaleMap2xToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles UpscaleMap2xToolStripMenuItem.Click
        If myGLView.Model Is Nothing Then Return
        For i = 0 To myGLView.Model.Points.Count - 1
            myGLView.Model.Points(i) *= 2
        Next

        For i = 0 To myGLView.Model.Nodes.Count - 1
            Dim node = myGLView.Model.Nodes(i)
            node.Plane.W *= 2
            myGLView.Model.Nodes(i) = node

            If node.iCollisionBound <> -1 Then
                Dim iNodes As List(Of Integer) = Nothing
                Dim bounds As BoundingBox
                myGLView.Model.GetLeafHull(node.iCollisionBound, iNodes, bounds)
                bounds.Min *= 2
                bounds.Max *= 2
                myGLView.Model.SetLeafHull(node.iCollisionBound, iNodes, bounds)
            End If
        Next

        Dim isVectorScaled As New BitArray(myGLView.Model.Vectors.Count)

        For Each surf In myGLView.Model.Surfs
            If Not isVectorScaled(surf.vTextureU) Then
                myGLView.Model.Vectors(surf.vTextureU) /= 2
                isVectorScaled(surf.vTextureU) = True
            End If

            If Not isVectorScaled(surf.vTextureV) Then
                myGLView.Model.Vectors(surf.vTextureV) /= 2
                isVectorScaled(surf.vTextureV) = True
            End If
        Next

        For i = 0 To myGLView.Model.Bounds.Count - 1
            Dim bound = myGLView.Model.Bounds(i)
            bound.Min *= 2
            bound.Max *= 2
            myGLView.Model.Bounds(i) = bound
        Next

    End Sub

    Private Sub ListLeafHullsToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs)
        If myGLView.Model Is Nothing Then Return
        Dim str As New System.Text.StringBuilder
        str.AppendLine("Parent,...,Node | LeafHull (a.k.a. CollisionBound)")
        Dim maxThing = 0
        For i = 0 To myGLView.Model.Nodes.Count - 1
            Dim node = myGLView.Model.Nodes(i)
            If node.iCollisionBound <> -1 Then

                ' Get list of parents
                Dim hierarchy As New List(Of Integer)
                Dim iNode = i
                Do
                    hierarchy.Add(iNode)
                    iNode = myGLView.Model.Nodes(iNode).iParent
                Loop Until iNode = -1

                For j = hierarchy.Count - 1 To 0 Step -1
                    str.Append(hierarchy(j))
                    If j <> 0 Then str.Append(" ")
                Next

                str.Append(" | ")
                Dim lhNodes As List(Of Integer) = Nothing
                Dim box As BoundingBox
                myGLView.Model.GetLeafHull(node.iCollisionBound, lhNodes, box)

                For j = 0 To lhNodes.Count - 1
                    Dim lhNode = lhNodes(j)
                    ' &H40000000 means that the next child (if a node is skipped, it refers
                    ' to the next one in tree, NOT the next one in this list) is a front one.
                    ' Obviously, always absent from last node.
                    Dim nextFront = False
                    If lhNode And &H40000000 Then nextFront = True
                    lhNode = lhNode And Not &H40000000
                    maxThing = Math.Max(maxThing, lhNode)
                    str.Append(lhNode)
                    If nextFront Then
                        str.Append(" front")
                    Else
                        If j <> lhNodes.Count - 1 Then str.Append(" back")
                    End If

                    If j <> lhNodes.Count - 1 Then str.Append(" ")
                Next
                str.AppendLine()
            End If
        Next
        TextForm.ShowText(str.ToString, Me)
    End Sub

    Private Sub btnChangeZone_Click(sender As System.Object, e As System.EventArgs) Handles btnChangeZone.Click
        If myGLView.Model Is Nothing Then Return

        Dim iNode As Integer
        Dim front As Boolean
        Dim region = myGLView.Model.GetLocZone(myGLView.CameraPosition, iNode, front)

        Dim newZone = cmbChangeZone.SelectedIndex
        If newZone = cmbChangeZone.Items.Count - 1 Then
            newZone = 63
        End If

        Dim node = myGLView.Model.Nodes(iNode)
        If front Then
            node.iZone1 = newZone
        Else
            node.iZone0 = newZone
        End If
        myGLView.Model.Nodes(iNode) = node

        UpdateYoureHere()
    End Sub

    Private Sub btnRecalcCollision_Click(sender As System.Object, e As System.EventArgs) Handles btnRecalcCollision.Click
        If myGLView.Model Is Nothing Then Return
        If myGLView.iSelectedNode = -1 Then Return
        myGLView.Model.RecalcNodeCollision(myGLView.iSelectedNode)
    End Sub

    Private Sub btnRemoveCollision_Click(sender As System.Object, e As System.EventArgs) Handles btnRemoveCollision.Click
        If myGLView.Model Is Nothing Then Return
        If myGLView.iSelectedNode = -1 Then Return
        Dim node = myGLView.Model.Nodes(myGLView.iSelectedNode)
        node.iCollisionBound = -1
        myGLView.Model.Nodes(myGLView.iSelectedNode) = node
    End Sub

    Private Sub EnlargeRenderBoundsToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles EnlargeRenderBoundsToolStripMenuItem.Click
        If myGLView.Model Is Nothing Then Return

        For i = 0 To myGLView.Model.Bounds.Count - 1
            Dim bound = myGLView.Model.Bounds(i)
            Dim center = (bound.Max + bound.Min) / 2
            Dim radius = (bound.Max - bound.Min) / 2

            radius *= 2

            bound.Min = center - radius
            bound.Max = center + radius

            bound.Min.X = Math.Min(32768, Math.Max(-32768, bound.Min.X))
            bound.Min.Y = Math.Min(32768, Math.Max(-32768, bound.Min.Y))
            bound.Min.Z = Math.Min(32768, Math.Max(-32768, bound.Min.Z))
            bound.Max.X = Math.Min(32768, Math.Max(-32768, bound.Max.X))
            bound.Max.Y = Math.Min(32768, Math.Max(-32768, bound.Max.Y))
            bound.Max.Z = Math.Min(32768, Math.Max(-32768, bound.Max.Z))

            myGLView.Model.Bounds(i) = bound
        Next
    End Sub

    Private Sub RecalcRenderBoundsToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles RecalcRenderBoundsToolStripMenuItem.Click
        If myGLView.Model Is Nothing Then Return

        'myGLView.Model.Bounds.Clear()

        CalcRenderBounds(myGLView.Model, 0)

    End Sub

    Private Shared Function CalcRenderBounds(model As Model, iNode As Integer, Optional dontSet As Boolean = False) As BoundingBox
        Dim curBound As BoundingBox

        Dim node = model.Nodes(iNode)

        If node.iChild0 <> -1 Then curBound.Extend(CalcRenderBounds(model, node.iChild0))
        If node.iChild1 <> -1 Then curBound.Extend(CalcRenderBounds(model, node.iChild1))
        If node.iChild2 <> -1 Then curBound.Extend(CalcRenderBounds(model, node.iChild2, True))

        For i = 0 To node.NumVertices - 1
            curBound.Extend(model.Points(model.Verts(node.iVertPool + i).pVertex))
        Next

        If Not dontSet Then
            If node.iRenderBound <> -1 Then
                Dim origBound = model.Bounds(node.iRenderBound)
                origBound.Extend(curBound)
                model.Bounds(node.iRenderBound) = origBound
            End If
        End If

        Return curBound
    End Function

    Private Sub btnFlipZones_Click(sender As System.Object, e As System.EventArgs) Handles btnFlipZones.Click
        If myGLView.Model Is Nothing Then Return
        Dim node = myGLView.Model.Nodes(myGLView.iSelectedNode)
        Swap(node.iZone0, node.iZone1)
        myGLView.Model.Nodes(myGLView.iSelectedNode) = node
    End Sub

    Private Sub FixInvisibleSemisolidsToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles FixInvisibleSemisolidsToolStripMenuItem.Click
        If myGLView.Model Is Nothing Then Return

        FixInvisibleSemisolids(myGLView.Model, 0)

        For i As Integer = 0 To myGLView.Model.Nodes.Count - 1
            Dim node = myGLView.Model.Nodes(i)
            If node.iZone1 = 0 Then node.iZone1 = node.iZone0
            myGLView.Model.Nodes(i) = node
        Next

    End Sub

    Sub FixInvisibleSemisolids(model As Model, iNode As Integer)
        Dim node = model.Nodes(iNode)

        If node.iChild0 <> -1 Then FixInvisibleSemisolids(model, node.iChild0)
        If node.iChild1 <> -1 Then FixInvisibleSemisolids(model, node.iChild1)

        'Dim origNode = node
        Dim goodZone = node.iZone1
        If goodZone = 0 Then goodZone = node.iZone0
        While node.iChild2 <> -1
            iNode = node.iChild2
            node = model.Nodes(iNode)

            'If AlmostEqual(node.Plane.X, -origNode.Plane.X) And
            '   AlmostEqual(node.Plane.Y, -origNode.Plane.Y) And
            '   AlmostEqual(node.Plane.Z, -origNode.Plane.Z) Then

            If node.iZone1 = 0 Then
                node.iZone1 = goodZone
                model.Nodes(iNode) = node
            End If

            'End If
        End While
    End Sub

    Private Shared Function AlmostEqual(a As Single, b As Single) As Boolean
        Return Math.Abs(a - b) < 0.01
    End Function

    Private Sub FastLoadToolStripMenuItem_Click(sender As System.Object, e As System.EventArgs) Handles FastLoadToolStripMenuItem.Click
        FastLoading = Not FastLoading
    End Sub

    Private Sub HelpToolStripMenuItem1_Click(sender As System.Object, e As System.EventArgs) Handles HelpToolStripMenuItem1.Click
        MessageBox.Show(My.Resources.HelpText, "Help")
    End Sub

    Private Sub Form1_FormClosing(sender As System.Object, e As System.Windows.Forms.FormClosingEventArgs) Handles MyBase.FormClosing
        Try
            IO.File.WriteAllText("BSPTool.ini", "FastLoading=" & FastLoading)
        Catch
        End Try
    End Sub

End Class

Public Class GLViewControl
    Inherits OpenGLControl

    Public Model As Model
    Public iSelectedNode As Integer = -1

    Public CameraPosition As Vector = New Vector(256, 256, 256)
    Public CameraRotationH As Single
    Public CameraRotationV As Single

    Protected Overrides Sub initGLContext() 'Need to override this method
        GL.glEnable(GLFlags.GL_CULL_FACE) ' dont draw back
        GL.glShadeModel(GLFlags.GL_SMOOTH) 'Smooth Shading
        GL.glClearColor(0.0F, 0.0F, 0.0, 0.5F)               'Black Background
        GL.glClearDepth(1.0F)                                'Depth Buffer Setup
        GL.glEnable(GLFlags.GL_DEPTH_TEST) 'Enable Depth Testing
        GL.glDepthFunc(GLFlags.GL_LEQUAL)  'Type of Depth Testing to do
        'does this do anything?
        GL.glHint(GLFlags.GL_PERSPECTIVE_CORRECTION_HINT, GLFlags.GL_NICEST) 'Nice Perspective Calculations
    End Sub

    Protected Overrides Sub OnSizeChanged(ByVal e As EventArgs)
        MyBase.OnSizeChanged(e)
        Dim S As Size = Me.Size

        GL.glMatrixMode(GLFlags.GL_PROJECTION)
        GL.glLoadIdentity()
        GL.gluPerspective(90, S.Width / S.Height, 1, 65536.0 * Math.Sqrt(3))
        GL.glMatrixMode(GLFlags.GL_MODELVIEW)
        GL.glLoadIdentity()
    End Sub

    Private Sub DrawLine3D(x1 As Single, y1 As Single, z1 As Single, x2 As Single, y2 As Single, z2 As Single, col As Color)
        GL.glBegin(GLFlags.GL_LINES)
        GL.glColor3ub(col.R, col.G, col.B)
        GL.glVertex3f(x1, y1, z1)
        GL.glVertex3f(x2, y2, z2)
        GL.glEnd()
    End Sub

    Private Sub DrawLine3D(v1 As Vector, v2 As Vector, col As Color)
        DrawLine3D(v1.X, v1.Y, v1.Z, v2.X, v2.Y, v2.Z, col)
    End Sub

    Private Sub DrawRadius(rad As Single)
        Const numSegments = 32
        Dim mult = 1 / numSegments * 2 * Math.PI
        GL.glBegin(GL.GL_LINE_LOOP)
        For i = 0 To numSegments
            GL.glVertex3f(Math.Cos(i * mult) * rad, Math.Sin(i * mult) * rad, 0)
        Next
        GL.glEnd()
        GL.glBegin(GL.GL_LINE_LOOP)
        For i = 0 To numSegments
            GL.glVertex3f(Math.Cos(i * mult) * rad, 0, Math.Sin(i * mult) * rad)
        Next
        GL.glEnd()
        GL.glBegin(GL.GL_LINE_LOOP)
        For i = 0 To numSegments
            GL.glVertex3f(0, Math.Cos(i * mult) * rad, Math.Sin(i * mult) * rad)
        Next
        GL.glEnd()
    End Sub

    Sub DrawSprite(size As Single)
        GL.glPushMatrix()
        GL.glRotatef(CameraRotationH, 0.0, 1.0, 0.0)
        GL.glRotatef(CameraRotationV, 1.0, 0.0, 0.0)

        GL.glBegin(GL.GL_QUADS)
        GL.glTexCoord2f(0, 0)
        GL.glVertex3f(-size, -size, 0)
        GL.glTexCoord2f(0, 1)
        GL.glVertex3f(-size, size, 0)
        GL.glTexCoord2f(1, 1)
        GL.glVertex3f(size, size, 0)
        GL.glTexCoord2f(1, 0)
        GL.glVertex3f(size, -size, 0)
        GL.glEnd()

        GL.glPopMatrix()
    End Sub

    Sub DrawPlane(plane As Plane)
        Dim normal = New Vector(plane.X, plane.Y, plane.Z)
        Dim origin = normal * plane.W
        Dim planar1 As Vector
        If normal.X = 0 And normal.Y = 0 Then
            planar1 = Vector.Cross(normal, New Vector(1, 0, 0))
        Else
            planar1 = Vector.Cross(normal, New Vector(0, 0, 1))
        End If
        Dim planar2 = Vector.Cross(normal, planar1)

        ' Arrow
        DrawLine3D(origin, origin + normal * 256, Color.Lime)
        DrawLine3D(origin + normal * 256, origin + normal * 192 + planar1 * 32, Color.Lime)
        DrawLine3D(origin + normal * 256, origin + normal * 192 - planar1 * 32, Color.Lime)
        DrawLine3D(origin + normal * 256, origin + normal * 192 + planar2 * 32, Color.Lime)
        DrawLine3D(origin + normal * 256, origin + normal * 192 - planar2 * 32, Color.Lime)

        ' Grid
        Dim darkGreen = Color.FromArgb(0, 64, 0)
        For i = -16 To 16
            DrawLine3D(origin - planar1 * 32768 + planar2 * i * 2048, origin + planar1 * 32768 + planar2 * i * 2048, darkGreen)
            DrawLine3D(origin - planar2 * 32768 + planar1 * i * 2048, origin + planar2 * 32768 + planar1 * i * 2048, darkGreen)
        Next
    End Sub

    Sub DrawSurfs(node As BspNode, col As Color, children As Boolean)
        GL.glColor3ub(col.R, col.G, col.B)
        GL.glBegin(GL.GL_LINE_LOOP)
        For i = 0 To node.NumVertices - 1
            Dim v = Model.Points(Model.Verts(node.iVertPool + i).pVertex)
            GL.glVertex3f(v.X, v.Y, v.Z)
        Next
        GL.glEnd()

        If children Then
            If node.iChild0 <> -1 Then DrawSurfs(Model.Nodes(node.iChild0), col, True)
            If node.iChild1 <> -1 Then DrawSurfs(Model.Nodes(node.iChild1), col, True)
            If node.iChild2 <> -1 Then DrawSurfs(Model.Nodes(node.iChild2), col, True)
        End If
    End Sub

    Private Sub DrawCube(v1 As Vector, v2 As Vector, col As Color)
        GL.glBegin(GLFlags.GL_LINES)
        GL.glColor3ub(col.R, col.G, col.B)
        GL.glVertex3f(v1.X, v1.Y, v1.Z)
        GL.glVertex3f(v2.X, v1.Y, v1.Z)
        GL.glVertex3f(v1.X, v1.Y, v1.Z)
        GL.glVertex3f(v1.X, v2.Y, v1.Z)
        GL.glVertex3f(v1.X, v1.Y, v1.Z)
        GL.glVertex3f(v1.X, v1.Y, v2.Z)

        GL.glVertex3f(v2.X, v2.Y, v2.Z)
        GL.glVertex3f(v1.X, v2.Y, v2.Z)
        GL.glVertex3f(v2.X, v2.Y, v2.Z)
        GL.glVertex3f(v2.X, v1.Y, v2.Z)
        GL.glVertex3f(v2.X, v2.Y, v2.Z)
        GL.glVertex3f(v2.X, v2.Y, v1.Z)

        GL.glVertex3f(v2.X, v1.Y, v2.Z)
        GL.glVertex3f(v1.X, v1.Y, v2.Z)
        GL.glVertex3f(v1.X, v1.Y, v2.Z)
        GL.glVertex3f(v1.X, v2.Y, v2.Z)

        GL.glVertex3f(v2.X, v1.Y, v1.Z)
        GL.glVertex3f(v2.X, v2.Y, v1.Z)
        GL.glVertex3f(v2.X, v2.Y, v1.Z)
        GL.glVertex3f(v1.X, v2.Y, v1.Z)

        GL.glVertex3f(v2.X, v1.Y, v1.Z)
        GL.glVertex3f(v2.X, v1.Y, v2.Z)

        GL.glVertex3f(v1.X, v2.Y, v1.Z)
        GL.glVertex3f(v1.X, v2.Y, v2.Z)
        GL.glEnd()
    End Sub

    Private Sub DrawBackgroundGrid()
        Dim normal = Color.FromArgb(64, 64, 64)
        Dim dark = Color.FromArgb(48, 48, 48)

        Dim a, b As Vector

        GL.glBindTexture(GL.GL_TEXTURE_2D, 0)

        GL.glColor3b(32, 32, 32)
        GL.glutWireCube(65536)

        For i = -32768 To 32768 Step 1024
            a.X = i
            b.X = i
            a.Y = -32768
            b.Y = 32768
            DrawLine3D(a, b, If(i = 0, normal, dark))
            a.X = -32768
            b.X = 32768
            a.Y = i
            b.Y = i
            DrawLine3D(a, b, If(i = 0, normal, dark))
        Next
    End Sub

    Private Sub SetSelectionCode(code As Integer)
        GL.glColor3ub(code And 255, (code >> 8) And 255, (code >> 16 And 255))
    End Sub

    Public Overrides Sub GLDraw()
        GL.glDisable(GL.GL_BLEND)
        GL.glDepthMask(GL.GL_TRUE)
        GL.glClearColor(0, 0, 0, 1)
        GL.glClear(GLFlags.GL_COLOR_BUFFER_BIT Or GLFlags.GL_DEPTH_BUFFER_BIT)

        GL.glMatrixMode(GLFlags.GL_MODELVIEW)
        GL.glLoadIdentity()

        'GL.glMultMatrixf({1, 0, 0, 0, 0, 1, 0, 0, 0, 0, -1, 0, 0, 0, 0, 1})
        GL.glRotatef(CameraRotationV, 1, 0, 0) 'up/down
        GL.glRotatef(CameraRotationH, 0, 1, 0) 'left/right
        GL.glMultMatrixf({1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1}) 'Use Unreal coords
        GL.glTranslatef(-CameraPosition.X, -CameraPosition.Y, -CameraPosition.Z)

        DrawBackgroundGrid()
        GL.glClear(GLFlags.GL_DEPTH_BUFFER_BIT)
        GL.glEnable(GL.GL_TEXTURE_2D)

        ' Draw stuff...
        If iSelectedNode <> -1 Then
            Dim node = Model.Nodes(iSelectedNode)

            If node.iRenderBound <> -1 Then
                Dim bounds = Model.Bounds(node.iRenderBound)
                DrawCube(bounds.Min, bounds.Max, Color.Red)
            End If

            If node.iCollisionBound <> -1 Then 'TODO: Brush polys?
                Dim bounds As BoundingBox
                Model.GetLeafHull(node.iCollisionBound, Nothing, bounds)
                DrawCube(bounds.Min, bounds.Max, Color.Blue)
            End If

            DrawPlane(node.Plane)

            If node.iChild0 <> -1 Then DrawSurfs(Model.Nodes(node.iChild0), Color.Gray, True)
            If node.iChild1 <> -1 Then DrawSurfs(Model.Nodes(node.iChild1), Color.White, True)
            If node.iChild2 <> -1 Then DrawSurfs(Model.Nodes(node.iChild2), Color.Green, True)
            DrawSurfs(node, Color.Lime, False)
        End If

        GL.glFlush()
    End Sub

End Class

' BSPNode but easier to use. Not sure if should use.
Public Class NiceNode
    Public Plane As Plane
    Public FrontChild As NiceNode
    Public BackChild As NiceNode
    Public iFrontLeaf As Integer
    Public iBackLeaf As Integer
    Public iFrontZone As Byte
    Public iBackZone As Byte
    Public Surfs_iSurf As New List(Of Integer)
    Public Surfs_iVertPool As New List(Of Integer)
    Public Surfs_NumVertices As New List(Of Integer)
End Class

