﻿Public Module Models
    Class Polygon
        Public Texture As Integer
        Public Flags As Integer
        Public Actor As Integer
        Public Item As Integer
        Public Link As Integer
        Public iBrushPoly As Integer

        Public Origin As Vector
        Public Normal As Vector
        Public TextureU As Vector
        Public TextureV As Vector
        Public UPan As Int16
        Public VPan As Int16
        Public VertList As New List(Of Vector)
    End Class

    ' Based on ase2t3d by Daire Stockdale
    Sub ConvertUV(u0 As Double, v0 As Double, u1 As Double, v1 As Double, u2 As Double, v2 As Double, _
                  pt0 As Vector, pt1 As Vector, pt2 As Vector, _
                  ByRef origin As Vector, ByRef textureU As Vector, ByRef textureV As Vector, _
                  Optional texture_width As Double = 128, Optional texture_height As Double = 128)
        Dim dpt1 As Vector = pt1 - pt0
        Dim dpt2 As Vector = pt2 - pt0

        Dim dv1 As New Vector(u1 - u0, v1 - v0, 0.0)
        Dim dv2 As New Vector(u2 - u0, v2 - v0, 0.0)

        Dim dpt11 As Double = Vector.Dot(dpt1, dpt1)
        Dim dpt12 As Double = Vector.Dot(dpt1, dpt2)
        Dim dpt22 As Double = Vector.Dot(dpt2, dpt2)
        Dim factor As Double = 1.0 / (dpt11 * dpt22 - dpt12 * dpt12)

        Dim g1 As Vector = ((dv1 * dpt22) - (dv2 * dpt12)) * factor
        Dim g2 As Vector = ((dv2 * dpt11) - (dv1 * dpt12)) * factor

        Dim p_grad_u As Vector = dpt1 * g1.X + dpt2 * g2.X
        Dim p_grad_v As Vector = dpt1 * g1.Y + dpt2 * g2.Y

        Dim dup1 As Double = Vector.Dot(dpt1, p_grad_u)
        Dim dup2 As Double = Vector.Dot(dpt2, p_grad_u)
        Dim dvp1 As Double = Vector.Dot(dpt1, p_grad_v)
        Dim dvp2 As Double = Vector.Dot(dpt2, p_grad_v)

        Dim divisor As Double = (dup1 * dvp2 - dvp1 * dup2)
        Dim impossible1 As Boolean = Math.Abs(divisor) <= 0.00000000001

        If impossible1 Then divisor = 1.0

        Dim fuctor As Double = 1.0 / divisor

        Dim b1 As Double = (u0 * dvp2 - v0 * dup2) * fuctor
        Dim b2 As Double = (v0 * dup1 - u0 * dvp1) * fuctor

        Dim p_base As Vector = pt0 - ((dpt1 * b1) + (dpt2 * b2))

        p_grad_u *= texture_width
        p_grad_v *= texture_height

        Dim pA As Vector = pt1 - pt0
        Dim pB As Vector = pt2 - pt0

        Dim pN As Vector = Vector.Cross(pA, pB)
        pN = pN.Normalized()

        ' Normals
        Dim dnx As Double = pN.X '-pN.X
        Dim dny As Double = pN.Y
        Dim dnz As Double = pN.Z

        '
        ' Check for error values
        '
        Dim impossible2 As Boolean = Single.IsNaN(p_base.X) Or Single.IsNaN(p_base.Y) Or Single.IsNaN(p_base.Z) Or _
             Single.IsNaN(p_grad_v.X) Or Single.IsNaN(p_grad_v.Y) Or Single.IsNaN(p_grad_v.Z) Or _
             Single.IsNaN(p_grad_v.X) Or Single.IsNaN(p_grad_v.Y) Or Single.IsNaN(p_grad_v.Z)

        Dim impossible As Boolean = impossible1 Or impossible2

        Assert(Not impossible)

        ' If the texture system is inconsistent then use the
        ' default first coord as the origin.
        If impossible Then
            origin.X = pt1.X
            origin.Y = pt1.Y
            origin.Z = pt1.Z
        Else
            origin.X = p_base.X
            origin.Y = p_base.Y
            origin.Z = p_base.Z
        End If

        If Not impossible Then
            textureU.X = p_grad_u.X
            textureU.Y = p_grad_u.Y
            textureU.Z = p_grad_u.Z

            textureV.X = -p_grad_v.X '-p_grad_v.X
            textureV.Y = -p_grad_v.Y '-p_grad_v.Y
            textureV.Z = -p_grad_v.Z '-p_grad_v.Z
        Else
            textureU.X = dnx
            textureU.Y = dny
            textureU.Z = dnz

            textureV.X = dnx
            textureV.Y = dny
            textureV.Z = dnz
        End If
    End Sub

    Class Polys
        Private Pkg As Package
        Public Name As Integer
        Public PolyList As New List(Of Polygon)

        Sub New(pkg As Package)
            Me.Pkg = pkg
        End Sub

        Public Overrides Function ToString() As String
            Dim text As New System.Text.StringBuilder

            text.Append("Begin PolyList Name=").AppendLine(Pkg.GetName(Name))

            For Each p In PolyList
                text.Append("    Begin Polygon")
                If p.Item <> 0 Then text.Append(" Item=").Append(Pkg.GetName(p.Item))
                If p.Texture <> 0 Then text.Append(" Texture=").Append(Pkg.GetFullObjectName(p.Texture, False))
                If p.Flags <> 0 Then text.Append(" Flags=").Append(p.Flags)
                If p.Link <> 0 Then text.Append(" Link=").Append(p.Link)
                text.AppendLine()
                text.Append("        Origin   ").AppendLine(p.Origin.ToT3DString())
                text.Append("        Normal   ").AppendLine(p.Normal.ToT3DString())
                text.Append("        TextureU ").AppendLine(p.TextureU.ToT3DString())
                text.Append("        TextureV ").AppendLine(p.TextureV.ToT3DString())
                If p.UPan <> 0 Or p.VPan <> 0 Then
                    text.AppendLine("        Pan U=" & p.UPan & " V=" & p.VPan)
                End If
                For Each v In p.VertList
                    text.AppendLine("        Vertex   " & v.ToT3DString())
                Next
                text.AppendLine("    End Polygon")
            Next

            text.Append("End PolyList")

            Return text.ToString
        End Function
    End Class

    Class Model
        Public Level As Package
        Public Name As String

        Public PrimitiveBoundingBox As BoundingBox
        Public PrimitiveBoundingSphere As BoundingSphere

        Public Vectors As New List(Of Vector)
        ''' <summary>Only for visuals?</summary>
        Public Points As New List(Of Vector)
        Public Nodes As New List(Of BspNode)
        Public Surfs As New List(Of BspSurf)
        Public Verts As New List(Of Vert)
        Public NumSharedSides As UInteger
        Public Zones As New List(Of Zone)
        Public Polys As Integer
        Public LightMaps As New List(Of LightMap)
        Public LightBits As Byte()
        Public Bounds As New List(Of BoundingBox)

        ''' <summary>
        ''' An int array with some floats.
        ''' iCollisionBound points into this. Keep reading int iBrushPoly's until
        ''' you encounter -1. What follows is vector min, max.
        ''' </summary>
        Public LeafHulls As BinaryBuffer

        ''' <summary>Optional? (didn't test zone behavior)</summary>
        Public Leaves As New List(Of Leaf)
        Public Lights As New List(Of Integer)
        Public RootOutside, Linked As UInt32

        Sub New(myLevel As Package)
            Level = myLevel
        End Sub

        Sub RecalcNodeParents()
            For i = 0 To Nodes.Count - 1
                Dim node = Nodes(i)
                node.iParent = -1
                Nodes(i) = node
            Next

            For i = 0 To Nodes.Count - 1
                Dim node = Nodes(i)

                If node.iChild0 <> -1 Then
                    Dim child0 = Nodes(node.iChild0)
                    child0.iParent = i
                    Nodes(node.iChild0) = child0
                End If

                If node.iChild1 <> -1 Then
                    Dim child1 = Nodes(node.iChild1)
                    child1.iParent = i
                    Nodes(node.iChild1) = child1
                End If

                If node.iChild2 <> -1 Then
                    Dim child2 = Nodes(node.iChild2)
                    child2.iParent = i
                    Nodes(node.iChild2) = child2
                End If
            Next
        End Sub

        Function ToT3D() As String
            Dim str As New System.Text.StringBuilder

            str.AppendLine("Begin Brush Name=" & Name)
            str.AppendLine("Begin PolyList")

            For Each n In Nodes
                If n.NumVertices > 0 Then
                    Dim surf = Surfs(n.iSurf)

                    str.Append("Begin Polygon")
                    If surf.Texture <> 0 Then str.Append(" Texture=" & Level.GetFullObjectName(surf.Texture, False))
                    If surf.PolyFlags <> 0 Then str.Append(" Flags=" & surf.PolyFlags)
                    str.AppendLine()

                    str.AppendLine("Origin " & Points(surf.pBase).ToT3DString())
                    str.AppendLine("Normal " & Vectors(surf.vNormal).ToT3DString())
                    str.AppendLine("TextureU " & Vectors(surf.vTextureU).ToT3DString())
                    str.AppendLine("TextureV " & Vectors(surf.vTextureV).ToT3DString())
                    If surf.PanU <> 0 Or surf.PanV <> 0 Then
                        str.AppendLine("Pan U=" & surf.PanU & " V=" & surf.PanV)
                    End If

                    For i = n.iVertPool To n.iVertPool + n.NumVertices - 1
                        str.AppendLine("Vertex " & Points(Verts(i).pVertex).ToT3DString)
                    Next

                    str.AppendLine("End Polygon")
                End If
            Next

            str.AppendLine("End PolyList")
            str.AppendLine("End Brush")

            Return str.ToString()
        End Function

        Sub GetLeafHull(iCollisionBound As Integer, ByRef iNodes As List(Of Integer), ByRef bounds As BoundingBox)
            LeafHulls.Position = iCollisionBound * 4

            iNodes = New List(Of Integer)

            Do
                Dim num = LeafHulls.R.ReadInt32
                If num <> -1 Then
                    iNodes.Add(num)
                Else
                    Exit Do
                End If
            Loop

            bounds.Min = LeafHulls.R.ReadVector
            bounds.Max = LeafHulls.R.ReadVector
            bounds.IsValid = (bounds.Min.X <= bounds.Max.X And bounds.Min.Y <= bounds.Max.Y And bounds.Min.Z <= bounds.Max.Z)
         End Sub

        Function AddLeafHull(iNodes As List(Of Integer), bounds As BoundingBox) As Integer
            Dim oldLength = LeafHulls.Size
            
            LeafHulls.Position = oldLength

            For Each iNode In iNodes
                LeafHulls.W.Write(iNode)
            Next
            LeafHulls.W.Write(-1)

            LeafHulls.W.WriteVector(bounds.Min)
            LeafHulls.W.WriteVector(bounds.Max)


            Return oldLength / 4
        End Function

        ''' <summary>
        ''' Make sure you don't exceed original node count, otherwise you'll overwrite
        ''' the next leaf hull.
        ''' </summary>
        Sub SetLeafHull(iCollisionBound As Integer, iNodes As List(Of Integer), bounds As BoundingBox)
            LeafHulls.Position = iCollisionBound * 4

            For Each iNode In iNodes
                LeafHulls.W.Write(iNode)
            Next
            LeafHulls.W.Write(-1)

            LeafHulls.W.WriteVector(bounds.Min)
            LeafHulls.W.WriteVector(bounds.Max)
        End Sub

        Sub RecalcNodeCollision(iNode As Integer)
            Dim node = Nodes(iNode)
            node.iRenderBound = -1

            If node.NumVertices <> 0 Then
                ' Get list of parents
                Dim hierarchy As New List(Of Integer)
                Dim iCurNode = iNode
                Do
                    hierarchy.Insert(0, iCurNode)
                    iCurNode = Nodes(iCurNode).iParent
                Loop Until iCurNode = -1

                ' Set front bit
                For j = 0 To hierarchy.Count - 2
                    If Nodes(hierarchy(j)).iChild1 = hierarchy(j + 1) Then
                        hierarchy(j) = hierarchy(j) Or &H40000000
                    End If
                Next

                Dim bbox As BoundingBox
                bbox.Min = New Vector(-32768, -32768, -32768) 'TODO: Make more accurate
                bbox.Max = New Vector(32768, 32768, 32768)
                node.iCollisionBound = AddLeafHull(hierarchy, bbox)
            Else
                node.iCollisionBound = -1
            End If

            Nodes(iNode) = node
        End Sub

        ''' <summary>
        ''' Works like in 227, with one difference -- it returns 0 instead of LevelInfo.
        ''' </summary>
        Function GetLocZone(pos As Vector, Optional ByRef iNode As Integer = -1, Optional ByRef nodeFront As Boolean = False) As PointRegion
            Dim pr As PointRegion
            Dim iCurNode = 0
            pr.ZoneNumber = 0
goDeeper:
            Dim node = Nodes(iCurNode)
            If node.Plane.TestPoint(pos) > 0 Then
                If node.iChild1 <> -1 Then
                    iCurNode = node.iChild1
                    GoTo goDeeper
                Else
                    pr.iLeaf = node.iLeaf1
                    pr.ZoneNumber = node.iZone1
                    iNode = iCurNode
                    nodeFront = True
                End If
            Else
                If node.iChild0 <> -1 Then
                    iCurNode = node.iChild0
                    GoTo goDeeper
                Else
                    pr.iLeaf = node.iLeaf0
                    pr.ZoneNumber = node.iZone0
                    iNode = iCurNode
                    nodeFront = False
                End If
            End If

            If pr.ZoneNumber > 0 And pr.ZoneNumber <> 63 Then
                pr.Zone = Zones(pr.ZoneNumber).ZoneActor
            End If

            Return pr
        End Function
    End Class

    'TODO: Replace XYZ with Vector. This would break compat so goes into UPLib2
    Structure Plane
        Dim X, Y, Z, W As Single

        ''' <summary>
        ''' Returns 0 if point is on plane, &gt;0 if in front of plane, &lt;0 if behind.
        ''' </summary>
        Function TestPoint(pt As Vector) As Single
            Return X * pt.X + Y * pt.Y + Z * pt.Z - W
        End Function
    End Structure

    Structure BspNode
        Dim Plane As Plane
        Dim ZoneMask As ULong
        Dim NodeFlags As Byte
        Dim iVertPool As Integer
        Dim iSurf As Integer
        ''' <summary>Back child node.</summary>
        Dim iChild0 As Integer
        ''' <summary>Front child node.</summary>
        Dim iChild1 As Integer
        ''' <summary>Coplanars. Nodes referenced in iChild2 have no iChild 0 &amp; 1.</summary>
        Dim iChild2 As Integer
        Dim iZone0, iZone1 As Byte
        Dim NumVertices As Byte
        Dim iLeaf0, iLeaf1 As Integer
        ''' <summary>Index into LeafHulls. Read this using GetLeafHull.
        ''' 
        ''' NOT optional, without this the player will fall through everything
        ''' (although for some reason the projectiles still work).
        ''' 
        ''' Some nodes with solid surfaces don't have this. It's unknown why -- perhaps
        ''' because it's redundant?
        ''' 
        ''' A collision bound (a.k.a. leaf hull) consists of a list of node indices and
        ''' a bounding box. These nodes are always a subset of this node and its parents,
        ''' in order from parentmost. It's unknown, which nodes should be omitted.
        ''' There's probably a bigger picture here that we can't see by just looking at
        ''' the numbers.
        ''' 
        ''' Some node indices are ORed with 0x40000000. This means that the next node
        ''' (in BSP, NOT in this incomplete list) is its front child node.
        ''' 
        ''' The bounding box is probably the bounding box of the convex polyhedron
        ''' made up by the nodes's planes. Sometimes a part of it may extend infinitely.
        ''' Root node's b. box fills the entire space.
        ''' </summary>
        Dim iCollisionBound As Integer
        ''' <summary>Index into Bounds. Bounding box of given node's (child) polygons.
        ''' Sometimes, for unknown reason, includes parent polygons.
        ''' Not seen in childmost nodes.</summary>
        Dim iRenderBound As Integer

        Dim iParent As Integer 'temporary -- remove later!
    End Structure

    Structure BspSurf
        Dim Texture, PolyFlags As Integer
        Dim pBase, vNormal, vTextureU, vTextureV As Integer
        Dim iLightMap, iBrushPoly As Integer
        Dim PanU, PanV As Short
        Dim Actor As Integer
    End Structure

    Structure Vert
        Dim pVertex, iSide As Integer

        Sub New(pVertex As Integer, iSide As Integer)
            Me.pVertex = pVertex
            Me.iSide = iSide
        End Sub
    End Structure

    Structure LightMap
        Dim DataOffset As Integer
        Dim Pan As Vector
        Dim UClamp, VClamp As Integer
        Dim UScale, VScale As Single
        Dim iLightActors As Integer
    End Structure

    Function GetPolys(obj As ExportTableItem) As Polys
        obj.Data.Position = 0
        obj.SkipStack()
        obj.GetProperties()

        Dim result As New Polys(obj.Pkg)

        result.Name = obj.ObjectName

        Dim NumSides = obj.Data.R.ReadUInt32()
        obj.Data.R.ReadUInt32()

        For s = 0 To NumSides - 1
            Dim poly As New Polygon
            Dim NumVerts = obj.Data.R.ReadByte()

            poly.Origin = obj.Data.R.ReadVector()
            poly.Normal = obj.Data.R.ReadVector()
            poly.TextureU = obj.Data.R.ReadVector()
            poly.TextureV = obj.Data.R.ReadVector()
            For v = 0 To NumVerts - 1
                poly.VertList.Add(obj.Data.R.ReadVector())
            Next
            poly.Flags = obj.Data.R.ReadInt32()
            If obj.Pkg.PackageVersion = 69 AndAlso (obj.Pkg.LicenseeMode = 63 Or obj.Pkg.LicenseeMode = 31) Then
                obj.Data.R.ReadInt32() 'Extended PolyFlags, found in some .un2 files
            End If
            poly.Actor = obj.Data.R.ReadCompact()
            poly.Texture = obj.Data.R.ReadCompact()
            poly.Item = obj.Data.R.ReadCompact()
            poly.Link = obj.Data.R.ReadCompact()
            poly.iBrushPoly = obj.Data.R.ReadCompact()
            poly.UPan = obj.Data.R.ReadInt16()
            poly.VPan = obj.Data.R.ReadInt16()

            result.PolyList.Add(poly)
        Next

        Return result
    End Function

    ' These are copied from NuPlayer's UnExtract:
    Function FindLevelModel(pkg As Package) As ExportTableItem
        ' This isn't perfect, but should work most of the time.
        Dim candidates As New List(Of ExportTableItem)

        For Each item In pkg.ExportTable
            If StrEq(item.ClassNameStr, "Model") Then
                If (item.ObjectFlags = (EObjectFlags.RF_Transactional Or EObjectFlags.RF_LoadForClient Or EObjectFlags.RF_LoadForServer Or EObjectFlags.RF_LoadForEdit)) Then
                    If StrEq(item.ObjectNameStr, "Model1") Then
                        Return item
                    End If
                    candidates.Add(item)
                End If
            End If
        Next

        ' Try another heuristic...
        Dim maxSize = 0
        Dim maxItem As ExportTableItem = Nothing
        For i = 0 To candidates.Count - 1
            If candidates(i).OriginalSize > maxSize Then
                maxSize = candidates(i).OriginalSize
                maxItem = candidates(i)
            End If
        Next

        Return maxItem
    End Function

    ' Only for >61 package versions.
    Function GetModel(mdl As ExportTableItem) As Model
        Dim pkg = mdl.Pkg
        If pkg.PackageVersion <= 61 Then
            Throw New Exception("The level model uses the old format. Resave the map in a later engine build, then try again.")
            Return Nothing
        End If

        Dim r = mdl.Data.R

        Dim result As New Model(mdl.Pkg)
        result.Name = mdl.ObjectNameStr

        r.BaseStream.Position = 0
        mdl.SkipStack()
        r.ReadProperties()

        result.PrimitiveBoundingBox.Min = r.ReadVector
        result.PrimitiveBoundingBox.Max = r.ReadVector
        result.PrimitiveBoundingBox.IsValid = r.ReadByte

        result.PrimitiveBoundingSphere.Center = r.ReadVector
        result.PrimitiveBoundingSphere.Radius = r.ReadSingle

        Dim NumVectors = r.ReadCompact()
        For i = 0 To NumVectors - 1
            result.Vectors.Add(r.ReadVector)
        Next

        Dim NumPoints = r.ReadCompact()
        For i = 0 To NumPoints - 1
            result.Points.Add(r.ReadVector)
        Next

        Dim NumNodes = r.ReadCompact
        For i = 0 To NumNodes - 1
            Dim node As New BspNode

            'If pkg.PackageVersion < 129 Then
            If pkg.PackageVersion < 128 Then 'FIXME: Exact version?
                node.Plane.X = r.ReadSingle
                node.Plane.Y = r.ReadSingle
                node.Plane.Z = r.ReadSingle
                node.Plane.W = r.ReadSingle

                node.ZoneMask = r.ReadUInt64
                node.NodeFlags = r.ReadByte

                node.iVertPool = r.ReadCompact
                node.iSurf = r.ReadCompact

                node.iChild0 = r.ReadCompact
                node.iChild1 = r.ReadCompact
                node.iChild2 = r.ReadCompact
                node.iCollisionBound = r.ReadCompact
                node.iRenderBound = r.ReadCompact

                node.iZone0 = r.ReadByte
                node.iZone1 = r.ReadByte

                node.NumVertices = r.ReadByte

                node.iLeaf0 = r.ReadInt32
                node.iLeaf1 = r.ReadInt32
            Else
                node.Plane.X = r.ReadSingle
                node.Plane.Y = r.ReadSingle
                node.Plane.Z = r.ReadSingle
                node.Plane.W = r.ReadSingle

                node.ZoneMask = r.ReadUInt64
                node.NodeFlags = r.ReadByte

                node.iVertPool = r.ReadCompact
                node.iSurf = r.ReadCompact
                Assert(node.iSurf >= -1 And node.iSurf < 32768)

                node.iChild0 = r.ReadCompact
                node.iChild1 = r.ReadCompact
                node.iChild2 = r.ReadCompact
                node.iCollisionBound = r.ReadCompact
                node.iRenderBound = r.ReadCompact
                Assert(node.iChild0 >= -1 And node.iChild0 < NumNodes)
                Assert(node.iChild1 >= -1 And node.iChild1 < NumNodes)
                Assert(node.iChild2 >= -1 And node.iChild2 < NumNodes)

                r.BaseStream.Position += 4 * 4 'FSphere ExclusiveSphereBound
                If pkg.PackageVersion < 134 Then
                    r.BaseStream.Position += 4 * 4 'FPlane DeadSphere
                End If

                node.iZone0 = r.ReadByte
                node.iZone1 = r.ReadByte
                Assert(node.iZone0 >= -1 And node.iZone0 <= 63)
                Assert(node.iZone1 >= -1 And node.iZone1 <= 63)

                node.NumVertices = r.ReadByte
                Assert(node.NumVertices = 0 Or (node.NumVertices >= 3 And node.NumVertices <= 32))

                node.iLeaf0 = r.ReadInt32
                node.iLeaf1 = r.ReadInt32
                Assert(node.iLeaf0 >= -1 And node.iLeaf0 < 32768)
                Assert(node.iLeaf1 >= -1 And node.iLeaf1 < 32768)

                r.ReadInt32() 'iSection
                r.ReadInt32() 'iFirstVertex
                r.ReadInt32() 'iLightmap
            End If

                result.Nodes.Add(node)
        Next

        result.RecalcNodeParents()

        Dim NumSurfs = r.ReadCompact
        For i = 0 To NumSurfs - 1
            Dim surf As New BspSurf

            surf.Texture = r.ReadCompact
            surf.PolyFlags = r.ReadInt32
            surf.pBase = r.ReadCompact
            surf.vNormal = r.ReadCompact
            surf.vTextureU = r.ReadCompact
            surf.vTextureV = r.ReadCompact
            'If pkg.PackageVersion < 129 Then
            If pkg.PackageVersion < 128 Then 'FIXME: Exact version?
                surf.iLightMap = r.ReadCompact
            Else
                surf.iLightMap = -1
            End If
            surf.iBrushPoly = r.ReadCompact

            'If pkg.PackageVersion < 129 Then
            If pkg.PackageVersion < 128 Then 'FIXME: Exact version?
                surf.PanU = r.ReadInt16
                surf.PanV = r.ReadInt16
                surf.Actor = r.ReadCompact
            Else
                surf.Actor = r.ReadCompact
                r.BaseStream.Position += 4 * 4 'FPlane Plane
                r.ReadSingle() 'LightMapScale
            End If

            result.Surfs.Add(surf)
        Next

        Dim NumVerts = r.ReadCompact()
        For i = 0 To NumVerts - 1
            Dim vert As New Vert
            vert.pVertex = r.ReadCompact
            vert.iSide = r.ReadCompact
            result.Verts.Add(vert)
        Next

        result.NumSharedSides = r.ReadUInt32() 'usually 0?

        Dim NumZones = r.ReadUInt32
        For i = 0 To NumZones - 1
            Dim zone As Zone
            zone.ZoneActor = r.ReadCompact
            zone.Connectivity = r.ReadUInt64
            zone.Visibility = r.ReadUInt64
            result.Zones.Add(zone)
        Next

        result.Polys = r.ReadCompact

        'Try 'At least we have Polys. The rest isn't as important.
        'If pkg.PackageVersion < 129 Then
        If pkg.PackageVersion < 128 Then 'FIXME: Exact version?
            Dim NumLightMaps = r.ReadCompact
            For i = 0 To NumLightMaps - 1
                Dim lmap As New LightMap
                lmap.DataOffset = r.ReadInt32
                lmap.Pan = r.ReadVector
                lmap.UClamp = r.ReadCompact
                lmap.VClamp = r.ReadCompact
                lmap.UScale = r.ReadSingle
                lmap.VScale = r.ReadSingle
                lmap.iLightActors = r.ReadInt32
                result.LightMaps.Add(lmap)
            Next

            Dim NumLightbitBytes = r.ReadCompact
            result.LightBits = r.ReadBytes(NumLightbitBytes)
        End If

        Dim NumBounds = r.ReadCompact
        For i = 0 To NumBounds - 1
            Dim bb As BoundingBox
            bb.Min = r.ReadVector
            bb.Max = r.ReadVector
            bb.IsValid = r.ReadByte
            result.Bounds.Add(bb)
        Next

        Dim NumLeafHulls = r.ReadCompact
        result.LeafHulls = New BinaryBuffer(r.ReadBytes(NumLeafHulls * 4))

        Dim Leaves = r.ReadCompact
        For i = 0 To Leaves - 1
            Dim leaf As Leaf
            leaf.iZone = r.ReadCompact
            leaf.iPermeating = r.ReadCompact
            leaf.iVolumetric = r.ReadCompact
            leaf.VisibleZones = r.ReadUInt64
            result.Leaves.Add(leaf)
        Next

        Dim NumLights = r.ReadCompact
        For i = 0 To NumLights - 1
            result.Lights.Add(r.ReadCompact)
        Next

        result.RootOutside = r.ReadUInt32
        result.Linked = r.ReadUInt32
        'Catch ex As Exception
        'End Try

        Return result
    End Function

    Sub SetModel(obj As ExportTableItem, mdl As Model)
        obj.ObjectFlags = obj.ObjectFlags And Not EObjectFlags.RF_HasStack

        obj.Data.Clear()

        Dim w = obj.Data.W
        w.BaseStream.Position = 0
        obj.Data.W.WriteCompact(obj.Pkg.AddName("None")) 'props

        w.WriteVector(mdl.PrimitiveBoundingBox.Min)
        w.WriteVector(mdl.PrimitiveBoundingBox.Max)
        w.WriteByte(mdl.PrimitiveBoundingBox.IsValid)

        w.WriteVector(mdl.PrimitiveBoundingSphere.Center)
        w.WriteSingle(mdl.PrimitiveBoundingSphere.Radius)

        w.WriteCompact(mdl.Vectors.Count)
        For Each v In mdl.Vectors
            w.WriteVector(v)
        Next

        w.WriteCompact(mdl.Points.Count)
        For Each p In mdl.Points
            w.WriteVector(p)
        Next

        w.WriteCompact(mdl.Nodes.Count)
        For Each node In mdl.Nodes
            w.Write(node.Plane.X)
            w.Write(node.Plane.Y)
            w.Write(node.Plane.Z)
            w.Write(node.Plane.W)

            w.Write(node.ZoneMask)
            w.Write(node.NodeFlags)

            w.WriteCompact(node.iVertPool)
            w.WriteCompact(node.iSurf)

            w.WriteCompact(node.iChild0)
            w.WriteCompact(node.iChild1)
            w.WriteCompact(node.iChild2)
            w.WriteCompact(node.iCollisionBound)
            w.WriteCompact(node.iRenderBound)

            w.Write(node.iZone0)
            w.Write(node.iZone1)

            w.Write(node.NumVertices)

            w.Write(node.iLeaf0)
            w.Write(node.iLeaf1)
        Next

        w.WriteCompact(mdl.Surfs.Count)
        For Each surf In mdl.Surfs
            w.WriteCompact(surf.Texture)
            w.Write(surf.PolyFlags)
            w.WriteCompact(surf.pBase)
            w.WriteCompact(surf.vNormal)
            w.WriteCompact(surf.vTextureU)
            w.WriteCompact(surf.vTextureV)
            w.WriteCompact(surf.iLightMap)
            w.WriteCompact(surf.iBrushPoly)
            w.Write(surf.PanU)
            w.Write(surf.PanV)
            w.WriteCompact(surf.Actor)
        Next

        w.WriteCompact(mdl.Verts.Count)
        For Each vert In mdl.Verts
            w.WriteCompact(vert.pVertex)
            w.WriteCompact(vert.iSide)
        Next

        w.Write(mdl.NumSharedSides)

        w.WriteUInt32(mdl.Zones.Count)
        For Each zone In mdl.Zones
            w.WriteCompact(zone.ZoneActor)
            w.Write(zone.Connectivity)
            w.Write(zone.Visibility)
        Next

        w.WriteCompact(mdl.Polys)

        w.WriteCompact(mdl.LightMaps.Count)
        For Each lmap In mdl.LightMaps
            w.Write(lmap.DataOffset)
            w.WriteVector(lmap.Pan)
            w.WriteCompact(lmap.UClamp)
            w.WriteCompact(lmap.VClamp)
            w.Write(lmap.UScale)
            w.Write(lmap.VScale)
            w.Write(lmap.iLightActors)
        Next

        w.WriteCompact(mdl.LightBits.Length)
        w.Write(mdl.LightBits)

        w.WriteCompact(mdl.Bounds.Count)
        For Each bb In mdl.Bounds
            w.WriteVector(bb.Min)
            w.WriteVector(bb.Max)
            w.WriteByte(bb.IsValid)
        Next

        w.WriteCompact(mdl.LeafHulls.Size / 4)
        w.Write(mdl.LeafHulls.ToArray)

        w.WriteCompact(mdl.Leaves.Count)
        For Each leaf In mdl.Leaves
            w.WriteCompact(leaf.iZone)
            w.WriteCompact(leaf.iPermeating)
            w.WriteCompact(leaf.iVolumetric)
            w.Write(leaf.VisibleZones)
        Next

        w.WriteCompact(mdl.Lights.Count)
        For Each light In mdl.Lights
            w.WriteCompact(light)
        Next

        w.Write(mdl.RootOutside)
        w.Write(mdl.Linked)
    End Sub

End Module

Public Structure BoundingBox
    Dim Min As Vector
    Dim Max As Vector
    Dim IsValid As Boolean

    Sub New(min As Vector, max As Vector)
        Me.Min = min
        Me.Max = max
    End Sub

    Sub Extend(pt As Vector)
        If Not IsValid Then
            Min = pt
            Max = pt
            IsValid = True
        Else
            If pt.X < Min.X Then Min.X = pt.X
            If pt.Y < Min.Y Then Min.Y = pt.Y
            If pt.Z < Min.Z Then Min.Z = pt.Z
            If pt.X > Max.X Then Max.X = pt.X
            If pt.Y > Max.Y Then Max.Y = pt.Y
            If pt.Z > Max.Z Then Max.Z = pt.Z
        End If
    End Sub

    Sub Extend(bounds As BoundingBox)
        If bounds.IsValid Then
            Extend(bounds.Min)
            Extend(bounds.Max)
        End If
    End Sub

    Function PointInside(pt As Vector) As Boolean
        If Not IsValid Then Return False
        If pt.X < Min.X Then Return False
        If pt.Y < Min.Y Then Return False
        If pt.Z < Min.Z Then Return False
        If pt.X > Max.X Then Return False
        If pt.Y > Max.Y Then Return False
        If pt.Z > Max.Z Then Return False
        Return True
    End Function
End Structure

Public Structure BoundingSphere
    Dim Center As Vector
    Dim Radius As Single
End Structure

Public Structure Leaf
    Dim iZone As Integer
    Dim iPermeating As Integer
    Dim iVolumetric As Integer
    Dim VisibleZones As UInt64
End Structure

Public Structure Zone
    Dim ZoneActor As Integer
    Dim Connectivity As UInt64
    Dim Visibility As UInt64
End Structure

Public Class StaticMesh
    Public Pkg As Package
    Public Name As String

    Public Properties As Properties
    Public Materials As New List(Of String)

    Public PrimitiveBoundingBox As BoundingBox
    Public PrimitiveBoundingSphere As BoundingSphere

    Public Sections As New List(Of StaticMeshSection)

    Public CompressionScale, CompressionInvScale As Vector
    Public UseTriStrips, OldUseTriStrips, bCacheOptimize, OldbCacheOptimize As Boolean
    Public CompressionOffset As Vector

    Public VertexStream As New StaticMeshVertexStream
    Public ColorStream As New RawColorStream
    Public AlphaStream As New RawColorStream
    Public UVStreams As New List(Of StaticMeshUVStream)
    Public IndexBuffer As New RawIndexBuffer
    Public WireframeIndexBuffer As New RawIndexBuffer

    Public CollisionModel As Integer
    Public kDOPTree As New kDOPTree
    Public RawTriangles As New List(Of StaticMeshTriangle)

    ''' <summary>Is value of BoundingVolume suspicious?</summary>
    Private Shared Function IsBVNumWeird(num As Single) As Boolean
        ' For some reason I can't just fucking write If num = 3.40282347E+38
        If Math.Abs(Math.Abs(num) - 3.40282347E+38) < 1.0E+32 Then Return False
        Return IsNumWeird(num)
    End Function

    Private Function UC2004PropHint(prop As ObjectProperty, ByRef result As String) As Boolean
        If prop.Type = EPropertyType.ArrayProperty Then
            If prop.NameStr = "Materials" Then
                result = "("
                Dim buff = New BinaryBuffer(prop.RawValue)
                buff.R.Pkg = prop.Props.Pkg
                Dim count = buff.R.ReadCompact()
                For i = 0 To count - 1
                    result &= buff.R.ReadProperties().ToSingleLine(AddressOf UC2004PropHint)
                    If i < count - 1 Then result &= ","
                Next
                result &= ")"

                Return True
            End If
        ElseIf prop.Type = EPropertyType.StructProperty Then
            Select Case Pkg.GetName(prop.StructName)
                Case "Vector", "Rotator", "Color"
                    Return False
                Case Else
                    ' Hierarchical structs
                    Dim buff = New BinaryBuffer(prop.RawValue)
                    buff.R.Pkg = prop.Props.Pkg
                    Dim innerProps = buff.R.ReadProperties()
                    result = innerProps.ToSingleLine(AddressOf UC2004PropHint)

                    Return True
            End Select
        End If
        Return False
    End Function

    Sub New(exp As ExportTableItem)
        Dim pkg = exp.Pkg

        Dim r = exp.Data.R

        Name = exp.ObjectNameStr

        r.BaseStream.Position = 0
        exp.SkipStack()
        Properties = r.ReadProperties()

        Dim iPropMat = Properties.FindProp("Materials")
        If iPropMat <> -1 Then
            Dim matsStr = Properties.GetValueStr(iPropMat, AddressOf UC2004PropHint)

            Dim curPos = 0
            Do
                Dim ind1 = matsStr.IndexOf("Material=", curPos)
                If ind1 = -1 Then Exit Do
                Dim ind2 = matsStr.IndexOf(")", ind1)
                Dim fullName = matsStr.Substring(ind1 + 9, ind2 - ind1 - 9)
                If StrEq(fullName, "None") Then
                    Materials.Add("None")
                Else
                    Dim iQuote1 = fullName.IndexOf("'")
                    Dim iQuote2 = fullName.LastIndexOf("'")
                    Assert(iQuote1 <> -1)
                    Assert(iQuote2 <> -1)
                    Materials.Add(fullName.Substring(iQuote1 + 1, iQuote2 - iQuote1 - 1))
                End If

                curPos = ind2 + 1
            Loop
        End If

        PrimitiveBoundingBox.Min = r.ReadVector
        PrimitiveBoundingBox.Max = r.ReadVector
        Assert(Not PrimitiveBoundingBox.Min.IsWeird)
        Assert(Not PrimitiveBoundingBox.Max.IsWeird)
        PrimitiveBoundingBox.IsValid = EnsureBool(r.ReadByte)

        PrimitiveBoundingSphere.Center = r.ReadVector
        PrimitiveBoundingSphere.Radius = r.ReadSingle
        Assert(Not PrimitiveBoundingSphere.Center.IsWeird)
        Assert(Not IsNumWeird(PrimitiveBoundingSphere.Radius))

        Dim numSections = r.ReadCompact
        For i = 0 To numSections - 1
            Dim sec As StaticMeshSection
            sec.IsStrip = EnsureBool(r.ReadInt32) '???
            sec.FirstIndex = r.ReadUInt16
            sec.MinVertexIndex = r.ReadUInt16
            sec.MaxVertexIndex = r.ReadUInt16
            sec.NumTriangles = r.ReadUInt16
            sec.NumPrimitives = r.ReadUInt16
            Assert(sec.FirstIndex >= 0)
            Assert(sec.MinVertexIndex >= 0)
            Assert(sec.MaxVertexIndex >= 0)
            Assert(sec.MaxVertexIndex >= sec.MinVertexIndex)
            Assert(sec.NumTriangles >= 0)
            Assert(sec.NumPrimitives >= 0)
            Sections.Add(sec)
        Next

        'BoundingBox again?
        Dim UnusedBoxMin = r.ReadVector()
        Dim UnusedBoxMax = r.ReadVector()
        Assert(Not UnusedBoxMin.IsWeird)
        Assert(Not UnusedBoxMax.IsWeird)
        Assert(UnusedBoxMin = PrimitiveBoundingBox.Min)
        Assert(UnusedBoxMax = PrimitiveBoundingBox.Max)
        Dim UnusedBoxValid = EnsureBool(r.ReadByte())

        If pkg.PackageVersion >= 135 Then
            CompressionScale = r.ReadVector
            CompressionInvScale = r.ReadVector
            If pkg.PackageVersion >= 137 Then 'TODO: Exact version?
                UseTriStrips = EnsureBool(r.ReadInt32)
                OldUseTriStrips = EnsureBool(r.ReadInt32)
                bCacheOptimize = EnsureBool(r.ReadInt32)
                OldbCacheOptimize = EnsureBool(r.ReadInt32)
            Else
                UseTriStrips = EnsureBool(r.ReadByte)
                OldUseTriStrips = EnsureBool(r.ReadByte)
                bCacheOptimize = EnsureBool(r.ReadByte)
                OldbCacheOptimize = EnsureBool(r.ReadByte)
            End If
            Assert(Not CompressionScale.IsWeird)
            Assert(Not CompressionInvScale.IsWeird)
            If pkg.PackageVersion >= 137 Then
                CompressionOffset = r.ReadVector
                Assert(Not CompressionOffset.IsWeird)
            End If
        Else
            CompressionScale = New Vector(1, 1, 1)
            CompressionInvScale = New Vector(1, 1, 1)
        End If

        If pkg.PackageVersion >= 112 Then
            Dim numVertexStreamVerts = r.ReadCompact
            For i = 0 To numVertexStreamVerts - 1
                Dim vert As StaticMeshVertex
                vert.Position = r.ReadVector
                vert.Normal = r.ReadVector
                Assert(Not vert.Position.IsWeird)
                Assert(Not vert.Normal.IsWeird)
                VertexStream.Vertices.Add(vert)
            Next
            VertexStream.Revision = r.ReadInt32
            Assert(VertexStream.Revision >= 0 And VertexStream.Revision < 200) 'temp

            'ColorStream
            Dim numColorStreamColors = r.ReadCompact
            Assert(numColorStreamColors >= 0 And numColorStreamColors < 99999)
            Assert(numColorStreamColors = VertexStream.Vertices.Count)
            For i = 0 To numColorStreamColors - 1
                ColorStream.Colors.Add(r.ReadColor)
            Next
            ColorStream.Revision = r.ReadInt32
            Assert(ColorStream.Revision >= 0 And ColorStream.Revision < 200) 'temp

            'AlphaStream
            Dim numAlphaStreamColors = r.ReadCompact
            Assert(numAlphaStreamColors >= 0 And numAlphaStreamColors < 99999)
            Assert(numAlphaStreamColors = VertexStream.Vertices.Count)
            For i = 0 To numAlphaStreamColors - 1
                AlphaStream.Colors.Add(r.ReadColor)
            Next
            AlphaStream.Revision = r.ReadInt32
            Assert(AlphaStream.Revision >= 0 And AlphaStream.Revision < 200) 'temp

            'UVStreams
            Dim numUVStreams = r.ReadCompact
            Assert(numUVStreams >= 0 And numUVStreams < 99999)
            For i = 0 To numUVStreams - 1
                Dim stream As New StaticMeshUVStream
                Dim numUVs = r.ReadCompact
                For j = 0 To numUVs - 1
                    Dim uv As StaticMeshUV
                    uv.U = r.ReadSingle
                    uv.V = r.ReadSingle
                    Assert(Not IsNumWeird(uv.U))
                    Assert(Not IsNumWeird(uv.V))
                    stream.UVs.Add(uv)
                Next
                stream.CoordinateIndex = r.ReadInt32
                stream.Revision = r.ReadInt32
                Assert(stream.Revision >= 0 And stream.Revision < 200) 'temp
                UVStreams.Add(stream)
            Next

            'IndexBuffer
            Dim numIndexBufferIndices = r.ReadCompact
            Assert(numIndexBufferIndices >= 0 And numIndexBufferIndices < 99999)
            For i = 0 To numIndexBufferIndices - 1
                IndexBuffer.Indices.Add(r.ReadUInt16)
            Next
            IndexBuffer.Revision = r.ReadInt32
            Assert(IndexBuffer.Revision >= 0 And IndexBuffer.Revision < 200) 'temp

            'WireframeIndexBuffer
            Dim numWireframeIndexBufferIndices = r.ReadCompact
            Assert(numWireframeIndexBufferIndices >= 0 And numWireframeIndexBufferIndices < 99999)
            For i = 0 To numWireframeIndexBufferIndices - 1
                WireframeIndexBuffer.Indices.Add(r.ReadUInt16)
            Next
            WireframeIndexBuffer.Revision = r.ReadInt32
            Assert(WireframeIndexBuffer.Revision >= 0 And WireframeIndexBuffer.Revision < 200) 'temp

            'CollisionModel
            CollisionModel = r.ReadCompact
            Assert(CollisionModel = 0 OrElse StrEq(pkg.ExportTable(CollisionModel - 1).ClassNameStr, "Model"))

            'kDOPTree
            Dim numkDOPNodes = r.ReadCompact
            Assert(numkDOPNodes >= 0 And numkDOPNodes < 99999)
            For i = 0 To numkDOPNodes - 1
                Dim node As New kDOPNode

                node.BoundingVolume.Min0 = r.ReadSingle
                node.BoundingVolume.Min1 = r.ReadSingle
                node.BoundingVolume.Min2 = r.ReadSingle
                node.BoundingVolume.Max0 = r.ReadSingle
                node.BoundingVolume.Max1 = r.ReadSingle
                node.BoundingVolume.Max2 = r.ReadSingle
                Assert(Not IsBVNumWeird(node.BoundingVolume.Min0))
                Assert(Not IsBVNumWeird(node.BoundingVolume.Min1))
                Assert(Not IsBVNumWeird(node.BoundingVolume.Min2))
                Assert(Not IsBVNumWeird(node.BoundingVolume.Max0))
                Assert(Not IsBVNumWeird(node.BoundingVolume.Max1))
                Assert(Not IsBVNumWeird(node.BoundingVolume.Max2))

                node.bIsLeaf = EnsureBool(r.ReadInt32) '???
                If node.bIsLeaf Then
                    node.LeftNode = r.ReadUInt16
                    node.RightNode = r.ReadUInt16
                Else
                    node.NumTriangles = r.ReadUInt16
                    node.StartIndex = r.ReadUInt16
                End If
                kDOPTree.Nodes.Add(node)
            Next
            Dim numkDOPTriangles = r.ReadCompact
            For i = 0 To numkDOPTriangles - 1
                Dim t As kDOPCollisionTriangle
                t.v1 = r.ReadUInt16
                t.v2 = r.ReadUInt16
                t.v3 = r.ReadUInt16
                t.MaterialIndex = r.ReadUInt16
                Assert(t.MaterialIndex < Materials.Count)
                kDOPTree.Triangles.Add(t)
            Next
        End If

        r.ReadInt32() '???

        Dim numRawTriangles = r.ReadCompact
        If numRawTriangles = 0 AndAlso (PrimitiveBoundingBox.Min <> New Vector(0, 0, 0) Or PrimitiveBoundingBox.Max <> New Vector(0, 0, 0) Or VertexStream.Vertices.Count > 0 Or Sections(0).MinVertexIndex <> 65535 Or Sections(0).MaxVertexIndex <> 65535) Then
            ' No RawTriangles, but other structures have triangles?
            ' Then recreate RawTriangles based on them.
            Dim iMinTri = 0
            Dim iMaxTri = 0
            For iSection = 0 To Sections.Count - 1
                iMaxTri = iMinTri + Sections(iSection).NumTriangles - 1
                For iTri = iMinTri To iMaxTri
                    Dim t As New StaticMeshTriangle
                    t.V0 = VertexStream.Vertices(IndexBuffer.Indices(iTri * 3)).Position
                    t.V1 = VertexStream.Vertices(IndexBuffer.Indices(iTri * 3 + 1)).Position
                    t.V2 = VertexStream.Vertices(IndexBuffer.Indices(iTri * 3 + 2)).Position
                    t.NumUVs = UVStreams.Count
                    For iUV = 0 To UVStreams.Count - 1
                        t.UVs.Add(UVStreams(iUV).UVs(IndexBuffer.Indices(iTri * 3)))
                        t.UVs.Add(UVStreams(iUV).UVs(IndexBuffer.Indices(iTri * 3 + 1)))
                        t.UVs.Add(UVStreams(iUV).UVs(IndexBuffer.Indices(iTri * 3 + 2)))
                    Next
                    t.MaterialIndex = iSection
                    RawTriangles.Add(t)
                Next
                iMinTri = iMaxTri + 1
            Next
        Else
            For i = 0 To numRawTriangles - 1
                Dim t As New StaticMeshTriangle
                t.V0 = r.ReadVector
                t.V1 = r.ReadVector
                t.V2 = r.ReadVector
                Assert(Not t.V0.IsWeird)
                Assert(Not t.V1.IsWeird)
                Assert(Not t.V2.IsWeird)
                t.NumUVs = r.ReadInt32
                For j = 0 To t.NumUVs * 3 - 1
                    Dim uv As StaticMeshUV
                    uv.U = r.ReadSingle
                    uv.V = r.ReadSingle
                    Assert(Not IsNumWeird(uv.U))
                    Assert(Not IsNumWeird(uv.V))
                    t.UVs.Add(uv)
                Next
                t.Color0 = r.ReadColor
                t.Color1 = r.ReadColor
                t.Color2 = r.ReadColor
                t.MaterialIndex = r.ReadInt32
                t.SmoothingMask = r.ReadUInt32
                RawTriangles.Add(t)
            Next
        End If


    End Sub

    Sub SaveASE(file As String)
        Using w As New IO.StreamWriter(file)
            w.WriteLine("*3DSMAX_ASCIIEXPORT	200")
            w.WriteLine("*COMMENT ""AsciiExport Version  2.00 - Mon Oct 22 13:32:00 2001""")
            w.WriteLine("*SCENE {")
            w.WriteLine("	*SCENE_FILENAME """ & Name.ToUpper & ".max""")
            w.WriteLine("	*SCENE_FIRSTFRAME 0")
            w.WriteLine("	*SCENE_LASTFRAME 100")
            w.WriteLine("	*SCENE_FRAMESPEED 30")
            w.WriteLine("	*SCENE_TICKSPERFRAME 160")
            w.WriteLine("	*SCENE_BACKGROUND_STATIC 0.0000	0.0000	0.0000")
            w.WriteLine("	*SCENE_AMBIENT_STATIC 0.6039	0.6039	0.6039")
            w.WriteLine("}")
            w.WriteLine("*MATERIAL_LIST {")
            w.WriteLine("	*MATERIAL_COUNT " & Materials.Count)
            For i = 0 To Materials.Count - 1
                w.WriteLine("	*MATERIAL " & i & " {")
                w.WriteLine("		*MATERIAL_NAME """ & Materials(i) & """")
                w.WriteLine("		*MATERIAL_CLASS ""Standard""")
                w.WriteLine("		*MATERIAL_AMBIENT 0.1000	0.1000	0.1000")
                w.WriteLine("		*MATERIAL_DIFFUSE 0.5000	0.5000	0.5000")
                w.WriteLine("		*MATERIAL_SPECULAR 0.9000	0.9000	0.9000")
                w.WriteLine("		*MATERIAL_SHINE 0.0000")
                w.WriteLine("		*MATERIAL_SHINESTRENGTH 0.0000")
                w.WriteLine("		*MATERIAL_TRANSPARENCY 0.0000")
                w.WriteLine("		*MATERIAL_WIRESIZE 1.0000")
                w.WriteLine("		*MATERIAL_SHADING Blinn")
                w.WriteLine("		*MATERIAL_XP_FALLOFF 0.0000")
                w.WriteLine("		*MATERIAL_SELFILLUM 0.0000")
                w.WriteLine("		*MATERIAL_FALLOFF In")
                w.WriteLine("		*MATERIAL_XP_TYPE Filter")
                w.WriteLine("		*MAP_DIFFUSE {")
                w.WriteLine("			*MAP_NAME ""Map #" & i + 1 & """")
                w.WriteLine("			*MAP_SUBNO 1")
                w.WriteLine("			*MAP_AMOUNT 1.0000")
                w.WriteLine("			*BITMAP """ & Materials(i) & ".JPG""")
                w.WriteLine("			*MAP_TYPE Screen")
                w.WriteLine("			*UVW_U_OFFSET 0.0000")
                w.WriteLine("			*UVW_V_OFFSET 0.0000")
                w.WriteLine("			*UVW_U_TILING 1.0000")
                w.WriteLine("			*UVW_V_TILING 1.0000")
                w.WriteLine("			*UVW_ANGLE 0.0000")
                w.WriteLine("			*UVW_BLUR 1.0000")
                w.WriteLine("			*UVW_BLUR_OFFSET 0.0000")
                w.WriteLine("			*UVW_NOUSE_AMT 1.0000")
                w.WriteLine("			*UVW_NOISE_SIZE 1.0000")
                w.WriteLine("			*UVW_NOISE_LEVEL 1")
                w.WriteLine("			*UVW_NOISE_PHASE 0.0000")
                w.WriteLine("			*BITMAP_FILTER Pyramidal")
                w.WriteLine("		}")
                w.WriteLine("	}")
            Next
            w.WriteLine("}")
            w.WriteLine("*GEOMOBJECT {")
            w.WriteLine("	*NODE_NAME ""Loft01""")
            w.WriteLine("	*NODE_TM {")
            w.WriteLine("		*NODE_NAME ""Loft01""")
            w.WriteLine("		*INHERIT_POS 0 0 0")
            w.WriteLine("		*INHERIT_ROT 0 0 0")
            w.WriteLine("		*INHERIT_SCL 0 0 0")
            w.WriteLine("		*TM_ROW0 1.0000	0.0000	0.0000")
            w.WriteLine("		*TM_ROW1 0.0000	1.0000	0.0000")
            w.WriteLine("		*TM_ROW2 0.0000	0.0000	1.0000")
            w.WriteLine("		*TM_ROW3 0.0000	0.0000	0.0000")
            w.WriteLine("		*TM_POS 0.0000	0.0000	0.0000")
            w.WriteLine("		*TM_ROTAXIS 0.0000	0.0000	0.0000")
            w.WriteLine("		*TM_ROTANGLE 0.0000")
            w.WriteLine("		*TM_SCALE 1.0000	1.0000	1.0000")
            w.WriteLine("		*TM_SCALEAXIS 0.0000	0.0000	0.0000")
            w.WriteLine("		*TM_SCALEAXISANG 0.0000")
            w.WriteLine("	}")
            w.WriteLine("	*MESH {")
            w.WriteLine("		*TIMEVALUE 0")
            w.WriteLine("		*MESH_NUMVERTEX " & RawTriangles.Count * 3)
            w.WriteLine("		*MESH_NUMFACES " & RawTriangles.Count)
            w.WriteLine("		*MESH_VERTEX_LIST {")
            For i = 0 To RawTriangles.Count - 1
                w.WriteLine("			*MESH_VERTEX    " & i * 3 & "	" & NumToStr(RawTriangles(i).V0.X) & "	" & NumToStr(RawTriangles(i).V0.Y) & "	" & NumToStr(RawTriangles(i).V0.Z))
                w.WriteLine("			*MESH_VERTEX    " & i * 3 + 1 & "	" & NumToStr(RawTriangles(i).V1.X) & "	" & NumToStr(RawTriangles(i).V1.Y) & "	" & NumToStr(RawTriangles(i).V1.Z))
                w.WriteLine("			*MESH_VERTEX    " & i * 3 + 2 & "	" & NumToStr(RawTriangles(i).V2.X) & "	" & NumToStr(RawTriangles(i).V2.Y) & "	" & NumToStr(RawTriangles(i).V2.Z))
            Next
            w.WriteLine("		}")
            w.WriteLine("		*MESH_FACE_LIST {")
            For i = 0 To RawTriangles.Count - 1
                w.WriteLine("			*MESH_FACE    " & i & ":    A:    " & i * 3 & " B:    " & i * 3 + 1 & " C:    " & i * 3 + 2 & " AB:    1 BC:    1 CA:    0	 *MESH_SMOOTHING 1 	*MESH_MTLID " & RawTriangles(i).MaterialIndex)
            Next
            w.WriteLine("		}")
            w.WriteLine("		*MESH_NUMTVERTEX " & RawTriangles.Count * 3)
            w.WriteLine("		*MESH_TVERTLIST {")
            For i = 0 To RawTriangles.Count - 1
                w.WriteLine("			*MESH_TVERT " & i * 3 & "	" & NumToStr(RawTriangles(i).UVs(0).U) & "	" & NumToStr(RawTriangles(i).UVs(0).V) & "	0.0000")
                w.WriteLine("			*MESH_TVERT " & i * 3 + 1 & "	" & NumToStr(RawTriangles(i).UVs(1).U) & "	" & NumToStr(RawTriangles(i).UVs(1).V) & "	0.0000")
                w.WriteLine("			*MESH_TVERT " & i * 3 + 2 & "	" & NumToStr(RawTriangles(i).UVs(2).U) & "	" & NumToStr(RawTriangles(i).UVs(2).V) & "	0.0000")
            Next
            w.WriteLine("		}")
            w.WriteLine("		*MESH_NUMTVFACES " & RawTriangles.Count)
            w.WriteLine("		*MESH_TFACELIST {")
            For i = 0 To RawTriangles.Count - 1
                w.WriteLine("			*MESH_TFACE " & i & "	" & i * 3 & "	" & i * 3 + 1 & "	" & i * 3 + 2)
            Next
            w.WriteLine("		}")
            w.WriteLine("	}")
            w.WriteLine("	*PROP_MOTIONBLUR 0")
            w.WriteLine("	*PROP_CASTSHADOW 1")
            w.WriteLine("	*PROP_RECVSHADOW 1")
            w.WriteLine("	*MATERIAL_REF 0")
            w.WriteLine("}")
        End Using
    End Sub

    Function ToT3D() As String
        Dim s As New System.Text.StringBuilder
        s.AppendLine("Begin StaticMesh Name=" & Name)
        s.Append("    Version=2.000000")
        s.Append(" BoundingBox.Min.X=" & NumToStr(PrimitiveBoundingBox.Min.X))
        s.Append(" BoundingBox.Min.Y=" & NumToStr(PrimitiveBoundingBox.Min.Y))
        s.Append(" BoundingBox.Min.Z=" & NumToStr(PrimitiveBoundingBox.Min.Z))
        s.Append(" BoundingBox.Max.X=" & NumToStr(PrimitiveBoundingBox.Max.X))
        s.Append(" BoundingBox.Max.Y=" & NumToStr(PrimitiveBoundingBox.Max.Y))
        s.AppendLine(" BoundingBox.Max.Z=" & NumToStr(PrimitiveBoundingBox.Max.Z))
        For Each tri In RawTriangles
            s.AppendLine("    Begin Triangle")
            s.AppendLine("        Texture " & Materials(tri.MaterialIndex))
            s.AppendLine("        PolyFlags 0")
            s.AppendLine("        SmoothingMask " & tri.SmoothingMask)
            s.AppendLine("        Vertex 0 " & NumToStr(tri.V0.X) & " " & NumToStr(tri.V0.Y) & " " & NumToStr(tri.V0.Z) & " " & NumToStr(tri.UVs(0).U) & " " & NumToStr(tri.UVs(0).V))
            s.AppendLine("        Vertex 1 " & NumToStr(tri.V1.X) & " " & NumToStr(tri.V1.Y) & " " & NumToStr(tri.V1.Z) & " " & NumToStr(tri.UVs(1).U) & " " & NumToStr(tri.UVs(1).V))
            s.AppendLine("        Vertex 2 " & NumToStr(tri.V2.X) & " " & NumToStr(tri.V2.Y) & " " & NumToStr(tri.V2.Z) & " " & NumToStr(tri.UVs(2).U) & " " & NumToStr(tri.UVs(2).V))
            s.AppendLine("    End Triangle")
        Next
        s.AppendLine("End StaticMesh")
        Return s.ToString
    End Function
End Class

Public Structure StaticMeshSection
    Dim IsStrip As Boolean
    Dim FirstIndex As UShort
    Dim MinVertexIndex As UShort
    Dim MaxVertexIndex As UShort
    Dim NumTriangles As UShort
    Dim NumPrimitives As UShort
End Structure

Public Class StaticMeshVertexStream
    Public Vertices As New List(Of StaticMeshVertex)
    Public Revision As Integer
End Class

Public Structure StaticMeshVertex
    Dim Position, Normal As Vector
End Structure

Public Class RawColorStream
    Public Colors As New List(Of Drawing.Color)
    Public Revision As Integer
End Class

Public Class StaticMeshUVStream
    Public UVs As New List(Of StaticMeshUV)
    Public CoordinateIndex As Integer
    Public Revision As Integer
End Class

Public Structure StaticMeshUV
    Dim U, V As Single
End Structure

Public Class RawIndexBuffer
    Public Indices As New List(Of UShort)
    Public Revision As Integer
End Class

Public Class kDOPTree
    Public Nodes As New List(Of kDOPNode)
    Public Triangles As New List(Of kDOPCollisionTriangle)
End Class

Public Structure kDOPNode
    Dim BoundingVolume As kDOP
    Dim bIsLeaf As Boolean
    Dim LeftNode, RightNode, NumTriangles, StartIndex As UShort
End Structure

Public Structure kDOP
    Dim Min0, Min1, Min2, Max0, Max1, Max2 As Single
End Structure

Public Structure kDOPCollisionTriangle
    Dim v1, v2, v3, MaterialIndex As UShort
End Structure

Public Class StaticMeshTriangle
    Public V0, V1, V2 As Vector
    Public NumUVs As Integer = 1
    Public UVs As New List(Of StaticMeshUV)
    Public Color0, Color1, Color2 As Drawing.Color
    Public MaterialIndex As Integer
    Public SmoothingMask As UInteger = 1

    Sub New()
        Color0 = Drawing.Color.White
        Color1 = Drawing.Color.White
        Color2 = Drawing.Color.White
    End Sub
End Class