﻿Public Module PackageClasses

    Enum EPackageFlags
        PKG_AllowDownload = &H1  ' Allow downloading package.
        PKG_ClientOptional = &H2 ' Purely optional for clients.
        PKG_ServerSideOnly = &H4 ' Only needed on the server side.
        PKG_BrokenLinks = &H8    ' Loaded from linker with broken import links.
        PKG_Unsecure = &H10      ' Not trusted.
        PKG_Need = &H8000        ' Client needs to download this package.
    End Enum

    Enum EObjectFlags
        RF_Transactional = &H1   ' Object is transactional.
        RF_Unreachable = &H2     ' Object is not reachable on the object graph.
        RF_Public = &H4          ' Object is visible outside its package.
        RF_TagImp = &H8          ' Temporary import tag in load/save.
        RF_TagExp = &H10         ' Temporary export tag in load/save.
        RF_SourceModified = &H20 ' Modified relative to source files.
        RF_TagGarbage = &H40     ' Check during garbage collection.
        '
        '
        RF_NeedLoad = &H200            ' During load indicates object needs loading.
        RF_HighlightedName = &H400     ' A hardcoded name which should be syntax-highlighted.
        RF_EliminateObject = &H400     ' NULL out references to this during garbage collecion.
        RF_InSingularFunc = &H800      ' In a singular function.
        RF_RemappedName = &H800        ' Name is remapped.
        RF_Suppress = &H1000           'warning: Mirrored in UnName.h. Suppressed log name.
        RF_StateChanged = &H1000       ' Object did a state change.
        RF_InEndState = &H2000         ' Within an EndState call.
        RF_Transient = &H4000          ' Don't save object.
        RF_Preloading = &H8000         ' Data is being preloaded from file.
        RF_LoadForClient = &H10000     ' In-file load for client.
        RF_LoadForServer = &H20000     ' In-file load for client.
        RF_LoadForEdit = &H40000       ' In-file load for client.
        RF_Standalone = &H80000        ' Keep object around for editing even if unreferenced.
        RF_NotForClient = &H100000     ' Don't load this object for the game client.
        RF_NotForServer = &H200000     ' Don't load this object for the game server.
        RF_NotForEdit = &H400000       ' Don't load this object for the editor.
        RF_Destroyed = &H800000        ' Object Destroy has already been called.
        RF_NeedPostLoad = &H1000000    ' Object needs to be postloaded.
        RF_HasStack = &H2000000        ' Has execution stack.
        RF_Native = &H4000000          ' Native (UClass only).
        RF_Marked = &H8000000          ' Marked (for debugging).
        RF_ErrorShutdown = &H10000000  ' ShutdownAfterError called.
        RF_DebugPostLoad = &H20000000  ' For debugging Serialize calls.
        RF_DebugSerialize = &H40000000 ' For debugging Serialize calls.
        RF_DebugDestroy = &H80000000   ' For debugging Destroy calls.
        RF_ContextFlags = RF_NotForClient Or RF_NotForServer Or RF_NotForEdit ' All context flags.
        RF_LoadContextFlags = RF_LoadForClient Or RF_LoadForServer Or RF_LoadForEdit ' Flags affecting loading.
        RF_Load = RF_ContextFlags Or RF_LoadContextFlags Or RF_Public Or RF_Standalone Or RF_Native Or RF_SourceModified Or RF_Transactional Or RF_HasStack ' Flags to load from Unrealfiles.
        RF_Keep = RF_Native Or RF_Marked ' Flags to persist across loads.
        RF_ScriptMask = RF_Transactional Or RF_Public Or RF_Transient Or RF_NotForClient Or RF_NotForServer Or RF_NotForEdit ' Script-accessible flags.
    End Enum

    Enum EPolyFlags
        ' Regular in-game flags.
        PF_Invisible = &H1    ' Poly is invisible.
        PF_Masked = &H2    ' Poly should be drawn masked.
        PF_Translucent = &H4    ' Poly is transparent.
        PF_NotSolid = &H8    ' Poly is not solid doesn't block.
        PF_Environment = &H10    ' Poly should be drawn environment mapped.
        PF_Semisolid = &H20    ' Poly is semi-solid = collision solid Csg nonsolid.
        PF_Modulated = &H40    ' Modulation transparency.
        PF_FakeBackdrop = &H80    ' Poly looks exactly like backdrop.
        PF_TwoSided = &H100    ' Poly is visible from both sides.
        PF_AutoUPan = &H200    ' Automatically pans in U direction.
        PF_AutoVPan = &H400    ' Automatically pans in V direction.
        PF_NoSmooth = &H800    ' Don't smooth textures.
        PF_BigWavy = &H1000    ' Poly has a big wavy pattern in it.
        PF_SmallWavy = &H2000    ' Small wavy pattern (for water/enviro reflection).
        PF_Flat = &H4000    ' Flat surface.
        PF_LowShadowDetail = &H8000    ' Low detaul shadows.
        PF_NoMerge = &H10000    ' Don't merge poly's nodes before lighting when rendering.
        PF_CloudWavy = &H20000    ' Polygon appears wavy like clouds.
        PF_DirtyShadows = &H40000    ' Dirty shadows.
        PF_BrightCorners = &H80000    ' Brighten convex corners.
        PF_SpecialLit = &H100000    ' Only speciallit lights apply to this poly.
        PF_Gouraud = &H200000    ' Gouraud shaded.
        PF_Unlit = &H400000    ' Unlit.
        PF_HighShadowDetail = &H800000    ' High detail shadows.
        PF_Portal = &H4000000    ' Portal between iZones.
        PF_Mirrored = &H8000000    ' Reflective surface.

        ' Editor flags.
        PF_Memorized = &H1000000    ' Editor: Poly is remembered.
        PF_Selected = &H2000000    ' Editor: Poly is selected.
        PF_Highlighted = &H10000000    ' Editor: Poly is highlighted.   
        PF_FlatShaded = &H40000000    ' FPoly has been split by SplitPolyWithPlane.   

        ' Internal.
        PF_EdProcessed = &H40000000    ' FPoly was already processed in editorBuildFPolys.
        PF_EdCut = &H80000000    ' FPoly has been split by SplitPolyWithPlane.  
        PF_RenderFog = &H40000000    ' Render with fogmapping.
        PF_Occlude = &H80000000    ' Occludes even if PF_NoOcclude.
        PF_RenderHint = &H1000000   ' Rendering optimization hint.
    End Enum

    Enum EPropertyType
        ByteProperty = 1
        IntegerProperty
        BooleanProperty
        FloatProperty
        ObjectProperty
        NameProperty
        StringProperty
        ClassProperty
        ArrayProperty
        StructProperty
        VectorProperty
        RotatorProperty
        StrProperty
        MapProperty
        FixedArrayProperty
    End Enum

    Enum EPropertySize
        Size1
        Size2
        Size4
        Size12
        Size16
        CheckNextByte
        CheckNextWord
        CheckNextInt
    End Enum

    Public Class Package
        Public Signature As Int32
        Public PackageVersion As Int16
        Public LicenseeMode As Int16
        Public PackageFlags As EPackageFlags

        Public NameTable As New List(Of NameTableItem)
        Public ExportTable As New List(Of ExportTableItem)
        Public ImportTable As New List(Of ImportTableItem)
        Public HeritageTable As New List(Of HeritageTableItem)

        Public GUID(16 - 1) As Byte
        Public Generations As New List(Of Generation)

        Sub New()
            Signature = &H9E2A83C1
            PackageVersion = 68
            PackageFlags = EPackageFlags.PKG_AllowDownload
        End Sub

        Sub New(file As String)
            Dim stream As New IO.FileStream(file, IO.FileMode.Open, IO.FileAccess.Read)

            Dim r As New UnBinaryReader(stream)

            r.Pkg = Me

            Signature = r.ReadInt32
            If Signature <> &H9E2A83C1 Then
                Throw New Exception("Not an Unreal file.")
            End If
            PackageVersion = r.ReadInt16
            LicenseeMode = r.ReadInt16
            If PackageVersion >= 34 Then
                PackageFlags = r.ReadInt32
            End If
            Dim NameCount = r.ReadInt32
            Dim NameOffset = r.ReadInt32
            Dim ExportCount = r.ReadInt32
            Dim ExportOffset = r.ReadInt32
            Dim ImportCount = r.ReadInt32
            Dim ImportOffset = r.ReadInt32
            Dim HeritageCount = 0
            Dim HeritageOffset = 0
            If PackageVersion < 68 Then
                HeritageCount = r.ReadInt32
                HeritageOffset = r.ReadInt32
            Else
                GUID = r.ReadBytes(16)
                Dim GenerationCount = r.ReadInt32
                For i = 0 To GenerationCount - 1
                    Dim gen As New Generation
                    gen.ExportCount = r.ReadInt32
                    gen.NameCount = r.ReadInt32
                    Generations.Add(gen)
                Next
            End If

            Dim name As String
            Dim curByte As Byte
            Dim flags As Integer

            r.BaseStream.Position = NameOffset
newname:
            If NameTable.Count >= NameCount Then GoTo endnametable
            If PackageVersion = 127 And LicenseeMode = 34 Then 'Only seen in M01A_sounds.uax from https://www.oldunreal.com/phpBB3/viewtopic.php?f=33&p=99358
                r.ReadInt32()
            ElseIf PackageVersion >= 64 Then
                r.ReadByte()
            End If

            name = ""
newbyte:
            curByte = r.ReadByte
            If curByte <> 0 Then
                name &= Char.ConvertFromUtf32(curByte)
                GoTo newbyte
            Else
                flags = r.ReadInt32
                Dim item As New NameTableItem
                item.Name = name
                item.Flags = flags
                NameTable.Add(item)
                GoTo newname
            End If
endnametable:

            Dim isDreamcast = (PackageVersion = 69 And LicenseeMode = 0 And IO.Path.GetExtension(file).ToLower = ".dat")
            If Not isDreamcast Then
                r.BaseStream.Position = ImportOffset
            End If

            For i = 0 To ImportCount - 1
                Dim newItem As New ImportTableItem(Me)

                newItem.ClassPackage = r.ReadCompact
                newItem.ClassName = r.ReadCompact
                If PackageVersion >= 50 Then 'oldver
                    newItem.Package = r.ReadInt32
                Else
                    newItem._PackageName = r.ReadCompact
                End If
                newItem.ObjectName = r.ReadCompact

                ImportTable.Add(newItem)
            Next

            If Not isDreamcast Then
                r.BaseStream.Position = ExportOffset
            End If

            For i = 0 To ExportCount - 1
                Dim newItem As New ExportTableItem(Me)

                newItem.ClassRef = r.ReadCompact
                newItem.Super = r.ReadCompact
                If PackageVersion >= 50 Then 'oldver
                    newItem.Package = r.ReadInt32
                End If
                newItem.ObjectName = r.ReadCompact
                newItem.ObjectFlags = r.ReadInt32
                Dim SerialSize = r.ReadCompact
                newItem.OriginalSize = SerialSize
                If SerialSize Then
                    Dim SerialOffset = r.ReadCompact
                    newItem.OriginalOffset = SerialOffset
                    Dim OldOffset = r.BaseStream.Position
                    r.BaseStream.Position = SerialOffset
                    newItem.Data = New BinaryBuffer(r.ReadBytes(SerialSize))
                    r.BaseStream.Position = OldOffset
                Else
                    newItem.Data = New BinaryBuffer(0)
                End If

                newItem.Data.R.Pkg = Me
                newItem.Data.W.Package = Me

                ExportTable.Add(newItem)
            Next

            If PackageVersion < 68 Then
                r.BaseStream.Position = HeritageOffset
                For i = 0 To HeritageCount - 1
                    Dim item As New HeritageTableItem
                    item.GUID = r.ReadBytes(16)
                    HeritageTable.Add(item)
                Next
            End If

            r.Close()
        End Sub

        Sub SetRandomGUID()
            Dim rn As New Security.Cryptography.RNGCryptoServiceProvider
            Dim newGuid(16 - 1) As Byte
            rn.GetBytes(newGuid)

            If PackageVersion < 68 Then
                If HeritageTable.Count > 0 Then
                    HeritageTable(HeritageTable.Count - 1).GUID = newGuid
                Else
                    HeritageTable.Add(New HeritageTableItem With {.GUID = newGuid})
                End If
            Else
                GUID = newGuid
            End If
        End Sub

        Sub Save(file As String)
            If Not IO.Directory.Exists(IO.Path.GetDirectoryName(file)) Then
                IO.Directory.CreateDirectory(IO.Path.GetDirectoryName(file))
            End If
            Dim w As New UnBinaryWriter(New IO.FileStream(file, IO.FileMode.Create))

            Dim NameOffsetPtr = 0
            Dim ExportOffsetPtr = 0
            Dim ImportOffsetPtr = 0
            Dim HeritageOffsetPtr = 0
            Dim ObjectOffsets(ExportTable.Count - 1) As Integer

            w.WriteInt32(Signature)
            w.WriteInt16(PackageVersion)
            w.WriteInt16(LicenseeMode)
            If PackageVersion >= 34 Then 'oldver
                w.WriteInt32(PackageFlags)
            End If
            w.WriteInt32(NameTable.Count)
            NameOffsetPtr = w.BaseStream.Position
            w.WriteInt32(0) 'later...
            w.WriteInt32(ExportTable.Count)
            ExportOffsetPtr = w.BaseStream.Position
            w.WriteInt32(0) 'later...
            w.WriteInt32(ImportTable.Count)
            ImportOffsetPtr = w.BaseStream.Position
            w.WriteInt32(0) 'later...
            If PackageVersion < 68 Then
                w.WriteInt32(HeritageTable.Count)
                HeritageOffsetPtr = w.BaseStream.Position
                w.WriteInt32(0) 'later...
            Else
                w.Write(GUID)
                w.WriteInt32(Generations.Count)
                For i = 0 To Generations.Count - 1
                    w.WriteInt32(Generations(i).ExportCount)
                    w.WriteInt32(Generations(i).NameCount)
                Next
            End If

            Dim changeOffset(ExportTable.Count - 1) As Boolean
            ' rare (or impossible?) case of first object overlapping
            If ExportTable(0).OriginalOffset <> 0 And w.BaseStream.Position > ExportTable(0).OriginalOffset Then
                changeOffset(0) = True
            End If
            For i = 0 To ExportTable.Count - 1
                If ExportTable(i).OriginalOffset = 0 Or ExportTable(i).Data.Size > ExportTable(i).OriginalSize Then
                    changeOffset(i) = True
                End If
            Next

            'Objects with unchanged offsets
            For i = 0 To ExportTable.Count - 1
                If Not changeOffset(i) Then
                    w.BaseStream.Position = ExportTable(i).OriginalOffset
                    ObjectOffsets(i) = w.BaseStream.Position
                    w.Write(ExportTable(i).Data.ToArray())
                End If
            Next

            'Objects with changed offsets
            For i = 0 To ExportTable.Count - 1
                If changeOffset(i) Then
                    ObjectOffsets(i) = w.BaseStream.Position
                    ExportTable(i).UpdateAbsoluteOffsets(w.BaseStream.Position)
                    w.Write(ExportTable(i).Data.ToArray())
                End If
            Next

            'Name table
            Dim NameOffset = w.BaseStream.Position
            w.BaseStream.Position = NameOffsetPtr
            w.WriteInt32(NameOffset)
            w.BaseStream.Position = NameOffset
            For Each name In NameTable
                If PackageVersion >= 64 Then
                    w.WriteByte(name.Name.Length + 1)
                End If
                w.WriteZString(name.Name)
                w.WriteInt32(name.Flags)
            Next

            'Import table
            Dim ImportOffset = w.BaseStream.Position
            w.BaseStream.Position = ImportOffsetPtr
            w.WriteInt32(ImportOffset)
            w.BaseStream.Position = ImportOffset
            For Each import In ImportTable
                w.WriteCompact(import.ClassPackage)
                w.WriteCompact(import.ClassName)
                If PackageVersion >= 50 Then 'oldver
                    w.WriteInt32(import.Package)
                Else
                    w.WriteCompact(import._PackageName)
                End If
                w.WriteCompact(import.ObjectName)
            Next

            'Export table
            Dim ExportOffset = w.BaseStream.Position
            w.BaseStream.Position = ExportOffsetPtr
            w.WriteInt32(ExportOffset)
            w.BaseStream.Position = ExportOffset
            For i = 0 To ExportTable.Count - 1
                w.WriteCompact(ExportTable(i).ClassRef)
                w.WriteCompact(ExportTable(i).Super)
                If PackageVersion >= 50 Then
                    w.WriteInt32(ExportTable(i).Package)
                End If
                w.WriteCompact(ExportTable(i).ObjectName)
                w.WriteInt32(ExportTable(i).ObjectFlags)
                w.WriteCompact(ExportTable(i).Data.Size)
                If ExportTable(i).Data.Size Then
                    w.WriteCompact(ObjectOffsets(i))
                End If
            Next

            'Heritage table
            If PackageVersion < 68 Then
                Dim HeritageOffset = w.BaseStream.Position
                w.BaseStream.Position = HeritageOffsetPtr
                w.WriteInt32(HeritageOffset)
                w.BaseStream.Position = HeritageOffset
                For i = 0 To HeritageTable.Count - 1
                    w.Write(HeritageTable(i).GUID)
                Next
            End If

            w.Close()
        End Sub

        Function GetObjectPackageName(objRef As Integer) As String
            If objRef = 0 Then
                Return "None"
            ElseIf objRef < 0 Then
                Dim index = -objRef - 1
                Dim packageInd = -ImportTable(index).Package - 1
                Dim package = ImportTable(packageInd).ObjectName
                Dim outerPackageInd = -ImportTable(packageInd).Package - 1
                Dim outerPackage = ""
                If outerPackageInd <> -1 Then
                    outerPackage = ImportTable(outerPackageInd).ObjectName
                End If

                If outerPackage = "" Then
                    Return package
                Else
                    Return outerPackage
                End If
            Else 'If objRef > 0 Then
                Return "MyLevel"
            End If
        End Function

        'Function GetFullObjectName(objRef As Integer, includeClass As Boolean) As String
        '    Dim objName = ""
        '    Dim className = ""

        '    If objRef = 0 Then
        '        Return "None"
        '    ElseIf objRef < 0 Then
        '        Dim index = -objRef - 1
        '        Dim name = ImportTable(index).ObjectNameStr
        '        Dim packageInd = -ImportTable(index).Package - 1
        '        Dim package = ImportTable(packageInd).ObjectNameStr
        '        Dim outerPackageInd = -ImportTable(packageInd).Package - 1
        '        Dim outerPackage = ""
        '        If outerPackageInd <> -1 Then
        '            outerPackage = ImportTable(outerPackageInd).ObjectNameStr
        '        End If

        '        If outerPackage = "" Then
        '            objName = package & "." & name
        '        Else
        '            objName = outerPackage & "." & package & "." & name
        '        End If

        '        className = ImportTable(index).ClassNameStr
        '    ElseIf objRef > 0 Then
        '        objName = ExportTable(objRef - 1).ObjectNameStr
        '        className = GetShortObjectName(ExportTable(objRef - 1).ClassRef)
        '        If className = "None" Then
        '            className = "Class"
        '        End If
        '    End If

        '    If includeClass Then
        '        Return className & "'" & objName & "'"
        '    Else
        '        Return objName
        '    End If
        'End Function

        Function GetFullObjectName(objRef As Integer, includeClass As Boolean) As String
            Dim objName = ""
            Dim className = ""

            If objRef = 0 Then
                Return "None"
            ElseIf objRef < 0 Then
                Dim index = -objRef - 1
                Dim name = ImportTable(index).ObjectNameStr
                Dim packageInd = -ImportTable(index).Package - 1
                Dim package = ImportTable(packageInd).ObjectNameStr
                Dim outerPackageInd = -ImportTable(packageInd).Package - 1
                Dim outerPackage = ""
                If outerPackageInd <> -1 Then
                    outerPackage = ImportTable(outerPackageInd).ObjectNameStr
                End If

                If outerPackage = "" Then
                    objName = package & "." & name
                Else
                    objName = outerPackage & "." & package & "." & name
                End If

                className = ImportTable(index).ClassNameStr
            ElseIf objRef > 0 Then
                If ExportTable(objRef - 1).Package <> 0 Then
                    objName = GetFullObjectName(ExportTable(objRef - 1).Package, False) & "."
                End If
                objName &= ExportTable(objRef - 1).ObjectNameStr
                className = GetShortObjectName(ExportTable(objRef - 1).ClassRef)
                If className = "None" Then
                    className = "Class"
                End If
            End If

            If includeClass Then
                Return className & "'" & objName & "'"
            Else
                Return objName
            End If
        End Function

        Function GetShortObjectName(objRef As Integer) As String
            If objRef = 0 Then
                Return "None"
            ElseIf objRef < 0 Then
                Dim item = ImportTable(-objRef - 1)
                Return item.ObjectNameStr
            Else
                Dim item = ExportTable(objRef - 1)
                Return item.ObjectNameStr
            End If
        End Function

        Function GetObjectClassName(objRef As Integer) As String
            If objRef = 0 Then
                Return "None"
            ElseIf objRef < 0 Then
                Dim item = ImportTable(-objRef - 1)
                Return item.ClassNameStr
            Else
                Dim item = ExportTable(objRef - 1)
                Return item.ClassNameStr
            End If
        End Function

        Function IsClass(obj As ExportTableItem) As Boolean
            Return NameEq(obj.ClassNameStr, "None") AndAlso Not NameEq(obj.ObjectNameStr, "None")
        End Function

        Function GetName(num As Integer) As String
            If num < 0 Or num >= NameTable.Count Then
                Return "(Bad name " & num & ")"
            Else
                Return NameTable(num).Name
            End If
        End Function

        ' Finds name in NameTable, and if it doesn't exists, adds it. Returns name index.
        Function AddName(name As String) As Integer
            If name = "" Then
                name = "None"
            End If

            'If name = "" Then
            '    Return AddName("None")
            'End If

            'For i = 0 To NameTable.Count - 1
            '    If StrEq(name, NameTable(i).Name) Then
            '        Return i
            '    End If
            'Next

            For i = 0 To NameTable.Count - 1
                If NameEq(name, NameTable(i).Name) Then
                    Return i
                End If
            Next

            NameTable.Add(New NameTableItem With {.Name = name, .Flags = EObjectFlags.RF_TagExp Or EObjectFlags.RF_LoadForClient Or EObjectFlags.RF_LoadForServer Or EObjectFlags.RF_LoadForEdit})
            Return NameTable.Count - 1
        End Function

        Function AddUniqueName(ByRef name As String) As Integer
            If name = "" Then
                name = "None"
            End If
            Dim origName = name
            Dim curName = name
            Dim suffix = 2

            Do
                If FindName(curName) <> -1 Then
                    curName = origName & "_" & suffix
                    suffix += 1
                Else
                    Exit Do
                End If
            Loop

            NameTable.Add(New NameTableItem With {.Name = curName, .Flags = EObjectFlags.RF_TagExp Or EObjectFlags.RF_LoadForClient Or EObjectFlags.RF_LoadForServer Or EObjectFlags.RF_LoadForEdit})
            name = curName
            Return NameTable.Count - 1
        End Function

        ' Returns index of a given name.
        Function FindName(name As String) As Integer
            'If name = "" Then
            '    Return FindName("None")
            'End If

            'For i = 0 To NameTable.Count - 1
            '    If StrEq(name, NameTable(i).Name) Then
            '        Return i
            '    End If
            'Next

            For i = 0 To NameTable.Count - 1
                If NameEq(name, NameTable(i).Name) Then
                    Return i
                End If
            Next

            Return -1
        End Function

        ' Find object with given name and class. Can be both export and import.
        Function FindObject(name As String, className As String) As Integer
            If NameEq(name, "None") Then
                Return 0
            End If

            For i = 0 To ExportTable.Count - 1
                If NameEq(ExportTable(i).ObjectNameStr, name) AndAlso
                    NameEq(ExportTable(i).ClassNameStr, className) Then
                    Return i + 1
                End If
            Next

            For i = 0 To ImportTable.Count - 1
                If NameEq(ImportTable(i).ObjectNameStr, name) AndAlso
                    NameEq(ImportTable(i).ClassNameStr, className) Then
                    Return -i - 1
                End If
            Next

            Return 0
        End Function

        Function FindObject(name As String) As Integer
            If NameEq(name, "None") Then
                Return 0
            End If

            For i = 0 To ExportTable.Count - 1
                If NameEq(ExportTable(i).ObjectNameStr, name) Then
                    Return i + 1
                End If
            Next

            For i = 0 To ImportTable.Count - 1
                If NameEq(ImportTable(i).ObjectNameStr, name) Then
                    Return -i - 1
                End If
            Next

            Return 0
        End Function

        Function FindObjectLong(name As String) As Integer
            If NameEq(name, "None") Then
                Return 0
            End If

            For i = 1 To ExportTable.Count
                If NameEq(Me.GetFullObjectName(i, False), name) Then
                    Return i
                End If
            Next

            For i = -1 To -ImportTable.Count Step -1
                If NameEq(Me.GetFullObjectName(i, False), name) Then
                    Return i
                End If
            Next

            Return 0
        End Function

        ' Add an import or, if it already exists, return its objRef.
        Function AddImport(classPkg As String, className As String, pkg As String, name As String) As Integer
            For i = 0 To ImportTable.Count - 1
                Dim import = ImportTable(i)
                If NameEq(import.ClassPackageStr, classPkg) And
                   NameEq(import.ClassNameStr, className) And
                   NameEq(import.ObjectNameStr, name) And
                   NameEq(GetShortObjectName(import.Package), pkg) Then
                    Return -i - 1
                End If
            Next

            Dim imp As New ImportTableItem(Me)
            imp.ClassPackage = AddName(classPkg)
            imp.ClassName = AddName(className)
            imp.Package = FindObject(pkg, "Package")
            imp.ObjectName = AddName(name)
            ImportTable.Add(imp)
            Return -ImportTable.Count
        End Function

        ' Add an export or, if it already exists, return its objRef.
        Function AddExport(name As String, className As String, pkg As String) As Integer
            For i = 0 To ExportTable.Count - 1
                Dim export = ExportTable(i)
                If NameEq(export.ObjectNameStr, name) And
                   NameEq(export.ClassNameStr, className) And
                   NameEq(GetShortObjectName(export.Package), pkg) Then
                    Return i + 1
                End If
            Next

            Dim ex As New ExportTableItem(Me)
            ex.ObjectName = AddName(name)
            ex.ClassRef = FindObject(className)
            ex.Package = FindObject(pkg, "Package")
            ex.ObjectFlags = EObjectFlags.RF_Public Or
                             EObjectFlags.RF_LoadForClient Or
                             EObjectFlags.RF_LoadForServer Or
                             EObjectFlags.RF_LoadForEdit Or
                             EObjectFlags.RF_Standalone
            ExportTable.Add(ex)
            Return ExportTable.Count
        End Function

        Function AddExport(name As String, className As String) As Integer
            For i = 0 To ExportTable.Count - 1
                Dim export = ExportTable(i)
                If NameEq(export.ObjectNameStr, name) And
                   NameEq(export.ClassNameStr, className) Then
                    'If existingGroup <> "" AndAlso Not NameEq(GetShortObjectName(export.Package), existingGroup) Then
                    '    Continue For
                    'End If
                    Return i + 1
                End If
            Next

            Dim ex As New ExportTableItem(Me)
            ex.ObjectName = AddName(name)
            ex.ClassRef = FindObject(className)
            ex.ObjectFlags = EObjectFlags.RF_Public Or
                             EObjectFlags.RF_LoadForClient Or
                             EObjectFlags.RF_LoadForServer Or
                             EObjectFlags.RF_LoadForEdit Or
                             EObjectFlags.RF_Standalone
            ExportTable.Add(ex)
            Return ExportTable.Count
        End Function

        Function GetActorList() As Integer()
            Dim levelInd = -1
            For i = 0 To ExportTable.Count - 1
                If StrEq(ExportTable(i).ObjectNameStr, "MyLevel") And StrEq(ExportTable(i).ClassNameStr, "Level") Then
                    levelInd = i
                    Exit For
                End If
            Next

            If levelInd = -1 Then
                Return Nothing
            End If

            Dim buff = ExportTable(levelInd).Data

            buff.Position = 0
            buff.R.ReadByte()
            Dim numActors = buff.R.ReadUInt32
            buff.R.ReadUInt32()
            Dim actors(numActors - 1) As Integer
            For i = 0 To numActors - 1
                actors(i) = buff.R.ReadCompact
            Next

            Return actors
        End Function

        Function IsValidPolysNum(num As Integer) As Boolean
            If num < 0 Or num >= ExportTable.Count Then
                Return False
            End If

            If ExportTable(num).ClassNameStr <> "Polys" Then
                Return False
            End If

            Return True
        End Function

    End Class


    Class NameTableItem
        Public Name As String
        Public Flags As EObjectFlags
    End Class

    Class ExportTableItem
        Public Pkg As Package

        Public ClassRef As Integer
        Public Super As Integer
        Public Package As Integer
        Public ObjectName As Integer
        Public ObjectFlags As EObjectFlags

        Public Data As New BinaryBuffer(0)

        Public OriginalOffset As Integer
        Public OriginalSize As Integer

        ReadOnly Property ObjectNameStr() As String
            Get
                Return Pkg.GetName(ObjectName)
            End Get
        End Property

        ReadOnly Property ClassNameStr() As String
            Get
                Return Pkg.GetShortObjectName(ClassRef)
            End Get
        End Property

        Sub New(myPkg As Package)
            Pkg = myPkg
            Data.R.Pkg = myPkg
            Data.W.Package = myPkg
        End Sub

        Sub NullOut()
            ClassRef = 0
            Super = 0
            Package = 0
            ObjectName = 0
            ObjectFlags = 0
            Data.Clear()
        End Sub

        Sub SkipStack()
            If Pkg.PackageVersion < 52 Then 'oldver
                Dim OldClass = Data.R.ReadCompact()
                If OldClass Then
                    Data.Position += 4
                End If
                Data.R.ReadCompact()
                Data.Position += 8
            Else
                If ObjectFlags And EObjectFlags.RF_HasStack Then
                    Data.R.ReadCompact()
                    Data.R.ReadCompact()
                    Data.Position += 12
                    Data.R.ReadCompact()
                End If
            End If

        End Sub

        Function GetProperties() As Properties
            Data.Position = 0
            SkipStack()

            Dim props = Data.R.ReadProperties()
            props.Obj = Me
            Return props
        End Function

        Sub SkipAfterPropsOld() 'oldver
            If Pkg.PackageVersion < 57 Then 'TempState
                Data.R.ReadCompact()
            End If
            If Pkg.PackageVersion < 58 Then 'TempGroup
                Data.R.ReadCompact()
            End If
        End Sub

        Sub UpdateAbsoluteOffsets(filePos As Integer)
            Select Case ClassNameStr
                Case "Texture"
                    If Pkg.PackageVersion >= 63 Then
                        Dim props = GetProperties() 'Parse the properties
                        Dim lists = 1
                        If props.GetBool("bHasComp") Then lists = 2

                        For l = 0 To lists - 1
                            Dim mipmaps = Data.R.ReadByte()
                            For i = 0 To mipmaps - 1
                                'We are at WidthOffset
                                Dim returnHere = Data.Position

                                'Calculate WidthOffset
                                Data.R.ReadInt32()
                                Dim mipmapSize = Data.R.ReadCompact()
                                Dim widthOffset = filePos + Data.Position + mipmapSize

                                'Write WidthOffset value 
                                Data.Position = returnHere
                                Data.W.WriteInt32(widthOffset)

                                'Go to next mipmap
                                Data.Position = widthOffset + 10 - filePos
                            Next
                        Next
                    End If

                Case "Mesh"

                Case "Sound"
                    If Pkg.PackageVersion >= 63 Then
                        GetProperties() 'Skip the properties
                        Data.R.ReadCompact() 'name 'WAV'

                        Dim nextOffsetOffset = Data.Position
                        Data.R.ReadUInt32() 'NextOffset
                        Dim WAVSize = Data.R.ReadCompact()
                        Dim nextOffset = filePos + Data.Position + WAVSize

                        Data.Position = nextOffsetOffset
                        Data.W.WriteInt32(nextOffset)

                        'Data.W.WriteInt32(Pkg.ExportTable(Pkg.ExportTable.IndexOf(Me) + 1).OriginalOffset)
                    End If
            End Select
        End Sub

        '============================
        ' Class-specific stuff:
        ' Obj_<class>_Get/Set<what>
        '============================

        Function Obj_Enum_GetValues() As String()
            Data.Position = 2 'Null property that is not zero?
            Data.R.ReadCompact()
            Data.R.ReadCompact()
            Dim count = Data.R.ReadCompact()
            Dim values(count - 1) As String
            For i = 0 To count - 1
                values(i) = Pkg.GetName(Data.R.ReadCompact())
            Next
            Return values
        End Function

        Function Obj_ByteProperty_GetEnum() As Integer
            Data.Position = 2
            Data.R.ReadCompact()
            Data.R.ReadCompact()
            Data.Position += 8
            Data.R.ReadCompact()
            Return Data.R.ReadCompact()
        End Function

    End Class

    Class ImportTableItem
        Private Pkg As Package

        Public ClassPackage As Integer
        Public ClassName As Integer
        Public Package As Integer
        Public _PackageName As Integer 'oldver
        Public ObjectName As Integer

        ReadOnly Property ClassPackageStr() As String
            Get
                Return Pkg.GetName(ClassPackage)
            End Get
        End Property

        ReadOnly Property ClassNameStr() As String
            Get
                Return Pkg.GetName(ClassName)
            End Get
        End Property

        ReadOnly Property PackageStr() As String
            Get
                If Pkg.PackageVersion >= 50 Then
                    Return Pkg.GetShortObjectName(Package)
                Else
                    Return Pkg.GetName(_PackageName)
                End If

            End Get
        End Property

        ReadOnly Property ObjectNameStr() As String
            Get
                Return Pkg.GetName(ObjectName)
            End Get
        End Property

        Sub New(myPkg As Package)
            Pkg = myPkg
        End Sub
    End Class

    Structure Generation
        Dim ExportCount, NameCount As Integer
    End Structure

    Class HeritageTableItem
        Public GUID As Byte()
    End Class

    ' Type:				    Value stored in:
    ' ByteProperty		    IntValue
    ' IntegerProperty		IntValue
    ' BooleanProperty		IntValue
    ' FloatProperty	    	FloatValue
    ' ObjectProperty		IntValue
    ' NameProperty	    	IntValue
    ' StringProperty		StringValue
    ' ClassProperty		    IntValue
    ' ArrayProperty		    RawValue
    ' StructProperty		RawValue
    ' VectorProperty		RawValue
    ' RotatorProperty		RawValue
    ' StrProperty			StringValue
    ' MapProperty			RawValue
    ' FixedArrayProperty	RawValue
    Class ObjectProperty
        Public Props As Properties

        Public Name As Integer
        Public Type As EPropertyType
        Public IntValue As Integer
        Public FloatValue As Single
        Public StringValue As String
        Public RawValue As Byte()
        Public ArrayIndex As Integer
        Public StructName As Integer

        ReadOnly Property NameStr As String
            Get
                Return Props.Pkg.GetName(Name)
            End Get
        End Property

        Sub New(props As Properties)
            Me.Props = props
        End Sub

    End Class

    ' High-level, easily readable form of ObjectProperty.
    Structure ObjectPropertyStr
        Dim Name As String
        Dim Type As String
        Dim Value As String
    End Structure

    ''' <summary>
    ''' Intercepts property to string conversion. Use this to help UPLib reading unusual properties like structs, enums, arrays, etc.
    ''' </summary>
    ''' <param name="result">String representing value of property.</param>
    ''' <returns>True to override, False to use default conversion routine.</returns>
    Delegate Function PropReadingHint(prop As ObjectProperty, ByRef result As String) As Boolean

    Class Properties
        Public Pkg As Package
        Public Obj As ExportTableItem
        Public Props As New List(Of ObjectProperty)

        Default ReadOnly Property GetProp(name As String) As String
            Get
                Dim iProp = FindProp(name)
                If iProp <> -1 Then
                    Return GetValueStr(iProp)
                End If
                Return ""
            End Get
        End Property

        ''' <summary>
        ''' Converts value of a given property to string.
        ''' </summary>
        ''' <param name="num">Number of property</param>
        ''' <param name="hint">Hint for reading unusual properties like structs, enums, ArrayProperties, etc.</param>
        Function GetValueStr(num As Integer, Optional hint As PropReadingHint = Nothing) As String
            Dim p = Props(num)

            If hint IsNot Nothing Then
                Dim result = ""
                If hint(p, result) Then
                    Return result
                End If
            End If

            Select Case p.Type
                Case EPropertyType.IntegerProperty, EPropertyType.ByteProperty
                    Return p.IntValue
                Case EPropertyType.BooleanProperty
                    Return If(p.IntValue, "True", "False")
                Case EPropertyType.FloatProperty
                    Return NumToStr(p.FloatValue)
                Case EPropertyType.ObjectProperty
                    Return Pkg.GetFullObjectName(p.IntValue, True)
                Case EPropertyType.NameProperty
                    Return Pkg.GetName(p.IntValue)
                Case EPropertyType.StrProperty, EPropertyType.StringProperty
                    Return """"c & p.StringValue & """"c
                Case EPropertyType.StructProperty
                    Dim buff = New BinaryBuffer(p.RawValue)
                    buff.R.Pkg = Pkg
                    Select Case Pkg.GetName(p.StructName)
                        Case "Vector"
                            Return buff.R.ReadVector().ToPropertyString()
                        Case "Rotator", "Rotation"
                            Dim str = "(Pitch=" & buff.R.ReadInt32
                            str &= ",Yaw=" & buff.R.ReadInt32
                            str &= ",Roll=" & buff.R.ReadInt32 & ")"c
                            Return str
                        Case "Scale"
                            Dim str = "(Scale=(X=" & NumToStr(buff.R.ReadSingle)
                            str &= ",Y=" & NumToStr(buff.R.ReadSingle)
                            str &= ",Z=" & NumToStr(buff.R.ReadSingle)
                            str &= "),SheerRate=" & NumToStr(buff.R.ReadSingle)
                            str &= ",SheerAxis=" & buff.R.ReadByte & ")"c
                            Return str
                        Case "PointRegion"
                            Dim str = "(Zone=" & Pkg.GetFullObjectName(buff.R.ReadCompact, True)
                            str &= ",iLeaf=" & buff.R.ReadInt32
                            str &= ",ZoneNumber=" & buff.R.ReadByte & ")"c
                            Return str
                        Case "Color"
                            Dim str = "(R=" & buff.R.ReadByte
                            str &= ",G=" & buff.R.ReadByte
                            str &= ",B=" & buff.R.ReadByte
                            str &= ",A=" & buff.R.ReadByte & ")"c
                            Return str
                        Case Else
                            'Throw New FormatException("Unknown struct: " & Pkg.GetName(p.StructName))
                            Return BytesToHex(p.RawValue)
                    End Select
                Case Else
                    'Throw New FormatException("Unknown property: " & p.Type.ToString & " " & p.NameStr)
                    Return BytesToHex(p.RawValue)
            End Select

        End Function

        ' Converts all properties to easily readable format.
        Function ToStringList(Optional hint As PropReadingHint = Nothing) As List(Of ObjectPropertyStr)
            Dim l As New List(Of ObjectPropertyStr)

            For i = 0 To Props.Count - 1
                Dim item As New ObjectPropertyStr
                item.Name = Pkg.GetName(Props(i).Name)
                If Props(i).ArrayIndex > 0 Then
                    item.Name &= "("c & Props(i).ArrayIndex & ")"c
                End If
                item.Type = Props(i).Type.ToString
                If Props(i).Type = EPropertyType.StructProperty Then
                    item.Type &= " "c & Pkg.GetName(Props(i).StructName)
                End If
                item.Value = GetValueStr(i, hint)
                l.Add(item)
            Next

            Return l
        End Function

        Function FindProp(name As String) As Integer
            For i = 0 To Props.Count - 1
                If NameEq(Props(i).NameStr, name) Then
                    Return i
                End If
            Next
            Return -1
        End Function

        Function GetInt(var As String, Optional defaultVal As Integer = 0) As Integer
            Dim varNum = FindProp(var)
            If varNum <> -1 Then
                Return Props(varNum).IntValue
            End If
            Return defaultVal
        End Function

        Function GetFloat(var As String, Optional defaultVal As Single = 0) As Single
            Dim varNum = FindProp(var)
            If varNum <> -1 Then
                Return Props(varNum).FloatValue
            End If
            Return defaultVal
        End Function

        Function GetBool(var As String, Optional defaultVal As Boolean = False) As Boolean
            Dim varNum = FindProp(var)
            If varNum <> -1 Then
                Return Props(varNum).IntValue
            End If
            Return defaultVal
        End Function

        Function GetVector(var As String) As Vector
            Dim varNum = FindProp(var)
            If varNum <> -1 Then
                Dim buff = New BinaryBuffer(Props(varNum).RawValue)
                Return buff.R.ReadVector()
            End If

            Return New Vector(0, 0, 0)
        End Function

        Function GetVector(var As String, defaultVal As Vector) As Vector
            Dim varNum = FindProp(var)
            If varNum <> -1 Then
                Dim buff = New BinaryBuffer(Props(varNum).RawValue)
                Return buff.R.ReadVector()
            End If
            Return defaultVal
        End Function

        Function GetVectorArray(var As String) As List(Of Vector)
            Dim result As New List(Of Vector)

            Dim name = Pkg.FindName(var)

            For Each p In Props
                If p.Name = name Then
                    Dim buff = New BinaryBuffer(p.RawValue)
                    ListSet(result, p.ArrayIndex, buff.R.ReadVector())
                End If
            Next

            Return result
        End Function

        Function GetRotatorArray(var As String) As List(Of Rotator)
            Dim result As New List(Of Rotator)

            Dim name = Pkg.FindName(var)

            For Each p In Props
                If p.Name = name Then
                    Dim buff = New BinaryBuffer(p.RawValue)
                    ListSet(result, p.ArrayIndex, buff.R.ReadRotator())
                End If
            Next

            Return result
        End Function

        Function GetPointRegion(var As String) As PointRegion
            Dim varNum = FindProp(var)
            If varNum <> -1 Then
                Dim buff = New BinaryBuffer(Props(varNum).RawValue)
                Dim result As PointRegion
                result.Zone = buff.R.ReadCompact
                result.iLeaf = buff.R.ReadInt32
                result.ZoneNumber = buff.R.ReadByte
                Return result
            End If
            Return New PointRegion
        End Function

        Function GetRotator(var As String) As Rotator
            Dim varNum = FindProp(var)
            If varNum <> -1 Then
                Dim buff = New BinaryBuffer(Props(varNum).RawValue)
                Dim result As Rotator
                result.Pitch = buff.R.ReadInt32
                result.Yaw = buff.R.ReadInt32
                result.Roll = buff.R.ReadInt32
                Return result
            End If
            Return New Rotator
        End Function

        Function GetScale(var As String) As scale
            Dim varNum = FindProp(var)
            If varNum <> -1 Then
                Dim buff = New BinaryBuffer(Props(varNum).RawValue)
                Dim result As Scale
                result.Scale = buff.R.ReadVector
                result.SheerRate = buff.R.ReadSingle
                result.SheerAxis = buff.R.ReadByte
                Return result
            End If
            Return New Scale With {.Scale = New Vector(1, 1, 1)}
        End Function

        Overloads Function ToString(Optional detailed As Boolean = False, Optional hint As PropReadingHint = Nothing) As String
            Dim text As New System.Text.StringBuilder

            Dim textprops = ToStringList(hint)

            For i = 0 To textprops.Count - 1
                Dim p = textprops(i)
                If detailed Then
                    text.Append("("c)
                    text.Append(p.Type)
                    text.Append(") ")
                End If
                text.Append(p.Name)
                text.Append("="c)
                text.Append(p.Value)
                If i <> textprops.Count - 1 Then
                    text.AppendLine()
                End If
            Next

            Return text.ToString()
        End Function

        Function ToSingleLine(Optional hint As PropReadingHint = Nothing) As String
            Dim text As New System.Text.StringBuilder

            Dim textprops = ToStringList(hint)

            text.Append("("c)

            For i = 0 To textprops.Count - 1
                Dim p = textprops(i)
                text.Append(p.Name)
                text.Append("="c)
                text.Append(p.Value)
                If i <> textprops.Count - 1 Then
                    text.Append(","c)
                End If
            Next

            text.Append(")"c)

            Return text.ToString()
        End Function

        Private Function AddProp(name As String) As ObjectProperty
            Dim ind = FindProp(name)
            If ind = -1 Then
                Dim op As New ObjectProperty(Me)
                op.Name = Pkg.AddName(name)
                Props.Add(op)
                Return op
            Else
                Return Props(ind)
            End If
        End Function

        Sub SetProp(name As String, type As EPropertyType, value As Integer)
            Dim prop = AddProp(name)
            prop.Type = type
            prop.IntValue = value
        End Sub

        Sub SetPropFloat(name As String, value As Single)
            Dim prop = AddProp(name)
            prop.Type = EPropertyType.FloatProperty
            prop.FloatValue = value
        End Sub
    End Class

End Module

Public Structure Rotator
    Dim Pitch As Integer
    Dim Yaw As Integer
    Dim Roll As Integer
End Structure

Public Structure PointRegion
    Dim Zone As Integer
    Dim iLeaf As Integer
    Dim ZoneNumber As Byte
End Structure

Public Structure Scale
    Dim Scale As Vector
    Dim SheerRate As Single
    Dim SheerAxis As Byte
End Structure