﻿Imports System.Drawing

Public Module ResExporters
    Function GetTexture(obj As ExportTableItem) As Bitmap
        Dim pkg = obj.Pkg
        obj.Data.Position = 0
        obj.SkipStack()
        Dim props = obj.GetProperties()
        Dim palObjRef = -1
        Dim format = 0
        palObjRef = props.GetInt("Palette")
        format = props.GetInt("Format")
        If format <> 0 Then
            Throw New Exception("Only P8 textures are supported.")
        End If
        obj.Data.R.ReadByte()
        If pkg.PackageVersion >= 63 Then obj.Data.R.ReadInt32()
        Dim size = obj.Data.R.ReadCompact()
        Dim data = obj.Data.R.ReadBytes(size)
        Dim width = obj.Data.R.ReadInt32()
        Dim height = obj.Data.R.ReadInt32()

        Dim palColors = GetPalette(pkg.ExportTable(palObjRef - 1))

        Dim bmp As New Bitmap(width, height, Imaging.PixelFormat.Format8bppIndexed)

        Dim bmpPal = bmp.Palette
        For i = 0 To palColors.GetUpperBound(0)
            bmpPal.Entries(i) = palColors(i)
        Next
        bmp.Palette = bmpPal

        Dim rect As New Rectangle(0, 0, width, height)
        Dim locked = bmp.LockBits(rect, Imaging.ImageLockMode.ReadWrite, Imaging.PixelFormat.Format8bppIndexed)

        ' Can't just copy the data because of (potential) padding.
        Dim bmpSize = locked.Height * locked.Stride
        Dim bmpData(bmpSize - 1) As Byte
        For y = 0 To height - 1
            For x = 0 To width - 1
                bmpData(locked.Stride * y + x) = data(width * y + x)
            Next
        Next

        System.Runtime.InteropServices.Marshal.Copy(bmpData, 0, locked.Scan0, bmpSize)
        bmp.UnlockBits(locked)

        Return bmp
    End Function

    Function GetSound(obj As ExportTableItem) As Byte()
        Dim pkg = obj.Pkg
        obj.Data.Position = 0
        obj.SkipStack()
        obj.GetProperties()
        obj.Data.R.ReadCompact() 'Format
        If pkg.PackageVersion >= 63 Then obj.Data.R.ReadInt32() 'OffsetNext
        If pkg.PackageVersion = 127 And pkg.LicenseeMode = 34 Then
            'Only seen in M01A_sounds.uax from https://www.oldunreal.com/phpBB3/viewtopic.php?f=33&p=99358. That one weird file seems to not use compacts.
            obj.Data.R.ReadInt32()
        End If
        Dim size = obj.Data.R.ReadCompact()
        Return obj.Data.R.ReadBytes(size)
    End Function

    Function GetTextBuffer(obj As ExportTableItem) As String
        obj.Data.Position = 0
        obj.SkipStack()
        obj.GetProperties()
        obj.Data.Position += 8 'Pos, Top
        Dim size = obj.Data.R.ReadCompact()
        Return obj.Data.R.ReadFixedZString(size)
    End Function

    Function GetClassScript(obj As ExportTableItem) As String
        obj.Data.Position = 0
        obj.SkipStack()
        obj.Data.R.ReadCompact()
        obj.Data.R.ReadCompact()
        Dim script = obj.Data.R.ReadCompact()
        Dim pkg = obj.Pkg
        Assert(StrEq(pkg.GetObjectClassName(script), "TextBuffer"), "Not a TextBuffer")
        Dim buffObj = pkg.ExportTable(script - 1)
        Return GetTextBuffer(buffObj)
    End Function

    Function GetMesh(obj As ExportTableItem) As Mesh
        'TODO: Are the latter jumps really present in <=61?
        Dim pkg = obj.Pkg

        Dim r = obj.Data.R

        Dim mesh As New Mesh

        Dim isNewVersion = (pkg.PackageVersion > 61)
        Dim boundingSphereSize = If(isNewVersion, 16, 12)

        obj.Data.Position = 0
        obj.SkipStack()
        obj.GetProperties()
        obj.Data.Position += 25 'BoundingBox
        obj.Data.Position += boundingSphereSize 'BoundingSphere
        If isNewVersion Then
            obj.Data.Position += 4 'VertsJump
        End If

        Dim VertArraySize = r.ReadCompact()
        ReDim mesh.VertData(VertArraySize - 1)
        For i = 0 To VertArraySize - 1
            mesh.VertData(i) = r.ReadInt32
        Next
        If isNewVersion Then r.ReadInt32()
        Dim NumTris = r.ReadCompact()
        ReDim mesh.Tris(NumTris - 1)
        For i = 0 To NumTris - 1
            mesh.Tris(i).Vert1Num = r.ReadUInt16()
            mesh.Tris(i).Vert2Num = r.ReadUInt16()
            mesh.Tris(i).Vert3Num = r.ReadUInt16()
            mesh.Tris(i).Vert1U = r.ReadByte()
            mesh.Tris(i).Vert1V = r.ReadByte()
            mesh.Tris(i).Vert2U = r.ReadByte()
            mesh.Tris(i).Vert2V = r.ReadByte()
            mesh.Tris(i).Vert3U = r.ReadByte()
            mesh.Tris(i).Vert3V = r.ReadByte()
            mesh.Tris(i).Flags = r.ReadInt32()
            mesh.Tris(i).TextureIndex = r.ReadInt32()
        Next
        Dim NumAnimSeqs = r.ReadCompact()
        For i = 0 To NumAnimSeqs - 1
            r.ReadCompact()
            r.ReadCompact()
            r.ReadInt32()
            r.ReadInt32()
            Dim NumFuncs = r.ReadCompact()
            For j = 0 To NumFuncs - 1
                r.ReadInt32()
                r.ReadCompact()
            Next
            r.ReadSingle()
        Next

        If isNewVersion Then r.ReadInt32()
        Dim NumConnects = r.ReadCompact()
        obj.Data.Position += 8 * NumConnects

        'obj.Data.Position += 37
        obj.Data.Position += 25 'BoundingBox
        obj.Data.Position += boundingSphereSize 'BoundingSphere

        'If isNewVersion Then r.ReadInt32()
        obj.Data.Position += 4 'VertLinks_Jump

        Dim NumVertLinks = r.ReadCompact()
        obj.Data.Position += 4 * NumVertLinks

        Dim NumTextures = r.ReadCompact()
        For i = 0 To NumTextures - 1
            r.ReadCompact()
        Next

        Dim NumBoundingBoxes = r.ReadCompact()
        obj.Data.Position += 25 * NumBoundingBoxes

        Dim NumBoundingSpheres = r.ReadCompact()
        obj.Data.Position += boundingSphereSize * NumBoundingSpheres

        mesh.NumVerts = r.ReadInt32()
        mesh.NumFrames = r.ReadInt32()

        Return mesh
    End Function

    'TODO: Reuse GetMesh.
    Function GetLodMesh(obj As ExportTableItem) As Mesh

        Dim pkg = obj.Pkg

        Dim r = obj.Data.R

        Dim mesh As New Mesh

        Dim isNewVersion = (pkg.PackageVersion > 61)
        Dim boundingSphereSize = If(isNewVersion, 16, 12)

        obj.Data.Position = 0
        obj.SkipStack()
        obj.GetProperties()
        obj.Data.Position += 25 'BoundingBox
        obj.Data.Position += boundingSphereSize 'BoundingSphere
        If isNewVersion Then
            obj.Data.Position += 4 'VertsJump
        End If

        Dim VertArraySize = r.ReadCompact()
        ReDim mesh.VertData(VertArraySize - 1)
        For i = 0 To VertArraySize - 1
            mesh.VertData(i) = r.ReadInt32
        Next
        If isNewVersion Then r.ReadInt32()
        Dim NumTris = r.ReadCompact()
        ReDim mesh.Tris(NumTris - 1)
        For i = 0 To NumTris - 1
            mesh.Tris(i).Vert1Num = r.ReadUInt16()
            mesh.Tris(i).Vert2Num = r.ReadUInt16()
            mesh.Tris(i).Vert3Num = r.ReadUInt16()
            mesh.Tris(i).Vert1U = r.ReadByte()
            mesh.Tris(i).Vert1V = r.ReadByte()
            mesh.Tris(i).Vert2U = r.ReadByte()
            mesh.Tris(i).Vert2V = r.ReadByte()
            mesh.Tris(i).Vert3U = r.ReadByte()
            mesh.Tris(i).Vert3V = r.ReadByte()
            mesh.Tris(i).Flags = r.ReadInt32()
            mesh.Tris(i).TextureIndex = r.ReadInt32()
        Next
        Dim NumAnimSeqs = r.ReadCompact()
        For i = 0 To NumAnimSeqs - 1
            r.ReadCompact()
            r.ReadCompact()
            r.ReadInt32()
            r.ReadInt32()
            Dim NumFuncs = r.ReadCompact()
            For j = 0 To NumFuncs - 1
                r.ReadInt32()
                r.ReadCompact()
            Next
            r.ReadSingle()
        Next

        If isNewVersion Then r.ReadInt32()
        Dim NumConnects = r.ReadCompact()
        obj.Data.Position += 8 * NumConnects

        'obj.Data.Position += 37
        obj.Data.Position += 25 'BoundingBox
        obj.Data.Position += boundingSphereSize 'BoundingSphere

        'If isNewVersion Then r.ReadInt32()
        obj.Data.Position += 4 'VertLinks_Jump

        Dim NumVertLinks = r.ReadCompact()
        obj.Data.Position += 4 * NumVertLinks

        Dim NumTextures = r.ReadCompact()
        For i = 0 To NumTextures - 1
            r.ReadCompact()
        Next

        Dim NumBoundingBoxes = r.ReadCompact()
        obj.Data.Position += 25 * NumBoundingBoxes

        Dim NumBoundingSpheres = r.ReadCompact()
        obj.Data.Position += boundingSphereSize * NumBoundingSpheres

        mesh.NumVerts = r.ReadInt32()
        mesh.NumFrames = r.ReadInt32()

        ' DWORD   ANDFlags
        ' DWORD   ORFlags
        ' Vector  Scale
        ' Vector  Origin
        ' Rotator RotOrigin
        ' DWORD   CurPoly
        ' DWORD   CurVertex
        obj.Data.Position += 52

        If pkg.PackageVersion = 65 Then
            ' FLOAT TextureLOD?
            obj.Data.Position += 4
        ElseIf pkg.PackageVersion >= 66 Then
            Dim TextureLOD_Count = r.ReadCompact()
            ' FLOAT TextureLOD
            obj.Data.Position += 4 * TextureLOD_Count
        End If

        ' Class LodMesh starts here...

        Dim CollapsePointThus_Count = r.ReadCompact()
        ' WORD CollapsePointThus
        obj.Data.Position += 2 * CollapsePointThus_Count

        Dim FaceLevel_Count = r.ReadCompact()
        ' WORD FaceLevel
        obj.Data.Position += 2 * FaceLevel_Count

        Dim Faces_Count = r.ReadCompact()
        Dim Faces(FaceLevel_Count - 1) As LodMeshFace
        For i = 0 To Faces_Count - 1
            Faces(i).WedgeIndex1 = r.ReadUInt16()
            Faces(i).WedgeIndex2 = r.ReadUInt16()
            Faces(i).WedgeIndex3 = r.ReadUInt16()
            Faces(i).MaterialIndex = r.ReadUInt16()
        Next

        Dim CollapseWedgeThus_Count = r.ReadCompact()
        ' WORD CollapseWedgeThus
        obj.Data.Position += 2 * CollapseWedgeThus_Count

        Dim Wedges_Count = r.ReadCompact()
        Dim Wedges(Wedges_Count - 1) As LodMeshWedge
        For i = 0 To Wedges_Count - 1
            Wedges(i).VertexIndex = r.ReadUInt16()
            Wedges(i).U = r.ReadByte()
            Wedges(i).V = r.ReadByte()
        Next

        Dim Materials_Count = r.ReadCompact()
        Dim Materials(Materials_Count - 1) As LodMeshMaterial
        For i = 0 To Materials_Count - 1
            Materials(i).Flags = r.ReadInt32()
            Materials(i).TextureIndex = r.ReadInt32()
        Next

        Dim SpecialFaces_Count = r.ReadCompact()
        Dim SpecialFaces(SpecialFaces_Count - 1) As LodMeshFace
        For i = 0 To SpecialFaces_Count - 1
            SpecialFaces(i).WedgeIndex1 = r.ReadUInt16()
            SpecialFaces(i).WedgeIndex2 = r.ReadUInt16()
            SpecialFaces(i).WedgeIndex3 = r.ReadUInt16()
            SpecialFaces(i).MaterialIndex = r.ReadUInt16()
        Next

        Dim NormalVerts = r.ReadInt32()
        Dim SpecialVerts = r.ReadInt32()

        Dim newVerts(VertArraySize - 1) As Integer
        For f = 0 To mesh.NumFrames - 1
            Dim dstFrameStartPtr = f * mesh.NumVerts

            For v = 0 To SpecialVerts - 1
                newVerts(dstFrameStartPtr + v) = mesh.VertData(f * SpecialVerts + v)
            Next

            For v = 0 To NormalVerts - 1
                newVerts(dstFrameStartPtr + SpecialVerts + v) = mesh.VertData(SpecialVerts * mesh.NumFrames + f * NormalVerts + v)
            Next
        Next

        If StrEq(obj.ObjectNameStr, "GasBagM") Then
            Dim y
            y = 5
            y += 1
        End If
        mesh.VertData = newVerts

        Dim FaceToTri = Function(face As LodMeshFace) As MeshTriangle
                            Dim tri As New MeshTriangle
                            tri.Vert1Num = Wedges(face.WedgeIndex1).VertexIndex
                            tri.Vert2Num = Wedges(face.WedgeIndex2).VertexIndex
                            tri.Vert3Num = Wedges(face.WedgeIndex3).VertexIndex

                            tri.Vert1U = Wedges(face.WedgeIndex1).U
                            tri.Vert2U = Wedges(face.WedgeIndex2).U
                            tri.Vert3U = Wedges(face.WedgeIndex3).U

                            tri.Vert1V = Wedges(face.WedgeIndex1).V
                            tri.Vert2V = Wedges(face.WedgeIndex2).V
                            tri.Vert3V = Wedges(face.WedgeIndex3).V

                            tri.Flags = Materials(face.MaterialIndex).Flags
                            tri.TextureIndex = Materials(face.MaterialIndex).TextureIndex
                            Return tri
                        End Function

        ReDim mesh.Tris(SpecialFaces_Count + Faces_Count - 1)
        For i = 0 To SpecialFaces_Count - 1
            Dim tri As MeshTriangle
            tri = FaceToTri(SpecialFaces(i))
            mesh.Tris(i) = tri
        Next
        For i = 0 To Faces_Count - 1
            Dim tri As MeshTriangle
            tri = FaceToTri(Faces(i))
            tri.Vert1Num += SpecialVerts
            tri.Vert2Num += SpecialVerts
            tri.Vert3Num += SpecialVerts
            mesh.Tris(SpecialFaces_Count + i) = tri
        Next

        Return mesh
    End Function

    Private Structure LodMeshFace
        Dim WedgeIndex1, WedgeIndex2, WedgeIndex3, MaterialIndex As UShort
    End Structure

    Private Structure LodMeshWedge
        Dim VertexIndex As UShort
        Dim U, V As Byte
    End Structure

    Private Structure LodMeshMaterial
        Dim Flags, TextureIndex As Integer
    End Structure

    Class Mesh
        Public Scale As Vector
        Public Origin As Vector
        Public RotOrigin As Rotator
        Public Textures As New List(Of Integer)

        Public NumVerts As Integer
        Public NumFrames As Integer
        Public VertData As Integer()
        Public Tris As MeshTriangle()

        Sub Save(anivfile As String, datafile As String)
            Dim aw As New UnBinaryWriter(New IO.FileStream(anivfile, IO.FileMode.Create))
            aw.Write(CUShort(NumFrames))
            aw.Write(CUShort(NumVerts * 4))
            For i = 0 To VertData.Length - 1
                aw.Write(VertData(i))
            Next
            aw.Close()

            Dim dw As New UnBinaryWriter(New IO.FileStream(datafile, IO.FileMode.Create))
            dw.Write(CUShort(Tris.GetLength(0)))
            dw.Write(CUShort(NumVerts))
            dw.Seek(44, IO.SeekOrigin.Current)
            For i = 0 To Tris.GetUpperBound(0)
                dw.Write(CUShort(Tris(i).Vert1Num))
                dw.Write(CUShort(Tris(i).Vert2Num))
                dw.Write(CUShort(Tris(i).Vert3Num))
                dw.WriteByte(TranslateFlags(Tris(i).Flags))
                dw.WriteByte(0) 'PolyColor
                dw.WriteByte(Tris(i).Vert1U)
                dw.WriteByte(Tris(i).Vert1V)
                dw.WriteByte(Tris(i).Vert2U)
                dw.WriteByte(Tris(i).Vert2V)
                dw.WriteByte(Tris(i).Vert3U)
                dw.WriteByte(Tris(i).Vert3V)
                dw.WriteByte(Tris(i).TextureIndex)
                dw.WriteByte(0) 'Flags (unused)
            Next
            dw.Close()

        End Sub

        Private Function TranslateFlags(inPackage As EPolyFlags) As Byte
            Dim result As Byte
            If inPackage And EPolyFlags.PF_Invisible Then
                result = 8 'weapon tri
            ElseIf inPackage And EPolyFlags.PF_Masked Then
                result = 3
            ElseIf inPackage And EPolyFlags.PF_Modulated Then
                result = 4
            ElseIf inPackage And EPolyFlags.PF_Translucent Then
                result = 2
            ElseIf inPackage And EPolyFlags.PF_TwoSided Then
                result = 1
            End If

            If inPackage And EPolyFlags.PF_Unlit Then result = result Or 16
            If inPackage And EPolyFlags.PF_Flat Then result = result Or 32
            If inPackage And EPolyFlags.PF_Environment Then result = result Or 64
            If inPackage And EPolyFlags.PF_NoSmooth Then result = result Or 128

            Return result
        End Function
    End Class

    Structure MeshTriangle
        Dim Vert1Num, Vert2Num, Vert3Num As UInt16
        Dim Vert1U, Vert1V, Vert2U, Vert2V, Vert3U, Vert3V As Byte
        Dim Flags, TextureIndex As Integer
    End Structure

End Module
