﻿Public Class T3DExporter
    Public Package As Package
    Public KnownEnums As New List(Of EnumInfo)
    Public KnownEnumVars As New List(Of EnumVarInfo)
    Public KnownStructs As New List(Of SimpleStructs.StructInfo)

    Sub New(pkg As Package)
        Package = pkg
        For i = 0 To TypicalStructs.GetUpperBound(0)
            KnownStructs.Add(New StructInfo(TypicalStructs(i)))
        Next
    End Sub

    Sub GetEnums(pkg As Package)
        For Each obj In pkg.ExportTable
            Select Case obj.ClassNameStr
                Case "Enum"
                    Dim enumInfo As New EnumInfo
                    enumInfo.Name = obj.ObjectNameStr
                    enumInfo.Values = obj.Obj_Enum_GetValues()
                    KnownEnums.Add(enumInfo)
                Case "ByteProperty"
                    Dim enumNum = obj.Obj_ByteProperty_GetEnum()
                    If enumNum Then
                        Dim enumVarInfo As New EnumVarInfo
                        enumVarInfo.VarName = obj.ObjectNameStr
                        enumVarInfo.EnumName = pkg.GetShortObjectName(enumNum)
                        KnownEnumVars.Add(enumVarInfo)
                    End If
            End Select

        Next
    End Sub

    Function PropertiesToT3D(props As List(Of ObjectProperty)) As String
        Dim text As New System.Text.StringBuilder

        For Each p In props
            Dim name = Package.GetName(p.Name)

            Dim isUnknownStruct = True
            Dim structType As StructInfo = Nothing
            If p.Type <> EPropertyType.StructProperty Then
                isUnknownStruct = False
            Else
                Dim structName = Package.GetName(p.StructName)
                For Each s In KnownStructs
                    If StrEq(s.Name, structName) Then
                        structType = s
                        isUnknownStruct = False
                        Exit For
                    End If
                Next
            End If

            Select Case p.Type
                Case EPropertyType.ArrayProperty, EPropertyType.VectorProperty, EPropertyType.RotatorProperty,
                     EPropertyType.MapProperty, EPropertyType.FixedArrayProperty
                    text.Append(";" & name & "=(" & p.Type.ToString() & ") ")
                    text.AppendLine(BytesToHex(p.RawValue))
                    Continue For
            End Select

            If p.Type = EPropertyType.StructProperty And isUnknownStruct Then
                text.Append(";" & name & "=(struct " & Package.GetName(p.StructName) & ") ")
                text.AppendLine(BytesToHex(p.RawValue))
                Continue For
            End If

            text.Append(p.Name)
            text.Append("=")

            Select Case p.Type
                Case EPropertyType.IntegerProperty
                    text.AppendLine(p.IntValue)
                Case EPropertyType.ByteProperty
                    'ENUMS!!!
                    text.AppendLine(p.IntValue)
                Case EPropertyType.BooleanProperty
                    text.AppendLine(If(p.IntValue, "True", "False"))
                Case EPropertyType.FloatProperty
                    text.AppendLine(NumToStr(p.FloatValue))
                Case EPropertyType.StrProperty, EPropertyType.StringProperty
                    text.Append(""""c)
                    text.Append(p.StringValue)
                    text.AppendLine(""""c)
                Case EPropertyType.NameProperty
                    text.AppendLine(Package.GetName(p.IntValue))
                Case EPropertyType.ObjectProperty
                    text.AppendLine(Package.GetFullObjectName(p.IntValue, True))
                Case EPropertyType.StructProperty

            End Select
        Next

        Return text.ToString
    End Function

End Class

' Enums. In beta versions this was simple because of property layouts included in map files,
' but in final Unreal there's no simple way to check whether a given property is enum
' (and you can't simply put it's value because stupid UnrealEd 2.0 won't accept that)

Public Module SimpleEnums

    ''' <summary>
    ''' Identifies a variable which is of an enum type.
    ''' </summary>
    Public Class EnumVarInfo
        ' Name of variable (eg. CsgOper)
        Public VarName As String
        ' Which enum the variable uses
        Public EnumName As String

        Sub New()
        End Sub

        Sub New(definition As String)
            Dim tokens = definition.Split({" "c}, StringSplitOptions.RemoveEmptyEntries)
            EnumName = tokens(0)
            VarName = tokens(1)
        End Sub
    End Class

    ''' <summary>
    ''' Definition of an enum.
    ''' </summary>
    Public Class EnumInfo
        Public Name As String
        Public Values As String()

        Sub New()
        End Sub

        Sub New(definition As String)
            Dim tokens = definition.Split({" "c}, StringSplitOptions.RemoveEmptyEntries)
            Name = tokens(0)
            ReDim Values(tokens.GetUpperBound(0) - 1)
            For i = 0 To Values.GetUpperBound(0)
                Values(i) = tokens(i + 1)
            Next
        End Sub
    End Class

    Public TypicalEnumVars As String() =
        {
            "ECsgOper CsgOper",
            "ENetRole Role",
            "EPhysics Physics",
            "EDrawType DrawType",
            "ERenderStyle Style",
            "EMusicTransition Transition",
            "ELightType LightType",
            "ELightEffect LightEffect"
        }

    Public TypicalEnums As String() =
        {
            "ECsgOper CSG_Active CSG_Add CSG_Subtract CSG_Intersect CSG_Deintersect",
            "ENetRole ROLE_None ROLE_DumbProxy ROLE_SimulatedProxy ROLE_AutonomousProxy ROLE_Authority",
            "EPhysics PHYS_None PHYS_Walking PHYS_Falling PHYS_Swimming PHYS_Flying PHYS_Rotating PHYS_Projectile PHYS_Rolling PHYS_Interpolating PHYS_MovingBrush PHYS_Spider PHYS_Trailer",
            "EDrawType DT_None DT_Sprite DT_Mesh DT_Brush DT_RopeSprite DT_VerticalSprite DT_Terraform DT_SpriteAnimOnce",
            "ERenderStyle STY_None STY_Normal STY_Masked STY_Translucent STY_Modulated STY_AlphaBlend",
            "EMusicTransition MTRAN_None MTRAN_Instant MTRAN_Segue MTRAN_Fade MTRAN_FastFade MTRAN_SlowFade",
            "ELightType LT_None LT_Steady LT_Pulse LT_Blink LT_Flicker LT_Strobe LT_BackdropLight LT_SubtlePulse T_TexturePaletteOnce LT_TexturePaletteLoop",
            "ELightEffect LE_None LE_TorchWaver LE_FireWaver LE_WateryShimmer LE_Searchlight LE_SlowWave LE_FastWave LE_CloudCast LE_StaticSpot LE_Shock LE_Disco LE_Warp LE_Spotlight LE_NonIncidence LE_Shell LE_OmniBumpMap LE_Interference LE_Cylinder LE_Rotor"
        }

    Public ESheerAxisInfo As New EnumInfo("ESheerAxis SHEER_None SHEER_XY SHEER_XZ SHEER_YX SHEER_YZ SHEER_ZX SHEER_ZY")
End Module

' Structs. Here is a system for parsing them that is simple and good enough for most cases.
Public Module SimpleStructs
    Enum StructVarType
        T_Byte
        T_Int
        T_Float
        T_Object
        T_ESheerAxis 'Mega fucking hack
    End Enum

    Structure StructFormatToken
        Dim IsText As Boolean
        Dim VarType As StructVarType
        Dim Text As String
    End Structure

    Public Class StructInfo
        Public Name As String
        Public Format As New List(Of StructFormatToken)

        Sub New(definition As String)
            'example:
            'Vector (X=<float>,Y=<float>,Z=<float>)

            definition = definition.Trim()
            Dim spcInd = definition.IndexOf(" "c)
            If spcInd = -1 Then
                Throw New Exception("Bad struct definition")
                Return
            End If
            Me.Name = definition.Substring(0, spcInd)
            definition = definition.Substring(spcInd + 1)
            definition = definition.Trim()

            Dim IsVar = False
            Dim CurToken = ""
            For i = 0 To definition.Length - 1
                If definition(i) = "<"c Then
                    If Not IsVar Then
                        Me.Format.Add(New StructFormatToken With {.IsText = True, .Text = CurToken})
                        IsVar = True
                        CurToken = ""
                    Else
                        Throw New Exception("Bad struct definition")
                        Return
                    End If
                ElseIf definition(i) = ">"c Then
                    If IsVar Then
                        Dim varKind As StructVarType
                        Select Case CurToken.ToLower
                            Case "byte"
                                varKind = StructVarType.T_Byte
                            Case "int"
                                varKind = StructVarType.T_Int
                            Case "float"
                                varKind = StructVarType.T_Float
                            Case "object"
                                varKind = StructVarType.T_Object
                            Case "esheeraxis"
                                varKind = StructVarType.T_ESheerAxis
                            Case Else
                                Throw New Exception("Bad struct definition")
                                Return
                        End Select
                        Me.Format.Add(New StructFormatToken With {.IsText = False, .VarType = varKind})
                        IsVar = False
                        CurToken = ""
                    Else
                        Throw New Exception("Bad struct definition")
                        Return
                    End If
                Else
                    CurToken &= definition(i)
                End If
            Next
            Me.Format.Add(New StructFormatToken With {.IsText = True, .Text = CurToken})
        End Sub
    End Class

    Function Parse(rawData As Byte(), structInfo As StructInfo, pkg As Package) As String
        Dim r As New UnBinaryReader(New IO.MemoryStream(rawData))

        Dim text As New System.Text.StringBuilder

        For Each t In structInfo.Format
            If t.IsText Then
                text.Append(t.Text)
            Else
                Dim var = ""
                Select Case t.VarType
                    Case StructVarType.T_Byte
                        var = r.ReadByte()
                    Case StructVarType.T_Int
                        var = r.ReadInt32()
                    Case StructVarType.T_Float
                        var = NumToStr(r.ReadSingle())
                    Case StructVarType.T_Object
                        var = pkg.GetFullObjectName(r.ReadCompact(), True)
                    Case StructVarType.T_ESheerAxis
                        var = SimpleEnums.ESheerAxisInfo.Values(r.ReadByte())
                End Select
                text.Append(var)
            End If
        Next

        Return text.ToString
    End Function

    Public TypicalStructs As String() =
        {
            "Vector      (X=<float>,Y=<float>,Z=<float>)",
            "Color       (R=<byte>,G=<byte>,B=<byte>,A=<byte>)",
            "Rotator     (Pitch=<int>,Yaw=<int>,Roll=<int>)",
            "Scale       (Scale=(X=<float>,Y=<float>,Z=<float>),SheerRate=<float>,SheerAxis=<byte>)",
            "PointRegion (Zone=<object>,iLeaf=<int>,ZoneNumber=<byte>)",
            "Coords      (Origin=(X=<float>,Y=<float>,Z=<float>),XAxis=(X=<float>,Y=<float>,Z=<float>),YAxis=(X=<float>,Y=<float>,Z=<float>),ZAxis=(X=<float>,Y=<float>,Z=<float>))"
        }

End Module
