﻿Imports System.Drawing
Imports System.Runtime.InteropServices

Public Module Textures

    Class Texture
        Public Pkg As Package
        Public MyExport As ExportTableItem

        Public ReadOnly Property USize As Integer
            Get
                Return Props.GetInt("USize")
            End Get
        End Property

        Public ReadOnly Property VSize As Integer
            Get
                Return Props.GetInt("VSize")
            End Get
        End Property

        Public ReadOnly Property UClamp As Integer
            Get
                Return Props.GetInt("UClamp")
            End Get
        End Property

        Public ReadOnly Property VClamp As Integer
            Get
                Return Props.GetInt("VClamp")
            End Get
        End Property

        Public ReadOnly Property Palette As Integer
            Get
                Return Props.GetInt("Palette")
            End Get
        End Property

        Public ReadOnly Property Format As ETextureFormat
            Get
                Return Props.GetInt("Format")
            End Get
        End Property

        Public ReadOnly Property CompFormat As ETextureFormat
            Get
                Return Props.GetInt("CompFormat")
            End Get
        End Property

        ''' <summary>
        ''' Workaround for the 227 vs UT2k4 ETextureFormat inconsistency.
        ''' </summary>
        Public ReadOnly Property FormatName As String
            Get
                If Pkg.PackageVersion <= 69 Then '227
                    Dim fmt As ETextureFormat = Props.GetInt("Format")
                    Return fmt.ToString.Substring(5)
                Else
                    If Format = ETextureFormat2k4.TEXF_NODATA And bHasComp Then 'Some textures in UTX-Build
                        Dim fmt As ETextureFormat2k4 = Props.GetInt("CompFormat")
                        Return fmt.ToString.Substring(5)
                    Else 'UT2k4
                        Dim fmt As ETextureFormat2k4 = Props.GetInt("Format")
                        Return fmt.ToString.Substring(5)
                    End If
                End If
            End Get
        End Property

        Public ReadOnly Property bHasComp As Boolean
            Get
                Return Props.GetBool("bHasComp")
            End Get
        End Property

        Public ReadOnly Property bMasked As Boolean
            Get
                Return Props.GetBool("bMasked")
            End Get
        End Property

        Public MipMaps As New List(Of MipMap)
        Public CompMipMaps As New List(Of MipMap)

        ''' <summary>
        ''' Can be either MipMaps or CompMipMaps.
        ''' </summary>
        Public ReadOnly Property Mips As List(Of MipMap)
            Get
                If Pkg.PackageVersion > 69 AndAlso Format = ETextureFormat2k4.TEXF_NODATA AndAlso bHasComp Then
                    Return CompMipMaps
                Else
                    Return MipMaps
                End If
            End Get
        End Property

        Public Props As New Properties

        Sub New()
        End Sub

        Sub New(pkg As Package, width As Integer, height As Integer, format As ETextureFormat)
            Me.Pkg = pkg
            Me.Props.Pkg = pkg
            AddByteProp(Props, "Format", format)
            AddIntProp(Props, "UBits", Math.Log(width, 2))
            AddIntProp(Props, "VBits", Math.Log(height, 2))
            AddIntProp(Props, "USize", width)
            AddIntProp(Props, "VSize", height)
            AddIntProp(Props, "UClamp", width)
            AddIntProp(Props, "VClamp", height)

            ' Do we need this  at all?
            'Dim mipZero As Byte() = {0, 0, 0, 0}
            'Dim maxColor As Byte() = {255, 255, 255, 255}
            'Props.Props.Add(New ObjectProperty With {.Name = pkg.AddName("MipZero"),
            '                                         .Type = EPropertyType.StructProperty,
            '                                         .StructName = pkg.AddName("Color"),
            '                                         .RawValue = mipZero})
            'Props.Props.Add(New ObjectProperty With {.Name = pkg.AddName("MaxColor"),
            '                                         .Type = EPropertyType.StructProperty,
            '                                         .StructName = pkg.AddName("Color"),
            '                                         .RawValue = maxColor})
        End Sub

        Private Shared Sub AddIntProp(props As Properties, name As String, value As Integer)
            props.SetProp(name, EPropertyType.IntegerProperty, value)
        End Sub

        Private Shared Sub AddByteProp(props As Properties, name As String, value As Integer)
            props.SetProp(name, EPropertyType.ByteProperty, value)
        End Sub

        Function ToBitmap(Optional mip As Byte = 0, Optional ForceMasked As Boolean = False) As Bitmap
            Dim mips = Me.Mips
            Dim fmt = Me.FormatName

            Dim w = mips(mip).Width
            Dim h = mips(mip).Height
            Select Case fmt
                Case "P8"
                    If Palette <= 0 Then Throw New Exception("P8 texture without palette.")
                    Dim pal = GetPalette(Pkg.ExportTable(Palette - 1))
                    If bMasked Then
                        pal(0) = Color.FromArgb(0, 0, 0, 0)
                    ElseIf ForceMasked Then 'may make no difference, but I don't wanna break anything
                        pal(0) = Color.FromArgb(0, pal(0))
                    End If
                    Return MakeIndexedBitmap(mips(mip).Data, w, h, pal)
                Case "DXT1"
                    Return DecodeDXT1(mips(mip).Data, w, h)
                Case "DXT3"
                    Return DecodeDXT3(mips(mip).Data, w, h)
                Case "DXT5"
                    Return DecodeDXT5(mips(mip).Data, w, h)
                Case "RGB8"
                    Dim data = mips(mip).Data
                    Dim index = 0
                    Dim bmp As New Bitmap(w, h, Imaging.PixelFormat.Format24bppRgb)
                    Dim fastBmp As New FastPixel(bmp)
                    fastBmp.Lock()
                    For y = 0 To h - 1
                        For x = 0 To w - 1
                            ' Not sure about order
                            fastBmp.SetPixel(x, y, Color.FromArgb(data(index + 2), data(index + 1), data(index)))
                            index += 3
                        Next
                    Next
                    fastBmp.Unlock(True)
                    Return bmp

                Case "RGBA8"
                    Dim data = mips(mip).Data
                    Dim index = 0
                    Dim bmp As New Bitmap(w, h, Imaging.PixelFormat.Format32bppArgb)
                    Dim fastBmp As New FastPixel(bmp)
                    fastBmp.Lock()
                    For y = 0 To h - 1
                        For x = 0 To w - 1
                            fastBmp.SetPixel(x, y, Color.FromArgb(data(index + 3), data(index + 2), data(index + 1), data(index)))
                            index += 4
                        Next
                    Next
                    fastBmp.Unlock(True)
                    Return bmp

            End Select

            Throw New Exception("Unsupported texture format.")
        End Function

    End Class

    Class MipMap
        Public Width, Height As Integer
        Public Data As Byte()
    End Class

    Enum ETextureFormat
        TEXF_P8
        TEXF_RGBA7
        TEXF_RGB16
        TEXF_DXT1
        TEXF_RGB8
        TEXF_RGBA8
        TEXF_DXT3 '<---- !!!
        TEXF_DXT5
    End Enum

    Enum ETextureFormat2k4
        TEXF_P8
        TEXF_RGBA7
        TEXF_RGB16
        TEXF_DXT1
        TEXF_RGB8
        TEXF_RGBA8
        TEXF_NODATA '<---- !!!
        TEXF_DXT3
        TEXF_DXT5
        TEXF_L8
        TEXF_G16
        TEXF_RRRGGGBBB
    End Enum

    Function GetPalette(obj As ExportTableItem) As Color()
        If obj.ClassNameStr <> "Palette" Then
            Throw New Exception("Not a palette.")
        End If

        obj.Data.Position = 0
        obj.SkipStack()
        obj.GetProperties()
        obj.SkipAfterPropsOld()
        Dim size = obj.Data.R.ReadCompact()
        Dim pal(size - 1) As Color
        Dim R, G, B As Byte
        For i = 0 To size - 1
            R = obj.Data.R.ReadByte()
            G = obj.Data.R.ReadByte()
            B = obj.Data.R.ReadByte()
            obj.Data.R.ReadByte()
            pal(i) = Color.FromArgb(R, G, B)
        Next
        Return pal
    End Function

    Sub SetPalette(obj As ExportTableItem, pal As Color())
        obj.ObjectFlags = obj.ObjectFlags And Not EObjectFlags.RF_HasStack

        obj.Data.Clear()
        obj.Data.W.WriteCompact(obj.Pkg.AddName("None")) 'props
        obj.Data.W.WriteCompact(pal.GetLength(0))

        For i = 0 To pal.GetUpperBound(0)
            obj.Data.W.WriteByte(pal(i).R)
            obj.Data.W.WriteByte(pal(i).G)
            obj.Data.W.WriteByte(pal(i).B)
            obj.Data.W.WriteByte(pal(i).A)
        Next
    End Sub

    Function MakeIndexedBitmap(data As Byte(), width As Integer, height As Integer, pal As Color()) As Bitmap
        Dim bmp As New Bitmap(width, height, Imaging.PixelFormat.Format8bppIndexed)

        Dim bmpPal = bmp.Palette
        For i = 0 To pal.GetUpperBound(0)
            bmpPal.Entries(i) = pal(i)
        Next
        bmp.Palette = bmpPal

        Dim rect As New Rectangle(0, 0, width, height)
        Dim locked = bmp.LockBits(rect, Imaging.ImageLockMode.WriteOnly, Imaging.PixelFormat.Format8bppIndexed)

        'Can't just copy because of padding.
        Dim newData(locked.Height * locked.Stride - 1) As Byte

        For y = 0 To bmp.Height - 1
            For x = 0 To bmp.Width - 1
                newData(y * locked.Stride + x) = data(y * bmp.Width + x)
            Next
        Next

        System.Runtime.InteropServices.Marshal.Copy(newData, 0, locked.Scan0, locked.Stride * height)
        bmp.UnlockBits(locked)

        Return bmp
    End Function

    Function Get565Color(col As UShort) As Color
        Return Color.FromArgb((col >> 11) << 3, ((col >> 5) And 63) << 2, (col And 31) << 3)
    End Function

    Function DecodeDXT1(data As Byte(), width As Integer, height As Integer) As Bitmap
        Dim buff As New BinaryBuffer(data)

        Dim bmp As New Bitmap(width, height)
        Dim fastBmp As New FastPixel(bmp)
        fastBmp.Lock()

        Dim pal(4 - 1) As Color
        Dim col0i, col1i As UShort
        Dim pixels As UInt32
        Dim tmp As PreciseColor 'for blending

        Dim blockWidth, blockHeight As Byte

        Dim maxbx, maxby As Integer
        maxbx = Math.Ceiling(width / 4) - 1
        maxby = Math.Ceiling(height / 4) - 1

        For by = 0 To maxby
            For bx = 0 To maxbx
                col0i = buff.R.ReadUInt16
                col1i = buff.R.ReadUInt16

                pal(0) = Get565Color(col0i)
                pal(1) = Get565Color(col1i)

                If col0i > col1i Then
                    tmp = pal(0)
                    tmp = (tmp * 2 + pal(1)) / 3
                    pal(2) = tmp

                    tmp = pal(1)
                    tmp = (tmp * 2 + pal(0)) / 3
                    pal(3) = tmp
                Else
                    tmp = pal(0)
                    tmp = (tmp + pal(1)) / 2
                    pal(2) = tmp

                    pal(3) = Color.Transparent
                End If

                pixels = buff.R.ReadUInt32

                blockWidth = If(bx < maxbx, 4, maxbx Mod 4 + 1)
                blockHeight = If(by < maxby, 4, maxby Mod 4 + 1)

                For y = 0 To blockHeight - 1
                    For x = 0 To blockWidth - 1
                        fastBmp.SetPixel(bx * 4 + x, by * 4 + y, pal((pixels >> ((y * 4 + x) * 2)) And 3))
                    Next
                Next
            Next
        Next

        fastBmp.Unlock(True)

        Return bmp

    End Function

    Function DecodeDXT3(data As Byte(), width As Integer, height As Integer) As Bitmap
        Dim buff As New BinaryBuffer(data)

        Dim bmp As New Bitmap(width, height)
        Dim fastBmp As New FastPixel(bmp)
        fastBmp.Lock()

        Dim pal(4 - 1) As Color
        Dim col0i, col1i As UShort
        Dim alpha As UInt64
        Dim pixels As UInt32
        Dim tmp As PreciseColor 'for blending

        Dim blockWidth, blockHeight As Byte

        Dim maxbx, maxby As Integer
        maxbx = Math.Ceiling(width / 4) - 1
        maxby = Math.Ceiling(height / 4) - 1

        For by = 0 To maxby
            For bx = 0 To maxbx
                alpha = buff.R.ReadUInt64

                col0i = buff.R.ReadUInt16
                col1i = buff.R.ReadUInt16

                pal(0) = Get565Color(col0i)
                pal(1) = Get565Color(col1i)

                tmp = pal(0)
                tmp = (tmp * 2 + pal(1)) / 3
                pal(2) = tmp

                tmp = pal(1)
                tmp = (tmp * 2 + pal(0)) / 3
                pal(3) = tmp

                pixels = buff.R.ReadUInt32

                blockWidth = If(bx < maxbx, 4, maxbx Mod 4 + 1)
                blockHeight = If(by < maxby, 4, maxby Mod 4 + 1)

                For y = 0 To blockHeight - 1
                    For x = 0 To blockWidth - 1
                        Dim col = pal((pixels >> ((y * 4 + x) * 2)) And 3)
                        col = Color.FromArgb(((alpha >> ((y * 4 + x) * 4)) And CULng(15)) * (255 / 15),
                                             col.R, col.G, col.B)
                        fastBmp.SetPixel(bx * 4 + x, by * 4 + y, col)
                    Next
                Next
            Next
        Next

        fastBmp.Unlock(True)

        Return bmp

    End Function

    Function DecodeDXT5(data As Byte(), width As Integer, height As Integer) As Bitmap
        Dim buff As New BinaryBuffer(data)

        Dim bmp As New Bitmap(width, height)
        Dim fastBmp As New FastPixel(bmp)
        fastBmp.Lock()

        Dim pal(4 - 1) As Color
        Dim col0i, col1i As UShort
        Dim alphapal(8 - 1) As Byte
        Dim alpha As UInt64
        Dim pixels As UInt32
        Dim tmp As PreciseColor 'for blending

        Dim blockWidth, blockHeight As Byte

        Dim maxbx, maxby As Integer
        maxbx = Math.Ceiling(width / 4) - 1
        maxby = Math.Ceiling(height / 4) - 1

        For by = 0 To maxby
            For bx = 0 To maxbx
                alphapal(0) = buff.R.ReadByte
                alphapal(1) = buff.R.ReadByte

                If alphapal(0) > alphapal(1) Then
                    For i = 2 To 7
                        alphapal(i) = ((8 - i) * alphapal(0) + (i - 1) * alphapal(1) + 3) / 7
                    Next
                Else
                    For i = 2 To 5
                        alphapal(i) = ((6 - i) * alphapal(0) + (i - 1) * alphapal(1) + 2) / 5
                    Next
                    alphapal(6) = 0
                    alphapal(7) = 255
                End If

                alpha = buff.R.ReadUInt32
                alpha = alpha Or (CULng(buff.R.ReadUInt16) << 32)

                col0i = buff.R.ReadUInt16
                col1i = buff.R.ReadUInt16

                pal(0) = Get565Color(col0i)
                pal(1) = Get565Color(col1i)

                tmp = pal(0)
                tmp = (tmp * 2 + pal(1)) / 3
                pal(2) = tmp

                tmp = pal(1)
                tmp = (tmp * 2 + pal(0)) / 3
                pal(3) = tmp

                pixels = buff.R.ReadUInt32

                blockWidth = If(bx < maxbx, 4, maxbx Mod 4 + 1)
                blockHeight = If(by < maxby, 4, maxby Mod 4 + 1)

                For y = 0 To blockHeight - 1
                    For x = 0 To blockWidth - 1
                        Dim col = pal((pixels >> ((y * 4 + x) * 2)) And 3)
                        col = Color.FromArgb(alphapal((alpha >> ((y * 4 + x) * 3)) And CULng(7)),
                                             col.R, col.G, col.B)
                        fastBmp.SetPixel(bx * 4 + x, by * 4 + y, col)
                    Next
                Next
            Next
        Next

        fastBmp.Unlock(True)

        Return bmp

    End Function
    Private Function ColorFromInt(col As UInteger) As Color
        Return Color.FromArgb(col >> 24, col And 255, (col >> 8) And 255, (col >> 16) And 255)
    End Function

    Friend Structure PreciseColor
        Dim R As Single
        Dim G As Single
        Dim B As Single
        Dim A As Single

        ReadOnly Property Avg As Single
            Get
                Return (R + G + B) / 3
            End Get
        End Property

        Shared Narrowing Operator CType(input As PreciseColor) As Color
            Return Color.FromArgb(ClipByte(input.A * 255), ClipByte(input.R * 255), ClipByte(input.G * 255), ClipByte(input.B * 255))
        End Operator

        Shared Widening Operator CType(input As Color) As PreciseColor
            Return New PreciseColor(input.A / 255, input.R / 255, input.G / 255, input.B / 255)
        End Operator

        Shared Widening Operator CType(input As Single) As PreciseColor
            Return New PreciseColor(input, input, input)
        End Operator

        Sub New(_r As Single, _g As Single, _b As Single)
            A = 1
            R = _r
            G = _g
            B = _b
        End Sub

        Sub New(_a As Single, _r As Single, _g As Single, _b As Single)
            A = _a
            R = _r
            G = _g
            B = _b
        End Sub

        Shared Function Gray(brightness As Single) As PreciseColor
            Return New PreciseColor(brightness, brightness, brightness)
        End Function

        Shared Operator +(lhs As PreciseColor, rhs As PreciseColor) As PreciseColor
            Return New PreciseColor(lhs.A + rhs.A, lhs.R + rhs.R, lhs.G + rhs.G, lhs.B + rhs.B)
        End Operator

        Shared Operator -(lhs As PreciseColor, rhs As PreciseColor) As PreciseColor
            Return New PreciseColor(lhs.A - rhs.A, lhs.R - rhs.R, lhs.G - rhs.G, lhs.B - rhs.B)
        End Operator

        Shared Operator *(lhs As PreciseColor, rhs As Single) As PreciseColor
            Return New PreciseColor(lhs.A * rhs, lhs.R * rhs, lhs.G * rhs, lhs.B * rhs)
        End Operator

        Shared Operator *(lhs As PreciseColor, rhs As PreciseColor) As PreciseColor
            Return New PreciseColor(lhs.A * rhs.A, lhs.R * rhs.R, lhs.G * rhs.G, lhs.B * rhs.B)
        End Operator

        Shared Operator /(lhs As PreciseColor, rhs As Single) As PreciseColor
            Return New PreciseColor(lhs.A / rhs, lhs.R / rhs, lhs.G / rhs, lhs.B / rhs)
        End Operator

        Shared Operator /(lhs As PreciseColor, rhs As PreciseColor) As PreciseColor
            Return New PreciseColor(lhs.A / rhs.A, lhs.R / rhs.R, lhs.G / rhs.G, lhs.B / rhs.B)
        End Operator

        Shared Operator ^(lhs As PreciseColor, rhs As Single) As PreciseColor
            Return New PreciseColor(lhs.A ^ rhs, lhs.R ^ rhs, lhs.G ^ rhs, lhs.B ^ rhs)
        End Operator

        Function AbsSqr() As Single
            Return R * R + G * G + B * B
        End Function

        Function MaxValue() As Single
            Dim max As Single = R
            max = Math.Max(max, G)
            max = Math.Max(max, B)
            Return max
        End Function

        Function MinValue() As Single
            Dim min As Single = R
            min = Math.Min(min, G)
            min = Math.Min(min, B)
            Return min
        End Function

        Shared Function Lerp(c1 As PreciseColor, c2 As PreciseColor, v As Single) As PreciseColor
            Return c1 * (1 - v) + c2 * v
        End Function

        Private Shared Function ClipByte(x As Integer) As Byte
            Return Math.Min(255, Math.Max(0, x))
        End Function

    End Structure

    Function GetTexture(obj As ExportTableItem) As Texture
        Dim tex As New Texture
        tex.MyExport = obj

        Dim pkg = obj.Pkg
        tex.Pkg = pkg
        obj.Data.Position = 0
        obj.SkipStack()
        tex.Props = obj.GetProperties()
        obj.SkipAfterPropsOld()

        Dim lists = 1
        If tex.Props.GetBool("bHasComp") Then lists = 2

        For l = 0 To lists - 1
            Dim mips As New List(Of MipMap)

            Dim mipsNum = obj.Data.R.ReadByte()
            For i = 0 To mipsNum - 1
                Dim mip As New MipMap
                If pkg.PackageVersion >= 63 Then
                    obj.Data.R.ReadInt32() 'WidthOffset
                End If
                Dim size = obj.Data.R.ReadCompact()
                mip.Data = obj.Data.R.ReadBytes(size)
                mip.Width = obj.Data.R.ReadInt32()
                mip.Height = obj.Data.R.ReadInt32()
                obj.Data.R.ReadByte() 'BitsWidth
                obj.Data.R.ReadByte() 'BitsHeight
                mips.Add(mip)
            Next

            If l = 0 Then
                tex.MipMaps = mips
            Else
                tex.CompMipMaps = mips
            End If
        Next

        Return tex
    End Function

    Sub SetTexture(obj As ExportTableItem, tex As Texture)
        ' Temp hack just in case of.
        obj.ObjectFlags = obj.ObjectFlags And Not EObjectFlags.RF_HasStack

        obj.Data.Clear()
        obj.Data.W.WriteProperties(tex.Props)

        Dim lists = 1
        If tex.Props.GetBool("bHasComp") Then lists = 2

        For l = 0 To lists - 1

            Dim curList As List(Of MipMap)

            If l = 0 Then
                curList = tex.MipMaps
            Else
                curList = tex.CompMipMaps
            End If

            obj.Data.W.WriteByte(curList.Count)
            For Each mip In curList
                If obj.Pkg.PackageVersion >= 63 Then
                    obj.Data.W.WriteInt32(0) 'WidthOffset
                End If
                obj.Data.W.WriteCompact(mip.Data.Length)
                obj.Data.W.Write(mip.Data)
                obj.Data.W.WriteInt32(mip.Width)
                obj.Data.W.WriteInt32(mip.Height)
                obj.Data.W.WriteByte(Math.Log(mip.Width, 2)) 'BitsWidth
                obj.Data.W.WriteByte(Math.Log(mip.Height, 2)) 'BitsHeight
            Next

        Next
    End Sub
End Module
