﻿Public Class UnBinaryReader
    Inherits IO.BinaryReader

    Public Pkg As Package

    Sub New(input As IO.Stream)
        MyBase.New(input)
    End Sub

    Sub New(file As String)
        MyBase.New(New IO.FileStream(file, IO.FileMode.Open, IO.FileAccess.Read))
    End Sub

    Function ReadInts(count As Integer) As Integer()
        Dim result(count - 1) As Integer

        For i = 0 To count - 1
            result(i) = ReadInt32()
        Next

        Return result
    End Function

    Function ReadCompact() As Integer

        If Pkg.PackageVersion = 127 And Pkg.LicenseeMode = 34 Then
            'Only seen in M01A_sounds.uax from https://www.oldunreal.com/phpBB3/viewtopic.php?f=33&p=99358. That one weird file seems to not use compacts.
            Return ReadInt32()
        End If

        Dim output As Integer = 0
        Dim signed As Boolean = False
        Dim x As Byte
        For i = 0 To 4
            x = ReadByte()
            If (i = 0) Then
                If ((x And 128) > 0) Then signed = True
                output = output Or (x And 63)
                If ((x And 64) = 0) Then Exit For
            ElseIf (i = 4) Then
                output = output Or ((x And 31) << (6 + (3 * 7)))
            Else
                output = output Or ((x And 127) << (6 + ((i - 1) * 7)))
                If ((x And 128) = 0) Then Exit For
            End If
        Next
        If signed Then output *= -1
        Return output
    End Function

    Function ReadZString() As String
        Dim s As New System.Text.StringBuilder
        Dim c As Byte

        Do
            c = ReadByte()

            If c <> 0 Then
                s.Append(Char.ConvertFromUtf32(c))
            Else
                Exit Do
            End If
        Loop

        Return s.ToString
    End Function

    ''' <summary>
    ''' Stope reading when either zero byte was reached or specified length was reached.
    ''' </summary>
    Function ReadFixedZString(length As Integer) As String
        Dim s As New System.Text.StringBuilder
        Dim origPos = BaseStream.Position
        Dim c As Byte

        For i = 0 To length - 1
            c = ReadByte()

            If c <> 0 Then
                s.Append(Char.ConvertFromUtf32(c))
            Else
                BaseStream.Position = origPos + length
                Exit For
            End If
        Next

        Return s.ToString
    End Function

    Function ReadFixedZString16(lengthChars As Integer) As String
        Dim s As New System.Text.StringBuilder
        Dim origPos = BaseStream.Position
        Dim c As UInt16

        For i = 0 To lengthChars - 1
            c = ReadUInt16()

            If c <> 0 Then
                s.Append(Char.ConvertFromUtf32(c))
            Else
                BaseStream.Position = origPos + lengthChars * 2
                Exit For
            End If
        Next

        Return s.ToString
    End Function

    Function ReadVector() As Vector
        Dim x, y, z As Single
        x = ReadSingle()
        y = ReadSingle()
        z = ReadSingle()
        Return New Vector(x, y, z)
    End Function

    Function ReadRotator() As Rotator
        Dim r As Rotator
        r.Pitch = ReadInt32()
        r.Yaw = ReadInt32()
        r.Roll = ReadInt32()
        Return r
    End Function

    Function ReadColor() As Drawing.Color
        Dim cR = ReadByte()
        Dim cG = ReadByte()
        Dim cB = ReadByte()
        Dim cA = ReadByte()
        Return Drawing.Color.FromArgb(cA, cR, cG, cB)
    End Function

    Function ReadProperties() As Properties
        Dim props As New Properties
        Dim propList As New List(Of ObjectProperty)

        Do
            Dim curProp As New ObjectProperty(props)
            curProp.Name = ReadCompact()
            If NameEq(Pkg.GetName(curProp.Name), "None") Then
                Exit Do
            End If

            Dim PropInfo = ReadByte()
            Dim PropType As EPropertyType
            PropType = PropInfo And 15
            curProp.Type = PropType

            If PropType = EPropertyType.StructProperty Then
                curProp.StructName = ReadCompact()
            End If

            Dim PropSizeCode As EPropertySize
            PropSizeCode = (PropInfo >> 4) And 7
            Dim PropSize As Integer
            Select Case PropSizeCode
                Case EPropertySize.Size1
                    PropSize = 1
                Case EPropertySize.Size2
                    PropSize = 2
                Case EPropertySize.Size4
                    PropSize = 4
                Case EPropertySize.Size12
                    PropSize = 12
                Case EPropertySize.Size16
                    PropSize = 16
                Case EPropertySize.CheckNextByte
                    PropSize = ReadByte()
                Case EPropertySize.CheckNextWord
                    PropSize = ReadUInt16()
                Case EPropertySize.CheckNextInt
                    PropSize = ReadUInt32()
            End Select

            Dim ArrayFlag As Boolean
            ArrayFlag = PropInfo >> 7

            If ArrayFlag And PropType <> EPropertyType.BooleanProperty Then
                'BIG ENDIAN!
                Dim arrayIndex As Integer

                Dim firstByte = ReadByte()
                If (firstByte And 192) = 192 Then
                    'array index is integer
                    BaseStream.Position -= 1
                    arrayIndex = SwapEndianness(ReadInt32() And Not 192)
                ElseIf (firstByte And 128) = 128 Then
                    'array index is word
                    BaseStream.Position -= 1
                    arrayIndex = SwapEndianness(ReadInt16() And Not 128S)
                Else
                    arrayIndex = firstByte
                End If

                curProp.ArrayIndex = arrayIndex
            End If

            Select Case PropType
                Case EPropertyType.ByteProperty
                    curProp.IntValue = ReadByte()
                Case EPropertyType.IntegerProperty
                    curProp.IntValue = ReadInt32()
                Case EPropertyType.BooleanProperty
                    curProp.IntValue = ArrayFlag
                Case EPropertyType.FloatProperty
                    curProp.FloatValue = ReadSingle()
                Case EPropertyType.NameProperty
                    curProp.IntValue = ReadCompact()
                Case EPropertyType.ObjectProperty
                    curProp.IntValue = ReadCompact()
                Case EPropertyType.StrProperty
                    Dim length = ReadCompact()
                    If length < 0 Then
                        curProp.StringValue = ReadFixedZString16(-length)
                    Else
                        curProp.StringValue = ReadFixedZString(length)
                    End If
                Case EPropertyType.StringProperty
                    curProp.StringValue = ReadFixedZString(PropSize)
                Case EPropertyType.StructProperty
                    curProp.RawValue = ReadBytes(PropSize)
                Case Else
                    curProp.RawValue = ReadBytes(PropSize)
            End Select

            curProp.Props = props
            propList.Add(curProp)
        Loop


        props.Pkg = Pkg
        props.Props = propList
        Return props
    End Function

End Class
