#include "extract_conv.h"

#include "misc.h"
#include "packdata.h"
#include "packfind.h"
#include "objtools.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

static const export_entry *(*const refp)(const export_entry*, const char*) = getobjexprefprop;
static const char *(*const strp)(const export_entry*, const char*) = getobjstrprop;
static long int (*const intp)(const export_entry*, const char*, int*) = getobjintprop;
static const char *(*const nstrp)(const export_entry*, const char*) = getobjnameprop;

typedef enum{
	CET_SPEECH,
	CET_CHOICE,
	CET_SETFLAG,
	CET_CHECKFLAG,
	CET_CHECKOBJECT,
	CET_TRANSFEROBJECT,
	CET_MOVECAMERA,
	CET_ANIMATION,
	CET_TRADE,
	CET_JUMP,
	CET_RANDOMLABEL,
	CET_TRIGGER,
	CET_ADDGOAL,
	CET_ADDNOTE,
	CET_ADDSKILLPOINTS,
	CET_ADDCREDITS,
	CET_CHECKPERSONA,
	CET_COMMENT,
	CET_END,
	
	CETS_NUM
}coneventtype;

static const char *const eventtypestrs[CETS_NUM] = {
	"speech",
	"choice",
	"set_flag",
	"check_flag",
	"check_object",
	"transfer_object",
	"move_camera",
	"animation",
	"trade",
	"jump",
	"random_label",
	"trigger",
	"add_goal",
	"add_note",
	"add_skillpoints",
	"add_credits",
	"check_persona",
	"comment",
	"end"
};

static const char *const eventtypecolors[CETS_NUM] = {
	"#dddddd", // speech - harmaa
	"#aaaaaa", // choice - tummempi harmaa
	"#6666ff", // set flag - sinine
	"#aaaaff", // check flag - hämysempi sinine
	"#99ff99", // check object - hämynen vihree
	"#11ff11", // transfer object - vähemmän hämyne vihree
	"#ffffff", // camera
	"#ffffff", // animation
	"#ffffff", // trade
	"#ffffff", // jump
	"#cc77cc", // random - violetti
	"#44aaaa", // trigger - syaani
	"#ffcccc", // goal - punane
	"#ccffff", // note - vaalee syaani
	"#bb4444", // skillpoints - hämypunane
	"#cccc22", // credits - keltane
	"#aaaa55", // check persona - hämykeltane
	"#ffffff", // comment
	"#ffffff" // end
};

static const char *const choicecolor = "#dddddd";

static const char *const startcolor = "#dd2222";

static const char *const normaledgecolor = "#000000";
static const char *const yesedgecolor = "#44aa44";
static const char *const noedgecolor = "#aa4444";

static const char *const normaledgewidth = "3.0";
static const char *const foreignconvedgewidth = "1.5";

typedef struct convnode convnode;
typedef struct conv conv;

struct convnode{
	const export_entry *obj;
	convnode **inlinks;
	long int inlinksnum;
	convnode **outlinks;
	long int outlinksnum;
	conv *conversation;
	int added_to_stack;
};

struct conv{
	const export_entry *obj;
	convnode *nodes;
	long int nodesnum;
};

static void write_escaped_str_wrap(FILE *out, const char *src, long int srclen, int wraplen, const char *delims){
	if(!src) src = "(null)";
	
	int i;
	if(srclen < 0) srclen = strlen(src);
	if(wraplen < 0){
		wraplen = srclen;
		delims = "";
	}
	
	while(srclen){
		int last;
		if(srclen <= wraplen) last = srclen - 1;
		else{
			last = wraplen - 1;
			while(last >= 0 && !strchr(delims, src[last])) last--;
			if(last < 0) last = wraplen - 1;
		}
		for(i = 0; i <= last; i++){
			if(src[i] == '"') fprintf(out, "\\");
			fprintf(out, "%c", src[i]);
		}
		src += last + 1;
		srclen -= last + 1;
		if(srclen) fprintf(out, "\\n");
	}
		
	return;
}

static void write_node_label(const convnode *node, FILE *out){
	const char *convname = nstrp(node->conversation->obj, "ConName");
	if(convname){
		fprintf(out, "(Conversation '");
		write_escaped_str_wrap(out, convname, -1, 60, " ");
		fprintf(out, "')\\n");
	}
	else fprintf(out, "(Unnamed conversation %s)\\n", node->conversation->obj->objname->str);
	
	coneventtype type = intp(node->obj, "EventType", NULL);
	
	if(type == CET_SPEECH){
		const char *speaker = strp(node->obj, "SpeakerName");
		const char *listener = strp(node->obj, "SpeakingToName");
		const char *speech = strp(refp(node->obj, "ConSpeech"), "Speech");
		write_escaped_str_wrap(out, speaker, -1, 60, " ");
		fprintf(out, " to ");
		write_escaped_str_wrap(out, listener, -1, 60, " ");
		fprintf(out, ":\\n\\\"");
		write_escaped_str_wrap(out, speech, -1, 60, " ");
		fprintf(out, "\\\"");
	}
	else if(type == CET_CHOICE){
		fprintf(out, "Choice");
	}
	else if(type == CET_SETFLAG){
		fprintf(out, "Set flag ");
		write_escaped_str_wrap(out, nstrp(refp(node->obj, "FlagRef"), "FlagName"), -1, 60, " ");
		fprintf(out, " to %s", intp(refp(node->obj, "FlagRef"), "Value", NULL) ? "TRUE" : "FALSE");
	}
	else if(type == CET_CHECKFLAG){
		fprintf(out, "Check if flag\\n");
		const export_entry *firstflag = refp(node->obj, "FlagRef");
		const export_entry *flagref;
		for(flagref = firstflag; flagref; flagref = refp(flagref, "NextFlag")){
			if(flagref != firstflag) fprintf(out, " and\\n");
			write_escaped_str_wrap(out, nstrp(flagref, "FlagName"), -1, 60, " ");
			fprintf(out, " is %s", intp(flagref, "Value", NULL) ? "TRUE" : "FALSE");
		}
	}
	else if(type == CET_CHECKOBJECT){
		fprintf(out, "Does the player have object '");
		write_escaped_str_wrap(out, strp(node->obj, "ObjectName"), -1, 60, " ");
		fprintf(out, "'?");
	}
	else if(type == CET_RANDOMLABEL){
		fprintf(out, "Random");
	}
	else if(type == CET_TRIGGER){
		fprintf(out, "Trigger tag '");
		write_escaped_str_wrap(out, nstrp(node->obj, "TriggerTag"), -1, 60, " ");
		fprintf(out, "'");
	}
	else if(type == CET_TRANSFEROBJECT){
		int count = intp(node->obj, "TransferCount", NULL);
		write_escaped_str_wrap(out, strp(node->obj, "FromName"), -1, 60, " ");
		fprintf(out, " gives %d '", count ? count : 1);
		write_escaped_str_wrap(out, strp(node->obj, "ObjectName"), -1, 60, " ");
		fprintf(out, "' object%s to ", count > 1 ? "s" : "");
		write_escaped_str_wrap(out, strp(node->obj, "ToName"), -1, 60, " ");
		if(node->outlinksnum > 1) fprintf(out, " - is there enough room?");
	}
	else if(type == CET_ADDGOAL){
		const char *goalname = nstrp(node->obj, "GoalName");
		if(intp(node->obj, "bGoalCompleted", NULL)){
			fprintf(out, "Set goal '");
			write_escaped_str_wrap(out, goalname, -1, 60, " ");
			fprintf(out, "' as completed");
		}
		else{
			fprintf(out, "Add goal called '");
			write_escaped_str_wrap(out, goalname, -1, 60, " ");
			fprintf(out, "':\\n\\\"");
			write_escaped_str_wrap(out, strp(node->obj, "GoalText"), -1, 60, " ");
			fprintf(out, "\\\"");
		}
	}
	else if(type == CET_ADDNOTE){
		fprintf(out, "Add note:\\n\\\"");
		write_escaped_str_wrap(out, strp(node->obj, "NoteText"), -1, 60, " ");
		fprintf(out, "\\\"");
	}
	else if(type == CET_ADDSKILLPOINTS){
		int points = intp(node->obj, "PointsToAdd", NULL);
		if(points < 0) fprintf(out, "Remove %d skill points from player", -points);
		else fprintf(out, "Add %d skill points to player", points);
	}
	else if(type == CET_ADDCREDITS){
		long int credits = intp(node->obj, "CreditsToAdd", NULL);
		if(credits < 0) fprintf(out, "Remove %ld credits from player", -credits);
		else fprintf(out, "Add %ld credits to player", credits);
	}
	else if(type == CET_CHECKPERSONA){
		int personatype = intp(node->obj, "PersonaType", NULL);
		int condition = intp(node->obj, "Condition", NULL);
		int value = intp(node->obj, "Value", NULL);
		fprintf(out, "Check if player");
		if(personatype == 0 || personatype == 2){
			static const char *const cmpstrs[] = {"less than", "no more than", "exactly", "at least", "more than"};
			fprintf(out, " has %s %d %s%s",
				(condition < 0 || condition > 4) ? "(unknown condition)" : cmpstrs[condition],
				value,
				personatype == 0 ? "credit" : "skill point",
				value > 1 ? "s" : "");
		}
		else{
			static const char *const cmpstrs[] = {"lower than", "no higher than", "exactly", "at least", "higher than"};
			fprintf(out, "'s health is %s %d",
				(condition < 0 || condition > 4) ? "(unknown condition)" : cmpstrs[condition],
				value);
		}
	}
	else if(type == CET_END){
		fprintf(out, "End of conversation");
	}
	else if(type == CET_COMMENT){
		fprintf(out, "Comment:\\n");
		// TODO: kommenttistringit, terminointi?
		//write_escaped_str_wrap(out, strp(node->obj, "CommentText"), -1, 60, " ");
	}
	else if(type < CETS_NUM) fprintf(out, "%s", eventtypestrs[type]);
	else fprintf(out, "UNKNOWN EVENT TYPE\nNUMBER %u", type);
		
	return;
}

void write_start_label(const export_entry *conversation, FILE *out){
	const char *conname = nstrp(conversation, "ConName");
	if(!conname) conname = "(null)";
	fprintf(out, "Start of conversation %s", conname);
	const export_entry *flagref;
	for(flagref = refp(conversation, "FlagRefList"); flagref; flagref = refp(flagref, "NextFlagRef")){
		fprintf(out, "\\nRequires that flag ");
		write_escaped_str_wrap(out, nstrp(flagref, "FlagName"), -1, 60, " ");
		fprintf(out, " is %s", intp(flagref, "Value", NULL) ? "TRUE" : "FALSE");
	}
}

static void write_choice_label(const export_entry *conevent, long int choiceid, FILE *out){
	const export_entry *ch = refp(conevent, "ChoiceList");
	long int i;
	for(i = 0; i < choiceid; i++) ch = refp(ch, "NextChoice");
	
	const char *text = strp(ch, "ChoiceText");
	
	fprintf(out, "\\\"");
	write_escaped_str_wrap(out, text, -1, 60, " ");
	fprintf(out, "\\\"");
	
	const export_entry *flagref;
	for(flagref = refp(ch, "FlagRef"); flagref; flagref = refp(flagref, "NextFlagRef")){
		fprintf(out, "\\nRequires that flag ");
		write_escaped_str_wrap(out, nstrp(flagref, "FlagName"), -1, 60, " ");
		fprintf(out, " is %s", intp(flagref, "Value", NULL) ? "TRUE" : "FALSE");
	}
	
	return;
}

static void addlink(convnode *from, convnode *to){
	convnode **newlist0 = realloc(from->outlinks, (from->outlinksnum + 1) * sizeof(convnode*));
	if(!newlist0) TODOEXIT("memory allocation failure");
	convnode **newlist1;
	if(to){
		newlist1 = realloc(to->inlinks, (to->inlinksnum + 1) * sizeof(convnode*));
		if(!newlist1) TODOEXIT("memory allocation failure");
	}
	from->outlinks = newlist0;
	from->outlinks[from->outlinksnum] = to;
	from->outlinksnum++;
	if(to){
		to->inlinks = newlist1;
		to->inlinks[to->inlinksnum] = from;
		to->inlinksnum++;
	}
	return;
}

static convnode *findlabel(convnode *nodes, int count, const char *label){
	long int i;
	for(i = 0; i < count; i++){
		if(EMATCH(nodes[i].obj, EM_STRPROP_C("Label", label))) return &nodes[i];
	}
	return NULL;
}

static int uninteresting_node(const convnode *node){
	coneventtype type = intp(node->obj, "EventType", NULL);
	return type == CET_COMMENT || type == CET_MOVECAMERA || type == CET_JUMP;
}

int extract_conversation(const export_entry *speechev, const char *filename){
	int result = 1;
	long int i, j, k;
	
	long int convsnum = COUNTEXPS(EM_CLASSIM(IM_NAME_C("Conversation")));
	conv *convs = malloc(convsnum * sizeof(conv));
	if(!convs) TODOEXIT("memory allocation failure");
	
	for(i = 0; i < convsnum; i++){
		convs[i].nodes = NULL;
		convs[i].nodesnum = 0;
	}
	
	printf("Initializing conversation graph...\n");
	
	j = 0;
	for(i = 0; i < pd.expcount; i++){
		if(EMATCH(&pd.exports[i], EM_CLASSIM(IM_NAME_C("Conversation")))){
			convs[j].obj = &pd.exports[i];
			convs[j].nodesnum = 0;
			
			const export_entry *eventlist = refp(&pd.exports[i], "EventList");
			const export_entry *event;
			for(event = eventlist; event; event = refp(event, "nextEvent")) convs[j].nodesnum++;
			
			convs[j].nodes = malloc(convs[j].nodesnum * sizeof(convnode));
			if(!convs[j].nodes) TODOEXIT("memory allocation failure");
			
			k = 0;
			for(event = eventlist; event; event = refp(event, "nextEvent")){
				convs[j].nodes[k].obj = event;
				convs[j].nodes[k].inlinks = convs[j].nodes[k].outlinks = NULL;
				convs[j].nodes[k].inlinksnum = convs[j].nodes[k].outlinksnum = 0;
				convs[j].nodes[k].conversation = &convs[j];
				k++;
			}
			
			j++;
		}
	}
	
	printf("Done.\nSetting graph relations...\n");
	
	for(i = 0; i < convsnum; i++){
		const export_entry *curconvev = convs[i].obj;
		convnode *curnodes = convs[i].nodes;
		long int curnodesnum = convs[i].nodesnum;
		
		//const char *conname = nstrp(curconvev, "ConName");
		//printf("Conversation %ld, %s (%s):\n", i, conname ? conname : "(no ConName found)", curconvev->objname->str);
		
		for(j = 0; j < curnodesnum; j++){
			convnode *curnode = &curnodes[j];
			const export_entry *curobj = curnode->obj;
			coneventtype type = intp(curobj, "EventType", NULL);
			convnode *nextnode = j < curnodesnum-1 ? &curnodes[j+1] : NULL;
			
			if(type >= CETS_NUM){
				//printf("WARNING: Unknown event type number %u.\n", type);
				continue;
			}
			
			//const char *label = strp(curobj, "Label");
			//printf("\t%-25s (type %-20s)", curobj->objname->str, eventtypestrs[type]);
			//if(label) printf(" (label '%s')", label);
			//printf("\n");
			
			if(type == CET_SPEECH){
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_CHOICE){
				const export_entry *choice = refp(curobj, "ChoiceList");
				//if(!choice) printf("\t\tWARNING: Event has no ChoiceList reference.\n");
				while(choice){
					const char *choicelabel = strp(choice, "ChoiceLabel");
					if(choicelabel){
						convnode *jumptonode = findlabel(curnodes, curnodesnum, choicelabel);
						//if(!jumptonode) printf("\t\tWARNING: Jump from choice to undefined label '%s'.\n", choicelabel);
						addlink(curnode, jumptonode);
					}
					else if(nextnode) addlink(curnode, nextnode);
					
					choice = refp(choice, "NextChoice");
				}
			}
			else if(type == CET_SETFLAG){
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_CHECKFLAG){
				const char *setlabel = strp(curobj, "SetLabel");
				if(setlabel){
					convnode *jumptonode = findlabel(curnodes, curnodesnum, setlabel);
					//if(!jumptonode) printf("\t\tWARNING: Jump from flag check to undefined label '%s'.\n", setlabel);
					addlink(curnode, jumptonode);
				}
				else{
					//printf("\t\tWARNING: Flag check has no SetLabel.\n");
					if(nextnode) addlink(curnode, nextnode);
				}
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_CHECKOBJECT){
				if(nextnode) addlink(curnode, nextnode);
				
				const char *faillabel = strp(curobj, "FailLabel");
				if(faillabel){
					convnode *jumptonode = findlabel(curnodes, curnodesnum, faillabel);
					//if(!jumptonode) printf("\t\tWARNING: Jump from object check to undefined label '%s'.\n", faillabel);
					addlink(curnode, jumptonode);
				}
				else{
					//printf("\t\tWARNING: Object check has no FailLabel.\n");
					if(nextnode) addlink(curnode, nextnode);
				}
			}
			else if(type == CET_TRANSFEROBJECT){
				if(nextnode) addlink(curnode, nextnode);
				const char *faillabel = strp(curobj, "FailLabel");
				if(faillabel){
					convnode *jumptonode = findlabel(curnodes, curnodesnum, faillabel);
					//if(!jumptonode) printf("\t\tWARNING: Jump from object transfer to undefined label '%s'.\n", faillabel);
					addlink(curnode, jumptonode);
				}
			}
			else if(type == CET_MOVECAMERA){
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_ANIMATION){
				//printf("\t\tWARNING: Animation event not supported.\n");
			}
			else if(type == CET_TRADE){
				//printf("\t\tWARNING: Trade event not supported.\n");
			}
			else if(type == CET_JUMP){
				const char *jumplabel = strp(curobj, "JumpLabel");
				if(jumplabel){
					const export_entry *jumpcon = refp(curobj, "JumpCon");					
					if(!jumpcon){
						//printf("\t\tWARNING: Jump has no JumpCon - assuming current conversation.\n");
						jumpcon = curconvev;
					}
					
					// TODO: binäärihaku?
					for(k = 0; k < convsnum; k++){
						if(convs[k].obj == jumpcon) break;
					}
					
					convnode *jumptonode = findlabel(convs[k].nodes, convs[k].nodesnum, jumplabel);
					//if(!jumptonode) printf("\t\tWARNING: Jump to undefined label '%s'.\n", jumplabel);
					addlink(curnode, jumptonode);
				}
				else{
					//printf("\t\tWARNING: Jump has no JumpLabel.\n");
					if(nextnode) addlink(curnode, nextnode);
				}
			}
			else if(type == CET_RANDOMLABEL){
				const unsigned char *randomevdata = getobjdata(curobj);
				
				int labelsnum = readcomp(randomevdata);
				randomevdata += complen(randomevdata);
				
				for(k = 0; k < labelsnum; k++){
					long int labelstrlen = readcomp(randomevdata);
					randomevdata += complen(randomevdata);
					convnode *jumptonode = findlabel(curnodes, curnodesnum, randomevdata);
					
					//if(!jumptonode) printf("\t\tWARNING: Jump from random jump to undefined label '%s'.\n", randomevdata);
					addlink(curnode, jumptonode);
					
					randomevdata += labelstrlen;
				}
			}
			else if(type == CET_TRIGGER){
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_ADDGOAL){
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_ADDNOTE){
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_ADDSKILLPOINTS){
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_ADDCREDITS){
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_CHECKPERSONA){
				const char *jumplabel = strp(curobj, "JumpLabel");
				if(jumplabel){
					convnode *jumptonode = findlabel(curnodes, curnodesnum, jumplabel);
					//if(!jumptonode) printf("\t\tWARNING: Jump from persona check to undefined label '%s'.\n", jumplabel);
					addlink(curnode, jumptonode);
				}
				else{
					//printf("\t\tWARNING: Checkpersona has no JumpLabel.\n");
					if(nextnode) addlink(curnode, nextnode);
				}
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_COMMENT){
				if(nextnode) addlink(curnode, nextnode);
			}
			else if(type == CET_END){
			}
		}
		
		//printf("\n");
	}
	
	printf("Done.\n");
	
	long int totalnodesnum = 0;
	convnode *wantednode = NULL;
	
	for(i = 0; i < convsnum; i++) totalnodesnum += convs[i].nodesnum;
	
	for(i = 0; i < convsnum; i++){
		totalnodesnum += convs[i].nodesnum;
		for(j = 0; j < convs[i].nodesnum; j++){
			if(convs[i].nodes[j].obj == speechev) wantednode = &convs[i].nodes[j];
			convs[i].nodes[j].added_to_stack = 0;
		}
	}
	
	if(wantednode == NULL) TODOEXIT("wanted node not inside any conversation");
	
	FILE *file = fopen(filename, "wb");
	if(!file){
		result = 0;
		goto fail_free_convs_and_end;
	}
	
	convnode **wanted_nodes_stack = malloc(totalnodesnum * sizeof(convnode*));
	if(!wanted_nodes_stack) TODOEXIT("memory allocation failure");
	
	printf("Building graphviz graph...\n");

	fprintf(file,
		"digraph{\n"
        "	graph [pad=\"0,0\", charset=latin1];\n"
        "	node [style=filled, fontsize=10, shape=ellipse, color=\"#000000\", fillcolor=\"#ffffff\"];\n"
        "	edge [color=\"#000000\", penwidth=\"3.0\", arrowhead=normal];\n"
	);

	wanted_nodes_stack[0] = wantednode;
	wantednode->added_to_stack = 1;
	
	long int stacklen = 1;
	while(stacklen){
		convnode *curnode = wanted_nodes_stack[stacklen-1];
		int uninteresting = uninteresting_node(curnode);
		stacklen--;
		
		coneventtype type = intp(curnode->obj, "EventType", NULL);
		if(type >= CETS_NUM) continue;
		
		long int nodeidx = curnode->obj - pd.exports;
		
		const char *curnodename = TEMPSPRINTF1024("node%ld", curnode->obj - pd.exports);
		
		if(curnode == curnode->conversation->nodes){
			convnode *node = curnode;
			while(node && uninteresting_node(node) && node->outlinksnum) node = node->outlinks[0];
			if(node && !uninteresting_node(node) && node->conversation == curnode->conversation){
				const char *startnodename = TEMPSPRINTF1024("start%ld", curnode->conversation->obj - pd.exports);
				
				fprintf(file, "\t%s [label=\"", startnodename);
				write_start_label(curnode->conversation->obj, file);
				fprintf(file, "\", style=bold, shape=invhouse, color=\"%s\"];\n", startcolor);
				
				fprintf(file, "\t%s -> node%ld [arrowhead=none, penwidth=\"1.0\"];\n", startnodename, node->obj - pd.exports);
			}
		}
		
		if(!uninteresting){
			fprintf(file, "\t%s [label=\"", curnodename);
			write_node_label(curnode, file);
			fprintf(file, "\", fillcolor=\"%s\"", eventtypecolors[type]);
			
			if(type == CET_SPEECH) fprintf(file, ", shape=box");
			else if(type == CET_CHOICE) fprintf(file, ", shape=diamond");
			
			fprintf(file, "];\n");
		}
		
		for(i = 0; i < curnode->inlinksnum; i++){
			convnode *innode = curnode->inlinks[i];
			if(innode && !innode->added_to_stack){
				wanted_nodes_stack[stacklen] = innode;
				stacklen++;
				innode->added_to_stack = 1;
			}
		}
		
		const char *unknownlabelnodename = NULL;
		for(i = 0; i < curnode->outlinksnum; i++){
			convnode *outnode = curnode->outlinks[i];
			if(outnode && !outnode->added_to_stack){
				wanted_nodes_stack[stacklen] = outnode;
				stacklen++;
				outnode->added_to_stack = 1;
			}
			
			if(uninteresting) continue;
			
			while(outnode && uninteresting_node(outnode) && outnode->outlinksnum) outnode = outnode->outlinks[0];
			
			if(outnode && uninteresting_node(outnode)) continue;

			if(!outnode && !unknownlabelnodename){
				unknownlabelnodename = TEMPSPRINTF1024("unknownlabel%ld", nodeidx);
				fprintf(file, "\t%s [shape=box, label=\"JUMP TO UNKNOWN LABEL,\\nCONVERSATION ENDS\"];\n", unknownlabelnodename);
			}
			
			const char *endnodename = outnode ? TEMPSPRINTF1024("node%ld", outnode->obj - pd.exports) : unknownlabelnodename;
			
			if(type == CET_CHOICE){
				const char *choicenodename = TEMPSPRINTF1024("choice%ld_%ld", nodeidx, i);
				
				fprintf(file, "\t%s [shape=octagon, fillcolor=\"%s\", label=\"", choicenodename, choicecolor);
				write_choice_label(curnode->obj, i, file);
				fprintf(file, "\"];\n");
				
				fprintf(file, "\t%s -> %s -> %s [", curnodename, choicenodename, endnodename);
				
				if(!outnode) fprintf(file, "penwidth=\"1.0\"");
				else if(curnode->conversation != outnode->conversation) fprintf(file, "penwidth=\"%s\"", foreignconvedgewidth);
				
				fprintf(file, "];\n");
			}
			else{
				fprintf(file, "\t%s -> %s [", curnodename, endnodename);
				
				if(!outnode) fprintf(file, "penwidth=\"1.0\", ");
				else if(curnode->conversation != outnode->conversation) fprintf(file, "penwidth=\"%s\", ", foreignconvedgewidth);
				
				if(curnode->outlinksnum > 1 &&
					(type == CET_CHECKFLAG || type == CET_CHECKOBJECT ||
					type == CET_CHECKPERSONA || type == CET_TRANSFEROBJECT)){
					
					fprintf(file, "color=\"%s\"", i == 0 ? yesedgecolor : noedgecolor);
				}
				
				fprintf(file, "];\n");
			}

		}
	}
	
	fprintf(file, "}\n");
	
	fclose(file);
	
	printf("Done.\n");
	
	free(wanted_nodes_stack);
	
	fail_free_convs_and_end:
	
	for(i = 0; i < convsnum; i++){
		for(j = 0; j < convs[i].nodesnum; j++){
			free(convs[i].nodes[j].inlinks);
			free(convs[i].nodes[j].outlinks);
		}
		free(convs[i].nodes);
	}
	free(convs);
	
	return result;
}
