#include "gui.h"
#if SUPPORT_GUI

#include "misc.h"
#include "packdata.h"
#include "packfind.h"
#include "objtools.h"
#include "extract_speech.h"
#include "extract_conv.h"
#include "visconv.h"
#include "user_command.h"

#include "SDL.h"
#include "SDL_ttf.h"

#include <string.h>
#include <ctype.h>

static const int scrwid = 800;
static const int scrhei = 600;

static const int miscborder = 20;
#define FRAMEWID_D 1
static const int framewid = FRAMEWID_D;
static const int fxborder = FRAMEWID_D + 2;
static const int fyborder = FRAMEWID_D + 2;
static const int slssize = 20;

static const int keyrepeatdelay = 500;
static const int keyrepeatinterval = 30;

static const int doubleclickinterval = 300;

static const int announcelifetime = 3000;

static const SDL_Color textcolorrgb = {255, 255, 255};
static const SDL_Color backgroundcolorrgb = {0, 20, 40};
static const SDL_Color framecolorrgb = {150, 150, 150};
static const SDL_Color fieldnofocuscolorrgb = {0, 70, 120};
static const SDL_Color fieldfocuscolorrgb = {0, 100, 171};
static const SDL_Color cursorcolorrgb = {200, 200, 200};
static const SDL_Color slcolor0rgb = {0, 35, 60};
static const SDL_Color slcolor1rgb = {0, 50, 80};
static const SDL_Color slnofocushicolorrgb = {0, 90, 150};
static const SDL_Color slfocushicolorrgb = {0, 128, 214};
static const SDL_Color sliderbackcolorrgb = {0, 50, 120};
static const SDL_Color sliderbtncolorrgb = {0, 20, 100};
static const SDL_Color disabledbtncolorrgb = {40, 40, 40};
static const SDL_Color defbtncolorrgb = {128, 0, 0};
static const SDL_Color disableddefbtncolorrgb = {90, 90, 90};
static const SDL_Color defbtnfocuscolorrgb = {170, 0, 0};

static Uint32 backgroundcolor;
static Uint32 framecolor;
static Uint32 fieldnofocuscolor;
static Uint32 fieldfocuscolor;
static Uint32 cursorcolor;
static Uint32 slcolor0;
static Uint32 slcolor1;
static Uint32 slnofocushicolor;
static Uint32 slfocushicolor;
static Uint32 sliderbackcolor;
static Uint32 sliderbtncolor;
static Uint32 disabledbtncolor;
static Uint32 defbtncolor;
static Uint32 disableddefbtncolor;
static Uint32 defbtnfocuscolor;

static TTF_Font *font;
static int fhei;
static SDL_Surface *screen;
static int tfwid;
static int tfhei;
static int slwid;
static int slhei;
static int slsx;
static int slsy;
static int slshei;
static int slrowsnum;
static int btnwid;
static int btnhei;
static int tfmaxkeycharwid;

static int mousehold = 0;

static int announcetxtx;
static int announcetxty;
static int announceset;
static Uint32 announcetxtstarttime;

typedef struct{
	int x, y;
	const char key;
	const char *name;
	char text[512];
	int textstartidx;
	int cursoridx;
	int hasfocus;
}textfield;

typedef struct{
	stringbuffer data;
	struct{
		long int dataoffset;
		int parity;
		long int payload;
	}*strings;
	long int stringsnum;
	long int stringscap;
}stringlist;

typedef struct{
	int x, y;
	stringlist rows;
	long int upidx;
	long int hilightpayload;
	int hasfocus;
	
	char packprefix[512];
	
	int scrollhold;
	Uint32 scrollholdnexttime;
	
	int sliderdrag;
	int mdy;
	long int mdupidx;
	
	int waitdoubleclick;
	Uint32 hilightedclicktime;
	int doubleclicked;
}speechlist;

typedef struct{
	int x, y;
	const char *name;
	int down;
	int hasfocus;
	int enabled;
}button;

static Uint32 surfclr(const SDL_Surface *surf, SDL_Color clr){
	return SDL_MapRGB(surf->format, clr.r, clr.g, clr.b);
}

static Uint32 scrclr(SDL_Color clr){
	return surfclr(screen, clr);
}

static void delete_chars(char *txt, int first, int last){
	if(first > last) return;
	int tailnum = strlen(txt) - last;
	int i;
	for(i = 0; i < tailnum; i++) txt[first+i] = txt[last+1+i];
	return;
}

static int charclass(char ch){
	if(isspace(ch)) return 1<<8;
	else if(ispunct(ch)) return ch;
	return (1<<8)+1;
}

static int charclassspan_fwd(const char *txt, int start){
	int startclass = charclass(txt[start]);
	int i;
	for(i = start+1; txt[i] && charclass(txt[i]) == startclass; i++);
	return i-1;
}

static int charclassspan_bwd(const char *txt, int start){
	int startclass = charclass(txt[start]);
	int i;
	for(i = start-1; i >= 0 && charclass(txt[i]) == startclass; i--);
	return i+1;
}

static int charspan_bwd(const char *txt, int start, int pred(int)){
	int nstartpred = !pred(txt[start]);
	int i;
	for(i = start-1; i >= 0 && !pred(txt[i]) == nstartpred; i--);
	return i+1;
}

static int latin1_printable(int c){
	return c > 31 && c < 255 && !(c >= 127 && c <= 159);
}

static int textwid(const char *text){
	int res;
	if(TTF_SizeText(font, text, &res, NULL)) TODOEXIT("TTF_SizeText failed");
	return res;
}

static int char_x(const char *text, int idx){
	if(idx < 0) return -1;
	char buf[1024];
	char *copy = idx+1 <= (int)sizeof(buf) ? buf : malloc(idx+1);
	strncpy(copy, text, idx+1);
	copy[idx] = '\0';
	int res = textwid(copy);
	if(copy != buf) free(copy);
	return res;
}

static int text_width_idx(const char *text, int wid, int find_closest){
	int low = 0, high = strlen(text);
	
	while(low < high){
		int mid = (low + high) / 2;
		
		if(char_x(text, mid) > wid) high = mid;
		else low = mid+1;
	}
	
	if(char_x(text, low) <= wid || find_closest && abs(char_x(text, low) - wid) < abs(char_x(text, low-1) - wid)){
		return low;
	}
	return low-1;
}

static int text_width_idx_wordwrap(const char *text, int wid){
	int initidx = text_width_idx(text, wid, 0);
	if(text[initidx] && !isspace(text[initidx])){
		int idx = charspan_bwd(text, initidx, isspace);
		if(idx == 0) return initidx;
		return idx;
	}
	return initidx;
}

static void unsetannounce(void){
	SDL_FillRect(screen, &(SDL_Rect){announcetxtx, announcetxty, scrwid, fhei}, backgroundcolor);
	announceset = 0;
	return;
}

static void announce(const char *txt, SDL_Color clr){
	unsetannounce();
	SDL_Surface *tsurf = TTF_RenderText_Solid(font, txt, clr);
	SDL_BlitSurface(tsurf, NULL, screen, &(SDL_Rect){announcetxtx, announcetxty});
	SDL_FreeSurface(tsurf);
	announceset = 1;
	announcetxtstarttime = SDL_GetTicks();
	return;
}

static void announceplain(const char *txt){
	announce(txt, (SDL_Color){255, 255, 255});
	return;
}

static void announceplz(const char *txt){
	announce(txt, (SDL_Color){255, 255, 0});
	return;
}

static void announcefail(const char *txt){
	announce(txt, (SDL_Color){255, 0, 0});
	return;
}

static const char *strl_str(const stringlist *sl, long int idx){
	return &sl->data.buf[sl->strings[idx].dataoffset];
}

static int strl_add_str(stringlist *sl, const char *str, long int payload){
	if(sl->stringsnum == sl->stringscap){
		sl->stringscap = sl->stringscap ? sl->stringscap*2 : 1;
		
		void *newstrings = realloc(sl->strings, sl->stringscap*sizeof(*sl->strings));
		if(!newstrings) TODOEXIT("memory allocation failure");
		
		sl->strings = newstrings;
	}
	int newdataidx = sl->data.len+1;
	sbprintf(&sl->data, "%c%s", '\0', str);
	sl->strings[sl->stringsnum].dataoffset = newdataidx;
	sl->strings[sl->stringsnum].parity =
		sl->stringsnum ?
		sl->strings[sl->stringsnum-1].parity ^ (sl->strings[sl->stringsnum-1].payload != payload) :
		0;
	sl->strings[sl->stringsnum].payload = payload;
	sl->stringsnum++;
	return 1;
}

static void strl_add_wrapped_rows(stringlist *dst, const char *text, long int payload, int wrapwid){
	stringbuffer resbuf = {0};
	
	while(*text){
		if(text[0] == ' '){
			text++;
			continue;
		}
		const char *nl = strchr(text, '\n');
		int wrapidx = text_width_idx_wordwrap(text, wrapwid);
		if(nl && nl-text <= wrapidx){
			resbuf.len = 0;
			sbprintf(&resbuf, "%.*s", nl-text, text);
			text = nl + 1;
		}
		else{
			resbuf.len = 0;
			if(text[wrapidx-1] == ' ') sbprintf(&resbuf, "%.*s", wrapidx-1, text);
			else sbprintf(&resbuf, "%.*s", wrapidx, text);
			text += wrapidx;
		}
		
		strl_add_str(dst, resbuf.buf, payload);
	}
	
	free(resbuf.buf);
	
	return;
}

static void tf_scroll_to_cursor(textfield *tf){
	int len = strlen(tf->text);
	int over = char_x(tf->text + tf->textstartidx, len - tf->textstartidx) + 2*fxborder >= tfwid;
	
	while(tf->textstartidx > 0 &&
		(char_x(tf->text + tf->textstartidx, tf->cursoridx - tf->textstartidx) + 2*fxborder < tfwid*1/6 ||
			char_x(tf->text + tf->textstartidx, len - tf->textstartidx) + 2*fxborder < tfwid)){
		
		tf->textstartidx--;
	}
	
	if(!over &&
		char_x(tf->text + tf->textstartidx, len - tf->textstartidx) + 2*fxborder >= tfwid &&
		tf->textstartidx+1 <= tf->cursoridx) tf->textstartidx++;

	while(tf->textstartidx <= len &&
		char_x(tf->text + tf->textstartidx, tf->cursoridx - tf->textstartidx) + 2*fxborder >= tfwid*5/6 &&
		char_x(tf->text + tf->textstartidx, len - tf->textstartidx) + 2*fxborder >= tfwid){
		
		tf->textstartidx++;
	}
	
	return;
}

static void tf_draw(const textfield *tf){
	SDL_Surface *tsurf = TTF_RenderText_Solid(font, (char[2]){tf->key, '\0'}, textcolorrgb);
	SDL_BlitSurface(tsurf, NULL, screen, &(SDL_Rect){tf->x, tf->y});
	SDL_FreeSurface(tsurf);
	
	int box_x = tf->x + tfmaxkeycharwid + miscborder;
	
	tsurf = TTF_RenderText_Solid(font, tf->name, textcolorrgb);
	SDL_BlitSurface(tsurf, NULL, screen, &(SDL_Rect){box_x + tfwid + miscborder, tf->y});
	SDL_FreeSurface(tsurf);
	
	SDL_FillRect(screen, &(SDL_Rect){box_x, tf->y, tfwid, tfhei}, framecolor);
	SDL_FillRect(screen, &(SDL_Rect){box_x + framewid, tf->y + framewid, tfwid - 2*framewid, tfhei - 2*framewid},
		tf->hasfocus ? fieldfocuscolor : fieldnofocuscolor);
	
	if(tf->hasfocus){
		int cursx = char_x(tf->text + tf->textstartidx, tf->cursoridx - tf->textstartidx);
		if(cursx >= 0 && cursx < tfwid){
			SDL_FillRect(screen, &(SDL_Rect){fxborder + box_x + cursx, tf->y + fyborder, 2, tfhei - 2*fyborder}, cursorcolor);
		}
	}
	
	tsurf = TTF_RenderText_Solid(font, tf->text + tf->textstartidx, textcolorrgb);
	SDL_SetClipRect(screen, &(SDL_Rect){box_x + fxborder, tf->y + fyborder, tfwid - 2*fxborder, tfhei - 2*fyborder});
	SDL_BlitSurface(tsurf, NULL, screen, &(SDL_Rect){box_x + fxborder, tf->y + fyborder});
	SDL_FreeSurface(tsurf);
	SDL_SetClipRect(screen, NULL);
	
	return;
}

static void tf_handle_key(textfield *tf, SDL_keysym keysym){
	int ctrl = keysym.mod & KMOD_CTRL;
	
	switch(keysym.sym){
		case SDLK_LEFT:
			if(ctrl){
				if(tf->cursoridx > 0) tf->cursoridx = charclassspan_bwd(tf->text, tf->cursoridx-1);
			}
			else tf->cursoridx = max(tf->cursoridx-1, 0);
			break;
		case SDLK_RIGHT:
			if(ctrl) tf->cursoridx = charclassspan_fwd(tf->text, tf->cursoridx);
			tf->cursoridx = min(tf->cursoridx+1, strlen(tf->text));
			break;
		case SDLK_HOME:
			tf->cursoridx = 0;
			break;
		case SDLK_END:
			tf->cursoridx = strlen(tf->text);
			break;
		case SDLK_BACKSPACE:
			if(ctrl){
				if(tf->cursoridx > 0){
					int from = charclassspan_bwd(tf->text, tf->cursoridx-1);
					delete_chars(tf->text, from, tf->cursoridx-1);
					tf->cursoridx = from;
				}
			}
			else if(tf->cursoridx > 0){
				delete_chars(tf->text, tf->cursoridx-1, tf->cursoridx-1);
				tf->cursoridx--;
			}
			break;
		case SDLK_DELETE:
			if(ctrl){
				if(tf->text[tf->cursoridx]){
					int to = charclassspan_fwd(tf->text, tf->cursoridx);
					delete_chars(tf->text, tf->cursoridx, to);
				}
			}
			else if(tf->text[tf->cursoridx]) delete_chars(tf->text, tf->cursoridx, tf->cursoridx);
			break;
		default:
			if((keysym.unicode & 0xff00) == 0){
				int latin1 = keysym.unicode & 0xff;
				if(latin1_printable(latin1)){
					if(strlen(tf->text)+1 < sizeof(tf->text)){
						int i;
						for(i = strlen(tf->text)+1; i > tf->cursoridx; i--){
							tf->text[i] = tf->text[i-1];
						}
						tf->text[tf->cursoridx] = latin1;
						tf->cursoridx++;
					}
				}
			}
	}
	
	tf_scroll_to_cursor(tf);
}

static void tf_handle_mousedown(textfield *tf, int x, int y){
	int box_x = tf->x + tfmaxkeycharwid + miscborder;
	if(x >= box_x + fxborder && x < box_x + tfwid - fxborder &&
		y >= tf->y + fyborder && y < tf->y + tfhei - fyborder){
		
		tf->hasfocus = 1;
		tf->cursoridx = tf->textstartidx + text_width_idx(tf->text + tf->textstartidx, x-box_x-fxborder, 1);
		tf_scroll_to_cursor(tf);
	}
	
	return;
}

static void spcl_draw(const speechlist *sl){
	Uint32 hiclr = sl->hasfocus ? slfocushicolor : slnofocushicolor;
	
	SDL_FillRect(screen, &(SDL_Rect){sl->x, sl->y, slwid, slhei}, framecolor);
	SDL_FillRect(screen, &(SDL_Rect){
		sl->x + framewid, sl->y + framewid, slwid - 2*framewid, slhei - 2*framewid},
		fieldnofocuscolor);
	
	long int i;
	for(i = 0; i < slrowsnum && i + sl->upidx < sl->rows.stringsnum; i++){
		int parity = sl->rows.strings[i + sl->upidx].parity;
		int hilight = sl->rows.strings[i + sl->upidx].payload == sl->hilightpayload;
		
		SDL_FillRect(screen,
			&(SDL_Rect){sl->x + framewid, sl->y + fyborder + i*fhei, slwid - 2*framewid - slssize, fhei},
			hilight ? hiclr : parity ? slcolor1 : slcolor0);
		SDL_Surface *tsurf = TTF_RenderText_Solid(font, strl_str(&sl->rows, i + sl->upidx), textcolorrgb);
		SDL_BlitSurface(tsurf, NULL, screen, &(SDL_Rect){sl->x + fxborder, sl->y + fyborder + i*fhei});
		SDL_FreeSurface(tsurf);
	}
	
	SDL_FillRect(screen, &(SDL_Rect){sl->x+slsx, sl->y+slsy-slssize, slssize, slssize}, sliderbtncolor);
	SDL_FillRect(screen, &(SDL_Rect){sl->x+slsx, sl->y+slsy+slshei, slssize, slssize}, sliderbtncolor);
	SDL_FillRect(screen, &(SDL_Rect){sl->x+slsx, sl->y+slsy, slssize, slshei}, sliderbackcolor);
	
	if(sl->rows.stringsnum > slrowsnum){
		int slidery = sl->y + slsy + sl->upidx * (slshei - slssize) / (sl->rows.stringsnum - slrowsnum) + 1;
		SDL_FillRect(screen, &(SDL_Rect){sl->x+slsx+1, slidery, slssize-2, slssize-2}, sliderbtncolor);
	}
	
	return;
}

static void spcl_scroll_to_hilight(speechlist *sl){
	long int i;
	for(i = 0; i < sl->rows.stringsnum; i++){
		if(sl->rows.strings[i].payload == sl->hilightpayload){
			long int j;
			for(j = i+1; j < sl->rows.stringsnum && sl->rows.strings[j].payload == sl->hilightpayload; j++);
			if(j > sl->upidx + slrowsnum) sl->upidx = j - slrowsnum;
			if(i < sl->upidx) sl->upidx = i;
			return;
		}
	}
	return;
}

static void spcl_to_next_block(speechlist *sl){
	long int pl = sl->hilightpayload;
	int i;
	for(i = sl->rows.stringsnum - 1; sl->rows.strings[i].payload != pl; i--);
	sl->hilightpayload = sl->rows.strings[min(sl->rows.stringsnum-1, i+1)].payload;
	return;
}

static void spcl_to_prev_block(speechlist *sl){
	long int pl = sl->hilightpayload;
	int i;
	for(i = 0; sl->rows.strings[i].payload != pl; i++);
	sl->hilightpayload = sl->rows.strings[max(0, i-1)].payload;
	return;
}

static void spcl_handle_key(speechlist *sl, SDL_keysym keysym){
	if(!sl->rows.stringsnum) return;
	
	int shift = keysym.mod & KMOD_SHIFT;
	switch(keysym.sym){
		case SDLK_HOME:
			sl->hilightpayload = sl->rows.strings[0].payload;
			spcl_scroll_to_hilight(sl);
			break;
		case SDLK_END:
			sl->hilightpayload = sl->rows.strings[sl->rows.stringsnum-1].payload;
			spcl_scroll_to_hilight(sl);
			break;
		case SDLK_DOWN:
			if(shift) sl->upidx = min(sl->rows.stringsnum - slrowsnum, sl->upidx+1);
			else{
				spcl_to_next_block(sl);
				spcl_scroll_to_hilight(sl);
			}
			break;
		case SDLK_UP:
			if(shift) sl->upidx = max(0, sl->upidx-1);
			else{
				spcl_to_prev_block(sl);
				spcl_scroll_to_hilight(sl);
			}
			break;
		default:;
	}
	
	return;
}

static void spcl_handle_scroll(speechlist *sl, int x, int y, int step){
	if(x >= sl->x + framewid && x < sl->x + slwid - framewid &&
		y >= sl->y + framewid && y < sl->y + slhei - framewid){
		
		sl->upidx = max(0, min(sl->rows.stringsnum - slrowsnum, sl->upidx + step));
	}
	
	return;
}

static void spcl_handle_mousedown(speechlist *sl, int x, int y){
	if(sl->rows.stringsnum > slrowsnum){
		int sliderbtnx = sl->x + slsx;
		int sliderupbtny = sl->y + slsy - slssize;
		int sliderdownbtny = sl->y + slsy + slshei;
		
		if(x >= sliderbtnx && x < sliderbtnx + slssize){
			if(y >= sliderupbtny && y < sliderupbtny + slssize){
				sl->scrollhold = -1;
				sl->scrollholdnexttime = SDL_GetTicks() + keyrepeatdelay;
				spcl_handle_scroll(sl, x, y, sl->scrollhold);
			}
			if(y >= sliderdownbtny && y < sliderdownbtny + slssize){
				sl->scrollhold = 1;
				sl->scrollholdnexttime = SDL_GetTicks() + keyrepeatdelay;
				spcl_handle_scroll(sl, x, y, sl->scrollhold);
			}
		}
		
		int sliderx = sl->x + slsx + 1;
		int slidery = sl->y + slsy + sl->upidx * (slshei - slssize) / (sl->rows.stringsnum - slrowsnum) + 1;
		if(x >= sliderx && x < sliderx + slssize - 2 && y >= slidery && y < slidery + slssize - 2){
			sl->sliderdrag = 1;
			sl->mdy = y;
			sl->mdupidx = sl->upidx;
		}
	}
	if(sl->rows.stringsnum && x >= sl->x && x < sl->x + slsx && y >= sl->y + fyborder && y < sl->y + slhei - fyborder){
		long int origpl = sl->hilightpayload;
		
		int row = sl->upidx + (y - sl->y - fyborder) / fhei;
		row = max(0, min(sl->rows.stringsnum - 1, row));
		sl->hilightpayload = sl->rows.strings[row].payload;
		
		if(sl->waitdoubleclick && sl->hilightpayload == origpl && (int)(SDL_GetTicks() - sl->hilightedclicktime) < doubleclickinterval){
			sl->waitdoubleclick = 0;
			sl->doubleclicked = 1;
		}
		else{
			sl->waitdoubleclick = 1;
			sl->hilightedclicktime = SDL_GetTicks();
		}
		
		sl->hasfocus = 1;
	}
	return;
}

static void spcl_handle_mousehold(speechlist *sl, int x, int y){
	if(sl->rows.stringsnum > slrowsnum){
		if(sl->sliderdrag){
			sl->upidx = sl->mdupidx + divround((y - sl->mdy) * (sl->rows.stringsnum - slrowsnum), slshei - slssize);
			sl->upidx = max(0, min(sl->rows.stringsnum - slrowsnum, sl->upidx));
		}
		
		int sliderbtnx = sl->x + slsx;
		int sliderupbtny = sl->y + slsy - slssize;
		int sliderdownbtny = sl->y + slsy + slshei;
		
		if(sl->scrollhold && sl->scrollholdnexttime <= SDL_GetTicks()){
			if(x >= sliderbtnx && x < sliderbtnx + slssize){
				if(y >= sliderupbtny && y < sliderupbtny + slssize && sl->scrollhold < 0){
					sl->scrollholdnexttime = SDL_GetTicks() + keyrepeatinterval;
					spcl_handle_scroll(sl, x, y, sl->scrollhold);
				}
				if(y >= sliderdownbtny && y < sliderdownbtny + slssize && sl->scrollhold > 0){
					sl->scrollholdnexttime = SDL_GetTicks() + keyrepeatinterval;
					spcl_handle_scroll(sl, x, y, sl->scrollhold);
				}
			}
		}
	}
	
	return;
}

static void spcl_populate_searchresults(
	speechlist *sl,
	const char *con_prefix,
	const char *spoken_text,
	const char *spoken_by,
	const char *spoken_at){
	
	static const char *const con_text_suffix = "Text.u";
	
	char con_text_filename[512];
	if(strlen(con_prefix) + strlen(con_text_suffix) >= sizeof(con_text_filename)) TODOEXIT("file name prefix too long");

	strcpy(con_text_filename, con_prefix);
	strcat(con_text_filename, con_text_suffix);
	
	if(!usepackage(con_text_filename)){
		announcefail("Package file couldn't be opened");
		return;
	}
	
	stringbuffer sbuf = {0};
	
	sl->rows.data.len = 0;
	sl->rows.stringsnum = 0;
	
	long int matchcount = 0;
	long int i;
	for(i = 0; i < pd.expcount; i++){
		if(speecheventobjmatches(&pd.exports[i], spoken_by, spoken_at, spoken_text)){
			const char *speakername = getobjstrprop(&pd.exports[i], "speakerName");
			const char *speakingtoname = getobjstrprop(&pd.exports[i], "speakingToName");
			const char *speechtext = getobjstrprop(getobjexprefprop(&pd.exports[i], "conSpeech"), "speech");
			
			matchcount++;
			
			sbuf.len = 0;
			sbprintf(&sbuf, "[%ld] %s says to %s:\n\"%s\"", matchcount, speakername, speakingtoname, speechtext);
			
			strl_add_wrapped_rows(&sl->rows, sbuf.buf, i, slwid - 2*fxborder - slssize);
			strl_add_str(&sl->rows, "", i);
		}
	}
	
	if(sl->rows.stringsnum) sl->hilightpayload = sl->rows.strings[0].payload;
	sl->upidx = 0;
	
	free(sbuf.buf);
	
	strcpy(sl->packprefix, con_prefix);
	
	return;
}

static void btn_draw(const button *btn, int isdefault){
	SDL_FillRect(screen, &(SDL_Rect){btn->x, btn->y, btnwid, btnhei}, framecolor);
	SDL_FillRect(screen, &(SDL_Rect){btn->x+framewid, btn->y+framewid, btnwid-2*framewid, btnhei-2*framewid},
		!btn->enabled ?
			(isdefault ? disableddefbtncolor : disabledbtncolor) :
			btn->hasfocus ?
				(isdefault ? defbtnfocuscolor : fieldfocuscolor) :
				(isdefault ? defbtncolor : fieldnofocuscolor)
		);
	
	SDL_Surface *tsurf = TTF_RenderText_Solid(font, btn->name, textcolorrgb);
	SDL_BlitSurface(tsurf, NULL, screen, &(SDL_Rect){btn->x + fxborder, btn->y + fyborder});
	SDL_FreeSurface(tsurf);
	
	return;
}

static void btn_handle_mousedown(button *btn, int x, int y){
	if(x >= btn->x && x < btn->x + btnwid && y >= btn->y && y < btn->y + btnhei) btn->down = 1;
	return;
}

static stringbuffer expand_defname(
	const char *unexpanded,
	const export_entry *speechev,
	const char *searchtext){
	
	char speechevidstr[64];
	sprintf(speechevidstr, "%ld", speechev - pd.exports);
	
	stringbuffer expanded = {0};
	
	stringbuffer searchtext_no_ws = {0};
	
	replace(&searchtext_no_ws, searchtext, (const char *const[]){" "}, (const char *const[]){"_"}, NULL, 1, 0);
	
	replace(&expanded, unexpanded,
		(const char *const[]){
			"%speechid",
			"%speaker",
			"%listener",
			"%searchtext"},
		(const char *const[]){
			speechevidstr,
			getobjstrprop(speechev, "speakerName"),
			getobjstrprop(speechev, "speakingToName"),
			searchtext_no_ws.buf},
		NULL, 4, 0);
	
	free(searchtext_no_ws.buf);
	
	return expanded;
}

static Uint32 pushuserevent(Uint32 interval, void *dummy){
	(void)dummy;
	SDL_PushEvent(&(SDL_Event){.type = SDL_USEREVENT});
	return interval;
}

void gui(
	const char *initial_con_prefix,
	const char *initial_spoken_text,
	const char *initial_spoken_by,
	const char *initial_spoken_at,
	const char *initial_soundoutfile,
	const char *initial_graphoutfile,
	const char *initial_usercmd,
	const char *fontfilename,
	int givenfontsize
	#if SUPPORT_VISCONV
	, const char *initial_dotcmd
	#endif
){
	if(!fontfilename) TODOEXIT("font file required (please specify with font-file argument, see readme)");
	
	int fontsize = givenfontsize > 0 ? givenfontsize : 11;
	
	int i;
	
	if(SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER) < 0){
		printf("SDL_GetError says: \"%s\"\n", SDL_GetError());
		TODOEXIT("SDL initialization failed");
	}
	
	SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);
	
	SDL_WM_SetCaption("ueconv version 0", NULL);
	
	SDL_EnableKeyRepeat(keyrepeatdelay, keyrepeatinterval);
	SDL_EnableUNICODE(1);
	
	screen = SDL_SetVideoMode(scrwid, scrhei, 32, SDL_DOUBLEBUF);
	
	backgroundcolor = scrclr(backgroundcolorrgb);
	framecolor = scrclr(framecolorrgb);
	fieldnofocuscolor = scrclr(fieldnofocuscolorrgb);
	fieldfocuscolor = scrclr(fieldfocuscolorrgb);
	cursorcolor = scrclr(cursorcolorrgb);
	slcolor0 = scrclr(slcolor0rgb);
	slcolor1 = scrclr(slcolor1rgb);
	slnofocushicolor = scrclr(slnofocushicolorrgb);
	slfocushicolor = scrclr(slfocushicolorrgb);
	sliderbackcolor = scrclr(sliderbackcolorrgb);
	sliderbtncolor = scrclr(sliderbtncolorrgb);
	disabledbtncolor = scrclr(disabledbtncolorrgb);
	defbtncolor = scrclr(defbtncolorrgb);
	disableddefbtncolor = scrclr(disableddefbtncolorrgb);
	defbtnfocuscolor = scrclr(defbtnfocuscolorrgb);
	
	SDL_FillRect(screen, NULL, backgroundcolor);
	
	TTF_Init();
	printf("Opening font file %s ...\n", fontfilename);
	font = TTF_OpenFont(fontfilename, fontsize);
	if(!font) TODOEXIT("couldn't open font");
	printf("Done.\n");
	TTF_SetFontStyle(font, TTF_STYLE_BOLD);
	fhei = TTF_FontHeight(font);
	
	tfwid = 300;
	tfhei = fhei + 2*fyborder;
	
	textfield tf_con_prefix = {.key = 'p', .name = "Package file name prefix (e.g. DeusExCon)"};
	textfield tf_spoken_text = {.key = 't', .name = "Part of spoken text (e.g. rotten w)"};
	textfield tf_spoken_by = {.key = 'a', .name = "Part of speaker name (e.g. jcd)"};
	textfield tf_spoken_at = {.key = 'b', .name = "Part of listener name (e.g. sandraren)"};
	textfield tf_soundoutfile = {.key = 's', .name = "Speech sound output file"};
	textfield tf_graphoutfile = {.key = 'g', .name = "Conversation graph DOT output file"};
	textfield tf_usercmd = {.key = 'u', .name = "User-defined external command"};
	#if SUPPORT_VISCONV
	textfield tf_dotcmd = {.key = 'd', .name = "DOT command"};
	#endif
	textfield *const tfields[] = {
		&tf_con_prefix, &tf_spoken_text, &tf_spoken_by, &tf_spoken_at,
		&tf_soundoutfile, &tf_graphoutfile, &tf_usercmd,
		#if SUPPORT_VISCONV
		&tf_dotcmd
		#endif
	};

	if(initial_con_prefix) strncpy(tf_con_prefix.text, initial_con_prefix, sizeof(tf_con_prefix.text)-1);
	if(initial_spoken_text) strncpy(tf_spoken_text.text, initial_spoken_text, sizeof(tf_spoken_text.text)-1);
	if(initial_spoken_by) strncpy(tf_spoken_by.text, initial_spoken_by, sizeof(tf_spoken_by.text)-1);
	if(initial_spoken_at) strncpy(tf_spoken_at.text, initial_spoken_at, sizeof(tf_spoken_at.text)-1);
	if(initial_soundoutfile) strncpy(tf_soundoutfile.text, initial_soundoutfile, sizeof(tf_soundoutfile.text)-1);
	if(initial_graphoutfile) strncpy(tf_graphoutfile.text, initial_graphoutfile, sizeof(tf_graphoutfile.text)-1);
	if(initial_usercmd) strncpy(tf_usercmd.text, initial_usercmd, sizeof(tf_usercmd.text)-1);
	#if SUPPORT_VISCONV
	if(initial_dotcmd) strncpy(tf_dotcmd.text, initial_dotcmd, sizeof(tf_dotcmd.text)-1);
	#endif
	
	tfmaxkeycharwid = -1;
	for(i = 0; i < ARRLEN(tfields); i++){
		tfmaxkeycharwid = max(tfmaxkeycharwid, textwid((char[2]){tfields[i]->key, '\0'}));
	}
	
	for(i = 0; i < ARRLEN(tfields); i++){
		tfields[i]->x = miscborder;
		tfields[i]->y = miscborder + i*(miscborder+tfhei);
	}
	
	announcetxtx = miscborder;
	announcetxty = miscborder + ARRLEN(tfields)*(miscborder+tfhei);
	
	slwid = 500;
	slrowsnum = (scrhei - (miscborder + ARRLEN(tfields)*(miscborder+tfhei) + fhei + miscborder + miscborder + 2*fyborder)) / fhei;
	slhei = slrowsnum*fhei + 2*fyborder;
	slsx = slwid - framewid - slssize;
	slsy = framewid + slssize;
	slshei = slhei - 2*framewid - 2*slssize;
	
	speechlist sl_searchresults = {.x = miscborder, .y = miscborder + ARRLEN(tfields)*(miscborder+tfhei) + fhei + miscborder };
	speechlist *const slists[] = {&sl_searchresults};
	
	btnhei = fhei + 2*fyborder;
	
	button btn_search = {.name = "Find matches", .enabled = 1};
	button btn_usercmd = {.name = "Invoke user-defined command"};
	button btn_savespeech = {.name = "Save speech"};
	button btn_savegraph = {.name = "Save conversation graph"};
	#if SUPPORT_VISCONV
	button btn_visconv = {.name = "Visualize conversation graph"};
	#endif
	button *const buttons[] = {&btn_search, &btn_usercmd, &btn_savespeech, &btn_savegraph,
	#if SUPPORT_VISCONV
		&btn_visconv
	#endif
	};
	
	
	btnwid = 0;
	for(i = 0; i < ARRLEN(buttons); i++) btnwid = max(btnwid, textwid(buttons[i]->name));
	btnwid += 2*fxborder;
	
	for(i = 0; i < ARRLEN(buttons); i++){
		buttons[i]->x = miscborder + slwid + miscborder;
		buttons[i]->y = miscborder + ARRLEN(tfields)*(miscborder+tfhei) + i*(miscborder+btnhei) + fhei + miscborder;
	}
	button *defbtn = &btn_usercmd;
	
	#define FOCUSABLESNUM (ARRLEN(tfields) + ARRLEN(slists) + ARRLEN(buttons))
	
	typedef enum{
		FT_TF,
		FT_SPCL,
		FT_BTN
	}focustype;
	
	struct{
		int *f;
		focustype type;
		union{
			textfield *tf;
			speechlist *sl;
			button *btn;
		}o;
	}focuses[FOCUSABLESNUM];
	
	for(i = 0; i < FOCUSABLESNUM; i++){
		int index = i;
		if(index < ARRLEN(tfields)){
			focuses[i].type = FT_TF;
			focuses[i].o.tf = tfields[index];
			focuses[i].f = &focuses[i].o.tf->hasfocus;
		}
		else if((index -= ARRLEN(tfields)) < ARRLEN(slists)){
			focuses[i].type = FT_SPCL;
			focuses[i].o.sl = slists[index];
			focuses[i].f = &focuses[i].o.sl->hasfocus;
		}
		else if((index -= ARRLEN(slists)) < ARRLEN(buttons)){
			focuses[i].type = FT_BTN;
			focuses[i].o.btn = buttons[index];
			focuses[i].f = &focuses[i].o.btn->hasfocus;
		}
	}
	*focuses[0].f = 1;
	
	for(i = 0; i < ARRLEN(slists); i++){
		if(slists[i]->rows.stringsnum) slists[i]->hilightpayload = slists[i]->rows.strings[0].payload;
	}
	
	SDL_PushEvent(&(SDL_Event){.type = SDL_USEREVENT});
	
	while(1){
		int focusidx = 0;
		while(!*focuses[focusidx].f) focusidx++;

		for(i = 0; i < ARRLEN(buttons); i++){
			btn_usercmd.enabled =
				btn_savespeech.enabled =
				btn_savegraph.enabled =
				#if SUPPORT_VISCONV
				btn_visconv.enabled =
				#endif
				!!sl_searchresults.rows.stringsnum;
		}
		if(focuses[focusidx].type == FT_BTN && !focuses[focusidx].o.btn->enabled){
			*focuses[focusidx].f = 0;
			focusidx = 0;
			*focuses[focusidx].f = 1;
		}
		
		SDL_TimerID announcetimer = NULL;
		if(announceset){
			Uint32 announcelived = SDL_GetTicks() - announcetxtstarttime;
			int timertime = max(0, announcelifetime - (int)announcelived);
			announcetimer = SDL_AddTimer(timertime, pushuserevent, NULL);
			if(!announcetimer) TODOEXIT("SDL_AddTimer failed");
		}

		if(mousehold) SDL_PushEvent(&(SDL_Event){.type = SDL_USEREVENT});
		
		SDL_Event ev;
		if(!SDL_WaitEvent(&ev)) TODOEXIT("SDL_WaitEvent failed.\n");
		if(announcetimer && !SDL_RemoveTimer(announcetimer)) TODOEXIT("SDL_RemoveTimer failed");
		int endloop = 0;
		do{
			if(ev.type == SDL_QUIT) endloop = 1;
			else if(ev.type == SDL_KEYDOWN){
				int shift = ev.key.keysym.mod & KMOD_SHIFT;
				
				if(ev.key.keysym.sym == SDLK_ESCAPE) endloop = 1;
				else if(ev.key.keysym.mod & KMOD_LALT){
					if((ev.key.keysym.unicode & 0xff00) == 0){
						int latin1 = ev.key.keysym.unicode & 0xff;
						for(i = 0; i < FOCUSABLESNUM; i++){
							if(focuses[i].type == FT_TF && focuses[i].o.tf->key == latin1){
								*focuses[focusidx].f = 0;
								focusidx = i;
								*focuses[focusidx].f = 1;
							}
						}
					}
				}
				else if(ev.key.keysym.sym == SDLK_TAB){
					*focuses[focusidx].f = 0;
					int ofidx = focusidx;
					do{
						if(shift) focusidx = (focusidx-1+FOCUSABLESNUM) % FOCUSABLESNUM;
						else focusidx = (focusidx+1) % FOCUSABLESNUM;
					}while(focusidx != ofidx && focuses[focusidx].type == FT_BTN && !focuses[focusidx].o.btn->enabled);
					*focuses[focusidx].f = 1;
				}
				else if(ev.key.keysym.sym == SDLK_RETURN){
					if(shift) btn_search.down = 1;
					else{
						if(focuses[focusidx].type == FT_BTN) focuses[focusidx].o.btn->down = 1;
						else if(focuses[focusidx].type == FT_SPCL && focuses[focusidx].o.sl == &sl_searchresults && defbtn->enabled){
							defbtn->down = 1;
						}
					}
				}
				else if(ev.key.keysym.sym == SDLK_SPACE && focuses[focusidx].type == FT_BTN){
					if(
						focuses[focusidx].o.btn == &btn_usercmd
						|| focuses[focusidx].o.btn == &btn_savespeech
						|| focuses[focusidx].o.btn == &btn_savegraph
						#if SUPPORT_VISCONV
						|| focuses[focusidx].o.btn == &btn_visconv
						#endif
						){
						
						defbtn = focuses[focusidx].o.btn;
					}
				}
				else if(focuses[focusidx].type == FT_TF || focuses[focusidx].type == FT_BTN){
					focustype ft = focuses[focusidx].type;
					if(ev.key.keysym.sym == SDLK_DOWN || ev.key.keysym.sym == SDLK_UP){
							*focuses[focusidx].f = 0;
							int ofidx = focusidx;
							do{
								if(ev.key.keysym.sym == SDLK_DOWN) focusidx = (focusidx+1) % FOCUSABLESNUM;
								else focusidx = (focusidx-1+FOCUSABLESNUM) % FOCUSABLESNUM;
							}while(focusidx != ofidx && (focuses[focusidx].type != ft || ft == FT_BTN && !focuses[focusidx].o.btn->enabled));
							*focuses[focusidx].f = 1;
					}
					else{
						if(ft == FT_TF) tf_handle_key(focuses[focusidx].o.tf, ev.key.keysym);
					}
				}
				else if(focuses[focusidx].type == FT_SPCL){
					spcl_handle_key(focuses[focusidx].o.sl, ev.key.keysym);
				}
			}
			else if(ev.type == SDL_MOUSEBUTTONDOWN){
				int x = ev.button.x, y = ev.button.y;
				
				if(ev.button.button == SDL_BUTTON_LEFT){
					for(i = 0; i < ARRLEN(tfields); i++) tf_handle_mousedown(tfields[i], x, y);
					for(i = 0; i < ARRLEN(slists); i++) spcl_handle_mousedown(slists[i], x, y);
					for(i = 0; i < ARRLEN(buttons); i++){
						if(buttons[i]->enabled) btn_handle_mousedown(buttons[i], x, y);
					}
				}
				else if(ev.button.button == SDL_BUTTON_RIGHT){
					for(i = 0; i < ARRLEN(buttons); i++){
						if(
							(buttons[i] == &btn_usercmd
							|| buttons[i] == &btn_savespeech
							|| buttons[i] == &btn_savegraph
							#if SUPPORT_VISCONV
							|| buttons[i] == &btn_visconv
							#endif
							) &&
							x >= buttons[i]->x &&
							x < buttons[i]->x + btnwid &&
							y >= buttons[i]->y &&
							y < buttons[i]->y + btnhei){
							
							defbtn = buttons[i];
						}
					}
				}
				else if(ev.button.button == SDL_BUTTON_WHEELUP){
					for(i = 0; i < ARRLEN(slists); i++) spcl_handle_scroll(slists[i], x, y, -1);
				}
				else if(ev.button.button == SDL_BUTTON_WHEELDOWN){
					for(i = 0; i < ARRLEN(slists); i++) spcl_handle_scroll(slists[i], x, y, 1);
				}
			}
			else if(ev.type == SDL_MOUSEBUTTONUP){
				if(ev.button.button == SDL_BUTTON_LEFT){
					for(i = 0; i < ARRLEN(slists); i++) slists[i]->sliderdrag = slists[i]->scrollhold = 0;
				}
			}
		}while(SDL_PollEvent(&ev));
		if(endloop) break;
		
		int mx, my;
		mousehold = (SDL_GetMouseState(&mx, &my) & SDL_BUTTON(1));
		
		if(mousehold){
			for(i = 0; i < ARRLEN(slists); i++) spcl_handle_mousehold(slists[i], mx, my);
		}
		
		for(i = 0; i < FOCUSABLESNUM; i++){
			if(*focuses[i].f && focusidx != i){
				*focuses[focusidx].f = 0;
				focusidx = i;
				break;
			}
		}
		
		if(sl_searchresults.doubleclicked && defbtn->enabled) defbtn->down = 1;
		
		if(btn_search.down){
			spcl_populate_searchresults(
				&sl_searchresults,
				tf_con_prefix.text,
				tf_spoken_text.text,
				tf_spoken_by.text,
				tf_spoken_at.text);
		}
		if(btn_usercmd.down){
			stringbuffer expanded = expand_defname(
				tf_usercmd.text,
				&pd.exports[sl_searchresults.hilightpayload],
				tf_spoken_text.text);
			
			if(!user_command(&pd.exports[sl_searchresults.hilightpayload], sl_searchresults.packprefix, expanded.buf)){
				announcefail("Output file couldn't be opened");
			}
			else announceplain("Executed");
			free(expanded.buf);
		}
		if(btn_savespeech.down){
            if(*tf_soundoutfile.text){
                stringbuffer expanded = expand_defname(
                    tf_soundoutfile.text,
                    &pd.exports[sl_searchresults.hilightpayload],
                    tf_spoken_text.text);
                
                if(!extract_speech(&pd.exports[sl_searchresults.hilightpayload], sl_searchresults.packprefix, expanded.buf)){
                    announcefail("Speech output file couldn't be opened");
                }
                else announceplain("Speech saved");
                free(expanded.buf);
            }
            else announceplz("Please enter output file name");
		}
		if(btn_savegraph.down){
            if(*tf_graphoutfile.text){
                stringbuffer expanded = expand_defname(
                    tf_graphoutfile.text,
                    &pd.exports[sl_searchresults.hilightpayload],
                    tf_spoken_text.text);
                
                if(!extract_conversation(&pd.exports[sl_searchresults.hilightpayload], expanded.buf)){
                    announcefail("Conversation graph output file couldn't be opened");
                }
                else announceplain("Graph saved");
                free(expanded.buf);
            }
            else announceplz("Please enter output file name");
		}
		#if SUPPORT_VISCONV
		if(btn_visconv.down){
			if(*tf_dotcmd.text){
				int success = visconv(&pd.exports[sl_searchresults.hilightpayload], sl_searchresults.packprefix,
					tf_usercmd.text, tf_dotcmd.text);
				
				screen = SDL_SetVideoMode(scrwid, scrhei, 32, SDL_DOUBLEBUF);
				SDL_FillRect(screen, NULL, backgroundcolor);
				
				if(!success) announcefail("Visual mode failed (please check console output for details)");
			}
			else announceplz("Please enter DOT command");
		}
		#endif
		
		for(i = 0; i < ARRLEN(tfields); i++) tf_draw(tfields[i]);
		for(i = 0; i < ARRLEN(slists); i++) spcl_draw(slists[i]);
		for(i = 0; i < ARRLEN(buttons); i++) btn_draw(buttons[i], defbtn == buttons[i]);
		
		if(announceset && (int)(SDL_GetTicks() - announcetxtstarttime) > announcelifetime) unsetannounce();
		
		for(i = 0; i < ARRLEN(slists); i++) slists[i]->doubleclicked = 0;
		for(i = 0; i < ARRLEN(buttons); i++) buttons[i]->down = 0;
		
		SDL_Flip(screen);
	}
	
	for(i = 0; i < ARRLEN(slists); i++){
		free(slists[i]->rows.data.buf);
		free(slists[i]->rows.strings);
	}
	
	TTF_CloseFont(font);
	TTF_Quit();
	
	SDL_Quit();
	
	return;
}

#else
typedef int nonempty_translation_unit;
#endif
