#include "misc.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#if !(UECONV_NO_SDL)
#include "SDL.h"
#endif

char *strcasestr(const char *haystack, const char *needle){
	int i, j;
	int hslen = strlen(haystack);
	int nlen = strlen(needle);
	for(i = 0; i < hslen - nlen + 1; i++){
		for(j = 0; j < nlen && toupper(haystack[i+j]) == toupper(needle[j]); j++);
		if(j == nlen) return (char*)&haystack[i];
	}
	return NULL;
}

int strcasecmp(const char *a, const char *b){
	int i;
	for(i = 0; toupper(a[i]) == toupper(b[i]) && a[i]; i++);
	return toupper(a[i]) - toupper(b[i]);
}

long int readfile_a(const char *filename, unsigned char **bufptr){
	FILE *file = fopen(filename, "rb");
	if(!file) return -1;
	
	fseek(file, 0, SEEK_END);
	
	long int size = ftell(file);
	*bufptr = malloc(size);
	if(!*bufptr) TODOEXIT("memory allocation failure");
	
	rewind(file);
	int i;
	for(i = 0; i < size; i++) (*bufptr)[i] = fgetc(file);
	
	fclose(file);
	
	return size;
}

int startswith(const char *text, const char *prefix){
	size_t plen = strlen(prefix);
	return strncmp(text, prefix, plen) == 0 && strlen(text) >= plen;
}

int startswithword(const char *text, const char *prefix, const char *delims){
	size_t plen = strlen(prefix);
	return strncmp(text, prefix, plen) == 0 && strlen(text) >= plen && (!text[plen] || strchr(delims, text[plen]));
}

const char *skipchars(const char *ptr, const char *chars){
	while(*ptr && strchr(chars, *ptr)) ptr++;
	return ptr;
}

const char *nextword(const char *ptr, const char *delims){
	while(*ptr && !strchr(delims, *ptr)) ptr++;
	return skipchars(ptr, delims);
}

void outbytes(const unsigned char *ptr, int count){
	printf("%d bytes at %p:\n", count, ptr);
	int i;
	for(i = 0; i < count; i++) printf("%02x ", ptr[i]);
	printf("\n");
	for(i = 0; i < count; i++){
		if(ptr[i] > 31) printf(" %c ", ptr[i]);
		else printf(".. ");
	}
	printf("\n");
	return;
}

const unsigned char *findchars(const unsigned char *haystack, int hslen, const unsigned char *needle){
	int i, j;
	int nlen = strlen(needle);
	for(i = 0; i < hslen - nlen + 1; i++){
		for(j = 0; j < nlen && haystack[i+j] == needle[j]; j++);
		if(j == nlen) return &haystack[i];
	}
	return NULL;
}

const unsigned char *findcharsi(const unsigned char *haystack, int hslen, const unsigned char *needle){
	int i, j;
	int nlen = strlen(needle);
	for(i = 0; i < hslen - nlen + 1; i++){
		for(j = 0; j < nlen && toupper(haystack[i+j]) == toupper(needle[j]); j++);
		if(j == nlen) return &haystack[i];
	}
	return NULL;
}

int prompt_specify_str(char *dst, int dstsize, const char *promptname){
	printf("Please specify %s:\n", promptname);
	fgets(dst, dstsize, stdin);
	char *nlidx = strchr(dst, '\n');
	if(!nlidx) TODOEXIT("string entered is too long");
	*nlidx = '\0';
	return 1;
}

long int intbytes(const unsigned char *ptr, int count){
	int i;
	unsigned long int sign = (0x80UL << (8 * (count-1)));
	unsigned long int mask = sign | (sign-1);
	unsigned long int u = 0;
	for(i = 0; i < count; i++) u |= ptr[i] << (8*i);
	if(u & sign){
		if((u & sign) == u) return 0x80L << (8 * (sizeof(long int) - 1));
		return -(long int)((~u + 1) & mask);
	}
	return (long int)u;
}

int complen(const unsigned char *ptr){
	int i;
	for(i = 1; i < 5 && ptr[i-1] & (i == 1 ? 0x40 : 0x80); i++);
	return i;
}

long int readcomp(const unsigned char *ptr){
	int i;
	int sign = (ptr[0] & 0x80) ? -1 : 1;
	int len = complen(ptr);
	long long int result = ptr[0] & 0x3f;
	for(i = 1; i < len; i++){
		if(i < 5) result |= (ptr[i] & 0x7f) << (6 + (i - 1) * 7);
		else result |= (ptr[i] & 0x1f) << (6 + 3*7);
	}
	return sign*result;
}

long int min(long int a, long int b){
	return a < b ? a : b;
}

long int max(long int a, long int b){
	return a > b ? a : b;
}

long int divround(long int n, long int d){
	if(n < 0){
		if(d < 0) return divround(-n, -d);
		else return -divround(-n, d);
	}
	if(d < 0) return -divround(n, -d);
	return (n + d/2) / d;
}

char *tempsprintf(char *dst, const char *format, ...){
	va_list alist;
	va_start(alist, format);
	vsprintf(dst, format, alist);
	va_end(alist);
	return dst;
}

void replace(stringbuffer *dst, const char *orig, const char *const *from, const char *const *to, int *used, int num, int consume_escapes){
	int i, j;
	
	sbprintf(dst, "");

	int escape = 0;
	for(i = 0; orig[i]; i++){
		if(escape){
			if(consume_escapes) sbprintf(dst, "%c", orig[i]);
			else sbprintf(dst, "\\%c", orig[i]);
			escape = 0;
		}
		else{
			if(orig[i] == '\\') escape = 1;
			else{
				for(j = 0; j < num; j++){
					if(startswith(&orig[i], from[j])) break;
				}
				if(j < num){
					if(used) used[j] = 1;
					sbprintf(dst, "%s", to[j]);
					i += strlen(from[j]) - 1;
				}
				else sbprintf(dst, "%c", orig[i]);
			}
		}
	}
	
	return;
}

int sbprintf(stringbuffer *dst, const char *format, ...){
	va_list alist;
	va_start(alist, format);
	int writtennum = vsnprintf(&dst->buf[dst->len], dst->cap - dst->len, format, alist);
	va_end(alist);
	
	if(dst->cap - dst->len < writtennum + 1){
		while(dst->cap - dst->len < writtennum + 1){
			dst->cap = dst->cap ? dst->cap*2 : 1;
			char *newbuf = realloc(dst->buf, dst->cap*sizeof(char));
			if(!newbuf) TODOEXIT("memory allocation failure");
			
			dst->buf = newbuf;
		}
		va_start(alist, format);
		vsnprintf(&dst->buf[dst->len], dst->cap - dst->len, format, alist);
		va_end(alist);
	}
	
	dst->len += writtennum;
	
	return 1;
}

void fgetl(stringbuffer *dst, FILE *in){
	char buf[512];
	while(1){
		if(!fgets(buf, sizeof(buf), in)){
			sbprintf(dst, "");
			break;
		}
		
		int buflen = strlen(buf);
		int hasnl = buflen && buf[buflen-1] == '\n';
		
		if(hasnl) buf[buflen-1] = '\0';
		
		sbprintf(dst, buf);
		
		if(hasnl || buflen+1 < (int)sizeof(buf)) break;
	}
	return;
}

void todofailmsg(const char *msg, int line, const char *file){
	printf("\nFAILURE: error occurred: %s (line %d, in source file %s).\n", msg, line, file);
	return;
}

void todoexit(const char *msg, int line, const char *file){
	#if !(UECONV_NO_SDL)
	if(SDL_WasInit(SDL_INIT_VIDEO)) SDL_Quit();
	#endif
	
	todofailmsg(msg, line, file);
	
	printf(
		"Proper error handling not implemented, program will exit. Apologies for this.\n"
		"If you don't understand why this error happened, or if you think the error is due to a"
		" malfunction in ueconv itself, please contact the author (see readme), describing the error message above,"
		" and what you did and under what circumstances, in as much detail as possible.\n"
		"\nPlease press enter to close the program.\n");
	
	getchar();
	
	exit(EXIT_FAILURE);
}
