#include "visconv.h"
#if SUPPORT_VISCONV

#include "misc.h"
#include "packdata.h"
#include "objtools.h"
#include "extract_conv.h"
#include "extract_speech.h"
#include "user_command.h"
#include "tmpfiles.h"

#include "SDL.h"
#include "SDL_image.h"

static const int scrwid = 800;
static const int scrhei = 600;

static void setpix(int x, int y, Uint32 c, SDL_Surface *s){
	if(x < 0 || x >= s->w || y < 0 || y >= s->h) return;
	*(Uint32*)((Uint8*)s->pixels + y*s->pitch + x*s->format->BytesPerPixel) = c;
	return;
}

static Uint32 getpix(int x, int y, const SDL_Surface *s){
	if(x < 0 || x >= s->w || y < 0 || y >= s->h) return 0;
	return *(Uint32*)((Uint8*)s->pixels + y*s->pitch + x*s->format->BytesPerPixel);
}

static SDL_Surface *dotgraph(const char *dotcmd){
	const char *dotcmdline = TEMPSPRINTF1024("%s %s -Tpng > %s", dotcmd, tmpfname_dot, tmpfname_png);
	int sysret = system(dotcmdline);
	if(sysret){
		printf("FAILURE: DOT command \"%s\" (without quotes) returned non-zero value %d.\n", dotcmdline, sysret);
		return NULL;
	}
	
	SDL_Surface *res = IMG_Load(tmpfname_png);
	if(!res) TODOEXIT("IMG_Load returned NULL");
	return res;
}

typedef struct{
	int l, u, r, d;
	const export_entry *event;
}rect;

static rect *doteventrects(long int *countstorage, int imgwid, int imghei, const char *dotcmd){
	const char *dotcmdline = TEMPSPRINTF1024("%s %s -Tplain > %s", dotcmd, tmpfname_dot, tmpfname_plain);
	int sysret = system(dotcmdline);
	if(sysret){
		printf("FAILURE: DOT command \"%s\" (without quotes) returned non-zero value %d.\n", dotcmdline, sysret);
		return NULL;
	}
	
	FILE *file = fopen(tmpfname_plain, "rb");
	if(!file) TODOEXIT("couldn't open input file");

	int count = 0;
	rect *rects = NULL;
	
	float graphwid = -1.0f, graphhei = -1.0f;
	
	stringbuffer logicalline = {0};
	stringbuffer physicalline = {0};
	
	while(1){
		physicalline.len = 0;
		fgetl(&physicalline, file);
		if(feof(file)) break;
		sbprintf(&logicalline, "%s", physicalline.buf);

		if(physicalline.len && physicalline.buf[physicalline.len-1] == '\\') logicalline.len--;
		else{
			if(startswith(logicalline.buf, "graph ")){
				if(graphwid > 0.0f || graphhei > 0.0f) TODOEXIT("duplicate \"graph\" line in plain-format graph file");
				if(sscanf(logicalline.buf, "graph 1 %f %f", &graphwid, &graphhei) < 2){
					TODOEXIT("invalid \"graph\" line in plain-format graph file");
				}
			}
			else if(startswith(logicalline.buf, "node node")){
				if(graphwid <= 0.0f || graphhei <= 0.0f) TODOEXIT("no \"graph\" line before \"node\" line in plain-format graph file");
				
				long int id = -1;
				float x, y, wid, hei;
				if(sscanf(logicalline.buf, "node node%ld %f %f %f %f", &id, &x, &y, &wid, &hei) < 5){
					TODOEXIT("invalid \"node\" line in plain-format graph file");
				}
				
				if(id < 0 || id >= pd.expcount) TODOEXIT("invalid event id for node in plain-format graph file");
				
				if(getobjintprop(&pd.exports[id], "EventType", NULL) == 0 /*speech*/){
					rect *newrects = realloc(rects, (count+1)*sizeof(rect));
					if(!newrects) TODOEXIT("memory allocation failure");
					rects = newrects;
					rects[count].l = (x - wid*0.5f) * imgwid / graphwid;
					rects[count].r = (x + wid*0.5f) * imgwid / graphwid;
					rects[count].u = imghei - (y + hei*0.5f) * imghei / graphhei;
					rects[count].d = imghei - (y - hei*0.5f) * imghei / graphhei;
					rects[count].event = &pd.exports[id];
					count++;
				}
			}
			
			logicalline.len = 0;
		}
	}
	
	free(physicalline.buf);
	free(logicalline.buf);
	
	fclose(file);
	
	*countstorage = count;
	
	return rects;
}

void drawrect(int l, int u, int r, int d, int width, SDL_Surface *s){
	int maxv = min(s->w - 1, r);
	int i, j;
	for(i = max(l, 0); i <= maxv; i++){
		for(j = -width/2; j < width/2 + width%2; j++){
			setpix(i, u+j, 0xff0000, s);
			setpix(i, d+j, 0xff0000, s);
		}
	}
	maxv = min(s->h - 1, d);
	for(i = max(u, 0); i <= maxv; i++){
		for(j = -width/2; j < width/2 + width%2; j++){
			setpix(l+j, i, 0xff0000, s);
			setpix(r+j, i, 0xff0000, s);
		}
	}
	return;
}

int getr(Uint32 c){
	return (c & 0xff0000) >> 16;
}

int getg(Uint32 c){
	return (c & 0x00ff00) >> 8;
}

int getb(Uint32 c){
	return c & 0x0000ff;
}

SDL_Surface **gen_scaled(SDL_Surface *base, int *numstorage){
	int i;
	int num = 1;
	
	int w = base->w, h = base->h;
	while(w > 1 || h > 1){
		w = (w + 1) / 2;
		h = (h + 1) / 2;
		num++;
	}
	
	SDL_Surface **array = malloc(num * sizeof(SDL_Surface*));
	if(!array) TODOEXIT("memory allocation failure");
	
	w = base->w;
	h = base->h;
	
	array[0] = base;
	
	const SDL_PixelFormat *fmt = base->format;
	
	for(i = 1; i < num; i++){
		w = (w + 1) / 2;
		h = (h + 1) / 2;
		
		array[i] = SDL_CreateRGBSurface(0, w, h, 32, fmt->Rmask, fmt->Gmask, fmt->Bmask, fmt->Amask);
		if(!array[i]) TODOEXIT("SDL_CreateRGBSurface returned NULL");
		
		const SDL_Surface *prev = array[i - 1];
		int x, y;
		for(y = 0; y < h; y++){
			for(x = 0; x < w; x++){
				Uint32 clr0 = getpix(2*x, 2*y, prev);
				Uint32 clr1 = getpix(2*x + 1, 2*y, prev);
				Uint32 clr2 = getpix(2*x, 2*y + 1, prev);
				Uint32 clr3 = getpix(2*x + 1, 2*y + 1, prev);
				int r = (getr(clr0) + getr(clr1) + getr(clr2) + getr(clr3)) / 4;
				int g = (getg(clr0) + getg(clr1) + getg(clr2) + getg(clr3)) / 4;
				int b = (getb(clr0) + getb(clr1) + getb(clr2) + getb(clr3)) / 4;
				Uint32 clr = (r << 16) | (g << 8) | b;
				setpix(x, y, clr, array[i]);
			}
		}
	}
	
	*numstorage = num;
	
	return array;
}

int visconv(const export_entry *speechev, const char *con_prefix, const char *usercmd, const char *dotcmd){
	int result = 1;
	
	if(!dotcmd){
		printf("FAILURE: DOT command unspecified.\n");
		goto fail_end;
	}
	
	Uint32 sdlwasinit = SDL_WasInit(SDL_INIT_VIDEO);
	
	if(!sdlwasinit){
		if(!SDL_Init(SDL_INIT_VIDEO) < 0){
			printf("SDL_GetError says: \"%s\"\n", SDL_GetError());
			TODOEXIT("SDL initialization failed");
		}
		
		SDL_WM_SetCaption("ueconv version 0", NULL);
	}
	
	printf("Generating graph...\n");
	if(!extract_conversation(speechev, tmpfname_dot)) TODOEXIT("couldn't open output file");
	printf("Done.\n");
	printf("Generating png from graph...\n");
	SDL_Surface *img = dotgraph(dotcmd);
	if(!img){
		result = 0;
		goto fail_quit_sdl_and_end;
	}
	printf("Done.\n");
	
	printf("Converting graph to plain format and reading box coordinates from it...\n");
	long int evrectcount;
	rect *eventrects = doteventrects(&evrectcount, img->w, img->h, dotcmd);
	if(!eventrects){
		result = 0;
		goto fail_free_img_and_end;
	}
	printf("Done.\n");
	
	SDL_Surface *s = SDL_SetVideoMode(scrwid, scrhei, 32, SDL_DOUBLEBUF);
	if(!s) TODOEXIT("SDL_SetVideoMode returned NULL");
	
	SDL_Surface *temp = SDL_DisplayFormat(img);
	if(!temp) TODOEXIT("SDL_DisplayFormat returned NULL");
	SDL_FreeSurface(img);
	img = temp;
	
	printf("Generating scaled images...\n");
	int scalednum;
	SDL_Surface **scaled_arr = gen_scaled(img, &scalednum);
	printf("Done.\n");
	
	Uint8 mousedown = 0;
	
	int viewx = 0, viewy = 0;
	int scalelevel = 0;
	int scale = 1;
	
	while((scaled_arr[scalelevel]->w > s->w || scaled_arr[scalelevel]->h > s->h) && scalelevel < scalednum - 1){
		scalelevel++;
		scale *= 2;
	}
	
	int m1px, m1py;
	
	printf("Drag with left button, invoke user-defined external command with right. Use mouse wheel to zoom.\n");
	
	Uint8 mousemotionorigstate = SDL_EventState(SDL_MOUSEMOTION, SDL_QUERY);
	SDL_EventState(SDL_MOUSEMOTION, SDL_ENABLE);
	
	SDL_PushEvent(&(SDL_Event){.type = SDL_USEREVENT});
	
	int prevviewx = viewx, prevviewy = viewy;
	int prevscalelevel = scalelevel;
	long int prevrectidx = -1;
	while(1){
		SDL_Event ev;
		if(!SDL_WaitEvent(&ev)) TODOEXIT("SDL_WaitEvent failed.\n");
		
		int endloop = 0;
		do{
			if(ev.type == SDL_QUIT) endloop = 1;
			else if(ev.type == SDL_KEYDOWN){
				if(ev.key.keysym.sym == SDLK_ESCAPE) endloop = 1;
			}
			else if(ev.type == SDL_MOUSEBUTTONDOWN){
				int origscale = scale;
				if(ev.button.button == SDL_BUTTON_WHEELUP && scalelevel > 0){
					scalelevel--;
					scale /= 2;
				}
				else if(ev.button.button == SDL_BUTTON_WHEELDOWN && scalelevel < scalednum - 1){
					scalelevel++;
					scale *= 2;
				}
				
				if(scale != origscale){
					int imgmx = viewx + ev.button.x*origscale;
					int imgmy = viewy + ev.button.y*origscale;
					
					viewx = imgmx - (imgmx - viewx)*scale/origscale;
					viewy = imgmy - (imgmy - viewy)*scale/origscale;
				}
			}
		}while(SDL_PollEvent(&ev));
		if(endloop) break;
		
		int mx, my;
		Uint8 mousestate = SDL_GetMouseState(&mx, &my);
		
		if(mousestate & SDL_BUTTON(1)){
			if(!(mousedown & SDL_BUTTON(1))){
				m1px = mx;
				m1py = my;
			}
			viewx += (m1px - mx) * scale;
			viewy += (m1py - my) * scale;
			m1px = mx;
			m1py = my;
			mousedown |= SDL_BUTTON(1);
		}
		else mousedown &= ~SDL_BUTTON(1);
		
		if(viewx < -s->w*scale/2) viewx = -s->w*scale/2;
		if(viewy < -s->h*scale/2) viewy = -s->h*scale/2;
		
		if(viewx > -s->w*scale/2 + img->w) viewx = -s->w*scale/2 + img->w;
		if(viewy > -s->h*scale/2 + img->h) viewy = -s->h*scale/2 + img->h;
		
		long int rectidx = -1;
		long int i;
		int imgmx = viewx + mx*scale, imgmy = viewy + my*scale;
		
		for(i = 0; i < evrectcount; i++){
			if(eventrects[i].l <= imgmx && imgmx <= eventrects[i].r &&
				eventrects[i].u <= imgmy && imgmy <= eventrects[i].d){
				rectidx = i;
			}
		}
		
		if(mousestate & SDL_BUTTON(3)){
			if(!(mousedown & SDL_BUTTON(3))){
				if(rectidx >= 0){
					if(usercmd) user_command(eventrects[rectidx].event, con_prefix, usercmd);
					else printf("No user-defined external command specified.\n");
				}
			}
			mousedown |= SDL_BUTTON(3);
		}
		else mousedown &= ~SDL_BUTTON(3);
		
		if(viewx != prevviewx || viewy != prevviewy || scalelevel != prevscalelevel || rectidx != prevrectidx){
			SDL_FillRect(s, NULL, 0);
			SDL_BlitSurface(scaled_arr[scalelevel], NULL, s, &(SDL_Rect){-viewx/scale, -viewy/scale});

			if(rectidx >= 0){
				drawrect(
					(eventrects[rectidx].l - viewx) / scale, (eventrects[rectidx].u - viewy) / scale,
					(eventrects[rectidx].r - viewx) / scale, (eventrects[rectidx].d - viewy) / scale,
					max(4/scale, 1),
					s
				);
			}
			
			SDL_Flip(s);
			
			prevviewx = viewx;
			prevviewy = viewy;
			prevscalelevel = scalelevel;
			prevrectidx = rectidx;
		}
	}
	
	SDL_EventState(SDL_MOUSEMOTION, mousemotionorigstate);
	
	int i;
	for(i = 1; i < scalednum; i++) SDL_FreeSurface(scaled_arr[i]);
	free(scaled_arr);
	
	free(eventrects);
	
	fail_free_img_and_end:
	
	SDL_FreeSurface(img);
	
	fail_quit_sdl_and_end:
	
	if(!sdlwasinit) SDL_Quit();
	
	fail_end:
	
	return result;
}

#else
typedef int nonempty_translation_unit;
#endif
