#include "misc.h"
#include "packdata.h"
#include "packfind.h"
#include "objtools.h"
#include "extract_conv.h"
#include "extract_speech.h"
#include "visconv.h"
#include "user_command.h"
#include "gui.h"
#include "tmpfiles.h"
#include "debug_tools.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

static long int findspeecheventobjs(const char *spoken_by, const char *spoken_at, const char *text, long int listlen){
	long int i;
	long int matchcount = 0;
	for(i = 0; i < pd.expcount; i++){
		if(speecheventobjmatches(&pd.exports[i], spoken_by, spoken_at, text)){
			const char *speakername = getobjstrprop(&pd.exports[i], "speakerName");
			const char *speakingtoname = getobjstrprop(&pd.exports[i], "speakingToName");
			const char *speechtext = getobjstrprop(getobjexprefprop(&pd.exports[i], "conSpeech"), "speech");
			
			matchcount++;
			
			printf("[%3ld] %s says to %s:\n\t\"%s\"\n\n", matchcount, speakername, speakingtoname, speechtext);
			if(listlen > 0 && matchcount % listlen == 0){
				printf("(Press <enter> to continue, or q<enter> to stop looking for more)\n");
				int c;
				while((c=getchar()) != 'q' && c != '\n');
				while(c != '\n' && getchar() != '\n');
				if(c == 'q') break;
			}
		}
	}
	return matchcount;
}

static export_entry *findnthspeecheventobj(const char *spoken_by, const char *spoken_at, const char *text, long int idx){
	long int i;
	long int matchcount = 0;
	for(i = 0; i < pd.expcount; i++){
		if(speecheventobjmatches(&pd.exports[i], spoken_by, spoken_at, text)){
			matchcount++;
			if(matchcount == idx) return &pd.exports[i];
		}
	}
	return NULL;
}

int main(int argc, char *argv[]){
	static const char *const con_text_suffix = "Text.u";
	
	inittmpfiles();
	atexit(removetmpfiles);
	atexit(freepackages);

	long int i, j;
	
	const char *con_prefix = NULL;
	const char *spoken_text = NULL;
	const char *spoken_by = NULL;
	const char *spoken_at = NULL;
	const char *usercmd = NULL;
	long int listlen = 0;
	
	const char *speechautoout = NULL;
	long int speechautoidx = 0;
	const char *graphautoout = NULL;
	long int graphautoidx = 0;
	#if SUPPORT_VISCONV
	const char *dotcmd = NULL;
	long int visconvautoidx = 0;
	#endif
	
	#if SUPPORT_GUI
	const char *fontfilename = NULL;
	long int fontsize = 0;
	#endif
	
	const char *debugfilename = NULL;
	
	int allowcache = 1;
	
	int use_gui =
	#if SUPPORT_GUI
	1
	#else
	0
	#endif
	;
	
	int ignoreunknownoptions = 0;
	
	const struct{
		const char *flag, **pointer;
	}strargs[] = {
		{"speaker=", &spoken_by},
		{"a=", &spoken_by},
		{"listener=", &spoken_at},
		{"b=", &spoken_at},
		{"spoken-text=", &spoken_text},
		{"t=", &spoken_text},
		{"con-file-prefix=", &con_prefix},
		{"p=", &con_prefix},
		{"user-command=", &usercmd},
		{"u=", &usercmd},
		{"speech-out-file=", &speechautoout},
		{"s=", &speechautoout},
		{"graph-out-file=", &graphautoout},
		{"g=", &graphautoout},
		#if SUPPORT_GUI
		{"font-file=", &fontfilename},
		{"f=", &fontfilename},
		#endif
		#if SUPPORT_VISCONV
		{"dot-command=", &dotcmd},
		{"d=", &dotcmd},
		#endif
		
		{"debug-file=", &debugfilename}
	};
	
	const struct{
		const char *flag;
		long int *pointer;
	}intargs[] = {
		{"list-length=", &listlen},
		{"L=", &listlen},
		{"speech-auto-idx=", &speechautoidx},
		{"S=", &speechautoidx},
		{"graph-auto-idx=", &graphautoidx},
		{"G=", &graphautoidx},
		#if SUPPORT_VISCONV
		{"visual-auto-idx=", &visconvautoidx},
		{"V=", &visconvautoidx},
		#endif
		#if SUPPORT_GUI
		{"font-size=", &fontsize},
		{"F=", &fontsize}
		#endif
	};
	
	const struct{
		const char *flag;
		int *pointer, setvalue;
	}valsetargs[] = {
		{"ignore-unknown-options", &ignoreunknownoptions, 1},
		{"z", &ignoreunknownoptions, 1},
		{"no-ignore-unknown-options", &ignoreunknownoptions, 0},
		{"nz", &ignoreunknownoptions, 0},
		
		{"cache", &allowcache, 1},
		{"c", &allowcache, 1},
		{"no-cache", &allowcache, 0},
		{"nc", &allowcache, 0},
		
		#if SUPPORT_GUI
		{"graphical-interface", &use_gui, 1},
		{"r", &use_gui, 1},
		#endif
		{"no-graphical-interface", &use_gui, 0},
		{"nr", &use_gui, 0}
	};
	
	int unknownoptionsfound = 0;
	for(i = 1; i < argc; i++){
		for(j = 0; j < ARRLEN(strargs); j++){
			if(startswith(argv[i], strargs[j].flag)){
				*strargs[j].pointer = argv[i] + strlen(strargs[j].flag);
				break;
			}
		}
		if(j < ARRLEN(strargs)) continue;
		
		for(j = 0; j < ARRLEN(intargs); j++){
			if(startswith(argv[i], intargs[j].flag)){
				*intargs[j].pointer = strtol(argv[i] + strlen(intargs[j].flag), NULL, 0);
				break;
			}
		}
		if(j < ARRLEN(intargs)) continue;
		
		for(j = 0; j < ARRLEN(valsetargs); j++){
			if(!strcmp(argv[i], valsetargs[j].flag)){
				*valsetargs[j].pointer = valsetargs[j].setvalue;
				break;
			}
		}
		if(j < ARRLEN(valsetargs)) continue;
		
		unknownoptionsfound = 1;
		printf("Unknown option: %s.\n", argv[i]);
	}
	
	if(unknownoptionsfound){
		if(ignoreunknownoptions) printf("Ignoring unknown options.\n");
		else{
			printf("FAILURE: Unknown options found.\n");
			goto end;
		}
	}
	
	pd_usecache = allowcache;
	
	if(debugfilename) debug_file(debugfilename);
	else if(use_gui){		
		#if SUPPORT_GUI
		gui(con_prefix, spoken_text, spoken_by, spoken_at,
			speechautoout, graphautoout,
			usercmd, fontfilename,
			fontsize
			#if SUPPORT_VISCONV
			, dotcmd
			#endif
		);
		#endif
	}
	else{
		char con_prefix_array[512];
		if(!con_prefix){
			if(!prompt_specify_str(con_prefix_array, sizeof(con_prefix_array), "the file name prefix for the packages")){
				goto end;
			}
			con_prefix = con_prefix_array;
		}
		
		char con_text_filename[512];
		if(strlen(con_prefix) + strlen(con_text_suffix) >= sizeof(con_text_filename)){
			printf("FAILURE: File name prefix too long.\n");
			goto end;
		}
		strcpy(con_text_filename, con_prefix);
		strcat(con_text_filename, con_text_suffix);
		
		if(!usepackage(con_text_filename)) goto end;
		
		char spoken_text_array[512];
		if(!spoken_text){
			if(!prompt_specify_str(spoken_text_array, sizeof(spoken_text_array), "the text to search (just <enter> to accept anything)")){
				goto end;
			}
			spoken_text = spoken_text_array;
		}
		
		char spoken_by_array[512];
		if(!spoken_by){
			if(!prompt_specify_str(spoken_by_array, sizeof(spoken_by_array), "the name of the speaker (just <enter> to accept anything)")){
				goto end;
			}
			spoken_by = spoken_by_array;
		}
		
		char spoken_at_array[512];
		if(!spoken_at){
			if(!prompt_specify_str(spoken_at_array, sizeof(spoken_at_array), "the name of the listener (just <enter> to accept anything)")){
				goto end;
			}
			spoken_at = spoken_at_array;
		}
		
		char usercmd_array[512];
		
		if(speechautoidx){
			if(speechautoidx <= 0) printf("Invalid (non-positive) index %ld.\n", speechautoidx);
			else{
				const export_entry *ev = findnthspeecheventobj(spoken_by, spoken_at, spoken_text, speechautoidx);
				if(!ev) printf("Index %ld too high.\n", speechautoidx);
				else{
					if(!speechautoout) printf("Speech auto-output file required.\n");
					else extract_speech(ev, con_prefix, speechautoout);
				}
			}
		}
		if(graphautoidx){
			if(graphautoidx <= 0) printf("Invalid (non-positive) index %ld.\n", graphautoidx);
			else{
				const export_entry *ev = findnthspeecheventobj(spoken_by, spoken_at, spoken_text, graphautoidx);
				if(!ev) printf("Index %ld too high.\n", graphautoidx);
				else{
					if(!graphautoout) printf("Graph auto-output file required.\n");
					else extract_conversation(ev, graphautoout);
				}
			}
		}
		#if SUPPORT_VISCONV
		if(visconvautoidx){
			if(visconvautoidx <= 0) printf("Invalid (non-positive index %ld.\n", visconvautoidx);
			else{
				const export_entry *ev = findnthspeecheventobj(spoken_by, spoken_at, spoken_text, visconvautoidx);
				if(!ev) printf("Index %ld too high.\n", visconvautoidx);
				else if(!visconv(ev, con_prefix, usercmd, dotcmd)) printf("FAILURE: Visual mode failed.\n");
			}
		}
		#endif
		if(!(speechautoidx || graphautoidx
			#if SUPPORT_VISCONV
			|| visconvautoidx
			#endif
		)){
			
			int printlist = 1;
			int printhelp = 1;
			int loop = 1;
			while(loop){
				if(printlist){
					printlist = 0;
					if(findspeecheventobjs(spoken_by, spoken_at, spoken_text, listlen) == 0) printf("(No matches found)\n");
				}
				
				if(printhelp){
					printf("Commands:\n");
					printf(
						"s <number> <filename> : write speech to file\n");
						
					if(usercmd){
						printf(
						"p <number>            : invoke user-defined external command\n");
					}
					
					printf(
						"g <number> <filename> : write graph of conversation in dot format to file\n"
						#if SUPPORT_VISCONV
						"v <number>            : go to visual mode with conversation\n"
						#endif
						"a <name>              : re-specify speaker name (now '%s')\n"
						"b <name>              : re-specify listener name (now '%s')\n"
						"t <text>              : re-specify speech text (now '%s')\n",
						spoken_by, spoken_at, spoken_text);
					printf(
						"u <text>              : ");
					if(usercmd) printf("re-specify user-defined external command (now '%s')\n", usercmd);
					else printf("specify user-defined external command\n");
					
					printf(
						"l                     : list search results\n"
						"h                     : print this help message\n"
						"q                     : quit\n");
					
					printhelp = 0;
				}
				else printf("(Press h<enter> for help)\n");
				
				int cmdchar;
				while(isspace(cmdchar=getchar()));
				
				if(cmdchar == 'p' && usercmd || cmdchar == 's' || cmdchar == 'g'
					#if SUPPORT_VISCONV
					|| cmdchar == 'v'
					#endif
				){
					
					long int chosen_idx = -1;
					scanf("%ld", &chosen_idx);
					
					const export_entry *speechev = findnthspeecheventobj(spoken_by, spoken_at, spoken_text, chosen_idx);
					
					if(!speechev) printf("Invalid number.\n");
					else{
						if(cmdchar == 's' || cmdchar == 'g'){
							
							getchar();
							char outfile[512];
							fgets(outfile, sizeof(outfile), stdin);
							char *newlptr = strchr(outfile, '\n');
							if(newlptr) *newlptr = '\0';
							else while(getchar() != '\n');
								
							if(cmdchar == 's') extract_speech(speechev, con_prefix, outfile);
							else extract_conversation(speechev, outfile);
						}
						else if(cmdchar == 'p'){
							user_command(speechev, con_prefix, usercmd);
							while(getchar() != '\n');
						}
						#if SUPPORT_VISCONV
						else{
							if(!visconv(speechev, con_prefix, usercmd, dotcmd)) printf("FAILURE: Visual mode failed.\n");
							while(getchar() != '\n');
						}
						#endif
					}
				}
				else if(cmdchar == 'a'){
					getchar();
					fgets(spoken_by_array, sizeof(spoken_by_array), stdin);
					char *newlptr = strchr(spoken_by_array, '\n');
					if(newlptr) *newlptr = '\0';
					else while(getchar() != '\n');
					spoken_by = spoken_by_array;
				}
				else if(cmdchar == 'b'){
					getchar();
					fgets(spoken_at_array, sizeof(spoken_at_array), stdin);
					char *newlptr = strchr(spoken_at_array, '\n');
					if(newlptr) *newlptr = '\0';
					else while(getchar() != '\n');
					spoken_at = spoken_at_array;
				}
				else if(cmdchar == 't'){
					getchar();
					fgets(spoken_text_array, sizeof(spoken_text_array), stdin);
					char *newlptr = strchr(spoken_text_array, '\n');
					if(newlptr) *newlptr = '\0';
					else while(getchar() != '\n');
					spoken_text = spoken_text_array;
				}
				else if(cmdchar == 'u'){
					getchar();
					fgets(usercmd_array, sizeof(usercmd_array), stdin);
					char *newlptr = strchr(usercmd_array, '\n');
					if(newlptr) *newlptr = '\0';
					else while(getchar() != '\n');
					usercmd = usercmd_array;
				}
				else if(cmdchar == 'l'){
					printlist = 1;
					while(getchar() != '\n');
				}
				else if(cmdchar == 'h'){
					printhelp = 1;
					while(getchar() != '\n');
				}
				else if(cmdchar == 'q'){
					loop = 0;
					while(getchar() != '\n');
				}
				else{
					printf("Invalid command.\n");
					while(getchar() != '\n');
				}
			}
		}
	}
	
	end:
	
	// atexit: freepackages();
	
	// atexit: removetmpfiles();
	
	return 0;
}
