//=============================================================================
// PrecipitationNode
//
// - spawns the precipitation for a PrecipitationShelter zone along the line
//	to the next node
//
// - bNullNode is used to allow breaks in the chain
//	in the following example, Xs are null nodes, Os are active nodes
//	the lines show the progression of the chain
//	numbers show the order, 1 being the node referenced by the zone info
//	dots show the range where precipitation will be spawned
//	(overhead view)
//
//
//          X2-------------O3
//         :|:            :|:
//         :|:            :|:
//         :|:            :|:
//          O1             X4
//
//
//	in this example, the first and second nodes may be placed over a doorway,
//	and the third and fourth placed over a window across the room
//
// - the "basic settings" from PrecipitationZone work essentially the same here:
//
//	PrecipRad - distance on either side of the line between this node
//	            to the next that precipitation will be randomly spawned
//
//	PrecipFreq    _ these will usually have to be different from the
//	PrecipDensity   matching PrecipitationZone, since the area covered
//	                between the two nodes will usually be different from
//	                that covering the player in the PrecipitationZone
//
// - all other precipitation settings are retrieved from the PrecipitationZone
//	that the node resides in
//	*MAKE SURE THAT EACH NODE _DOES_ RESIDE IN A PrecipitationZone,
//	 or it will become a null node.  An error message will be reported
//	 in the log if it does happen.*
//
// - To construct the chain of nodes, set each node's Event to the Tag of
//	the next node in the chain.
//	Nodes with no Event set will default bNullNode to true.
//
//	[It's no longer necessary to set NextNode manually.  If you used a
//	 previous version and have PrecipitationNodes with NextNode already
//	 set, it will be used instead of using the Event name.]
//
// ** DO NOT MAKE CHAINS THAT LINK TO THEMSELVES (e.g., making the last node
//	point to the first node), OR YOU WILL CRASH THE GAME WITH INFINITE RECURSION **
//
// v0.2 (c)2005-14 Smoke39 - smoke39@gmail.com
//=============================================================================
class PrecipitationNode extends Precipitator;

#exec TEXTURE IMPORT NAME=RainNode FILE=textures\RainNode.bmp GROUP=Icons Mips=Off


var(PrecipitationNode) bool bNullNode;  // if true, this node does not spawn precipitation between it and its next node;
                                        // it serves only to demark the end point of the previous node's precipitation line


// mappers no longer need to set this; it's only left visible in UED
// so users of the older versions won't be stuck unable to change it
var(_Precip_DEPRECATED_) PrecipitationNode NextNode;


function PreBeginPlay()
{
	local PrecipitationNode n;

	Super.PreBeginPlay();

	if ( NextNode==None && Event!='' )
	{
		foreach AllActors( class'PrecipitationNode', n, Event )
		{
			NextNode = n;
			break;
		}
	}
}

function Initialize()
{
	local PrecipitationZone zi;

	// create our proxy particle
	// particle generators crash without one, so make one regardless of whether we use it
	if ( proxy == None )
	{
		proxy = Spawn( class'ParticleProxy' );
		proxy.bHidden = true;
	}

	zi = PrecipitationZone( Region.Zone );
	// invalid if not in a PrecipitationZone
	if ( zi == None )
	{
		log( "PRECIPITATION ERROR:" @ self @ "not in a PrecipitationZone" );
		InitialState = 'broken';
		if ( NextNode != None )
			NextNode.Initialize();
		return;
	}

	ControllingZone = zi;
//	PrecipClass = zi.PrecipClass;

	if ( NextNode != None )
	{
		if ( !bNullNode )
		{
//			bImpactParticles = zi.bImpactParticles;
			if ( zi.bImpactParticles )
			{
				riseRate          = zi.riseRate;
				ejectSpeed        = zi.ejectSpeed;
				numPerSpawn       = zi.numPerSpawn;
				particleTexture   = zi.particleTexture;
				particleLifeSpan  = zi.particleLifeSpan;
				particleDrawScale = zi.particleDrawScale;
				bParticlesUnlit   = zi.bParticlesUnlit;
				bScale            = zi.bScale;
				bFade             = zi.bFade;
				bRandomEject      = zi.bRandomEject;
				bTranslucent      = zi.bTranslucent;
				bGravity          = zi.bGravity;
				bModulated        = zi.bModulated;
				SetProxyData();
				proxy.bHidden = false;
			}

//			bWaterParticles      = zi.bWaterParticles;
//			WaterImpactClass     = zi.WaterImpactClass;
//			WaterImpactSpawnProb = zi.WaterImpactSpawnProb;

			CalcDensity(zi);
		}

		NextNode.Initialize();
	}
	else
	{
		bNullNode = true;
		// last node handles slantiness
//		bSlanty = zi.bSlanty;
	}
}

// calc PrecipDensity based on the area this segment of the node chain relative to
// the area covered by our containing zone's Precipitation area around the player
function CalcDensity( PrecipitationZone zi )
{
	local float ziArea, area, f;

	PrecipFreq = zi.PrecipFreq;
	ziArea = square( 2*zi.PrecipRad );
	area = VSize( NextNode.Location - Location ) * 2*PrecipRad;
	f = float(zi.PrecipDensity) * area/ziArea + 0.5;
	PrecipDensity = Max( f, 1 );
}

function TurnOn( Actor newTarget )
{
	Target = newTarget;
	bOn = true;
	if ( !bNullNode )
		SetTimer( PrecipFreq, false );

	if ( NextNode != None )
		NextNode.TurnOn( newTarget );
}

function TurnOff()
{
	bOn = false;
	if ( NextNode != None )
		NextNode.TurnOff();
}

function vector RandomSpawn()
{
	local vector dir, v;

	dir = NextNode.Location - Location;

	v.X = FRand() * VSize(dir);
	v.Y = ( FRand() - 0.5 ) * PrecipRad * 2;

	return location + (v >> rotator(dir)) + (FRand()-0.5)*150 * vect(0,0,1);
}

simulated function Tick( float dt )
{
	if ( bOn && bNullNode )
	{
		// last node handles slantiness
		if ( NextNode == None && ControllingZone.bSlanty )
			MakeSlanty();

		return;
	}

	if ( ControllingZone == None )
		log( GetStateName() );
	if ( ControllingZone.bImpactParticles && ParticleIterator(RenderInterface) != None )
		ParticleIterator(RenderInterface).Update( dt );
}

// spawn proxy in Initialize() instead of here
// become invisible, too (setting bHidden to true prevents RenderInterface from initializing)
function PostBeginPlay()
{
	Super(Effects).PostBeginPlay();

	DrawType = DT_None;
	if ( ControllingZone == None )
	{
		log( "PRECIPITATION ERROR:" @ self @ "not initialized; make sure a PrecipitationShelter refers to this chain" );
		InitialState = 'broken';
	}
}

state broken
{
	simulated function Tick( float dt );
	function TurnOn( Actor newTarget );
	function TurnOff();
}

defaultproperties
{
	PrecipRad=325
	PrecipFreq=0.01
	PrecipDensity=5

	DrawType=DT_Sprite
	Texture=Texture'Precipitation.Icons.RainNode'
}
