//=============================================================================
// PrecipitationShelter
//
// - zone info for zones sheltered from precipitation
//
// - triggers precipitation generators (placed by the mapper) outside of
//	openings in the shelter (eg outside windows, doorways, etc)
//
// - FirstNodeTag is the tag of the first PrecipitationNode of the chain of nodes
//	(child class of Precipitator under ParticleGenerator under Effects)
//
// - SoundTag is the tag of ShelterSounds that should fade in when this zone is entered
//	(ShelterSounds is a child of AmbientSound)
//
//	[It's no longer necessary to set FirstNode manually.  If you used
//	 a previous version and have PrecipitationShelters with FirstNode
//	 already set, it will be used instead of using FirstNodeTag.]
//
// v0.2 (c)2005-14 Smoke39 - smoke39@gmail.com
//=============================================================================
class PrecipitationShelter extends ZoneInfo;

#exec TEXTURE IMPORT NAME=ShelterZone FILE=textures\ShelterZone.bmp GROUP=Icons Mips=Off Flags=2


var(PrecipShelter) Name FirstNodeTag;  // tag of the first PrecipitationNode in the chain
var(PrecipShelter) Name SoundTag;      // tag of ShelterSounds to toggle


// mappers no longer need to set this; it's only left visible in UED
// so users of the older versions won't be stuck unable to change it
var(_Precip_DEPRECATED_) PrecipitationNode FirstNode;


function PreBeginPlay()
{
	local PrecipitationNode n;

	Super.PreBeginPlay();

	if ( FirstNode==None && FirstNodeTag!='' )
	{
		foreach AllActors( class'PrecipitationNode', n, FirstNodeTag )
		{
			FirstNode = n;
			break;
		}
	}
}

function BeginPlay()
{
	Super.BeginPlay();

	if ( FirstNode != None )
		FirstNode.Initialize();
}

event ActorEntered( Actor Other )
{
	local ShelterSound s;

	Super.ActorEntered( Other );

	if ( FirstNode!=None && PrecipitationZone(FirstNode.Region.Zone)!=None && !PrecipitationZone(FirstNode.Region.Zone).bWeatherActive )
		return;

	// only react to players
	if ( !Other.IsA('PlayerPawn') )
		return;

	// turn on the nodes
	if ( FirstNode != None && !FirstNode.bOn )
		FirstNode.TurnOn( Other );

	// turn on the ambient sounds
	if ( SoundTag != '' )
		foreach AllActors( class'ShelterSound', s, SoundTag )
			s.FadeIn();
}

event ActorLeaving( Actor Other )
{
	local ShelterSound s;

	Super.ActorLeaving( Other );

	// only react to players
	if ( !Other.IsA('PlayerPawn') )
		return;

	// turn off the nodes
	if ( FirstNode != None && FirstNode.Target == Other && FirstNode.bOn )
		FirstNode.TurnOff();

	// turn off the sounds
	if ( SoundTag != '' )
		foreach AllActors( class'ShelterSound', s, SoundTag )
			s.FadeOut();
}

defaultproperties
{
	Texture=Texture'Precipitation.Icons.ShelterZone'
}
