//=============================================================================
// PrecipitationZone
//
// - zone info for zones with rain, snow, hail, etc.
//
// - default is medium-intensity rain
//
// - also includes a static footstep function for custom player/scripted pawn classes to use
//	(see PrecipDemo.precipDemoPlayer for an example of its use)
//
// v0.2 (c)2005-14 Smoke39 - smoke39@gmail.com
//=============================================================================
class PrecipitationZone extends ZoneInfo;

// outdoor rain sound courtesy of The Nameless Mod by Off Topic Productions
#exec AUDIO IMPORT FILE=sounds\RainCity.wav NAME=RainCity GROUP=Ambient

#exec TEXTURE IMPORT NAME=RainZone FILE=textures\RainZone.bmp GROUP=Icons Mips=Off Flags=2

// ----------------------------------------
// default properties for the mapper to set

// basic precipitation settings
var(Precipitation) bool bWeatherActive;
var(Precipitation) float        PrecipRad;         // radius around player where precipitation will be spawned
var(Precipitation) float        PrecipFreq;        // delay in seconds between batches of precipitation
var(Precipitation) byte         PrecipDensity;     // how many particles to spawn in each batch
var(Precipitation) bool         bSlanty;           // precipitation slants in the opposite direction of the player's velocity
var(Precipitation) bool         bImpactParticles;  // should precipitation make impact particles (eg rain drop splashes)?
var(Precipitation) vector       Wind;              // velocity added to precipitation
                                                   // bSlanty doesn't have to be true for precipitation to point
                                                   // in the direction it is falling if wind is on
var(Precipitation) class<Actor> PrecipClass;       // the class of the falling precipitation
var(Precipitation) class<Actor> AltPrecipClass;    // secondary precipitation type, e.g., hail to mix in with rain
var(Precipitation) float        AltPrecipRate;     // probability that an AltPrecipClass will be spawned instead of a PrecipClass
                                                   // 0=never, 1=always
var(Precipitation) class<Precipitator> PrecipitatorClass;  // advanced - allows for custom Precipitators (like the SnowPrecipitator,
                                                           // which spawns snowflakes differently than the standard Precipitator)

// particle emitter settings for PrecipClass impact (a max of 64 can exist at once)
var(PrecipImpact) float   riseRate;           // how fast do the particles rise
var(PrecipImpact) float   ejectSpeed;         // how fast do the particles get ejected
var(PrecipImpact) int     numPerSpawn;        // max number of particles to spawn per puff
var(PrecipImpact) texture particleTexture;    // replacement texture to use
var(PrecipImpact) float   particleLifeSpan;   // how long each particle lives
var(PrecipImpact) float   particleDrawScale;  // draw scale for each particle
var(PrecipImpact) bool    bParticlesUnlit;    // is each particle unlit?
var(PrecipImpact) bool    bScale;             // scale each particle as it rises?
var(PrecipImpact) bool    bFade;              // fade each particle as it rises?
var(PrecipImpact) bool    bRandomEject;       // random eject velocity vector
var(PrecipImpact) bool    bTranslucent;       // are these particles translucent?
var(PrecipImpact) bool    bGravity;           // are these particles affected by gravity?
var(PrecipImpact) bool    bModulated;         // are these particles modulated?

// settings for impact effect used when precipitation hits water
var(PrecpWtrImpct) bool         bWaterParticles;       // use the same particle effect as solid impacts?
var(PrecpWtrImpct) class<Actor> WaterImpactClass;      // class to spawn when precipitation hits water
var(PrecpWtrImpct) float        WaterImpactSpawnProb;  // probability of spawning this when precipitation htis water
                                                       // 0 = never, 1 = always

// sound settings
var(PrecipSound) Sound PrecipNoise;   // sound that will follow the player
var(PrecipSound) byte  NoiseVolume;   // volume of each sound following the player
var(PrecipSound) int   NoiseMult;     // how many ambient sounds should follow the player
                                      // set this >1 if a single sound at 255 volume isn't loud enough
var(PrecipSound) byte  NoisePitch;    // pitch of each sound following the player
var(PrecipSound) bool  bSplashyFeet;  // is it wet enough to make the player's footsteps splashy?


// -------------------------------------------------
// internal variables that do not concern the mapper

var Precipitator PrecipGen;    // the precipitator, stored so we can kill it when the player leaves the zone
var FollowingSound FSound[4];  // the ambient sound that follows the player around
                               // (so we can fade it out when player leaves)


// static function for making splashy footstep sounds in PrecipitationZones
// for use in custom player classes or scripted pawns
// if P is walking in a rainy zone, play a splashy sound on top of the regular step sound
// returns a volume scalar for the original footstep sound effect (not the AI noise event)
// (the more it gets reduced, the more splashy the footstep will sound overall)
// see PrecipDemo.precipDemoPlayer for an example of this function's use
static function float RainStep( Pawn P, name FloorMaterial, float volume, float range, float pitch )
{
	local Sound splashySound;
	local float splashVol, stepVol;

	if ( P.FootRegion.Zone.IsA('PrecipitationZone') && PrecipitationZone(P.FootRegion.Zone).bWeatherActive
		&& PrecipitationZone(P.FootRegion.Zone).bSplashyFeet
		&& class'Precipitator'.default.bWeatherEnabled && !P.FootRegion.Zone.bWaterZone )
	{
		switch ( Rand(3) )
		{
			case 0:  splashySound=Sound'WaterStep1';  break;
			case 1:  splashySound=Sound'WaterStep2';  break;
			case 2:  splashySound=Sound'WaterStep3';  break;
		}
		switch ( FloorMaterial )
		{
			// carpet soaks up water and gets all splashy
			case 'Textile':
				splashVol = 1.0;
				stepVol   = 0.75;
				break;
			// grass gets really splashy
			case 'Foliage':
				splashVol = 1.0;
				stepVol   = 0.5;
				break;
			// default is a bit splashy
			default:
				splashVol = 0.75;
				stepVol   = 1.0;
		}
		P.PlaySound( splashySound, SLOT_None, splashVol*volume,, range, 2.0*pitch );
		return stepVol;
	}
	return 1.0;
}

event ActorEntered( Actor Other )
{
	Super.ActorEntered( Other );

	if ( !bWeatherActive || PlayerPawn(Other)==None )
		return;

	// start the precipitation
	StartPrecipitator( Other );
	// start the noise
	StartSound( Other );
}

// this is gross, but reusing Precipitators is important for avoiding a crash when rapidly
// transitioning between zones, like when the jumping-out-of-water code spazzes out
function bool PrecipSettingsMatch( Precipitator p )
{
	local PrecipitationZone z;

	// first just make sure it's even the same type of Precipitator
	if ( PrecipitatorClass != p.Class )
		return false;

	z = p.ControllingZone;

	// make sure whether particles are enabled are the same
	if ( z.bImpactParticles != bImpactParticles ||
	     z.bWaterParticles  != bWaterParticles )
			return false;

	// make sure basic settings are the same
	if ( p.PrecipRad        != PrecipRad ||
	     p.PrecipFreq       != PrecipFreq ||
	     p.PrecipDensity    != PrecipDensity ||
	     z.PrecipClass      != PrecipClass ||
	     z.bSlanty          != bSlanty )
			return false;

	// make sure particle effects are the same, if enabled
	if ( bImpactParticles &&
	    (p.riseRate          != riseRate ||
	     p.ejectSpeed        != ejectSpeed ||
	     p.numPerSpawn       != numPerSpawn ||
	     p.particleTexture   != particleTexture ||
	     p.particleLifeSpan  != particleLifeSpan ||
	     p.particleDrawScale != particleDrawScale ||
	     p.bParticlesUnlit   != bParticlesUnlit ||
	     p.bScale            != bScale ||
	     p.bFade             != bFade ||
	     p.bRandomEject      != bRandomEject ||
	     p.bTranslucent      != bTranslucent ||
	     p.bGravity          != bGravity ||
	     p.bModulated        != bModulated) )
			return false;

	// make sure water impact effects are the same, if not particles
	if ( !bWaterParticles &&
	     (z.WaterImpactClass     != WaterImpactClass ||
		z.WaterImpactSpawnProb != WaterImpactSpawnProb) )
			return false;

	// if it made it through that gauntlet, accept it
	return true;
}

function StartPrecipitator( Actor Other )
{
	local Precipitator p;

	if ( PrecipGen != None )
	{
		PrecipGen.TurnOn( Other );
		return;
	}

	// try to recycle the Precipitator if transitioning between equally configured zones
	foreach AllActors( class'Precipitator', p )
	{
		if ( PrecipSettingsMatch(p) )
		{
			PrecipGen = p;
			PrecipGen.TurnOn( Other );
			return;
		}
	}

	PrecipGen = Spawn( PrecipitatorClass );  // Spawn( class'Precipitator' );

	PrecipGen.PrecipRad        = PrecipRad;
	PrecipGen.PrecipFreq       = PrecipFreq;
	PrecipGen.PrecipDensity    = PrecipDensity;
//	PrecipGen.PrecipClass      = PrecipClass;
//	PrecipGen.bSlanty          = bSlanty;
//	PrecipGen.bImpactParticles = bImpactParticles;

	if ( bImpactParticles )
	{
		PrecipGen.riseRate          = riseRate;
		PrecipGen.ejectSpeed        = ejectSpeed;
		PrecipGen.numPerSpawn       = numPerSpawn;
		PrecipGen.particleTexture   = particleTexture;
		PrecipGen.particleLifeSpan  = particleLifeSpan;
		PrecipGen.particleDrawScale = particleDrawScale;
		PrecipGen.bParticlesUnlit   = bParticlesUnlit;
		PrecipGen.bScale            = bScale;
		PrecipGen.bFade             = bFade;
		PrecipGen.bRandomEject      = bRandomEject;
		PrecipGen.bTranslucent      = bTranslucent;
		PrecipGen.bGravity          = bGravity;
		PrecipGen.bModulated        = bModulated;
	}

//	PrecipGen.bWaterParticles      = bWaterParticles;
//	PrecipGen.WaterImpactClass     = WaterImpactClass;
//	PrecipGen.WaterImpactSpawnProb = WaterImpactSpawnProb;

	PrecipGen.TurnOn( Other );
}

function StartSound( Actor Other )
{
	local FollowingSound s;
	local int i;

	if ( PrecipNoise==None || NoiseMult<1 )
		return;

	// although technically less robust, we're just going to assume that if one of them exists, then all of them do
	// because that's how I've coded the rest of the class and this is simpler
	if ( FSound[i] != None )
		return;

	// first look for existing FollowingSounds with the same properties
	// (to reuse them in the case of entering a PrecipitationZone from another PrecipitationZone with the same properties)
	foreach AllActors( class'FollowingSound', s )
	{
		if ( s.AmbientSound==PrecipNoise &&
		     s.FullVol==NoiseVolume &&
		     s.SoundPitch==NoisePitch )
		{
			FSound[i++] = s;
			s.FadeIn( Other );
		}

		if ( i>=NoiseMult || i>=ArrayCount(FSound) )
			return;
	}

	// if we didn't find any/enough to reuse, spawn new ones
	while ( i<NoiseMult && i<ArrayCount(FSound) )
	{
		if ( FSound[i] == None )
		{
			FSound[i] = Spawn( class'FollowingSound' );
			FSound[i].AmbientSound = PrecipNoise;
			FSound[i].FullVol      = NoiseVolume;
			FSound[i].SoundPitch   = NoisePitch;
		}

		FSound[i].FadeIn( Other );
		++i;
	}
}

event ActorLeaving( Actor Other )
{
	local int i;

	Super.ActorLeaving( Other );

	if ( PlayerPawn(Other) == None )
		return;

	// stop the rain
	if ( PrecipGen != None && PrecipGen.Target == Other )
	{
		PrecipGen.TurnOff();
		PrecipGen = None;
	}

	// stop the sound
	for ( i=0; i<ArrayCount(FSound); i++ )
	{
		if ( FSound[i] != None && FSound[i].Owner == Other )
		{
			FSound[i].FadeOut();
			FSound[i] = None;
		}
	}
}

defaultproperties
{
	bWeatherActive=true
	PrecipRad=650
	PrecipFreq=0.01
	PrecipDensity=7
	PrecipClass=class'Precipitation.Raindrop'
	AltPrecipClass=class'Precipitation.Hailstone'
	AltPrecipRate=0.0
	bSlanty=true
	bImpactParticles=true
	PrecipitatorClass=class'Precipitation.Precipitator'

	RiseRate=0
	ejectSpeed=35.0
	numPerSpawn=2
	particleTexture=Texture'Effects.Generated.WtrDrpSmall'
	particleLifeSpan=0.4
	particleDrawScale=0.1
	bParticlesUnlit=true
	bScale=false
	bFade=true
	bRandomEject=true
	bTranslucent=true
	bGravity=true

	bWaterParticles=false
	WaterImpactClass=class'Precipitation.RaindropRing'
	WaterImpactSpawnProb=1.0

	PrecipNoise=Sound'Precipitation.Ambient.RainCity'  // 'Ambient.ambient.ShowER'
	NoiseVolume=255
	NoiseMult=2
	NoisePitch=64  // 50
	bSplashyFeet=true

	Texture=Texture'Precipitation.Icons.RainZone'
}
