//=============================================================================
// Precipitator
//
// - spawns the actual precipitation of a PrecipitationZone
//
// - for internal use; mappers, see child class PrecipitationNode
//
// v0.2 (c)2005-14 Smoke39 - smoke39@gmail.com
//=============================================================================
class Precipitator extends ParticleGenerator;


// basic precipitation settings set by the mapper in the PrecipitationZone that spawned us
// (some made visible in UED for PrecipitationNode subclass for mappers; the rest come from the PrecipitationZone outside)
var(PrecipitationNode)   float        PrecipRad;         // radius around player where precipitation will be spawned
var(_Precip_DEPRECATED_) float        PrecipFreq;        // delay in seconds between batches of precipitation
var(_Precip_DEPRECATED_) byte	        PrecipDensity;     // how many particles to spawn in each batch
//var                class<Actor> PrecipClass;       // the class of the falling precipitation
//var                bool         bSlanty;           // precipitation slants in the opposite direction of the player's velocity
//var                bool         bImpactParticles;  // should precipitation make impact particles (eg rain drop splashes)?


// settings for impact effect used when precipitation hits water
//var bool         bWaterParticles;       // use the same particle effect as solid impacts?
//var class<Actor> WaterImpactClass;      // class to spawn when precipitation hits water
//var float        WaterImpactSpawnProb;  // probability of spawning this when precipitation htis water
                                        // 0 = never, 1 = always

// internal variables
var PrecipitationZone ControllingZone;  // which PrecipitationZone we're spawning precipitation for
var bool  bOn;     // allows us to wait for all precipitation to go away before we die
                   // also used for child to keep track of whether or not we're on
var float Height;  // used to start the precipitation low and move up so there's less pause before it starts hitting the ground


var config bool bWeatherEnabled;  // DeusEx.ini setting to disable weather effects


function TurnOn( Actor newTarget )
{
	ControllingZone = PrecipitationZone( newTarget.Region.Zone );
	proxy.bHidden = !ControllingZone.bImpactParticles;
	LifeSpan = 0;
	Height = 300;
	Target = newTarget;
	SetProxyData();
	SetTimer( PrecipFreq, false );
	bOn = true;
}

function TurnOff()
{
	LifeSpan = ControllingZone.PrecipClass.default.LifeSpan;
	bOn = false;
}

function Timer()
{
	if ( !bOn )
		return;

	// start low and get higher so it doesn't take as long for rain to appear at all altitudes
	if ( Height < 600 )
		Height = FMin( Height + PrecipFreq*600, 600 );

	Precipitate( PrecipDensity );
	SetTimer( PrecipFreq, false );
}

function vector RandomSpawn()
{
	return Target.Location + Target.Velocity*PrecipRad/1200 + PrecipRad/4*vector(Target.Rotation)
		- ControllingZone.Wind + Height*vect(0,0,1)
		+ (FRand()-0.5)*PrecipRad * vect(2,0,0)
		+ (FRand()-0.5)*PrecipRad * vect(0,2,0)
		+ (FRand()-0.5)*150 * vect(0,0,1);
}

function Precipitate( int density )
{
	local int i;

	if ( !bWeatherEnabled )
		return;

	for ( i=0; i<density; i++ )
	{
		if ( FRand() < ControllingZone.AltPrecipRate )
			Spawn( ControllingZone.AltPrecipClass, self,, RandomSpawn() );
		else
			Spawn( ControllingZone.PrecipClass, self,, RandomSpawn() );
	}
}

function bool Recycle( Actor particle )
{
	local int i;

	if ( !bOn )
		return false;

	for ( i=0; i<5; i++ )
	{
		if ( particle.SetLocation(RandomSpawn()) )
		{
			TimerRate += PrecipFreq / PrecipDensity;
			return true;
		}
	}

	return false;
}

function RecycleFailed()
{
	if ( bOn )
		Precipitate( 1 );
}

// find all precipitation and tilt it based on player's velocity
simulated function MakeSlanty()
{
	local vector vel;
	local Actor p;

	vel = 0.1*Target.Velocity;
	vel.Z = 0;
	foreach AllActors( ControllingZone.PrecipClass, p )
//		Precipitation(p).Slantify( vel );
		p.SetRotation( rotator(Velocity-vel) );
}

simulated function Tick( float dt )
{
	if ( ControllingZone.bImpactParticles && ParticleIterator(RenderInterface) != None )
		ParticleIterator(RenderInterface).Update( dt );

//	if ( ControllingZone.bSlanty )
//		MakeSlanty();
}

function PostBeginPlay()
{
	local Actor A;

	Super(Effects).PostBeginPlay();

	// create our proxy particle
	// particle generators crash the game without one, so make one even if we won't use it...
	if ( proxy == None )
	{
		proxy = Spawn( class'ParticleProxy' );
//		if ( !ControllingZone.bImpactParticles )
			proxy.bHidden = true;
	}
}

// called by precipitation, makes particles
function Impact( vector SplashLoc, vector HitNormal )
{
	local int i;

	if ( !ControllingZone.bImpactParticles )
		return;
	if ( ParticleIterator(RenderInterface) == None )
		return;

	pLoc = SplashLoc;
	for ( i=Rand(numPerSpawn); i<numPerSpawn; i++ )
	{
		if ( bRandomEject )
			pRot = rotator(HitNormal+VRand()/2);
		else
			pRot = rotator(HitNormal);

		ParticleIterator(RenderInterface).AddParticle();
	}
}

// called by precipitation, tries to make special water impact effect
function WaterImpact( vector SplashLoc, vector HitNormal )
{
	if ( ControllingZone.bWaterParticles )
	{
		Impact( SplashLoc, HitNormal );
	}
	else if ( ControllingZone.WaterImpactClass!=None && ControllingZone.WaterImpactSpawnProb>0 )
	{
		if ( ControllingZone.WaterImpactSpawnProb < 1 )
		{
			if ( FRand() > ControllingZone.WaterImpactSpawnProb )
				return;
		}

		Spawn( ControllingZone.WaterImpactClass,,, SplashLoc );
	}
}


defaultproperties
{
	bWeatherEnabled=true
	DrawType=DT_None
}
