//=============================================================================
// ShelterSound
//
// - an ambient sound that fades in/out when player enters/exits its controlling
//	PrecipitationShelter zone
//
// - for indoor areas, you'll want to put these with the default sound in places
//	the player should be able to hear the muffled sound of the rain outside,
//	plus some with the outdoor sound just outside doors and such, so that when
//	the player enters the indoor area, they'll still be able to hear the sound
//	of the rain coming through.  Since Deus Ex reduces the volume of sounds
//	that have no line of sight to the player, when the door is closed, the
//	outdoor sound will be reduced when the door is closed.
//
// - for more open shelters (e.g., an overpass, the bunker with the broken power
//	box by the back entrance to the statue on Liberty Island), only the latter,
//	outdoor sound is needed around the edges of the shelter
//
// v0.2 (c)2005-14 Smoke39 - smoke39@gmail.com
//=============================================================================
class ShelterSound extends AmbientSound;


// indoor rain sound courtesy of The Nameless Mod by Off Topic Productions
#exec AUDIO IMPORT FILE=sounds\RainInside.wav NAME=RainInside GROUP=Ambient

#exec TEXTURE IMPORT NAME=ShelterSound FILE=textures\ShelterSound.bmp GROUP=Icons Mips=Off Flags=2


var byte FullVol;  // the volume the sound plays at before beginning to fade out when turned off
var float time;


function BeginPlay()
{
	FullVol = SoundVolume;
	SoundVolume = 0;
	Super.BeginPlay();
}

// no longer set time here, so that when transitioning between two zones that use the same ShelterSounds,
// when the sound is turned off then immediately back on, it'll be fading in from full volume,
// instead of jumping to 0 first
simulated function FadeIn()
{
//	time = 0;
	GotoState( 'FadingIn' );
}

simulated function FadeOut()
{
//	time = 0.5;
	GotoState( 'FadingOut' );
}

auto state idle
{
}

state On
{
Begin:
	if ( class'Precipitator'.default.bWeatherEnabled )
		SoundVolume = FullVol;
	else
		SoundVolume = 0;
	Sleep( 0.1 );
	Goto( 'Begin' );
}

state FadingIn
{
	simulated function Tick( float dt )
	{
		if ( !class'Precipitator'.default.bWeatherEnabled )
		{
			time = 0.5;
			GotoState( 'On' );
			return;
		}

		time += dt;
		if ( time >= 0.5 )
		{
			time = 0.5;
			SoundVolume = FullVol;
//			GotoState( 'idle' );
			GotoState( 'On' );
			return;
		}

		SoundVolume = sqrt(time*2) * FullVol;
	}
}

state FadingOut
{
	simulated function Tick( float dt )
	{
		time -= dt;
		if ( time<=0 || !class'Precipitator'.default.bWeatherEnabled )
		{
			time = 0;
			SoundVolume = 0;
			GotoState( 'idle' );
			return;
		}

		SoundVolume = sqrt(time*2) * FullVol;
	}
}

defaultproperties
{
	AmbientSound=Sound'precipitation.Ambient.RainInside'  // 'Ambient.ambient.ShowER'
	SoundVolume=255
	SoundPitch=64  // 50
	SoundRadius=75

	bStatic=false
	Texture=Texture'Precipitation.Icons.ShelterSound'
}
