//
// UNR2DE
//
// Unreal to Deus Ex mesh converter.  Converts old style Unreal
// and Unreal Tournament _a.3d files into the higher precision
// format that Deus Ex expects.
//
// Takes 1 argument: the "base" filename of a pair of Unreal mesh
// filenames.  For example, if "RoboKitty" is passed in, the
// program will look at the "RoboKitty_a.3d" and "RoboKitty_d.3d"
// files.  It *will* overwrite the "_a.3d" file.
//
// Written by Steve Tack, 10/15/00
//
// Revision History
// ----------------
// v1.1 10/26/00 Changed datatype of NumFrames, FrameSize, NumPolys,
//               and NumVertices to "unsigned short" instead of just
//               "short" to allow for very large meshes to be processed.
//               (frame sizes over 32,000 were going negative).
//

#define WIN32_LEAN_AND_MEAN

#include <stdio.h>
#include <string>

//-----------------------------------------------------------------------

void main( int argc, char* argv[] )
{

	printf("\nUnreal to Deus Ex mesh converter, v1.1");
	printf("\nWritten by Steve Tack\n");

	// Display info if no command-line arguments passed in
	if(argc == 1) {

		printf("\nunr2de will convert an Unreal or Unreal Tournament mesh file");
		printf("\ninto a format suitable for Deus Ex. This allows use of");
		printf("\nexisting tools such as 3ds2unr that do not work with Deus Ex.\n");
		printf("\nUsage: unr2de <part of filename before the \"_\">\n");
		printf("\nFor example:");
		printf("\nunr2de BunnyDroid");
		printf("\nwill convert a file named BunnyDroid_a.3d");
		printf("\nYou will need to have your _d.3d file in the same directory,");
		printf("\nthough this program will only read from it.");
		printf("\nThis program *will* overwrite your _a.3d file, so make a backup");
		printf("\nof it if necessary.\n");
		return;
	}

	// Construct the "_a.3d" and "_d.3d" filenames.  Be sure to allocate an
	// extra 5 characters for the characters at the end.
	char *AnimFileName = new char[strlen(argv[1]) + 6];
	strcpy(AnimFileName, argv[1]);
	strcat(AnimFileName, "_a.3d");

	char *DataFileName = new char[strlen(argv[1]) + 6];
	strcpy(DataFileName, argv[1]);
	strcat(DataFileName, "_d.3d");

	// Open the _a.3d file (binary mode)
	FILE* fp = fopen(AnimFileName, "rb");
	if (fp == 0) {
		printf("\n\nCan't open file %s !\n", AnimFileName);
		return;
	}

	unsigned short NumFrames, FrameSize;
	unsigned short NumPolys, NumVertices;
	unsigned int BytesPerVertex;

	// Read in number of animation frames
	if (fread(&NumFrames, sizeof(NumFrames), 1, fp) != 1) {
		printf("\n\nProblem reading data file: could not read number of frames !\n");
		return;
	}

	// Time to rock and roll
	printf("\nConverting %s...\n", AnimFileName);

	// Display number of animation frames, just because we have it
	// and it's nice to tell the user something's happening
	if (NumFrames == 1)
		printf("\n  Single mesh (no animation)");
	else if (NumFrames > 1)
		printf("\n  %d animation frames", NumFrames);
	else if (NumFrames < 1) {
		printf("\n\nNumber of animation frames invalid !\n");
		return;
	}

	// Read in "frame size" (how many bytes each frame of animation is)
	// For Unreal files, this is 4 bytes * the number of vertices. Don't 
	// display this number to the user, cause he doesn't care
	if (fread(&FrameSize, sizeof(FrameSize), 1, fp) != 1) {
		printf("\n\nProblem reading data file: could not read frame size !\n");
		return;
	}

	// Now temporarily open the "_d.3d" file to grab the number of vertices.
	// The only reason we're doing this is to verify that the input file
	// has not already been converted.  (frame size / number of verticies
	// should be 4 for an Unreal file)
	FILE* fp_datafile = fopen( DataFileName, "rb");
	if (fp_datafile == 0) {
		printf("\n\nCan't open file %s !\n", DataFileName);
		return;
	}

	// Read number of polygons from _d.3d file
	if (fread(&NumPolys, sizeof(NumPolys), 1, fp_datafile) != 1) {
		printf("\n\nProblem reading data file: could not read number of polygons !\n");
		return;
	}
	else
		printf("\n  %d polygons", NumPolys);

	if (NumPolys < 1) {
		printf("\n\nNumber of polygons invalid !");
		return;
	}

	// Read number of vertices from _d.3d file
	if (fread(&NumVertices, sizeof(NumVertices), 1, fp_datafile) != 1) {
		printf("\n\nProblem reading data file: could not read number of vertices !\n");
		return;
	}
	else
		printf("\n  %d vertices", NumVertices);

	if (NumVertices < 1) {
		printf("\n\nNumber of vertices invalid !");
		return;
	}

	// We're done with the "_d.3d" file, so close up
	fclose(fp_datafile);

	// Make sure we've got an Unreal file that hasn't already
	// been converted (Unreal files are 4 bytes per vertex,
	// Deus Ex files are 8 bytes per vertex)
	BytesPerVertex = (FrameSize / NumVertices);

	if (BytesPerVertex == 8) {
		printf("\n\nMesh file has already been converted !\n");
		return;
	}
	else if (BytesPerVertex != 4) {
		printf("\n\nCound not determine vertex format !\n");
		return;
	}

	//
	// Start reading in the vertex coordinates and
	// stick 'em in an array.
	//
	int *VertexCoord = new int[sizeof(int) * NumVertices * NumFrames];
	int arrcnt = 0;
	int i = 0;
	short minX = 9999, maxX = -9999;
	short minY = 9999, maxY = -9999;
	short minZ = 9999, maxZ = -9999;

	while (fread(&VertexCoord[arrcnt], sizeof(VertexCoord), 1, fp) == 1) 
		arrcnt++;

	// Close the "_a.3d" file (no more reading needed)
	fclose(fp);

	// Open up the "_a.3d" for writing (binary mode).
	FILE* fp_write = fopen( AnimFileName, "wb");
	if (fp_write == 0) {
		printf("\n\nCan't open file %s for writing !\n", AnimFileName);
		delete [] VertexCoord;
		return;
	}

	// Write number of animation frames (this will not change from
	// original value)
	if (fwrite(&NumFrames, sizeof(NumFrames), 1, fp_write) != 1) { 
		printf("\n\nProblem writing to file %s !", AnimFileName);
		delete [] VertexCoord;
		return;
	}

	// Write new frame size (double the old one, since we're going
	// from 4 bytes per vertex to 8)
	FrameSize *= 2;
	if (fwrite(&FrameSize, sizeof(FrameSize), 1, fp_write) != 1) { 
		printf("\n\nProblem writing to file %s !\n", AnimFileName);
		delete [] VertexCoord;
		return;
	}

	struct DeusExVertex { 
        short x;
        short y;
        short z;
        short dummy;
    } st;

	memset( &st, '\0', sizeof(st) );
	st.dummy = 0;

	//
	// Convert the vertices and write 'em out!
	//
	for(i = 0; i < arrcnt; i++) {
	
		// Calculate x, y, and z values from the compressed Unreal format
		st.x = (VertexCoord[i] & 0x7FF) / 8;
		if (st.x > 128 ) { st.x -= 256; }
		st.x = -st.x;

		st.y = ((VertexCoord[i] >> 11) & 0x7FF) / 8;
		if (st.y > 128) { st.y -= 256; }
		st.y = -st.y;

		st.z = ((VertexCoord[i] >> 22) & 0x3FF) / 4;
		if (st.z > 128) { st.z -= 256; }

		//
		// Write Deus Ex vertex: x, y, z, and a dummy "0000"
		// For example, a Deus Ex vertex might look like
		// this: 3D00C4FFC3FF0000
		// The first 2 bytes are the X coordinate (will be
		// stored on disk in reversed format, so in this 
		// example, X is 0x003D), the second 2 bytes are 
		// the Y coordinate, the third 2 bytes are the Z 
		// coordinate, and the last 2 bytes are not used.
		//
		if (fwrite(&st, sizeof(st), 1, fp_write) != 1) {
			printf("\n\nCould not write vertex !\n");
			delete [] VertexCoord;
			return;
		}

		// Keep track of min/max values for display at end.  These
		// are for display only
		if (st.x < minX)
			minX = st.x;
		else if (st.x > maxX)
			maxX = st.x;

		if (st.y < minY)
			minY = st.y;
		else if (st.y > maxY)
			maxY = st.y;

		if (st.z < minZ)
			minZ = st.z;
		else if (st.z > maxZ)
			maxZ = st.z;
	}

	fclose(fp_write);

	// Display min/max x, y, and z values (just an FYI to the user)
	printf("\n  X coordinate range: %d to %d", minX, maxX);
	printf("\n  Y coordinate range: %d to %d", minY, maxY);
	printf("\n  Z coordinate range: %d to %d", minZ, maxZ);

	// Success!
	printf("\n\n*** Conversion complete ***\n");

	// Set our memory free
	delete [] VertexCoord;
	delete DataFileName;

}
