# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# (c) dart.weider 2013

import hou
import sys

POINT_IN_FRONT_OF_PLANE = 0
POINT_BEHIND_PLANE 	= 1
POINT_ON_PLANE 		= 2

g_unreal_tournament_99_scl = 52.5 #1 meter is equal to this value

def float_to_fixed_str(flt):
	fltAbs = abs(flt)
	sign = '+'
	if (flt < 0): sign = '-'
	return sign+format(fltAbs,"012f")

def convert_to_unreal(vec):
	x = vec[0]
	y = vec[2]
	z = vec[1]
	return hou.Vector3(x,y,z)

def convert_to_unreal_scaled(vec):
	return (convert_to_unreal(vec) * g_unreal_tournament_99_scl)

def get_point_side(normal, d, pnt):
	dist = normal.dot(pnt) - d
	if dist > 0:
		return POINT_IN_FRONT_OF_PLANE
	elif dist < 0:
		return POINT_BEHIND_PLANE
	else:
		return POINT_ON_PLANE

def get_grp_bbox(primGrp):
	defPos = primGrp.prims()[0].vertex(0).point().position();
	grpBBox = hou.BoundingBox(defPos[0], defPos[1], defPos[2], defPos[0], defPos[1], defPos[2])		
	for prim in primGrp.prims():
		for vtx in prim.vertices():
			pos = vtx.point().position()
			grpBBox.enlargeToContain(pos)
	return grpBBox
	
def get_prim_bbox(prim):
	defPos = prim.vertex(0).point().position()
	bbox = hou.BoundingBox(defPos[0], defPos[1], defPos[2], defPos[0], defPos[1], defPos[2])	
	for vtx in prim.vertices():
		pos = vtx.point().position()
		bbox.enlargeToContain(pos)
	return bbox	

class BrushPoly:
	def __init__(self):
		self.origin = []
		self.normal = []
		self.texU = []
		self.texV = []
		self.vertices = []
		self.texture = ""
		self.item = ""
		self.flags = 0
		self.link = -1
		
	def set(self, org, nrm, u, v, vertices):
		self.origin = org
		self.normal = nrm
		self.texU = u
		self.texV = v
		self.vertices = vertices
		
	def write(self, out):
		out.write("          Begin Polygon")
		if len(self.item):
			out.write(" Item="+self.item)
		if len(self.texture):
			out.write(" Texture="+self.texture)
		if self.flags > 0:
			out.write(" Flags="+str(self.flags))
		if self.link > 0:
			out.write(" Link="+str(self.link))
		out.write("\n")
		out.write("             Origin   " + float_to_fixed_str(self.origin[0]) + "," + float_to_fixed_str(self.origin[1]) + "," + float_to_fixed_str(self.origin[2]) + "\n")
		out.write("             Normal   " + float_to_fixed_str(self.normal[0]) + "," + float_to_fixed_str(self.normal[1]) + "," + float_to_fixed_str(self.normal[2]) + "\n")
		out.write("             TextureU " + float_to_fixed_str(self.texU[0]) + "," + float_to_fixed_str(self.texU[1]) + "," + float_to_fixed_str(self.texU[2]) + "\n")
		out.write("             TextureV " + float_to_fixed_str(self.texV[0]) + "," + float_to_fixed_str(self.texV[1]) + "," + float_to_fixed_str(self.texV[2]) + "\n")
		for vtx in self.vertices:
			out.write("             Vertex   " + float_to_fixed_str(vtx[0]) + "," + float_to_fixed_str(vtx[1]) + "," + float_to_fixed_str(vtx[2]) + "\n")				
		out.write("          End Polygon\n")		
	
class Brush:
	def __init__(self):
		self.name = "noname"
		self.operation = False
		self.subtract = True
		self.polys = []
		self.mainScale = []
		self.postScale = []
		self.dynLight = True
		self.level = ""
		self.tag = "Brush"
		self.location = [0,0,0]

	def init_from_grp(self, primGrp):
		self.name = primGrp.name()
		self.operation = True
		self.subtract = True
				
		uvsclAttr = primGrp.geometry().findPrimAttrib("uvscale")
		uvAttr = primGrp.geometry().findVertexAttrib("uv")
		tangentu = primGrp.geometry().findVertexAttrib("tangentu")
		tangentv = primGrp.geometry().findVertexAttrib("tangentv")
		shopAttr = primGrp.geometry().findPrimAttrib("shop_materialpath")
		
		grpBBox = get_grp_bbox(primGrp)				
		bboxCenter = grpBBox.center()
		nrm = org = hou.Vector3(0,0,0)
		for prim in primGrp.prims():
			poly = BrushPoly()
			positions = []
			nrm = prim.normal()
			org = get_prim_bbox(prim).center()
			
			texU = texV = hou.Vector3(0,0,0)
			uvscale = [1.0 , 1.0]			
			vlist = list(prim.vertices())
			if uvAttr:
				texU = vlist[0].attribValue(tangentu)
				texV = vlist[0].attribValue(tangentv)
				if uvsclAttr:
					uvscale = prim.attribValue(uvsclAttr)
		
		#Try to autodetect brush operation
			side = get_point_side(nrm, nrm.dot(org), bboxCenter)									
			if not side == POINT_IN_FRONT_OF_PLANE:
				self.subtract = False			

			for vtx in vlist:
				pos = vtx.point().position()
				pos = convert_to_unreal_scaled(pos)
				positions.append(pos)
				
			nrm = convert_to_unreal(nrm)
			org = convert_to_unreal_scaled(org)
			texU = convert_to_unreal(texU) * uvscale[0]  
			texV = convert_to_unreal(texV) * uvscale[1]
			
			poly.set(org, nrm, texU, texV, positions)				
		
			if shopAttr:
				mtlPath = prim.attribValue(shopAttr)
				mtlNode = hou.node(mtlPath)
				if mtlNode:
					colorMap = mtlNode.parm("dif_map").eval()
					texNode = mtlNode.node(colorMap)					
					if texNode:
						poly.texture = texNode.name()
					else:
						print "Not found ", colorMap

			self.polys.append(poly)

		#Force Brush operation
		if self.name.find("add") == 0:
			self.subtract = False			
		elif self.name.find("sub") == 0:
			self.subtract = True	

		if self.subtract:			
			for poly in self.polys: 
				poly.vertices = poly.vertices[::-1]				

		return
		
	def init_def_brush(self):
		self.name = "BrushDef"
		self.mainScale = [1,1,1]
		self.postScale = [1,1,1]
		poly0 = BrushPoly()
		poly1 = BrushPoly()		
		poly2 = BrushPoly()
		poly3 = BrushPoly()		
		poly4 = BrushPoly()		
		poly5 = BrushPoly()
		poly0.set([-1,-1,-1],	[-1,0,0],	[0,1,0],	[0,0,-1],	[[-1,-1,-1],	[-1,-1,1],	[-1,1,1],	[-1,1,-1]])		
		poly1.set([-1,1,-1],	[0,1,0],	[1,0,0],	[0,0,-1],	[[-1,1,-1],	[-1,1,1],	[1,1,1],	[1,1,-1]])
		poly2.set([1,1,-1],	[1,0,0],	[0,-1,0],	[0,0,-1],	[[1,1,-1],	[1,1,1],	[1,-1,1],	[1,-1,-1]])
		poly3.set([1,-1,-1],	[0,-1,0],	[-1,0,0],	[0,0,-1],	[[1,-1,-1],	[1,-1,1],	[-1,-1,1],	[-1,-1,-1]])
		poly4.set([-1,1,1],	[0,0,1],	[1,0,0],	[0,1,0],	[[-1,1,1],	[-1,-1,1],	[1,-1,1],	[1,1,1]])
		poly5.set([-1,-1,-1],	[0,0,-1],	[1,0,0],	[0,-1,0],	[[-1,-1,-1],	[-1,1,-1],	[1,1,-1],	[1,-1,-1]])
		self.polys = [poly0, poly1, poly2, poly3, poly4, poly5]

	def get_scale(self, scale):
		return "(SheerAxis=SHEER_ZX)"

	def get_location(self):
		return "(X="+float_to_fixed_str(self.location[0])+",Y="+float_to_fixed_str(self.location[1])+",Z="+float_to_fixed_str(self.location[2])+")"

	def write(self, out):
		print "Actor:", self.name, "Polys:", len(self.polys)
		out.write("Begin Actor Class=Brush Name="+self.name+"\n")
		if self.operation:
			if self.subtract:
				out.write("    CsgOper=CSG_Subtract\n")
			else:
				out.write("    CsgOper=CSG_Add\n")
		out.write("    MainScale="+self.get_scale(self.mainScale)+"\n")
		out.write("    PostScale="+self.get_scale(self.postScale)+"\n")
		out.write("    Tag=Brush\n")
		out.write("    Begin Brush Name=mdl_"+self.name+"\n")
		out.write("       Begin PolyList\n")
		for poly in self.polys:
			poly.write(out)
		out.write("       End PolyList\n")
		out.write("    End Brush\n")
		out.write("    Brush=Model'MyLevel.mdl_"+self.name+"'\n")
		out.write("    Name="+self.name+"\n")			
		out.write("End Actor\n")		
		return
		

class T3D:
	def __init__(self, node):
		self.name = node.parent().name()
		self.geo = node.geometry()
		self.actors = []
		self.brushesSub = []
		self.brushesAdd = []
		self.build()
		
	def build(self):
		defBrush = Brush()
		defBrush.init_def_brush()
		self.actors.append(defBrush)
		for primGrp in self.geo.primGroups():
			grpBrush = Brush()
			grpBrush.init_from_grp(primGrp)
			if grpBrush.subtract:
				self.brushesSub.append(grpBrush)
			else:
				self.brushesAdd.append(grpBrush)
		self.actors = self.actors + self.brushesSub + self.brushesAdd
		
	def write(self, path):
		filePath = path + "/" + self.name + ".t3d"
		outFile = open(filePath, "w")
		print "T3D:", filePath
		outFile.write("Begin Map\n")
		for actor in self.actors:
			actor.write(outFile)
		outFile.write("End Map\n")
		outFile.close()
		
if __name__ == "__main__" or __name__ == "__builtin__":
	expPath = g_geoPath + "/EXP"
	node = hou.node(expPath)
	t3dMap = T3D(node)
	path = hou.expandString(g_outPath)
	t3dMap.write(path)

#execfile("w:/houdini/Unreal/exp_t3d.py", {"g_geoPath":"/obj/BaseRoom", "g_outPath":"C:/Games/Unreal Tournament/Maps/"})
