/*-----------------------------------------------------------------------//
solosnakestartsolosnakestartsolosnakestartsolosnakestartsolosnakestartsol*/
/**
*   @file       ASE.cpp
*   @author     Dire Stockdale 
*   @author     <a href=http://www.solosnake.com>www.solosnake.com</a>
*   @date       09-Nov-01 21:04:23
*   @version    1.00
*   @brief      Functions, classes and structures for reading data
*   from 3ds Ascii Scene Exportor file format.
*           
*/
//-----------------------------------------------------------------------//
#include "ase.h"

using namespace std;

#define MYBUFFER 256

/**
*   Reads in the node header of a Geomobject. The first value read in is *NODE_NAME.
*/ 
bool ASE::GeomObjectNode::read(std::ifstream& ifile)
    {
    char buf[MYBUFFER];
    iErrorNo = 0;
    try{
        if( false == FindinFileBefore(ifile, szNODE_NAME, szGEOMOBJECT) )throw(int(-1));
        ifile.getline(buf, MYBUFFER, '"');
        ifile.getline(buf, MYBUFFER, '"');
        nodename = buf;

        if( false == FindinFileBefore(ifile, szINHERIT_POS, szGEOMOBJECT) )throw(int(-2));
        ifile >> dInheritPos[0] >> dInheritPos[1] >> dInheritPos[2];

        if( false == FindinFileBefore(ifile, szINHERIT_ROT, szGEOMOBJECT) )throw(int(-3));
        ifile >> dInheritRot[0] >> dInheritRot[1] >> dInheritRot[2];

        if( false == FindinFileBefore(ifile, szINHERIT_SCL, szGEOMOBJECT) )throw(int(-4));
        ifile >> dInheritScl[0] >> dInheritScl[1] >> dInheritScl[2];


        if( false == FindinFileBefore(ifile, szTM_ROW0, szGEOMOBJECT) )throw(int(-5));
        ifile >> dTMRow0[0] >> dTMRow0[1] >> dTMRow0[2];

        if( false == FindinFileBefore(ifile, szTM_ROW1, szGEOMOBJECT) )throw(int(-6));
        ifile >> dTMRow1[0] >> dTMRow1[1] >> dTMRow1[2];

        if( false == FindinFileBefore(ifile, szTM_ROW2, szGEOMOBJECT) )throw(int(-7));
        ifile >> dTMRow2[0] >> dTMRow2[1] >> dTMRow2[2];


        if( false == FindinFileBefore(ifile, szTM_POS, szGEOMOBJECT) )throw(int(-8));
        ifile >> dTMPos[0] >> dTMPos[1] >> dTMPos[2];

        if( false == FindinFileBefore(ifile, szTM_ROTAXIS, szGEOMOBJECT) )throw(int(-9));
        ifile >> dTMRotAxis[0] >> dTMRotAxis[1] >> dTMRotAxis[2];

        if( false == FindinFileBefore(ifile, szTM_ROTANGLE, szGEOMOBJECT) )throw(int(-10));
        ifile >> dTMRotAngle;


        if( false == FindinFileBefore(ifile, szTM_SCALE, szGEOMOBJECT) )throw(int(-11));
        ifile >> dTMScale[0] >> dTMScale[1] >> dTMScale[2];

        if( false == FindinFileBefore(ifile, szTM_SCALEAXIS, szGEOMOBJECT) )throw(int(-12));
        ifile >> dTMScaleAxis[0] >> dTMScaleAxis[1] >> dTMScaleAxis[2];

        if( false == FindinFileBefore(ifile, szTM_SCALEAXISANG, szGEOMOBJECT) )throw(int(-13));
        ifile >> dTMScaleAxisAng;
        }
    catch(int i){
        iErrorNo = i;
        return false;
        }
    catch(...){
        iErrorNo = UNKNOWNERROR;
        return false;
        }
    return true;
    }

/**
*   Reads in a single mesh values. The first value read in is *TIMEVALUE. 
*/
bool ASE::GeomObjectMesh::read(std::ifstream& ifile)
    {
    char buf[MYBUFFER];

    iErrorNo = 0;
    ulFlags = 0;

    vertex_list.clear();
    face_list.clear();
    face_listB.clear();
    face_list_materialID.clear();
    tvert_list.clear();
    tface_list.clear();
    facenormals.clear();
    vertexnormals.clear();

    try{
        if( FindinFileBefore(ifile, szTIMEVALUE, szGEOMOBJECT) )
            {
            ifile >> dTimeValue;
            ulFlags |= GTIMEVALUE;
            }


        if( false == FindinFileBefore(ifile, szMESH_NUMVERTEX, szGEOMOBJECT) )throw(int(-2));
        ifile >> iVertexCount;
        assert( iVertexCount || !"Error : Zero Vertices." );
        if( 0 == iVertexCount ) throw( NOVERTICES );


        if( false == FindinFileBefore(ifile, szMESH_NUMFACES, szGEOMOBJECT) )throw(int(-3));
        ifile >> iFaceCount;
        assert( iFaceCount || !"Error : Zero Faces." );
        if( 0 == iFaceCount ) throw( NOFACES );


        //
        // Dummy is a check variable. It should always equalt the number of the
        // data being read.
        //
        int iDummy(0);
        GeomPoint p;


        //
        // Find start of vertex list.
        //
        if( false == FindinFileBefore(ifile, szMESH_VERTEX_LIST, szGEOMOBJECT) )throw(int(-4));


        //
        // Reserve known spaces in vectors.
        //
        vertex_list.reserve( iVertexCount );


        //
        // Read in the vertex list.
        //
        for(int i=0; i<iVertexCount; ++i)
            {
            if( false == FindinFileBefore(ifile, szMESH_VERTEX, szGEOMOBJECT) )throw(int(-5 -i));

            ifile >> iDummy;
            assert( iDummy == i || !"Error : Mismatching dummy and mesh vertex number." );
            if( iDummy != i )throw(int(-5 -i));
            
            ifile >> p.x >> p.y >> p.z;

            vertex_list.push_back( p );
            }

        //
        // Reserve space
        face_list.reserve( 3 * iFaceCount );
        face_listB.reserve( 3 * iFaceCount );
        face_list_materialID.reserve( iFaceCount );


        //
        // Find start of face list.
        //
        if( false == FindinFileBefore(ifile, szMESH_FACE_LIST, szGEOMOBJECT) )throw(int(-6 -iVertexCount ));


        //
        // Read in the vertex list.
        //
        char cdummy;
        int ia, ib, ic;
        int iab, ibc, ica;
        int imid;
        for(int i=0; i<iFaceCount; ++i)
            {
            if( false == FindinFileBefore(ifile, szMESH_FACE, szGEOMOBJECT) )throw(int(-6 -iVertexCount -i));

            ifile >> iDummy;
            assert( iDummy == i || !"Error : Mismatching dummy and face vertex number." );
            if( iDummy != i )throw(int(-5 -i));
            
            // A line here looks like this:
            // *MESH_FACE    0:    A:    0 B:    1 C:    9 AB:    1 BC:    1 CA:    0	 *MESH_SMOOTHING 1,9,12 	*MESH_MTLID 2
			
            ifile >> cdummy; // the semi colon following dummy.
            ifile >> buf;    // The 'A:'.
            ifile >> ia;     // The integer face number.

            ifile >> buf;   // The 'B:'
            ifile >> ib;    // The integer face number.

            ifile >> buf;   // The 'C:'
            ifile >> ic;    // The integer face number.

            face_list.push_back( ia );
            face_list.push_back( ib );
            face_list.push_back( ic );

            ifile >> buf;   // The 'AB:'.
            ifile >> iab;   // The integer face number.

            ifile >> buf;   // The 'BC:'
            ifile >> ibc;   // The integer face number.

            ifile >> buf;   // The 'CA:'
            ifile >> ica;   // The integer face number.

            face_listB.push_back( iab );
            face_listB.push_back( ibc );
            face_listB.push_back( ica );

            //
            // Try and find the mesh material id number. I am not sure this is always
            // exported so we will not need this nor return an error if it is not there.
            //
            if( FindinFileBefore(ifile, szMESH_MTLID, szMESH_FACE) )
                {
                ifile >> imid;
                face_list_materialID.push_back( imid );
                }
            }

        //
        // Find start of texture vertex list.
        //
        int iTextureVCount = 0;
        if( FindinFileBefore(ifile, szMESH_NUMTVERTEX, szGEOMOBJECT) )
            {
            ifile >> iTextureVCount;
            }

        tface_list.reserve( 3 * iFaceCount );
        facenormals.reserve( iFaceCount );
        vertexnormals.reserve( 3 * iFaceCount );

        //
        // If the object has no texture list quit reading.
        //
        if( iTextureVCount > 0 )
            {
            ulFlags |= GTVERTICES;

            if( false == FindinFileBefore(ifile, szMESH_TVERTLIST, szGEOMOBJECT) )throw( NOTEXTUREFACES );
        
            tvert_list.reserve( iTextureVCount );

            //
            // Read in the vertex list.
            //
            for(int i=0; i<iTextureVCount; ++i )
                {
                if( false == FindinFileBefore(ifile, szMESH_TVERT, szGEOMOBJECT) )throw(int(-6-iFaceCount-i));
            
                ifile >> iDummy;
                assert( iDummy == i || !"Error: mismatched dummy and texture vertex." );
                if( iDummy != i )throw(int(-6-iFaceCount-i));

                ifile >> p.x >> p.y >> p.z;

                tvert_list.push_back( p );
                }

            
            //
            // Find start of texture face list.
            //
            if( false == FindinFileBefore(ifile, szMESH_TFACELIST, szGEOMOBJECT) )throw( NOTEXTUREFACES );
        
            //
            // Read in texture faces.
            //
            for(int i=0; i<iFaceCount; ++i )
                {
                if( false == FindinFileBefore(ifile, szMESH_TFACE, szGEOMOBJECT) )throw(int(-6-iFaceCount-iTextureVCount-i));

                ifile >> iDummy;
                assert( i == iDummy || !"Error: mismatched dummy and texture face number." );
                if( i != iDummy )throw(int(-6-iFaceCount-iTextureVCount-i));

                ifile >> ia >> ib >> ic;

                tface_list.push_back( ia );
                tface_list.push_back( ib );
                tface_list.push_back( ic );
                }
            }
        else
            {
            tvert_list.resize(0);
            tface_list.resize(0);
            }

        //
        // Find start of mesh normals, if there is any.
        //
        if( FindinFileBefore(ifile, szMESH_NORMALS, szGEOMOBJECT) )
            {
            ulFlags |= GNORMALS;

            for(int i=0; i<iFaceCount; ++i )
                {
                if( false == FindinFileBefore(ifile, szMESH_FACENORMAL, szGEOMOBJECT) )throw( INVALIDMESHNORMALS );

                ifile >> iDummy;
                assert( i == iDummy || !"Error: mismatched dummy and face normal.");
                if( i != iDummy )throw( INVALIDMESHNORMALS );
                ifile >> p.x >> p.y >> p.z;
                facenormals.push_back( p );

                //
                // The three normals at the vertices on the face.
                //
                if( false == FindinFileBefore(ifile, szMESH_VERTEXNORMAL, szMESH_FACENORMAL) )throw( INVALIDMESHVNORMAL );
                ifile >> iDummy;
                ifile >> p.x >> p.y >> p.z;
                vertexnormals.push_back( p );

                if( false == FindinFileBefore(ifile, szMESH_VERTEXNORMAL, szMESH_FACENORMAL) )throw( INVALIDMESHVNORMAL );
                ifile >> iDummy;
                ifile >> p.x >> p.y >> p.z;
                vertexnormals.push_back( p );

                if( false == FindinFileBefore(ifile, szMESH_VERTEXNORMAL, szMESH_FACENORMAL) )throw( INVALIDMESHVNORMAL );
                ifile >> iDummy;
                ifile >> p.x >> p.y >> p.z;
                vertexnormals.push_back( p );
                }
            }

        //
        // We now have all the arrays data. Now find the little portion 
        // of data at the end of the mesh.
        //
        if( false == FindinFileBefore(ifile, szPROP_MOTIONBLUR, szGEOMOBJECT) )throw( MESHERROR );
        ifile >> ia;
        bMotionBlur = (ia != 0);
        
        if( false == FindinFileBefore(ifile, szPROP_CASTSHADOW, szGEOMOBJECT) )throw( MESHERROR );      
        ifile >> ia;
        bCastShadow = (ia != 0);

        if( false == FindinFileBefore(ifile, szPROP_RECVSHADOW, szGEOMOBJECT) )throw( MESHERROR );
        ifile >> ia;
        bRecvShadow = (ia != 0);
        
        if( FindinFileBefore(ifile, szMATERIAL_REF, szGEOMOBJECT) )
            {
            ifile >> iMaterialRef;
            }
        else
            {
            iMaterialRef = -1;
            }
        }
    catch(ERRORCODE e){
        iErrorNo = e;
        return false;
        }    
    catch(int i){
        iErrorNo = i;
        return false;
        }

    catch(...){
        iErrorNo = UNKNOWNERROR;
        return false;
        }
    return true;
    }



/**
*   Reads in first the objects node then the mesh. Does not check for
*   the *GEOMOBJECT starting tag.
*/
bool ASE::GeomObject::read(std::ifstream& ifile)
    {
    iErrorNo = 0;
    if( false == node.read( ifile ) )
        {
        iErrorNo = node.GetError();
        return false;
        }

    if( false == mesh.read( ifile ) )
        {
        iErrorNo = mesh.GetError();
        return false;
        }

    return true;
    }

/**
*   Reads in the material. The first value read in is *MATERIAL_NAME.
*   Does not check for *MATERIAL.
*/
bool ASE::Material::read(std::ifstream& ifile)
    {
    char buf[MYBUFFER];
    
    ulFlags = 0L;
    iErrorNo = 0;

    try{
        const streampos pos = ifile.tellg();

        if(  FindinFileBefore(ifile, szMATERIAL_NAME, szMATERIAL) )
            {
            ifile.getline(buf, MYBUFFER, '"');
            ifile.getline(buf, MYBUFFER, '"');
            name = buf;
            ifile.seekg( pos );
            ulFlags |= MNAME;
            }

        if(  FindinFileBefore(ifile, szMATERIAL_AMBIENT, szMATERIAL) )
            {
            ifile >> dAmbient[0] >> dAmbient[1] >> dAmbient[2];
            ifile.seekg( pos );
            ulFlags |= MAMBIENT;
            }

        if(  FindinFileBefore(ifile, szMATERIAL_DIFFUSE, szMATERIAL) )
            {
            ifile >> dDiffuse[0] >> dDiffuse[1] >> dDiffuse[2];
            ifile.seekg( pos );
            ulFlags |= MDIFFUSE;
            }

        if(  FindinFileBefore(ifile, szMATERIAL_SPECULAR, szMATERIAL) )
            {
            ifile >> dSpecular[0] >> dSpecular[1] >> dSpecular[2];
            ifile.seekg( pos );
            ulFlags |= MSPECULAR;
            }

        if(  FindinFileBefore(ifile, szMATERIAL_SHINE, szMATERIAL) )
            {
            ifile >> dShine;
            ifile.seekg( pos );
            ulFlags |= MSHINE;
            }

        if(  FindinFileBefore(ifile, szMATERIAL_SHINESTRENGTH, szMATERIAL) )
            {
            ifile >> dShineStrength;
            ifile.seekg( pos );
            ulFlags |= MSHINESTRENGTH;
            }

        if(  FindinFileBefore(ifile, szMATERIAL_TRANSPARENCY, szMATERIAL) )
            {
            ifile >> dTransparency;
            ifile.seekg( pos );
            ulFlags |= MTRANSPARENCY;
            }

        if(  FindinFileBefore(ifile, szMATERIAL_WIRESIZE, szMATERIAL) )
            {
            ifile >> dWireSize;
            ifile.seekg( pos );
            ulFlags |= MWIRESIZE;
            }

        if(  FindinFileBefore(ifile, szMATERIAL_SHADING, szMATERIAL) )
            {
            ifile >> shading;
            ifile.seekg( pos );
            ulFlags |= MSHADING;
            }

        //
        // Only read in map value if one exists
        //
        if( FindinFileBefore(ifile, szMAP_DIFFUSE, szMATERIAL) )
            {
			ulFlags |= MMAPDIFFUSE;

            if(  FindinFileBefore(ifile, szMAP_NAME, szMATERIAL) )
                {
                ifile.getline(buf, MYBUFFER, '"');
                ifile.getline(buf, MYBUFFER, '"');
                mapname = buf;
                ifile.seekg( pos );
                ulFlags |= MMAPNAME;
                }

            if(  FindinFileBefore(ifile, szBITMAP, szMATERIAL) )
                {
                ifile.getline(buf, MYBUFFER, '"');
                ifile.getline(buf, MYBUFFER, '"');
                ExtractFile(buf);      // Only need filename
                bitmap = buf;
                ifile.seekg( pos );
                ulFlags |= MMAPBITMAP;
                }

            if(  FindinFileBefore(ifile, szMAP_AMOUNT, szMATERIAL) )
                {
                ifile >> mapamount;
                ifile.seekg( pos );
                ulFlags |= MMAPAMOUNT;
                }

            if(  FindinFileBefore(ifile, szUVW_U_OFFSET, szMATERIAL) )
                {
                ifile >> uvw_U_offset;
                ifile.seekg( pos );
                ulFlags |= MMAPUOFFSET;
                }

            if(  FindinFileBefore(ifile, szUVW_V_OFFSET, szMATERIAL) )
                {
                ifile >> uvw_V_offset;
                ifile.seekg( pos );
                ulFlags |= MMAPVOFFSET;
                }

            if(  FindinFileBefore(ifile, szUVW_U_TILING, szMATERIAL) )
                {
                ifile >> uvw_U_tiling;
                ifile.seekg( pos );
                ulFlags |= MMAPUTILING;
                }

            if(  FindinFileBefore(ifile, szUVW_V_TILING, szMATERIAL) )
                {
                ifile >> uvw_V_tiling;
                ifile.seekg( pos );
                ulFlags |= MMAPVTILING;
                }

            if(  FindinFileBefore(ifile, szUVW_ANGLE, szMATERIAL) )
                {
                ifile >> uvw_Angle;
                ifile.seekg( pos );
                ulFlags |= MMAPUVWANGLE;
                }

            if(  FindinFileBefore(ifile, szUVW_BLUR, szMATERIAL) )
                {
                ifile >> uvw_blur;
                ifile.seekg( pos );
                ulFlags |= MMAPBLUR;
                }

            if(  FindinFileBefore(ifile, szUVW_BLUR_OFFSET, szMATERIAL) )
                {
                ifile >> uvw_blur_offset;
                ulFlags |= MMAPBLUROFFSET;
                }
            }
        }
    catch(ERRORCODE e){
        iErrorNo = e;
        return false;
        }
    catch(int i){
        iErrorNo = i;
        return false;
        }
    catch(...){
        iErrorNo = UNKNOWNERROR;
        return false;
        }
    return true;
    }




ASE::ASEFile::ASEFile():iWarning(0),scenename("NO_NAME")
    {
    dBackgroundStatic[0] = dBackgroundStatic[1] = dBackgroundStatic[2] = 0.0;
    dAmbientStatic[0] = dAmbientStatic[1] = dAmbientStatic[2] = 0.0;
    }


/**
*   Opens the file specified and passes it immediately to 
*   read(std::ifstream&).
*/
int ASE::ASEFile::read(const char* szFileName)
    {
    using namespace std;

    ifstream ifile( szFileName );
    if( false == ifile.is_open() )
        {
        return 1;
        }
    
    char buf[MYBUFFER];
    ::strcpy( buf, szFileName );
    ExtractFile( buf );
    scenename = buf;

    return read(ifile);
    }



/**
*   Reads in an entire ASE file.
*/
int ASE::ASEFile::read(std::ifstream& ifile)
    {
    char buf[128];

    //
    // Clear contents.
    //
    dBackgroundStatic[0] = dBackgroundStatic[1] = dBackgroundStatic[2] = 0.0;
    dAmbientStatic[0] = dAmbientStatic[1] = dAmbientStatic[2] = 0.0;
    materials.clear();
    geomobjects.clear();
    //scenename = "";
    iWarning = 0;

    try{
        //
        // The file MUST have this header indentifier
        //
        ifile >> buf;
        assert( 0 == ::strcmp( buf, "*3DSMAX_ASCIIEXPORT" ) );
        if( ::strcmp( buf, "*3DSMAX_ASCIIEXPORT" ) )throw( INVALIDASEFILE );

        //
        // Look for the scene header.
        //
      //  if( false == FindinFileBefore(ifile, szSCENE, szMATERIAL_LIST) )throw( INVALIDASEFILE );

        if( FindinFileBefore(ifile, szSCENE_BACKGROUND_STATIC, szMATERIAL_LIST) )/*throw( INVALIDASEFILE );*/
            {
            ifile >> dBackgroundStatic[0] >> dBackgroundStatic[1] >> dBackgroundStatic[2];
            }

        if( FindinFileBefore(ifile, szSCENE_AMBIENT_STATIC, szMATERIAL_LIST) )/*throw( INVALIDASEFILE );*/
            {
            ifile >> dAmbientStatic[0] >> dAmbientStatic[1] >> dAmbientStatic[2];
            }

        //
        // Check to see if the file has materials. It is permitted not to.
        // Find and read in material list.
        //
        int imatcount = 0;
        if( FindinFile(ifile, szMATERIAL_LIST) && FindinFile(ifile, szMATERIAL_COUNT) )
            {
            ifile >> imatcount;
            }

        //
        // Go through the materials (which we are told must be here)
        // and read them in.
        //
        if( imatcount > 0 )
            {
            int iDummy = 0;
            for( int i=0; i<imatcount; ++i )
                {
                if( false == FindinFile(ifile, szMATERIAL) )throw( INVALIDASEFILE );
                ifile >> iDummy;
                assert( i == iDummy || !"Error mismatched material and dummy." );
                if( i != iDummy )throw( INVALIDASEFILE );

                ASE::Material m;

                if( false == m.read( ifile ) )throw( m.GetError() );

                materials.push_back( m );
                }
            }

        assert( imatcount == materials.size() );

        //
        // Find and read in all Geomobjects. There is no list of these so we just have to look until
        // the end of file.
        //
        while( FindinFile(ifile, szGEOMOBJECT) )
            {
            ASE::GeomObject gobject;
            if( false == gobject.read(ifile) )
                {
                switch( gobject.GetError() )
                    {
                    case NOFACES:               //  Objects with no faces are not considered 
                        ++iWarning;             //  to be fatal errors. These null objects
                        break;                  //  are not added to the lists.
                    case NOVERTICES:            //  Objects with no vertices are the same.
                        ++iWarning;
                        break;
                    default:
                        throw(gobject.GetError());
                    }
                }
            else                                //  Valid object found.
                {
                geomobjects.push_back( gobject );
                }
            }
        }
    catch(ERRORCODE e){
        return e;
        }
    catch(int i){
        return i;
        }
    catch(...){
        return -999;
        }
    return 0;
    }



int ASE::ASEFile::GetVertexCount(void)const throw()
    {
    int ret = 0;
    for(unsigned int i=0; i<geomobjects.size(); ++i )
        {
        ret += geomobjects[i].mesh.iFaceCount;
        }
    return ret;
    }


/*solosnakeendsolosnakeendsolosnakeendsolosnakeendsolosnakeendsolosnakeends
//-----------------------------------------------------------------------*/
