//-----------------------------------------------------------------------//
//solosnakestartsolosnakestartsolosnakestartsolosnakestartsolosnakestarts//
//
//   @file       T3D.cpp
//   @author     Dire Stockdale 
//   @author     <a href=http://www.solosnake.com>www.solosnake.com</a>
//   @date       11-Nov-01 10:39:07
//   @version    1.00
//   @brief      Code file for routines to convert an ASEFile object
//               to a TD file output.
//           
///
//-----------------------------------------------------------------------//
#include "t3d.h"  
#include <afxwin.h>
#include <io.h>
#include <sys/stat.h>


#define MINIMUM_DIVISOR 0.00000000001

using namespace std;

extern bool g_bFlip;

std::string StripSuffix(const std::string& s)
    {
    using namespace std;

    string strip;

    unsigned int i;

    for(i=0;i<s.size(); ++i )
        {
        if( s[i] == '.' )break;
        }

    strip.assign( s.begin(), s.begin()+i );

    return strip;
    }



/*
    Writes an untextured polygon point. This does not have any texture data,
    and so will use the default texture coords from UnrealEd.
*/
static
void WriteT3DPolygon(const ASE::ASEFile& ase, 
                     std::ofstream& ofile, 
                     const int i, const int v )
    {
    char buf[256] = {'\0'};

    //
    // Polygon data
    // 
    // Coords
    //
    ASE::GeomPoint pt0 = ase.geomobjects[i].mesh.vertex_list[ ase.geomobjects[i].mesh.face_list[v+0] ];
    ASE::GeomPoint pt1 = ase.geomobjects[i].mesh.vertex_list[ ase.geomobjects[i].mesh.face_list[v+1] ];
    ASE::GeomPoint pt2 = ase.geomobjects[i].mesh.vertex_list[ ase.geomobjects[i].mesh.face_list[v+2] ];

    //
    // Calculate Normals. These are ignored anyway but make an effort...
    //
    ASE::GeomPoint pA = pt1 - pt0;
    ASE::GeomPoint pB = pt2 - pt0;

    ASE::GeomPoint pN = pA ^ pB;
    pN = pN.normalized();//pN.Normalized();

    double dnx = -pN.x;
    double dny =  pN.y;
    double dnz =  pN.z;


    //
    // Print Origin. If the texture system is inconsistent then use the
    // default first coord as the origin.
    //
    ofile << szTab << szTab << szTab << szOrigin << szSpace << szSpace << szSpace;    
    
    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt1.x) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", static_cast<float>(-pt1.y) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt1.z) );
    ofile << buf << endl;

    //
    // Print Normals..
    //
    ofile << szTab << szTab << szTab << szNormal << szSpace << szSpace << szSpace;

    ::sprintf( buf, "%+013.6f", dnx );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", dny );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", dnz );
    ofile << buf << endl;

/*    //
    // Print normal texture UVs
    //
    ofile << szTab << szTab << szTab << szTextureU << szSpace;

    ::sprintf( buf, "%+013.6f",  dnx );
    ofile << buf << szComma; 

    ::sprintf( buf, "%+013.6f",  dny );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  dnz );
    ofile << buf << endl;

    ofile << szTab << szTab << szTab << szTextureV << szSpace;

    ::sprintf( buf, "%+013.6f",  dnx );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  dny );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  dnz );
    ofile << buf << endl;
*/



    //
    // Print Polygon points.
    //
    ofile << szTab << szTab << szTab << szVertex << szSpace << szSpace << szSpace;

    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt1.x) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", static_cast<float>(-pt1.y) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt1.z) );
    ofile << buf << endl;


    ofile << szTab << szTab << szTab << szVertex << szSpace << szSpace << szSpace;
    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt0.x) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", static_cast<float>(-pt0.y) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt0.z) );
    ofile << buf << endl;


    ofile << szTab << szTab << szTab << szVertex << szSpace << szSpace << szSpace;
    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt2.x) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", static_cast<float>(-pt2.y) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt2.z) );
    ofile << buf << endl;


    //
    // End Polygon
    //
    ofile << szTab << szTab << szEnd << szSpace << szPolygon << endl;
    }


static
void WriteTexturedT3DPolygon(const ASE::ASEFile& ase, 
                             std::ofstream& ofile, 
                             const int i, const int v,
                             const int txwidth, const int txheight )
    {
    char buf[256] = {'\0'};
    const double texture_width  = static_cast<double>(txwidth);
    const double texture_height = static_cast<double>(txheight);

    //
    // Polygon data
    //

    //
    // Texture Coords.
    //
    double dux(0.0), duy(0.0), duz(0.0), dvx(0.0), dvy(0.0), dvz(0.0);

    /*
    From the ASE - DXA file

    dScaleOU = -0.5 - dUF/2;
    dScaleVO = -0.5 - dVF/2;

    dt = (dUF * u));

    pFile << "\t" << ( -dUO * dUF + dScaleOU + dt ) << "\t";

    dt = (dVF * v);

    pFile <<  -( -dVO * dVF + dScaleVO + dt -1) << "\n"; 
    */
    const int mID = ase.geomobjects.at(i).mesh.iMaterialRef;

    const double dUF = ase.materials.at( mID ).uvw_U_tiling;
    const double dVF = ase.materials.at( mID ).uvw_V_tiling;
    const double dUO = ase.materials.at( mID ).uvw_U_offset;
    const double dVO = ase.materials.at( mID ).uvw_V_offset;

    double dScaleUO = -0.5 - ( dUF / 2.0 );
    double dScaleVO = -0.5 - ( dVF / 2.0 );

    double ds0 = ase.geomobjects.at(i).mesh.tvert_list[ ase.geomobjects.at(i).mesh.tface_list.at( v+0 ) ].x;
    double dt0 = ase.geomobjects.at(i).mesh.tvert_list[ ase.geomobjects.at(i).mesh.tface_list.at( v+0 ) ].y;

    double ds1 = ase.geomobjects.at(i).mesh.tvert_list[ ase.geomobjects.at(i).mesh.tface_list.at( v+1 ) ].x;
    double dt1 = ase.geomobjects.at(i).mesh.tvert_list[ ase.geomobjects.at(i).mesh.tface_list.at( v+1 ) ].y;

    double ds2 = ase.geomobjects.at(i).mesh.tvert_list[ ase.geomobjects.at(i).mesh.tface_list.at( v+2 ) ].x;
    double dt2 = ase.geomobjects.at(i).mesh.tvert_list[ ase.geomobjects.at(i).mesh.tface_list.at( v+2 ) ].y;

    //
    // Rotate texture coords about the point (0.5,0.5)
    //
    if( ase.materials.at( mID ).uvw_Angle != 0.0 )
        {
        const double a = /*(2.0 * 3.1415926535897932384626433832795) - */ase.materials.at( mID ).uvw_Angle;

        const double u0 = ds0 - 0.5;
        const double u1 = ds1 - 0.5;
        const double u2 = ds2 - 0.5;

        const double v0 = dt0 - 0.5;
        const double v1 = dt1 - 0.5;
        const double v2 = dt2 - 0.5;

        ds0 = 0.5 + ( u0 * cos( a ) - v0 * sin( a ) );
        dt0 = 0.5 + ( u0 * sin( a ) + v0 * cos( a ) );

        ds1 = 0.5 + ( u1 * cos( a ) - v1 * sin( a ) );
        dt1 = 0.5 + ( u1 * sin( a ) + v1 * cos( a ) );

        ds2 = 0.5 + ( u2 * cos( a ) - v2 * sin( a ) );
        dt2 = 0.5 + ( u2 * sin( a ) + v2 * cos( a ) );
        }

    //
    // Scale
    //
    ds0 *= dUF;
    dt0 *= dVF;

    ds1 *= dUF;
    dt1 *= dVF;

    ds2 *= dUF;
    dt2 *= dVF;

    ds0 =   ( -dUO * dUF ) + dScaleUO + ds0;        //  OpenGL / DirectX tex coords of first vertex
    dt0 = -(( -dVO * dVF ) + dScaleVO + dt0 -1.0);

    ds1 =   ( -dUO * dUF ) + dScaleUO + ds1;        //  OpenGL / DirectX tex coords of second vertex
    dt1 = -(( -dVO * dVF ) + dScaleVO + dt1 -1.0);

    ds2 =   ( -dUO * dUF ) + dScaleUO + ds2;        //  OpenGL / DirectX tex coords of third vertex
    dt2 = -(( -dVO * dVF ) + dScaleVO + dt2 -1.0);


    //
    // Translate so that coord one is minimum possible
    //
    const int translateU = static_cast<int>( ds0 );
    const int translateV = static_cast<int>( dt0 );

    const double ftranslateU = static_cast<double>( translateU );
    const double ftranslateV = static_cast<double>( translateV );

    ds0 -= ftranslateU;
    ds1 -= ftranslateU;
    ds2 -= ftranslateU;

    dt0 -= ftranslateV;
    dt1 -= ftranslateV;
    dt2 -= ftranslateV;


    // 
    // Coords
    //
    ASE::GeomPoint pt0 = ase.geomobjects.at(i).mesh.vertex_list[ ase.geomobjects.at(i).mesh.face_list[v+0] ];
    ASE::GeomPoint pt1 = ase.geomobjects.at(i).mesh.vertex_list[ ase.geomobjects.at(i).mesh.face_list[v+1] ];
    ASE::GeomPoint pt2 = ase.geomobjects.at(i).mesh.vertex_list[ ase.geomobjects.at(i).mesh.face_list[v+2] ];

    /*
    fn getTextureGrad pt0 pt1 pt2 val0 val1 val2 = (
    dpt1 = pt1 - pt0
    dpt2 = pt2 - pt0

    dv1 = val1 - val0
    dv2 = val2 - val0
    */
    ASE::GeomPoint dpt1 = pt1 - pt0;
    ASE::GeomPoint dpt2 = pt2 - pt0;

    ASE::GeomPoint dv1( ds1-ds0, dt1-dt0, 0.0 );
    ASE::GeomPoint dv2( ds2-ds0, dt2-dt0, 0.0 );

    /*
    -- Compute the 2D matrix values, and invert the matrix.
    dpt11 = dot dpt1 dpt1
    dpt12 = dot dpt1 dpt2
    dpt22 = dot dpt2 dpt2
    factor = 1.0 / (dpt11 * dpt22 - dpt12 * dpt12)
    */
    double dpt11 = dpt1 * dpt1;
    double dpt12 = dpt1 * dpt2;
    double dpt22 = dpt2 * dpt2;
	assert( ( dpt11 * dpt22 - dpt12 * dpt12 ) );
    double factor = 1.0 / ( dpt11 * dpt22 - dpt12 * dpt12 );

    /*
    -- Compute the two gradients.
    g1 = (dv1 * dpt22 - dv2 * dpt12) * factor
    g2 = (dv2 * dpt11 - dv1 * dpt12) * factor
    */
    ASE::GeomPoint g1 = ((dv1 * dpt22) - (dv2 * dpt12)) * factor;
    ASE::GeomPoint g2 = ((dv2 * dpt11) - (dv1 * dpt12)) * factor;

    /*
    p_grad_u = dpt1 * g1.x + dpt2 * g2.x
    p_grad_v = dpt1 * g1.y + dpt2 * g2.y
    */
    ASE::GeomPoint p_grad_u = dpt1 * g1.x + dpt2 * g2.x;
    ASE::GeomPoint p_grad_v = dpt1 * g1.y + dpt2 * g2.y;

    /*
    -- Repeat process above, computing just one vector in the plane.
    dup1 = dot dpt1 p_grad_u
    dup2 = dot dpt2 p_grad_u
    dvp1 = dot dpt1 p_grad_v
    dvp2 = dot dpt2 p_grad_v
    */
    double dup1 = dpt1 * p_grad_u;
    double dup2 = dpt2 * p_grad_u;
    double dvp1 = dpt1 * p_grad_v;
    double dvp2 = dpt2 * p_grad_v;


    /*
    fuctor = 1.0 / (dup1 * dvp2 - dvp1 * dup2)
    */

	/*  Impossible values may occur here, and cause divide by zero problems.
        Handle these by setting the divisor to a safe value then flagging
        thatr it is impossible. Impossible textured polygons use the normal 
        to the polygon, which makes no texture appear.
		26-12-01
	*/
	double divisor = (dup1 * dvp2 - dvp1 * dup2);
	bool impossible1 = fabs(divisor) <= MINIMUM_DIVISOR;

	if( impossible1 ) 
		{
		divisor = 1.0;
		}

    double fuctor  = 1.0 / divisor;

    /*
    b1 = (val0.x * dvp2 - val0.y * dup2) * fuctor
    b2 = (val0.y * dup1 - val0.x * dvp1) * fuctor
    */
    double b1 = (ds0 * dvp2 - dt0 * dup2) * fuctor;
    double b2 = (dt0 * dup1 - ds0 * dvp1) * fuctor;


    /*
    p_base = pt0 - (dpt1 * b1 + dpt2 * b2)
    */
    ASE::GeomPoint p_base = pt0 - ((dpt1 * b1) + (dpt2 * b2));


    /*
    p_grad_u *= 256
    p_grad_v *= 256
    */
    p_grad_u *= texture_width;
    p_grad_v *= texture_height;


    //
    // Calculate Normals. These are ignored anyway but make an effort...
    //
    ASE::GeomPoint pA = pt1 - pt0;
    ASE::GeomPoint pB = pt2 - pt0;

    ASE::GeomPoint pN = pA ^ pB;
    pN = pN.normalized(); //pN.Normalized();

    double dnx = -pN.x;
    double dny =  pN.y;
    double dnz =  pN.z;
    

    //
    // Check for error values
    //
    bool impossible2 = _isnan(p_base.x)   || _isnan(p_base.y)   || _isnan(p_base.z) ||
                       _isnan(p_grad_v.x) || _isnan(p_grad_v.y) || _isnan(p_grad_v.z) ||
                       _isnan(p_grad_v.x) || _isnan(p_grad_v.y) || _isnan(p_grad_v.z);

	const bool impossible = impossible1 || impossible2;

    //
    // Print Origin. If the texture system is inconsistent then use the
    // default first coord as the origin.
    //
    ofile << szTab << szTab << szTab << szOrigin << szSpace << szSpace << szSpace;    
    
    if( impossible )
        {
        ::sprintf( buf, "%+013.6f",  static_cast<float>(pt1.x) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f", static_cast<float>(-pt1.y) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(pt1.z) );
        ofile << buf << endl;
        }
    else
        {
        ::sprintf( buf, "%+013.6f",  static_cast<float>(p_base.x) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f", static_cast<float>(-p_base.y) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(p_base.z) );
        ofile << buf << endl;
        }


    //
    // Print Normals..
    //
    ofile << szTab << szTab << szTab << szNormal << szSpace << szSpace << szSpace;

    ::sprintf( buf, "%+013.6f", static_cast<float>(dnx) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", static_cast<float>(dny) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", static_cast<float>(dnz) );
    ofile << buf << endl;

    //
    // Print Texture axes
    // 
    if( ! impossible )
        {    
        ofile << szTab << szTab << szTab << szTextureU << szSpace;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(p_grad_u.x) );
        ofile << buf << szComma; 

        ::sprintf( buf, "%+013.6f", static_cast<float>(-p_grad_u.y) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(p_grad_u.z) );
        ofile << buf << endl;

        ofile << szTab << szTab << szTab << szTextureV << szSpace;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(p_grad_v.x) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f", static_cast<float>(-p_grad_v.y) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(p_grad_v.z) );
        ofile << buf << endl;
        }
    else
        {    
        ofile << szTab << szTab << szTab << szTextureU << szSpace;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(dnx) );
        ofile << buf << szComma; 

        ::sprintf( buf, "%+013.6f",  static_cast<float>(dny) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(dnz) );
        ofile << buf << endl;

        ofile << szTab << szTab << szTab << szTextureV << szSpace;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(dnx) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(dny) );
        ofile << buf << szComma;

        ::sprintf( buf, "%+013.6f",  static_cast<float>(dnz) );
        ofile << buf << endl;
        }



    //
    // Print Polygon points.
    //
    ofile << szTab << szTab << szTab << szVertex << szSpace << szSpace << szSpace;

    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt1.x) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", static_cast<float>(-pt1.y) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt1.z) );
    ofile << buf << endl;


    ofile << szTab << szTab << szTab << szVertex << szSpace << szSpace << szSpace;
    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt0.x) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", static_cast<float>(-pt0.y) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt0.z) );
    ofile << buf << endl;


    ofile << szTab << szTab << szTab << szVertex << szSpace << szSpace << szSpace;
    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt2.x) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f", static_cast<float>(-pt2.y) );
    ofile << buf << szComma;

    ::sprintf( buf, "%+013.6f",  static_cast<float>(pt2.z) );
    ofile << buf << endl;


    //
    // End Polygon
    //
    ofile << szTab << szTab << szEnd << szSpace << szPolygon << endl;
    }


static
void WriteT3DBrush(const ASE::ASEFile& ase,  const ASE::vTexSize& vtxsize, 
                   const unsigned int flags, const bool makeflags,  std::ofstream& ofile)
    {
    char buf[20] = {'\0'};
    unsigned int f = 0;
    //
    // Write polygon header:
    // Begin Brush Name=<ase scene name>
    // Begin PolyList
    //
    ofile << szBegin << szSpace << szBrush << szSpace << szName << szEquals <<  StripSuffix( ase.scenename ).c_str() << endl;
    ofile << szTab << szBegin << szSpace << szPolyList << endl;


    int ii = ase.geomobjects.size();

    for(unsigned int i=0; i<ase.geomobjects.size(); ++i )
        {
        for( int p=0, v=0; p<ase.geomobjects.at(i).mesh.iFaceCount; ++p, v += 3 )
            {
            //int ii = ase.geomobjects.at(i).mesh.iMaterialRef;

            const bool hasmaterialref = (ase.geomobjects.at(i).mesh.iMaterialRef > -1);
            bool hasdiffusetexturemap  = false;
            bool hasbitmaptexture      = false;

            if( hasmaterialref )
                {
                const int m = ase.geomobjects.at(i).mesh.iMaterialRef;
                hasdiffusetexturemap = (ase.materials.at( m ).GetFlags() & MMAPDIFFUSE) ? true : false;
                hasbitmaptexture     = (ase.materials.at( m ).GetFlags() & MMAPBITMAP)  ? true : false;
                }


            const bool hastexture = hasmaterialref && hasdiffusetexturemap && hasbitmaptexture;

            //
            // Begin Polygon Texture=<texture> Flags=<flags> Link=<i>
            //
            ofile << szTab << szTab << szBegin << szSpace << szPolygon << szSpace;

            //
            // Write out texture if one exists
            //
            if( hastexture )
                {
                string stripped = StripSuffix( ase.materials.at( ase.geomobjects.at(i).mesh.iMaterialRef ).bitmap );
                ofile << szTexture << szEquals << stripped.c_str() << szSpace;
                }

            //
            // Flags. Only write out when there is something to write out.
            //
            f = 0;
          
            if( makeflags )
                { 
                // 
                // Generate flags based upon as much data as we can from the ASE file
                //
                if( ase.geomobjects.at(i).mesh.bRecvShadow ) 
                    f |= PF_LowShadowDetail;

                if( hastexture && 0 == ::strcmp( ase.materials.at( ase.geomobjects.at(i).mesh.iMaterialRef ).shading.c_str(), "Blinn" ) )
                    f |= PF_Gouraud;

                if( hastexture && ase.materials.at( ase.geomobjects.at(i).mesh.iMaterialRef ).dTransparency > 0.0 ) 
                    f |= PF_Translucent;
                }
            else
                {
                f = flags;
                }

            if( f ) ofile << szFlags << szEquals << f << szSpace;

            //
            // Link number
            //
            ofile << szLink << szEquals << p << endl;

            //
            // Polygon data. Either the polygon has textures or not
            //
            if( ase.geomobjects.at(i).mesh.tvert_list.size() > 0 )
                {
                // Get texture width and height.
                int txwidth  = 256, 
                    txheight = 256;

                if( hastexture )
                    {
                    txwidth  = vtxsize.at( ase.geomobjects.at(i).mesh.iMaterialRef ).GetU();
                    txheight = vtxsize.at( ase.geomobjects.at(i).mesh.iMaterialRef ).GetV();
                    }

				WriteTexturedT3DPolygon( ase, ofile, i, v, txwidth, txheight );
                }
            else
                {
                WriteT3DPolygon( ase, ofile, i, v );
                }
            }
        }

    //
    // End PolyList
    // End Brush
    //
    ofile << szTab << szEnd << szSpace << szPolyList << endl;
    ofile << szEnd   << szSpace  << szBrush << endl;
    }



int ASE::WriteT3D(const ASEFile& ase, 
                  const char* szFileName,  
                  const vTexSize& vtxsize,
                  const unsigned int flags, 
                  const bool makeflags)
    {
    using namespace std;

    //
    // Change permission of file here to read/write. We have already
    // prompted the user so it should be okay to overwrite it.
    //
    _chmod( szFileName, _S_IWRITE );

    ofstream ofile;
    ofile.open( szFileName );

    if( false == ofile.is_open() )
        {
        assert( !"Unable to Open file" );
        return CANNOTOPENFILE;
        }

    try{
        WriteT3DBrush(ase, vtxsize, flags, makeflags, ofile);
        }
    catch(ERRORCODE e){
        ofile.close();
        return e;
        }
#ifdef _DEBUG
    catch(std::exception& e){
		ofile << e.what();
        ofile.close();
        return UNKNOWNERROR;
        }
#endif
    catch(...){
        ofile.close();
        return UNKNOWNERROR;
        }

    return 0;
    }



/*solosnakeendsolosnakeendsolosnakeendsolosnakeendsolosnakeendsolosnakeends
//-----------------------------------------------------------------------*/
