//-----------------------------------------------------------------------------
//!   @file       ASE.h
//!   @author     Dire Stockdale 
//!   @author     <a href=http://www.solosnake.com>www.solosnake.com</a>
//!   @date       09-Nov-01 21:04:23
//!   @version    1.00
//!   @brief      Functions, classes and structures for reading data
//!   from 3ds Ascii Scene Exportor file format.
//!           
//-----------------------------------------------------------------------//
#ifndef ASE_H
#define ASE_H 
#include <afxwin.h>
#include "ase_words.h" 
#include "ase2t3d_math.h" 
#include "ase2t3d_file.h"
#include <string>
#include <vector>
#include <fstream>
#include <assert.h>

//
// Flags detailing theinformation that is inthe material
//
#define MNAME           (0x00000001)
#define MAMBIENT        (0x00000002)
#define MDIFFUSE        (0x00000004)
#define MSPECULAR       (0x00000008)
#define MSHINE          (0x00000010)
#define MSHINESTRENGTH  (0x00000020)
#define MTRANSPARENCY   (0x00000040)
#define MWIRESIZE       (0x00000080)
#define MSHADING        (0x00000100)
#define MMAPDIFFUSE     (0x00000200)
#define MMAPNAME        (0x00000400)
#define MMAPBITMAP      (0x00000800)
#define MMAPAMOUNT      (0x00001000)
#define MMAPUOFFSET     (0x00002000)
#define MMAPVOFFSET     (0x00004000)
#define MMAPUTILING     (0x00008000)
#define MMAPVTILING     (0x00010000)
#define MMAPUVWANGLE    (0x00020000)
#define MMAPBLUR        (0x00040000)
#define MMAPBLUROFFSET  (0x00080000)

//
// Flags detailing what is in the mesh
//
#define GTIMEVALUE      (0x00000001)
#define GTVERTICES      (0x00000002)
#define GNORMALS        (0x00000004)


namespace ASE {

    typedef enum { 
           NOFACES              = 1,
           NOVERTICES           = 2,
           NOTEXTUREVERTICES    = 3,
           NOTEXTUREFACES       = 4,
           NOMESHNORMALS        = 5,
           INVALIDMESHNORMALS   = 6,
           INVALIDMESHVNORMAL   = 7,
           MESHERROR            = 8,
           NOMATERIALREF        = 9,
           INVALIDMATERIAL      = 10,
           INVALIDASEFILE       = 11,
           CANNOTOPENFILE       = 12,
           UNKNOWNERROR         = 65535 /* 0xFFFF */
        } ERRORCODE;

    typedef Vector3d<long double> GeomPoint;

    //!
    //!   Struct to represent the node data in a geomobject
    //!
    class GeomObjectNode {
        public:
            GeomObjectNode():iErrorNo(0),dTMRotAngle(0.0),dTMScaleAxisAng(0.0),nodename("")
                {
                dInheritPos[0] = dInheritPos[1] = dInheritPos[2] = 0.0;
                dInheritRot[0] = dInheritRot[1] = dInheritRot[2] = 0.0;
                dInheritScl[0] = dInheritScl[1] = dInheritScl[2] = 1.0;
                dTMRow0[0] = dTMRow0[1] = dTMRow0[2] = 0.0;
                dTMRow1[0] = dTMRow1[1] = dTMRow1[2] = 0.0;
                dTMRow2[0] = dTMRow2[1] = dTMRow2[2] = 0.0;
                dTMPos[0]  = dTMPos[1]  = dTMPos[2]  = 0.0;
                dTMRotAxis[0] = dTMRotAxis[1] = dTMRotAxis[2] = 0.0;
                dTMScale[0] = dTMScale[1] = dTMScale[2] = 1.0;
                dTMScaleAxis[0] = dTMScaleAxis[1] = dTMScaleAxis[2] = 0.0;
                }

            bool read(std::ifstream&);
            int  GetError(void)const throw(){return iErrorNo;};

            std::string     nodename;           //!< Name of Node.
            double          dInheritPos[3];     //!< Values of INHERIT_POS
            double          dInheritRot[3];     //!< Values of INHERIT_ROT
            double          dInheritScl[3];     //!< Values of INHERIT_SCL
            double          dTMRow0[3];         //!< Values of TM_ROW0
            double          dTMRow1[3];         //!< Values of TM_ROW1
            double          dTMRow2[3];         //!< Values of TM_ROW2
            double          dTMPos[3];          //!< Values of TM_POS
            double          dTMRotAxis[3];      //!< Values of TM_ROTAXIS
            double          dTMRotAngle;        //!< Value of  TM_ROTANGLE
            double          dTMScale[3];        //!< Values of TM_SCALE
            double          dTMScaleAxis[3];    //!< Values of TM_SCALEAXIS
            double          dTMScaleAxisAng;    //!< Value of TM_SCALEAXISANG
        private:
            int             iErrorNo;           //!< Error code. Zero if no error occurs.

        };



    //!
    //!   Class to represent the mesh data
    //!
    class GeomObjectMesh {
        public:
            GeomObjectMesh()
                :dTimeValue(0.0),iFaceCount(0),iVertexCount(0),
                bMotionBlur(false),bCastShadow(false),bRecvShadow(false),
                iMaterialRef(-1),iErrorNo(0)
                {};

            bool read(std::ifstream&);

            int  GetError(void)const throw()
                {return iErrorNo;};

            unsigned long GetFlags(void)const throw()
                {return ulFlags;};


            double                  dTimeValue;             //!< Value of the TIMEVALUE.
            int                     iFaceCount;             //!< Number of faces in mesh. Repeated data but for ease of use.
            int                     iVertexCount;           //!< Number of vertices held. Repeated data for ease of use.
            std::vector<GeomPoint>  vertex_list;            //!< The points in the mesh vertex list in order.
            std::vector<int>        face_list;              //!< The indexes into the mesh vertex list that define the 3D object.
            std::vector<int>        face_listB;             //!< The data set of 3-tuples with the indexes, whose purpose I am not sure.
            std::vector<int>        face_list_materialID;   //!< The material IDs associated with individual faces, again for unknown purpose.
            std::vector<GeomPoint>  tvert_list;             //!< Array of texture coordinates. The z-value is rarely used and should be zero.
            std::vector<int>        tface_list;             //!< Array of indexes into the texture coords, 3 per face.
            std::vector<GeomPoint>  facenormals;            //!< Array of normals to faces.
            std::vector<GeomPoint>  vertexnormals;          //!< Array of normals to each vertex in a face. There are thus 3x the number as in mesh_facenormals.
            bool                    bMotionBlur;            //!< Value of PROP_MOTIONBLUR, presumed to be the propert of the mesh concerning motion blur. In the files these seem to be only zero or one.
            bool                    bCastShadow;            //!< Value of PROP_CASTSHADOW, presumed to be the propert of the mesh concerning motion blur. In the files these seem to be only zero or one.
            bool                    bRecvShadow;            //!< Value of PROP_RECVSHADOW, presumed to be the propert of the mesh concerning motion blur. In the files these seem to be only zero or one.
            int                     iMaterialRef;           //!< Value of MATERIAL_REF, which refers to which material in the list to use.    

        private:
            int                     iErrorNo;               //!< Error code. Should be zero for no errors.
            unsigned long           ulFlags;                //!< Flags about the contents of mesh.
        };



    //!
    //!   Struct to represent a single *GEOMOBJECT in an ASE file
    //!
    class GeomObject {
        public:
            GeomObjectNode  node;
            GeomObjectMesh  mesh;

            bool read(std::ifstream&);

            int  GetError(void)const throw(){return iErrorNo;};

        private:
            int         iErrorNo;           //!< Error code. Zero if no error occurs.

        };


    //!
    //!   Class to represent a material referance in the ASE file.
    //!
    class Material {
        public:
           Material():name(""),
                      shading("Flat"),
                      mapname("NO_MAP"),
                      bitmap("NO_BITMAP"),
                      dShine(0.0), 
                      dShineStrength(0.0), 
                      dTransparency(0.0),
                      dWireSize(1.0), 
                      mapamount(1.0), 
                      uvw_U_offset(0.0), 
                      uvw_V_offset(0.0),
                      uvw_U_tiling(1.0), 
                      uvw_V_tiling(1.0),
                      uvw_Angle(0.0), 
                      uvw_blur(1.0), 
                      uvw_blur_offset(0.0),
                      ulFlags(0)
                {
                dAmbient[0]  = dAmbient[1]  = dAmbient[2]  = 0.2;   // OpenGL default.
                dDiffuse[0]  = dDiffuse[1]  = dDiffuse[2]  = 0.8;   // OpenGL defualt.
                dSpecular[0] = dSpecular[1] = dSpecular[2] = 0.0;   // OpenGL defualt.
                };

           bool read(std::ifstream&);

           int  GetError(void)const throw()
               {return iErrorNo;};

           unsigned long GetFlags(void)const throw()
               {return ulFlags;};

           std::string          name;                   //!< Value of MATERIAL_NAME.
           double               dAmbient[3];            //!< Values of MATERIAL_AMBIENT.
           double               dDiffuse[3];            //!< Values of MATERIAL_DIFFUSE.
           double               dSpecular[3];           //!< Values of MATERIAL_SPECULAR.
           double               dShine;                 //!< Value of MATERIAL_SHINE.
           double               dShineStrength;         //!< Value of MATERIAL_SHINESTRENGTH.
           double               dTransparency;          //!< Value of MATERIAL_TRANSPARENCY.
           double               dWireSize;              //!< Value of MATERIAL_WIRESIZE.
           std::string          shading;                //!< Value of MATERIAL_SHADING.
           std::string          mapname;                //!< Value of MAP_NAME.
           std::string          bitmap;                 //!< Value of BITMAP.
           double               mapamount;              //!< Value of MAP_AMOUNT.
           double               uvw_U_offset;           //!< Value of UVW_U_OFFSET.
           double               uvw_V_offset;           //!< Value of UVW_V_OFFSET.
           double               uvw_U_tiling;           //!< Value of UVW_U_TILING.
           double               uvw_V_tiling;           //!< Value of UVW_V_TILING.
           double               uvw_Angle;              //!< Value of UVW_ANGLE.
           double               uvw_blur;               //!< Value of UVW_BLUR.
           double               uvw_blur_offset;        //!< Value of UVW_BLUR_OFFSET.
        private:
            int                 iErrorNo;               //!< Error code. Zero if no error occurs.
            unsigned long       ulFlags;                //!< Flags

        };



    //!
    //!   Class to hold details of a single ase file.
    //!
    class   ASEFile {
        public:
            ASEFile();

            int read(const char*);
            int read(std::ifstream&);

            int GetWarnings(void)const throw()
                {return iWarning;};

            int GetVertexCount(void)const throw();

            int GetMaterialCount(void)const throw()
                {return materials.size();};

            int GetGeomObjectCount(void)const throw()
                {return geomobjects.size();};

            std::string                 scenename;
            double                      dBackgroundStatic[3];
            double                      dAmbientStatic[3];
            std::vector<Material>       materials;
            std::vector<GeomObject>     geomobjects;

        private:
            int                         iWarning;

        };



} // End namespace ASE;
#endif // ASE_H
//-----------------------------------------------------------------------------
