///////////////////////////////////////////////////////////////////////////////
//! \file       ase2t3d.h
//! \brief      Math functions and definitions used in conversion.
///////////////////////////////////////////////////////////////////////////////
#ifndef ASE2T3D_MATH_H
#define ASE2T3D_MATH_H
#include <afxwin.h>
#include <algorithm>
#include <cmath>

//
//  Radians
//
#define GE_TWO_PI           (6.28318530717958647692528676655901f)
#define GE_PI               (3.1415926535897932384626433832795f)
#define GE_TWO_PI_INV       (0.159154943091895335768883763372514f)
#define GE_PI_INV           (0.318309886183790671537767526745029f)
#define GE_3QUARTER_PI      (2.35619449019234492884698253745963f)
#define GE_2THIRD_PI        (2.09439510239319549230842892218634f)
#define GE_HALF_PI          (1.57079632679489661923132169163975f)
#define GE_THIRD_PI         (1.04719755119659774615421446109317f)
#define GE_QUARTER_PI       (0.785398163397448309615660845819876f)

//
//  Degrees
//
#define GE_TWO_PI_DEGREES       (360.0f)
#define GE_PI_DEGREES           (180.0f)
#define GE_TWO_PI_DEGREES_INV   (0.002777777777777777777777777777778f)
#define GE_PI_DEGREES_INV       (0.005555555555555555555555555555556f)
#define GE_3QUARTER_PI_DEGREES  (135.0f)
#define GE_2THIRD_PI_DEGREES    (120.0f)
#define GE_HALF_PI_DEGREES      (90.0f)
#define GE_THIRD_PI_DEGREES     (60.0f)
#define GE_QUARTER_PI_DEGREES   (45.0f)

//
// Factors for converting one to another
//
#define GE_DEGREES_FACTOR       (57.2957795130823208767981548141052f)
#define GE_RADIANS_FACTOR       (0.0174532925199432957692369076848861f)
#define get_degrees(radians) (GE_DEGREES_FACTOR * (radians))
#define get_radians(degrees) (GE_RADIANS_FACTOR * (degrees))
#define ge_sqrt(rhs)        static_cast<float>(sqrtf( static_cast<float>(rhs)))
#define ge_inv_sqrt(rhs)    (1.0f/sqrtf((rhs)))

//
// Templated versions of common routines offering faster versions for
// specialisations.
//
template<typename T> T inline ge_abs(const T rhs){ return fabs(rhs); }
template<typename T> T inline ge_cos(const T rhs){ return cos(rhs); }
template<typename T> T inline ge_sin(const T rhs){ return sin(rhs); }
template<typename T> T inline ge_tan(const T rhs){ return tan(rhs); }
template<typename T> T inline ge_acos(const T rhs){ return acos(rhs); }
template<typename T> T inline ge_asin(const T rhs){ return asin(rhs); }
template<typename T> T inline ge_atan(const T rhs){ return atan(rhs); }
template<typename T> T inline ge_atan2(const T rhs1, const T rhs2){ return atan2(rhs1,rhs2); }
template<typename T> T inline ge_cosh(const T rhs){ return cosh(rhs); }
template<typename T> T inline ge_sinh(const T rhs){ return sinh(rhs); }
template<typename T> T inline ge_tanh(const T rhs){ return tanh(rhs); }
template<typename T> T inline ge_ceil(const T rhs){ return ceil(rhs); }
template<typename T> T inline ge_floor(const T rhs){ return floor(rhs); }
template<typename T> T inline ge_fmod(const T rhs1,const T rhs2){ return fmod(rhs1,rhs2); }

/*
*   Specialised versions.
*/    
template<> inline float ge_abs(const float rhs){ return fabsf(rhs); }
template<> inline float ge_cos(const float rhs){ return cosf(rhs); }
template<> inline float ge_sin(const float rhs){ return sinf(rhs); }
template<> inline float ge_tan(const float rhs){ return tanf(rhs); }
template<> inline float ge_acos(const float rhs){ return acosf(rhs); }
template<> inline float ge_asin(const float rhs){ return asinf(rhs); }
template<> inline float ge_atan(const float rhs){ return atanf(rhs); }
template<> inline float ge_atan2(const float rhs1, const float rhs2){ return atan2f(rhs1,rhs2); }
template<> inline float ge_cosh(const float rhs){ return coshf(rhs); }
template<> inline float ge_sinh(const float rhs){ return sinhf(rhs); }
template<> inline float ge_tanh(const float rhs){ return tanhf(rhs); }
template<> inline float ge_ceil(const float rhs){ return ceilf(rhs); }
template<> inline float ge_floor(const float rhs){ return floorf(rhs); }
template<> inline float ge_fmod(const float rhs1, const float rhs2){ return fmodf(rhs1,rhs2); }


// Prototypes.
template<typename T> class Vector3d;
template<typename T>Vector3d<T> operator-(const Vector3d<T>& v);
template<typename T>Vector3d<T> operator*(const T f, const Vector3d<T>& v);
template<typename T>Vector3d<T> operator/(const T f, const Vector3d<T>& v);

template<typename T>
struct GETuple3 {
    T   x,y,z;
    };

template<typename T>
struct GETuple4 {
    T   x,y,z,w;
};

template<typename T>
class Vector3d {

#ifdef _MSC_VER
    friend inline Vector3d<T>  operator-(const Vector3d<T>& v);    // Negates vector.
    friend inline Vector3d<T>  operator*(const T, const Vector3d<T>&);
    friend inline Vector3d<T>  operator/(const T, const Vector3d<T>&);
#else
    friend inline Vector3d<T>  operator-<>(const Vector3d<T>& v);    // Negates vector.
    friend inline Vector3d<T>  operator*<>(const T, const Vector3d<T>&);
    friend inline Vector3d<T>  operator/<>(const T, const Vector3d<T>&);
#endif

    public:
        Vector3d():x(0.0f),y(0.0f),z(0.0f){};       
        Vector3d(const T a, const T b, const T c):x(a),y(b),z(c){};
        
        inline Vector3d(const T[3]);
        
        typedef T type;

        // Assignment operator
        inline Vector3d<T>& operator=(const T*const);

        // Binary operators
        inline Vector3d<T> operator+(const Vector3d<T>& rhs) const;
        inline Vector3d<T> operator-(const Vector3d<T>& rhs) const;
        inline Vector3d<T> operator*(const T rhs) const;
        inline Vector3d<T> operator/(const T rhs) const;

        // Unary operators
        inline Vector3d<T>& operator+=(const Vector3d<T>& rhs);
        inline Vector3d<T>& operator-=(const Vector3d<T>& rhs);
        inline Vector3d<T>& operator*=(const T rhs);
        inline Vector3d<T>& operator/=(const T rhs);

        // Logical operators
        inline bool operator==(const Vector3d<T>& rhs) const;
        inline bool operator!=(const Vector3d<T>& rhs) const; 
        inline bool operator <(const Vector3d<T>&)const;
        inline bool operator >(const Vector3d<T>&)const;

        // Note: Dot / Scalar product:
        inline T  operator*(const Vector3d<T>& rhs) const;

        // Note: Cross / Vector product:
        inline Vector3d<T>  operator^(const Vector3d<T>& rhs) const;
        inline Vector3d<T>& operator^=(const Vector3d<T>& rhs);

        // Return modulus of vector
        inline T modulus(void)const;

        // Returns the 'square-sum' of the vector (x*x + y*y + z*z)
        inline T square_sum(void)const;

        // Returns the distance between two TVectors.
        inline T distance(const Vector3d<T>&)const;

        // Returns the squared (no use of sqrt) distance between vectors.
        inline T distance_sq(const Vector3d<T>& a)const;

        // Returns the midpoint between the two vectors.
        inline Vector3d<T> midpoint(const Vector3d<T>& rhs)const;

        // Returns true if perpendiculer to vector
        inline bool perpendiculer(const Vector3d<T>& rhs) const;
        
        // Returns the point as spolar coordinates, as angle,angle,radius.
        inline Vector3d<T> polar()const;
        
        // Returns the polar point (angle,angle,radius) as cartesian (rectanglur) x,y,z.
        inline Vector3d<T> cartesian()const;
        
        Vector3d<T> normalize(void)
            { 
            const register T m = 1.0f / ge_sqrt( x*x + y*y + z*z );
            x *= m; 
            y *=m;
            z *=m; 
            return *this;   
            }
        
        // Divides all co-efs by modulus
        Vector3d<T> normalized(void)const
            { 
                const register T m = 1.0f / ge_sqrt( x*x + y*y + z*z );
                return Vector3d<T>(x*m, y*m, z*m);   
            }

        // Returns a vector interpolated between a & b by distance d from a (1.0 = 100%).
        inline static Vector3d<T> interpolate(const Vector3d<T>& a,const Vector3d<T>& b, const T d);

        // Return / set current precision:
        inline static T getPrecision(void);
        inline static void setPrecision(const T);

        // Added for compliance with other geometry types:
        // Returns true if vectors are parallel and of equal magnitude.
        inline bool equivalent(const Vector3d<T>& a)const;

        // Returns true if vectors are parallel.
        inline bool parallel(const Vector3d<T>& a)const;

        T x, y, z;

         static const Vector3d<T>      ORIGIN;
      
    protected:
        //! Precision for testing against
        static T                sm_precision;
};



//!
//!  Prefix unary minus: added 12 01 02.
//!
template<typename T>
Vector3d<T> operator-(const Vector3d<T>& v)
    {
    return Vector3d<T>( -v.x, -v.y, -v.z );
    }


//!
//!  Operators for scalar global multiplication.
//!  Added 19 01 02
//!
template<typename T>
Vector3d<T> operator*(const T f, const Vector3d<T>& v)
    {
    return v * f;
    }


//!
//!  Operators for scalar global division.
//!  Added 19 01 02
//!
template<typename T>
Vector3d<T> operator/(const T f, const Vector3d<T>& v)
    {
    return v / f;
    }
        
    
//!
//! Returns the vector of the absalute values of the rhs vector
//!
template<typename T>inline
Vector3d<T> abs(const Vector3d<T>& rhs)
    {
    return Vector3d<T>( ge_abs(rhs.x), ge_abs(rhs.y), ge_abs(rhs.z) );
    }


//!
//! Static variable to class:
//!
template<typename T>
T Vector3d<T>::sm_precision = PTHICKNESS;

//!
//! Static origin.
//!
template<typename T>
const Vector3d<T> Vector3d<T>::ORIGIN = Vector3d<T>(0.0f, 0.0f, 0.0f); 


template<typename T>inline 
Vector3d<T>::Vector3d(const T arr[3]):x(arr[0]),y(arr[1]),z(arr[2]){}


//!
//! Gets class precison
//!
template<typename T>inline
T Vector3d<T>::getPrecision(void)
    {
    return sm_precision;
    }


//!
//! Sets class precison.
//!
template<typename T>inline
void Vector3d<T>::setPrecision(const T prec)
    {
    sm_precision = prec;
    }


//!
//! Assignment operator added 16 04 01
//!
template<typename T>inline 
Vector3d<T>& Vector3d<T>::operator=(const T* rhs)
    {
    x=rhs[0];
    y=rhs[1];
    z=rhs[2];
    return *this;
    }


//!
//!   Add two vectors.
//!
template<typename T>inline 
Vector3d<T> Vector3d<T>::operator+(const Vector3d<T>& rhs) const
    {
    return Vector3d<T>(x+rhs.x, y+rhs.y, z+rhs.z);
    }


//!
//!   Subtract two vectors:
//!
template<typename T>inline 
Vector3d<T> Vector3d<T>::operator-(const Vector3d<T>& rhs) const
    {
    return Vector3d<T>(x-rhs.x, y-rhs.y, z-rhs.z);
    }


//!
//!  Multiply a vector by a scalar:
//!
template<typename T>inline 
Vector3d<T> Vector3d<T>::operator*(const T rhs) const
    {
    return Vector3d<T>(x*rhs, y*rhs, z*rhs);
    }


//!
//!  Divide a vector by a scalar.
//!
template<typename T>inline 
Vector3d<T> Vector3d<T>::operator/(const T rhs) const
    {
//    assert( rhs && "Divide by zero in Vector3d" );
    return Vector3d<T>(x/rhs, y/rhs, z/rhs);
    }


//!
//!  Unary operators. Add a vector to itself:
//!
template<typename T>inline 
Vector3d<T>& Vector3d<T>::operator+=(const Vector3d<T>& rhs)
    {
    x+=rhs.x;
    y+=rhs.y;
    z+=rhs.z;
    return *this;
    }


//!
//!  Unary operators. Subtract a vector from itself.
//!
template<typename T>inline 
Vector3d<T>& Vector3d<T>::operator-=(const Vector3d<T>& rhs)
    {
    x-=rhs.x;
    y-=rhs.y;
    z-=rhs.z;
    return *this;
    }


//!
//!  Unary operators. Multiply it by a scalar.
//!
template<typename T>inline 
Vector3d<T>& Vector3d<T>::operator*=(const T rhs)
    {
    x*=rhs;
    y*=rhs;
    z*=rhs;
    return *this;
    }


//!
//!  Unary operators. Divide it by a scalar.
//!
template<typename T>inline 
Vector3d<T>& Vector3d<T>::operator/=(const T rhs)
    {
    assert( rhs != T(0) || !"Dicide by zero in Vector3d" );

#if PREFER_FAST_GEVECTOR3D == 1
    const register T r = 1.0f/rhs;
    x*=r;
    y*=r;
    z*=r;
#else
    x/=rhs;
    y/=rhs;
    z/=rhs;
#endif        
    return *this;
    }


//!
//!  Logical operators. Equality.
//!
template<typename T>inline 
bool Vector3d<T>::operator==(const Vector3d<T>& rhs) const
    {
    return (x==rhs.x)&&(y==rhs.y)&&(z==rhs.z);
    }


//!
//!  Logical operators. Non-Equality.
//!
template<typename T>inline 
bool Vector3d<T>::operator!=(const Vector3d<T>& rhs) const
    {
    return (x!=rhs.x)||(y!=rhs.y)||(z!=rhs.z);
    }


//!
//!  Binary predicate for STL. Returns true if this < a.
//!
template<typename T>inline 
bool Vector3d<T>::operator<(const Vector3d<T>& a)const
    {
    if( x < a.x ) return true;
    else if( x == a.x )
        {
        if( y < a.y ) return true;
        else if( y == a.y )
            {
            if( z < a.z ) return true;
            }
        }
    return false;
    }


//!
//!  Binary predicate for STL. Returns true if this < a.
//!
template<typename T>inline 
bool Vector3d<T>::operator>(const Vector3d<T>& a)const
    {
    if( x > a.x ) return true;
    else if( x == a.x )
        {
        if( y > a.y ) return true;
        else if( y == a.y )
            {
            if( z > a.z ) return true;
            }
        }
    return false;
    }


//!
//! Dot / Scalar product operations.
//! The dot product of two vectors is scalar value, which can be thought of as
//! representative of the angle between the vectors, or a measure of their
//! magnitude.
//! Some properties of the DOT product:
//!
//! <pre>
//!
//!        a.b = |a|*|b|*Cos(A), A = angle between a & b in radians
//!        a.b = b.a
//!        a.(b+c) = a.b + a.c
//!        a.(R*b) = R*(a.b)
//!        a.a = |a|^2 >= 0
//!        a.a = 0 <-> a = 0
//!
//! Most importantly:
//! a.b = 0 <-> a & b are orthogonal (perpendicular)
//!</pre>
//! 
template<typename T>inline  
T  Vector3d<T>::operator*(const Vector3d<T>& rhs) const
    {
    return (x*rhs.x + y*rhs.y + z*rhs.z);
    }


//!
//!   Cross / Vector product operations.
//!   The cross product of two vectors is a third vector orthogonal to the other
//!   two. I have used the ^ operator to denote cross product operations after
//!   older notation used in maths texts similar to an inverted V: /\
//!   The example below shows a right handed coordinate sytem.
//!   OpenGL is a right handed sytem, DirectX is by default a left-handed system.
//!   Some properties of the CROSS product:
//!   <pre>
//!
//!        a^b =/= b^a 
//!            Except in certain special cases
//!
//!        a^b = 0
//!            Either a=0 or b=0. If neither a nor b are 0 
//!            then a||b (a parallel to b)
//!
//!        a^b = -b^a
//!
//!        a^(b^c) =/= (a^b)^c
//!
//!        a^b is a vector orthogonal to a & b:
//!                    
//!                   a^b
//!                    |
//!                    |
//!                    |_ _ _ _ a
//!                    /
//!                   /
//!                  /
//!                 b    
//!</pre>
//!
template<typename T>inline 
Vector3d<T> Vector3d<T>::operator^(const Vector3d<T>& rhs) const
    {
    return Vector3d<T>(  y*rhs.z - z*rhs.y,
                         z*rhs.x - x*rhs.z, // -( x*rhs.z - z*rhgs.x )
                         x*rhs.y - y*rhs.x );
    }


//!
//!  Unary Cross / Vector product operations.
//!  The cross product of the vectors is a third vector orthogonal to the other
//!  two. I have used the ^ operator to denote cross product operations after
//!  older notation used in maths texts similar to an inverted V: /\
//!  Some properties of the CROSS product:<pre>
//!
//!        a^b =/= b^a 
//!            Except in certain special cases
//!
//!        a^b = 0
//!            Either a=0 or b=0. If neither a nor b are 0 
//!            then a||b (a parallel to b)
//!
//!        a^b = -b^a
//!
//!        a^(b^c) =/= (a^b)^c
//!
//!        a^b is a vector orthogonal to a & b:
//!                    
//!                   a^b
//!                    |
//!                    |
//!                    |_ _ _ _ a
//!                    /
//!                   /
//!                  /
//!                 b    
//!</pre>
//!
template<typename T>inline 
Vector3d<T>& Vector3d<T>::operator^=(const Vector3d<T>& rhs)
    {
    return *this = Vector3d<T>(  y*rhs.z - z*rhs.y,
                                 z*rhs.x - x*rhs.z,
                                 x*rhs.y - y*rhs.x );
    }


//!
//!  Return modulus of vector.
//!
template<typename T>inline 
T Vector3d<T>::modulus()const
    {
    return ge_sqrt( x*x + y*y + z*z );
    }


//!
//!  Returns true if vector rhs is perpendicular to *this. Added 13 03 01
//!
template<typename T>inline 
bool Vector3d<T>::perpendiculer(const Vector3d<T>& rhs) const
    {
    return (ge_abs(*this * rhs) < sm_precision);
    }


//!
//!  Returns a point that is d times the distance between a & b away from a. 
//!  Added 20 03 01.
//!
template<typename T>inline 
Vector3d<T> Vector3d<T>::interpolate(const Vector3d<T>& a,const Vector3d<T>& b, const T d)
    {
    return Vector3d<T>( b.x > a.x ? a.x + (b.x - a.x)*d : a.x - (a.x - b.x)*d,
                        b.y > a.y ? a.y + (b.y - a.y)*d : a.y - (a.y - b.y)*d,
                        b.z > a.z ? a.z + (b.z - a.z)*d : a.z - (a.z - b.z)*d );
    }

//!
//!  Returns the distance between two TVectors. Added 08 04 01.
//!
template<typename T>inline 
T Vector3d<T>::distance(const Vector3d<T>& rhs)const
    {
    return ge_sqrt( (x-rhs.x)*(x-rhs.x)+(y-rhs.y)*(y-rhs.y)+(z-rhs.z)*(z-rhs.z));
    }


//!
//!  Returns the midpoint between two vectors. Added 08 04 01.
//!
template<typename T>inline 
Vector3d<T> Vector3d<T>::midpoint(const Vector3d<T>& rhs)const
    {
    return interpolate(*this, rhs, 0.5f);
    }
    
    
    
//!
//!  Returns true if vectors are parallel.
//!  Added 15 01 02
//!
template<typename T>inline
bool Vector3d<T>::parallel(const Vector3d<T>& a)const
    {
    return ((y * a.z - z * a.y) == 0.0f) && ((z * a.x - x * a.z) == 0.0f) && ((x * a.y - y * a.x) == 0.0f);
    }


//!
//!  Test for mathematical equivalence: added 15 01 02.
//!  Returns true if vectors are parallel & have same magnitude.
//!
template<typename T>inline
bool Vector3d<T>::equivalent(const Vector3d<T>& a)const
    {
    return parallel(a) && ((x * x + y * y + z * z) == (a.x * a.x + a.y * a.y + a.z * a.z));
    }




//!
//!  Returns the distacne between two vectors without the sqrt.
//!
template<typename T>inline
T Vector3d<T>::distance_sq(const Vector3d<T>& a)const
    {
    return (x-a.x)*(x-a.x) + (y-a.y)*(y-a.y) + (z-a.z)*(z-a.z);
    }


//!
//!  Returns the 'square-sum' of the vector (x*x + y*y + z*z)
//!  Added 05 02 02
//!
template<typename T>inline
T Vector3d<T>::square_sum(void)const
    {
    return x*x + y*y + z*z;
    }

   

//! Added 05 May 2002
template<typename T>inline
Vector3d<T> Vector3d<T>::polar()const
    {
//    const T r = ge_sqrt(x*x + y*y + z*z);
//    const T a = ge_abs( z ) > sm_precision ? ge_atan( x / z ) : GE_HALF_PI; 
//    const T b = ge_acos( y / r );        
//    return Vector3d<T>( a,b,r );
    
    const T r = ge_sqrt(x*x + y*y + z*z);
   
#if PREFER_FAST_GEVECTOR3D == 1
    T a = ge_atan( x / z );
#else 
    T a = ge_abs( z ) > sm_precision ? ge_atan( x / z ) : GE_HALF_PI;
#endif   

    a = z < 0 ? a + GE_PI : a;
    
    return Vector3d<T>( a, ge_acos( y / r ), r );     
    }
  
        
template<typename T>inline
Vector3d<T> Vector3d<T>::cartesian()const
    {
//    const T xxx = z * ge_sin( x ) * ge_sin( y );     
//    const T yyy = z * ge_cos( y );  
//    const T zzz = z * ge_sin( y ) * ge_cos( x );    
//    return Vector3d<T>( xxx, yyy, zzz );
    
    const T sa = ge_sin( y );
    return Vector3d<T>( z * sa * ge_sin( y ), z * ge_cos( y ), z * sa * ge_cos( x ) );    
    }
    
#endif // ASE2T3D_MATH_H
///////////////////////////////////////////////////////////////////////////////

