/******************************************************************************
Copyright (c) 2007 by Wormbo

Baseclass for animated effects.
******************************************************************************/

class AnimatedEffect extends Effects abstract;


//=============================================================================
// Enums
//=============================================================================

enum InterpStyle
{
	IS_Flat,
	IS_Linear,
	IS_Exp,
	IS_InvExp,
	IS_S
};


//=============================================================================
// Structs
//=============================================================================

struct Interpomatic
{
	var() float Start;
	var() float Mid;
	var() float End;
	var() float InTime;
	var() float OutTime;
	var() InterpStyle InStyle;
	var() InterpStyle OutStyle;
};


//=============================================================================
// Properties
//=============================================================================

var() Interpomatic FadeInterp;
var() Interpomatic ScaleInterp;


simulated function Tick(float DeltaTime)
{
	ScaleGlow = InterpInterp(FadeInterp, 1.0 - LifeSpan/default.LifeSpan);
	AmbientGlow = default.AmbientGlow * ScaleGlow;
	
	DrawScale = InterpInterp(ScaleInterp, 1.0 - LifeSpan/default.LifeSpan);
}

static final function float InterpInterp(Interpomatic Interp, float t)
{
	if (t < Interp.InTime) {
		t = t / Interp.InTime;
		switch (Interp.InStyle) {
			case IS_Linear:
				return Lerp(t, Interp.Start, Interp.Mid);
			case IS_Exp:
				return Lerp(t*t*t, Interp.Start, Interp.Mid);
			case IS_InvExp:
				return Lerp(1-(1-t)*(1-t)*(1-t), Interp.Start, Interp.Mid);
			case IS_S:
				return Smerp(t, Interp.Start, Interp.Mid);
			default:
				return Interp.Start;
		}
	}
	else if (t <= Interp.OutTime) {
		return Interp.Mid;
	}
	else {
		t = (t - Interp.OutTime) / (1.0 - Interp.OutTime);
		switch (Interp.OutStyle) {
			case IS_Linear:
				return Lerp(t, Interp.Mid, Interp.End);
			case IS_Exp:
				return Lerp(1-(1-t)*(1-t)*(1-t), Interp.Mid, Interp.End);
			case IS_InvExp:
				return Lerp(t*t*t, Interp.Mid, Interp.End);
			case IS_S:
				return Smerp(t, Interp.Mid, Interp.End);
			default:
				return Interp.Mid;
		}
	}
}


//=============================================================================
// Default values
//=============================================================================

defaultproperties
{
    FadeInterp=(Start=0.00,Mid=1.00,End=0.00,InTime=0.00,OutTime=1.00,InStyle=1,OutStyle=1),
    ScaleInterp=(Start=0.00,Mid=1.00,End=0.00,InTime=0.00,OutTime=1.00,InStyle=1,OutStyle=1),
    RemoteRole=0
    DrawType=1
    Style=3
    AmbientGlow=254
    bUnlit=True
}
