// --------------------------------------------------------------------
// CFtpClient.hxx
// Whatis:  FTP Client class library
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-FEB-2003     Created this source
// --------------------------------------------------------------------
#ifndef CFtpClient_hxx
#define CFtpClient_hxx

#include	"CTcpSocket.hxx"
#include	"CCpuMutex.hxx"
#include	"CFtpFile.hxx"

// --------------------------------------------------------------------
typedef enum {
	ftp_state_idle		= 0,	// Not connected
	ftp_state_connect	= 1,	// Connected and logged in
    ftp_state_login		= 2,	// Perform login
    ftp_state_cwd		= 3,	// Change working directory
	ftp_state_refresh	= 4,	// Do a directory refresh
    ftp_state_upload	= 5,	// Upload a file to server
    ftp_state_update	= 6,	// Update existing file
    ftp_state_remove	= 7,	// Remove existing file

	ftp_state_error		= 999
}	ftp_state_t;

// --------------------------------------------------------------------
typedef enum {
    ftp_type_ASCII 		= 'A',
    ftp_type_EBCDIC		= 'E',
    ftp_type_IMAGE		= 'I'
}	ftp_type_t;

// --------------------------------------------------------------------
typedef enum {
	ftp_struc_file		= 'F',
    ftp_struc_record	= 'R',
    ftp_struc_page		= 'P'
}	ftp_struc_t;

// --------------------------------------------------------------------
typedef enum {
	ftp_mode_stream		= 'S',
    ftp_mode_block		= 'B',
    ftp_mode_compressed	= 'C'
}	ftp_mode_t;

// --------------------------------------------------------------------
typedef struct {
	const char *	name;
}	ftp_system_t;

// --------------------------------------------------------------------
typedef struct {
	char			addr[128];
    unsigned short	port;
}	ftp_sockaddr_t;

// --------------------------------------------------------------------
class CFtpClient {

	// ----------------------------------------------------------------
	public:		// Constructor and destructor
	// ----------------------------------------------------------------
	CFtpClient	();
	~CFtpClient	();

	// ----------------------------------------------------------------
	private:	// Copy constructor and assignment operator - invalid
	// ----------------------------------------------------------------
	CFtpClient				( const CFtpClient & );
	CFtpClient & operator =	( const CFtpClient & );

	// ----------------------------------------------------------------
	public:		// Method called by the thread
	// ----------------------------------------------------------------
	void				RunThread	( void );

	// ----------------------------------------------------------------
	public:		// Interface
	// ----------------------------------------------------------------
	void				Clear		( void );
    void				Login		( void );
	const char *		Cwd			( void ) const;
    void				Cwd			( const char * );
    const CFtpFile_l &	Dir			( void ) const;
    void				Refresh		( void );
    void				Update		( const char * );
    void				Upload		( const char * );
    void				Remove		( const char * );
    void				Abort		( void );

	// ----------------------------------------------------------------
	public:		// Accessing instance data
	// ----------------------------------------------------------------
	const char *		Host		( void ) const;
	unsigned short		Port		( void ) const;
	const char *		User		( void ) const;
	const char *		Pass		( void ) const;
	const char *		Acnt		( void ) const;
    ftp_state_t			State		( void ) const;
    bool				KeepAlive	( void ) const;
    const char *		LastResponse( void ) const;
    ftp_system_t		System		( void ) const;
    bool				Pasv		( void ) const;

	void				Host		( const char * );
    void				Port		( unsigned short );
    void				User		( const char * );
    void			 	Pass		( const char * );
    void			 	Acnt		( const char * );
    void				KeepAlive	( bool );
    void				Pasv		( bool );

	// ----------------------------------------------------------------
	private:	// Private helpers
	// ----------------------------------------------------------------
	void				Cleanup					( void );
	void				Free					( void );

	// ----------------------------------------------------------------
	private:	// State machine
	// ----------------------------------------------------------------
    void				StateMachine			( void );
	void				StateMachineConnect		( void );
	void				StateMachineLogin		( void );
	void				StateMachineCwd			( void );
	void				StateMachineRefresh		( void );
	void				StateMachineUpload		( void );
	void				StateMachineUpdate		( void );
	void				StateMachineRemove		( void );
    void				StateMachineError		( void );

	// ----------------------------------------------------------------
	private:	// RFC 959 - http://www.faqs.org/rfcs/rfc959.html
	// ----------------------------------------------------------------
    int				USER	( const char * );
    int				PASS	( const char * );
    int				ACCT	( const char * );
    int				CWD		( const char * );
    int				CDUP	( void );
    int				SMNT	( const char * );
    int				REIN	( void );
    int				QUIT	( void );
    CTcpSocket *	PORT	( void );
    bool			PASV	( ftp_sockaddr_t & );
    int				TYPE	( ftp_type_t );
    int				TYPE	( size_t );
    int				STRU	( ftp_struc_t );
    int				MODE	( ftp_mode_t );
    int				RETR	( const char * );
    int				STOR	( const char *, FILE * );
    int				STOU	( void );
    int				APPE	( const char * );
    int				ALLO	( size_t );
    int				ALLO	( size_t, size_t );
    int				REST	( const char * );
    int				RNFR	( const char * );
    int				RNTO	( const char * );
    int				ABOR	( void );
    int				DELE	( const char * );
    int				RMD		( const char * );
    int				MKD		( const char * );
    int				PWD		( void );
    int				LIST	( const char * = NULL );
    int				NLST	( const char * = NULL );
    int				SITE  	( const char * );
    int				SYST	( void );
    int				STAT	( const char * = NULL );
    int				HELP	( const char * = NULL );
    int				NOOP	( void );

    int				GetReply		( void );
	int				ParseReply		( char *, const char * );
    void			SkipContinue	( char *, char *, size_t );
    void			GetReplyLine	( char *, size_t );
    void			GetSystem		( const char * );
    void			GetPwd			( const char * );
    bool			GetSockAddr		( ftp_sockaddr_t &, const char * );
    int				GetDirectory	( CTcpSocket * );
    void			ParseDirectory	( const char * );

	// ----------------------------------------------------------------
	private:	// Instance data
	// ----------------------------------------------------------------
	char *				itsHost;
	unsigned short		itsPort;
	char *				itsUser;
	char *				itsPass;
    char *				itsAcnt;

	CCpuMutex			itsMutex;
	CTcpSocket *		itsCtlSock;

	// These are for the state machine
    ftp_state_t			itsState;

	// These are for the thread control
	bool				itsIsRunning;
	bool				itsDoTerminate;
    bool				itsDoAbortion;
   	unsigned long		itsThreadId;

    // These are for the error state
	char				itsErrorMessage	[1024];
    time_t				itsErrorStart;

    // These are for the last response
    char				itsLastResponse	[1024];

    // Keepalive (NOOP) signalling timing and stuff
    time_t				itsLastKeepAlive;
    bool				itsKeepAlive;

    // System is interpreted during connection
    int					itsSystem;

    // PWD support
    bool				itsCanPwd;
    char				itsCwd	[1024];
    CFtpFile_l			itsDir;

    // PASV support
    bool				itsCanPasv;

    // File name for file transfers
    char				itsLocalFile	[1024];

};

#endif
// --------------------------------------------------------------------
// EOF:	CFtpClient.hxx
// --------------------------------------------------------------------
