// --------------------------------------------------------------------
// CFtp.cpp
// Whatis:  The FTP abstract
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 11-NOV-2003     Created this source
// --------------------------------------------------------------------
#include	"CFtp.hxx"

// --------------------------------------------------------------------
static unsigned long __stdcall __MyFtpThread( void * aFtp ) {
	try	{
		((CFtp *)aFtp)->Run();
    }
    catch	( ... ) {
    }
	::ExitThread( 0 );
	return	0;
}

// --------------------------------------------------------------------
CFtp::CFtp		() {
	CFtp::Cleanup();
	if	( ::CreateThread(
    			NULL,					// Security
                32768,					// Stack
                __MyFtpThread,			// Thread
                (void *)this,			// Argument
                0,						// Flags
    			&itsThreadId ) <= 0 ) {
		itsError = CError( "Unable to start thread" );
		itsState = ftp_state_error;
	}
}

// --------------------------------------------------------------------
CFtp::~CFtp 	() {
	while	( itsRunning ) {
		CFtp::Abort();
	}
	CFtp::Free();
}

// --------------------------------------------------------------
void	CFtp::Run			( void ) {
	itsRunning = true;
	while	( itsState != ftp_state_abort ) {
		itsMutex.Lock();
		try {
			switch	( itsState ) {
            	case	ftp_state_login:	CFtp::DoLogin();	break;
                case	ftp_state_logout:	CFtp::DoLogout();	break;
                case	ftp_state_cwd:		CFtp::DoCwd();		break;
                case	ftp_state_upload:	CFtp::DoUpload();	break;
                case	ftp_state_download:	CFtp::DoDownload();	break;
                case	ftp_state_remove:	CFtp::DoRemove();	break;
				default:
                break;
			}
		}
		catch ( CError e ) {
			itsError = e;
			::strcpy( itsMessage, e.Error() );
			if	( itsState != ftp_state_logout )	itsState = ftp_state_error;
		}
		catch ( ... ) {
			itsError = CError( "Unknown error" );
            ::strcpy( itsMessage, "Unknown error" );
			if	( itsState != ftp_state_logout )	itsState = ftp_state_error;
		}
		itsMutex.Unlock();
		switch	( itsState ) {
        	case	ftp_state_error:
			case	ftp_state_offline:
            case	ftp_state_online:	SLEEP( 100 );	break;
		}
	}
	itsRunning = false;
}

// --------------------------------------------------------------
void	CFtp::Abort		( void ) {
	itsMutex.Lock();
	itsState = ftp_state_abort;
	itsMutex.Unlock();
}

// --------------------------------------------------------------------
void	CFtp::Login	( 	const char *	aHost,
						unsigned short	aPort,
						const char *	aUser,
						const char * 	aPass,
                        const char *	aAcct ) {
	itsMutex.Lock();
	try	{
		if	( itsState != ftp_state_offline )	throw CError( "Wrong state" );
	    if	( itsHost )	delete [] itsHost;	itsHost = NULL;
    	if	( itsUser )	delete [] itsUser;	itsUser = NULL;
	    if	( itsPass )	delete [] itsPass;	itsPass = NULL;
	    if	( itsAcct )	delete [] itsAcct;	itsAcct = NULL;
	    ::memset( itsCwd, 0, sizeof( itsCwd ) );

		itsHost 	= ::my_private_strdup( aHost );
		itsPort 	= aPort;
		itsUser 	= aUser ? ::my_private_strdup( aUser ) : NULL;
		itsPass 	= aPass ? ::my_private_strdup( aPass ) : NULL;
		itsAcct		= aAcct ? ::my_private_strdup( aAcct ) : NULL;
		itsState	= ftp_state_login;
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
void	CFtp::Logout	( void ) {
	itsMutex.Lock();
	try	{
		itsState	= ftp_state_logout;
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
void			CFtp::Cwd	( const char *	aCwd ) {
	itsMutex.Lock();
	try	{
		if	( itsState != ftp_state_online )	throw CError( "Wrong state" );
		::strcpy( itsCwd, aCwd ? aCwd : "" );
		itsState = ftp_state_cwd;
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
const char *	CFtp::Cwd	( void ) {
	return	itsCwd;
}

// --------------------------------------------------------------------
const CFtpFile_l &	CFtp::Dir	( void ) {
	return	itsDir;
}

// --------------------------------------------------------------------
CFtpFile		CFtp::Dir	( const char * aFile ) {
	CFtpFile	myfile;
	for	( CFtpFile_lci loop = itsDir.begin(); loop != itsDir.end(); loop++ ) {
    	if	( ! ::strcmp( (*loop).Name(), aFile ) ) {
        	myfile = *loop;
            break;
        }
    }
	return	myfile;
}

// --------------------------------------------------------------------
void				CFtp::Upload( const char * aFile, ftp_type_t aType ) {
	itsMutex.Lock();
	try	{
		if	( itsState != ftp_state_online )	throw CError( "Wrong state" );
        ::strcpy( itsLocalFile, aFile );
        itsLocalType = aType;
		itsState = ftp_state_upload;
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
void				CFtp::Download( const char * aFile, ftp_type_t aType ) {
	itsMutex.Lock();
	try	{
		if	( itsState != ftp_state_online )	throw CError( "Wrong state" );
        ::strcpy( itsLocalFile, aFile );
        itsLocalType = aType;
        itsLocalSize = 0;
		itsState = ftp_state_download;
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
void				CFtp::Remove( const char * aFile ) {
	itsMutex.Lock();
	try	{
		if	( itsState != ftp_state_online )	throw CError( "Wrong state" );
        ::strcpy( itsLocalFile, aFile );
		itsState = ftp_state_remove;
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
const char *	CFtp::Host	( void ) const {
	return	itsHost ? itsHost : "";
}

// --------------------------------------------------------------------
unsigned short	CFtp::Port	( void ) const {
	return	itsPort;
}

// --------------------------------------------------------------------
const char *	CFtp::User	( void ) const {
	return	itsUser ? itsUser : "";
}

// --------------------------------------------------------------------
const char *	CFtp::Pass	( void ) const {
	return	itsPass ? itsPass : "";
}

// --------------------------------------------------------------------
const char *	CFtp::Acct	( void ) const {
	return	itsAcct ? itsAcct : "";
}

// --------------------------------------------------------------------
ftp_state_t		CFtp::State	( void ) const {
	return	itsState;
}

// --------------------------------------------------------------------
const char *	CFtp::Message	( void ) const {
	return	itsMessage;
}

// --------------------------------------------------------------------
CError	CFtp::Error		( void ) const {
	return	itsError;
}

// --------------------------------------------------------------------
void	CFtp::Pasv		( bool aV ) {
	itsPasv = aV;
}

// --------------------------------------------------------------------
bool	CFtp::Pasv		( void ) const {
	return itsPasv;
}

// --------------------------------------------------------------------
void	CFtp::DoLogin	( void ) {
	CFtpReply	myreply;
	int			mycount;

	::sprintf( itsMessage, "Connecting to %s:%d", itsHost, itsPort );
    itsEngine.Connect( itsHost, itsPort );

	mycount = 100;
    do {
    	myreply = itsEngine.Reply();
        if	( myreply == 0 ) {
			if	( mycount < 0 ) {
            	throw CError( itsHost, "No response" );
            }
			CFtp::BreakTest();
            mycount--;
        }
    }	while	( myreply == 0 );
    switch	( myreply.Code() ) {
        case	230:
		case	220:	break;
		default:		throw CError( itsHost, myreply.Data() );
    }

    ::sprintf( itsMessage, "Logging in %s@%s", itsUser ? itsUser : "", itsHost );
    while	( myreply != 230 ) {
    	switch	( myreply.Code() ) {
			case	220:	if	( ! itsUser ) throw CError( itsHost, "Needs user" );
            				myreply = itsEngine.USER( itsUser );	break;

			case	331:	if	( ! itsPass ) throw CError( itsHost, "Needs password" );
            				myreply = itsEngine.PASS( itsPass );	break;

            case	332:	if	( ! itsAcct ) throw CError( itsHost, "Needs account" );
            				myreply = itsEngine.ACCT( itsAcct );	break;

			default:		throw CError( itsHost, myreply.Data() );
        }
		CFtp::BreakTest();
    }

	try {
	    CFtp::DoCwd();
    }
    catch	( ... ) {
	    ::memset( itsCwd, 0, sizeof( itsCwd ) );
    }

    ::sprintf( itsMessage, "%s online", itsHost );
	itsState = ftp_state_online;
}

// --------------------------------------------------------------------
void	CFtp::DoLogout	( void ) {
    try	{
		itsEngine.Disconnect();
    }
    catch ( ... ) {
    }

    if	( itsHost )	delete [] itsHost;	itsHost = NULL;
   	if	( itsUser )	delete [] itsUser;	itsUser = NULL;
    if	( itsPass )	delete [] itsPass;	itsPass = NULL;
    if	( itsAcct )	delete [] itsAcct;	itsAcct = NULL;
    ::memset( itsCwd, 0, sizeof( itsCwd ) );
    ::memset( itsLocalFile, 0, sizeof( itsLocalFile ) );
    ::strcpy( itsMessage, "Offline" );
	itsState = ftp_state_offline;
}

// --------------------------------------------------------------------
void	CFtp::DoCwd ( void ) {
	CFtpReply	myreply;

    if	( itsCwd[0] ) {	// Set prior getting it back
	    ::sprintf( itsMessage, "Setting directory %s", itsCwd );
    	myreply = itsEngine.CWD( itsCwd );

	    // Erase existing current directory
	    ::memset( itsCwd, 0, sizeof( itsCwd ) );

        switch	( myreply.Code() ) {
			case	250:	break;
			default:		throw CError( itsHost, myreply.Data() );
        }
    }

    // Get currrent directory
    ::strcpy( itsMessage, "Getting directory" );
    myreply = itsEngine.PWD( itsCwd, sizeof( itsCwd ) );
    switch	( myreply.Code() ) {
		case	257:	break;
		default:		throw CError( itsHost, myreply.Data() );
	}

	CFtp::DoDir();

    ::sprintf( itsMessage, "%s online", itsHost );
	itsState = ftp_state_online;
}

// --------------------------------------------------------------------
void	CFtp::DoDir		( void ) {
	ftp_sockaddr_t	myaddr;
	CFtpReply		myreply;
    CTcpSocket *	mysock	= NULL;
	CFtpFile		myfile;
    int				mycount;
	char			myline	[1024];
	int				myoffs;
    int				rxch;

	itsDir.clear();

	// We use ascii mode in directories
    myreply = itsEngine.TYPE( ftp_type_ASCII );
    switch	( myreply.Code() ) {
		case	200:	break;
		default:		throw CError( itsHost, myreply.Data() );
	}

    ::strcpy( itsMessage, myreply.Data() );

    // Let's try out the PASV approach (if allowed)
    if	( itsPasv ) {
		myreply = itsEngine.PASV( myaddr );
		switch	( myreply.Code() ) {
 			case	227:						break;
            case	500:	itsPasv = false;	break;
			default:		throw CError( itsHost, myreply.Data() );
		}
    }

    // Fallback to PORT approach
    if	( ! itsPasv ) {
		myreply = itsEngine.PORT();
		switch	( myreply.Code() ) {
			case	200:	break;
			default:		throw CError( itsHost, myreply.Data() );
		}
    }

    ::strcpy( itsMessage, myreply.Data() );

    // In case of passive mode - we should have the sock now and here
    if	( itsPasv ) {
    	mycount = 100;
		mysock = new CTcpSocket( socket_type_stream );
        while	( ! mysock->Connect( myaddr.addr, myaddr.port ) ) {
           	if	( mycount < 0 ) {
               	throw CError( myaddr.addr, "No response" );
            }
            CFtp::BreakTest();
            mycount--;
		}
    }

    // Start up the directory thing
	myreply = itsEngine.LIST();
    switch	( myreply.Code() ) {
    	case	125:	break;
        case	150:	break;
		default:		throw CError( itsHost, myreply.Data() );
    }

    ::strcpy( itsMessage, myreply.Data() );

    try {
	    // Open up the stream
        mycount = 100;
    	if	( ! mysock  ) {
        	do {
                mysock = itsEngine.Listener()->Accept();
				if	( mysock == NULL ) {
	            	if	( mycount < 0 ) {
    	            	throw CError( myaddr.addr, "No response" );
        	        }
		            CFtp::BreakTest();
                    mycount--;
                }
            }	while	( mysock == NULL );
	    }

		// Since the sock is now open we can do the listing
        myreply.Code( 0 );
        myoffs	= 0;
		mycount	= 100;
        while	( mycount > 0 ) {
			rxch = mysock->Read();
            if		( rxch == SOCKET_READ_ERROR ) {
            	throw CError( "Comms error" );
            }
            else if	( rxch == SOCKET_READ_DISCONNECT ) {
            	break;
            }
            else if	( rxch == SOCKET_READ_IDLE ) {
				if	( myreply == 0 ) {
	            	myreply = itsEngine.Control()->Reply();
                }
	            CFtp::BreakTest();
                mycount--;
            }
            else if	( ( rxch == '\r' ) || ( rxch == '\n' ) ) {
            	if	( myoffs == 0 )	continue;
                myline[myoffs] = 0;
				myfile = myline;
                if	( myfile.Cmnd() != 0 ) {
                	itsDir.push_back( myfile );
                }
                myoffs = 0;
            	mycount = 10;
            }
            else {
            	myline[myoffs++] = (char)rxch;
                if	( myoffs > sizeof( myline ) - 2 ) {
   	            	throw CError( itsHost, "Invalid response" );
                }
            	mycount = 10;
            }
        }

        if	( mysock ) 	delete mysock;
		mysock = NULL;
    }

    catch	( ... ) {
    	if	( mysock )	delete mysock;
        throw;
    }

	if	( myreply == 0 ) {
        myreply = itsEngine.Reply();
    }
    switch	( myreply.Code() ) {
		case	226:
        case	250:	break;
		default:		throw CError( itsHost, myreply.Data() );
	}

    ::sprintf( itsMessage, "%s online", itsHost );
	itsState = ftp_state_online;
}

// --------------------------------------------------------------------
void	CFtp::DoUpload		( void ) {
	ftp_sockaddr_t	myaddr;
	CFtpReply		myreply;
    CTcpSocket *	mysock	= NULL;
    FILE *			mystrm	= NULL;
    int				mycount;
	char			myline	[1024];
	int				myoffs;
    struct stat		mystat;
    size_t			sent;

	// Set up transfer type
    myreply = itsEngine.TYPE( itsLocalType );
    switch	( myreply.Code() ) {
		case	200:	break;
		default:		throw CError( itsHost, myreply.Data() );
	}

    ::strcpy( itsMessage, myreply.Data() );

    // Let's try out the PASV approach (if allowed)
    if	( itsPasv ) {
		myreply = itsEngine.PASV( myaddr );
		switch	( myreply.Code() ) {
 			case	227:						break;
            case	500:	itsPasv = false;	break;
			default:		throw CError( itsHost, myreply.Data() );
		}
    }

    // Fallback to PORT approach
    if	( ! itsPasv ) {
		myreply = itsEngine.PORT();
		switch	( myreply.Code() ) {
			case	200:	break;
			default:		throw CError( itsHost, myreply.Data() );
		}
    }

    ::strcpy( itsMessage, myreply.Data() );

    // In case of passive mode - we should have the sock now and here
    if	( itsPasv ) {
    	mycount = 100;
		mysock = new CTcpSocket( socket_type_stream );
        while	( ! mysock->Connect( myaddr.addr, myaddr.port ) ) {
           	if	( mycount < 0 ) {
               	throw CError( myaddr.addr, "No response" );
            }
            CFtp::BreakTest();
            mycount--;
		}
    }

    // Start up the upload thing
    ::fnsplit( itsLocalFile, NULL, NULL, myline, NULL );
    ::fnsplit( itsLocalFile, NULL, NULL, NULL, myline + ::strlen( myline ) );
	myreply = itsEngine.STOR( myline );
    switch	( myreply.Code() ) {
    	case	125:	break;
        case	150:	break;
		default:		throw CError( itsHost, myreply.Data() );
    }

    ::strcpy( itsMessage, myreply.Data() );

    try {
		// Open up the file stream
        if	( ::stat( itsLocalFile, &mystat ) ) {
        	throw CError( ::strerror( errno ) );
        }
		if	( itsLocalType == ftp_type_ASCII ) {
	        mystrm = ::fopen( itsLocalFile, "rt" );
        }
        else {
	        mystrm = ::fopen( itsLocalFile, "rb" );
        }
		if	( ! mystrm ) {
        	throw CError( ::strerror( errno ) );
        }

	    // Open up the socket stream
        mycount = 100;
    	if	( ! mysock ) {
        	do {
                mysock = itsEngine.Listener()->Accept();
				if	( mysock == NULL ) {
	            	if	( mycount < 0 ) {
    	            	throw CError( itsHost, "No response" );
        	        }
	                CFtp::BreakTest();
                    mycount--;
                }
            }	while	( mysock == NULL );
	    }

		// Do the actual transfer
		sent = 0;
		while	( ! feof( mystrm ) ) {
        	// Read a bunch of data
           	::memset( myline, 0, sizeof( myline ) );
			if	( itsLocalType == ftp_type_ASCII ) {
				if	( ::fgets( myline, sizeof( myline ) - 1, mystrm ) == NULL ) {
                	if	( ferror( mystrm ) ) {
                    	throw CError( ::strerror( errno ) );
                    }
                    continue;
                }
                myoffs = ::strlen( myline );
    		}
            else {
            	myoffs = (int)::fread( myline, 1, sizeof( myline ), mystrm );
                if	( myoffs == 0 ) {
                	if	( ferror( mystrm ) ) {
                    	throw CError( ::strerror( errno ) );
                    }
                    continue;
                }
            }

            // Output the data into the socket
			mysock->Write( myline, myoffs );
            CFtp::BreakTest( false );
            sent = sent + myoffs;
            ::sprintf( 	itsMessage, "Sending - %d%% done",
                        ( 100 * sent ) / ( mystat.st_size + 1 ) );
        }
		mysock->Flush();

        if	( mysock ) 	delete mysock;
		mysock = NULL;
        if	( mystrm )  ::fclose( mystrm );
		mystrm = NULL;
    }

    catch	( ... ) {
    	if	( mysock )	delete mysock;
        if	( mystrm )  ::fclose( mystrm );
        throw;
    }

    myreply = itsEngine.Reply();
    switch	( myreply.Code() ) {
		case	226:
        case	250:	break;
		default:		throw CError( itsHost, myreply.Data() );
	}

    ::sprintf( itsMessage, "%s online", itsHost );
	itsState = ftp_state_online;
}

// --------------------------------------------------------------------
void	CFtp::DoDownload	( void ) {
	ftp_sockaddr_t	myaddr;
	CFtpReply		myreply;
    CTcpSocket *	mysock	= NULL;
    FILE *			mystrm	= NULL;
	char			myline	[1024];
    int				mycount;
    int				rxch;
    size_t			done;
    CFtpFile		myfile;

	// Set up transfer type
    myreply = itsEngine.TYPE( itsLocalType );
    switch	( myreply.Code() ) {
		case	200:	break;
		default:		throw CError( itsHost, myreply.Data() );
	}

    ::strcpy( itsMessage, myreply.Data() );

    // Let's try out the PASV approach (if allowed)
    if	( itsPasv ) {
		myreply = itsEngine.PASV( myaddr );
		switch	( myreply.Code() ) {
 			case	227:						break;
            case	500:	itsPasv = false;	break;
			default:		throw CError( itsHost, myreply.Data() );
		}
    }

    // Fallback to PORT approach
    if	( ! itsPasv ) {
		myreply = itsEngine.PORT();
		switch	( myreply.Code() ) {
			case	200:	break;
			default:		throw CError( itsHost, myreply.Data() );
		}
    }

    ::strcpy( itsMessage, myreply.Data() );

    // In case of passive mode - we should have the sock now and here
    if	( itsPasv ) {
    	mycount = 100;
		mysock = new CTcpSocket( socket_type_stream );
        while	( ! mysock->Connect( myaddr.addr, myaddr.port ) ) {
           	if	( mycount < 0 ) {
               	throw CError( myaddr.addr, "No response" );
            }
            CFtp::BreakTest();
            mycount--;
		}
    }

    // Start up the upload thing
    ::fnsplit( itsLocalFile, NULL, NULL, myline, NULL );
    ::fnsplit( itsLocalFile, NULL, NULL, NULL, myline + ::strlen( myline ) );
    myfile  = CFtp::Dir( myline );
    itsLocalSize = myfile.Size();
	myreply = itsEngine.RETR( myline );
    switch	( myreply.Code() ) {
    	case	125:	break;
        case	150:	break;
		default:		throw CError( itsHost, myreply.Data() );
    }

    ::strcpy( itsMessage, myreply.Data() );

    try {
		// Open up the file stream
		::unlink( itsLocalFile );
		if	( itsLocalType == ftp_type_ASCII ) {
	        mystrm = ::fopen( itsLocalFile, "wt" );
        }
        else {
	        mystrm = ::fopen( itsLocalFile, "wb" );
        }
		if	( ! mystrm ) {
        	throw CError( ::strerror( errno ) );
        }

	    // Open up the socket stream
        mycount = 100;
    	if	( ! mysock ) {
        	do {
                mysock = itsEngine.Listener()->Accept();
				if	( mysock == NULL ) {
	            	if	( mycount < 0 ) {
    	            	throw CError( itsHost, "No response" );
        	        }
	                CFtp::BreakTest();
                    mycount--;
                }
            }	while	( mysock == NULL );
	    }

		// Since the sock is now open we can do the transfer
        myreply.Code( 0 );
		mycount	= 100;
        done	= 0;
        while	( mycount > 0 ) {
			rxch = mysock->Read();
            if		( rxch == SOCKET_READ_ERROR ) {
            	throw CError( "Comms error" );
            }
            else if	( rxch == SOCKET_READ_DISCONNECT ) {
            	break;
            }
            else if	( rxch == SOCKET_READ_IDLE ) {
				if	( myreply == 0 ) {
	            	myreply = itsEngine.Control()->Reply();
                }
				if		( itsLocalSize > done ) {
		            CFtp::BreakTest();
                }
                else if	( ( itsLocalSize > 0 ) && ( itsLocalSize <= done ) ) {
                	break;
                }
                else {
	                mycount--;
                }
            }
            else {
				::fputc( rxch, mystrm );
                done++;
				if	( ( done % 1024 ) == 0 ) {
	                if	( itsLocalSize ) {
			            ::sprintf( 	itsMessage, "Receiving - %d%% done",
            		            ( 100 * done ) / itsLocalSize );
    	            }
        	        else {
                    	::sprintf( itsMessage, "Received %d k-bytes", done / 1024 );
            	    }
                }
            	mycount = 10;
            }
        }

        if	( mysock ) 	delete mysock;
		mysock = NULL;
        if	( mystrm )  ::fclose( mystrm );
		mystrm = NULL;

        // Last - change file date if available
        if	( myfile.Size() ) {
        	struct utimbuf	myutim;
            myutim.actime 	= ::time( NULL );
            myutim.modtime	= myfile.Time();
			::utime( itsLocalFile, &myutim );
        }
    }

    catch	( ... ) {
    	if	( mysock )	delete mysock;
        if	( mystrm )  ::fclose( mystrm );
        throw;
    }

    if	( myreply == 0 ) {
    	myreply = itsEngine.Reply();
    }
    switch	( myreply.Code() ) {
		case	226:
        case	250:	break;
		default:		throw CError( itsHost, myreply.Data() );
	}

    ::sprintf( itsMessage, "%s online", itsHost );
	itsState = ftp_state_online;
}

// --------------------------------------------------------------------
void	CFtp::DoRemove ( void ) {
	CFtpReply	myreply;

    ::strcpy( itsMessage, "Removing file" );
    myreply = itsEngine.DELE( itsLocalFile );
    switch	( myreply.Code() ) {
		case	250:	break;
		default:		throw CError( itsHost, myreply.Data() );
	}

    ::sprintf( itsMessage, "%s online", itsHost );
	itsState = ftp_state_online;
}

// --------------------------------------------------------------------
void	CFtp::BreakTest	( bool aPause ) {
	itsMutex.Unlock();
    if	( aPause )	SLEEP( 100 );
    itsMutex.Lock();
    if	(itsState == ftp_state_logout ) {
    	throw	CError( itsHost, "Bailing Out" );
    }
}

// --------------------------------------------------------------------
void	CFtp::Cleanup	( void ) {
	itsState	= ftp_state_offline;
   	itsThreadId	= 0;
	itsRunning	= false;
	itsHost		= NULL;
    itsPort		= 0;
    itsUser		= NULL;
    itsPass		= NULL;
    itsAcct		= NULL;
    itsPasv		= false;
    ::memset( itsCwd, 0, sizeof( itsCwd ) );
    ::memset( itsLocalFile, 0, sizeof( itsLocalFile ) );
    itsLocalType = ftp_type_IMAGE;
    ::strcpy( itsMessage, "Offline" );
    itsLocalSize = 0;
}

// --------------------------------------------------------------------
void	CFtp::Free		( void ) {
	itsEngine.Disconnect();
    itsDir.clear();
    if	( itsHost )	delete [] itsHost;
    if	( itsUser )	delete [] itsUser;
    if	( itsPass )	delete [] itsPass;
    if	( itsAcct )	delete [] itsAcct;
	CFtp::Cleanup();
}

// --------------------------------------------------------------------
// EOF: CFtp.hxx
// --------------------------------------------------------------------
