// --------------------------------------------------------------------
// CFtpClient.cxx
// Whatis:  FTP Client class library
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-FEB-2003     Created this source
// --------------------------------------------------------------------
#include	"CFtpClient.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// local:	RFC943 - http://www.faqs.org/rfcs/rfc943.html
// --------------------------------------------------------------------
static const ftp_system_t	_official_systems[] = {
	{ "ASP" },
	{ "AUGUST" },
	{ "BKY" },
	{ "CCP" },
	{ "DOS/360" },
	{ "ELF" },
	{ "EPOS" },
	{ "EXEC-8" },
	{ "GCOS" },
	{ "GPOS" },
	{ "ITS" },
	{ "INTERCOM" },
	{ "INTERLISP" },
	{ "KRONOS" },
	{ "MCP" },
	{ "MOS" },
	{ "MPX-RT" },
	{ "MULTICS" },
	{ "MVT" },
	{ "NOS" },
	{ "NOS/BE" },
	{ "OS/MVS" },
	{ "OS/MVT" },
	{ "RIG" },
	{ "RSX-11M" },
	{ "RT11" },
	{ "SCOPE" },
	{ "SIGNAL" },
	{ "SINTRAN" },
	{ "TAC" },
	{ "TENEX" },
	{ "TOPS-10" },
	{ "TOPS-20" },
	{ "TSS" },
	{ "UNIX" },
	{ "VM/370" },
	{ "VM/CMS" },
	{ "VMS" },
	{ "WAITS" },
	{ "XDE" },
	{ 0 }
};

// --------------------------------------------------------------------
// The thread itself
// --------------------------------------------------------------------
static unsigned long	__stdcall	__MyFtpThread( void * aFtpClient ) {
	try	{
		((CFtpClient *)aFtpClient)->RunThread();
    }
    catch	( ... ) {
    }
	::ExitThread( 0 );
	return	0;
}

// --------------------------------------------------------------------
CFtpClient::CFtpClient	() {
	CFtpClient::Cleanup();
	if	( ::CreateThread(
    			NULL,			// Security
                32768,			// Stack
                __MyFtpThread,	// Thread
                (void *)this,	// Argument
                0,				// Flags
    			&itsThreadId ) <= 0 ) {
		throw	CError( "Unable to start FTP thread" );
	}
}

// --------------------------------------------------------------------
CFtpClient::~CFtpClient	() {
	CFtpClient::Clear();
	while	( itsIsRunning ) {
		itsDoTerminate = true;
        SLEEP( 100 );
    }
	CFtpClient::Free();
}

// --------------------------------------------------------------------
void				CFtpClient::RunThread( void ) {
	itsIsRunning = true;
	while	( itsDoTerminate == false ) {
    	try {
	    	itsMutex.Lock();
			CFtpClient::StateMachine();
    		itsMutex.Unlock();
        }
        catch ( ... ) {
	    	itsMutex.Unlock();
        }
		SLEEP( 10 );
    }
	itsIsRunning = false;
}

// --------------------------------------------------------------------
const char *		CFtpClient::Host	( void ) const {
	return	itsHost ? itsHost : "";
}

// --------------------------------------------------------------------
void				CFtpClient::Host	( const char * aHost ) {
	try	{
    	itsMutex.Lock();
		if	( ::strcmp( CFtpClient::Host(), aHost ) ) {
	        CFtpClient::Clear();
            if	( itsHost )	delete [] itsHost;
            itsHost = ::my_private_strdup( aHost );
        }
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
unsigned short		CFtpClient::Port	( void ) const {
	return itsPort;
}

// --------------------------------------------------------------------
void				CFtpClient::Port	( unsigned short aPort ) {
	try	{
    	itsMutex.Lock();
		if	( itsPort != aPort ) {
	        CFtpClient::Clear();
            itsPort = aPort;
        }
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
const char *		CFtpClient::User	( void ) const {
	return itsUser ? itsUser : "";
}

// --------------------------------------------------------------------
void				CFtpClient::User	( const char * aUser ) {
	try	{
    	itsMutex.Lock();
		if	( ::strcmp( CFtpClient::User(), aUser ) ) {
	        CFtpClient::Clear();
            if	( itsUser )	delete [] itsUser;
            itsUser = ::my_private_strdup( aUser );
        }
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
const char *		CFtpClient::Pass	( void ) const {
	return itsPass ? itsPass : "";
}

// --------------------------------------------------------------------
void				CFtpClient::Pass	( const char * aPass ) {
	try	{
    	itsMutex.Lock();
		if	( ::strcmp( CFtpClient::User(), aPass ) ) {
	        CFtpClient::Clear();
            if	( itsPass )	delete [] itsPass;
            itsPass = ::my_private_strdup( aPass );
        }
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
const char *		CFtpClient::Acnt	( void ) const {
	return itsAcnt ? itsAcnt : "";
}

// --------------------------------------------------------------------
void				CFtpClient::Acnt	( const char * aAcnt ) {
	try	{
    	itsMutex.Lock();
		if	( ::strcmp( CFtpClient::Acnt(), aAcnt ) ) {
	        CFtpClient::Clear();
            if	( itsAcnt )	delete [] itsAcnt;
            itsAcnt = ::my_private_strdup( aAcnt );
        }
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
const char *		CFtpClient::LastResponse	( void ) const {
	return itsLastResponse;
}

// --------------------------------------------------------------------
ftp_system_t		CFtpClient::System		( void ) const {
	return	_official_systems[itsSystem];
}

// --------------------------------------------------------------------
ftp_state_t			CFtpClient::State	( void ) const {
	return	itsState;
}

// --------------------------------------------------------------------
bool				CFtpClient::KeepAlive( void ) const {
	return	itsKeepAlive;
}

// --------------------------------------------------------------------
void				CFtpClient::KeepAlive( bool aV ) {
	try	{
    	itsMutex.Lock();
        itsKeepAlive = aV;
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
bool				CFtpClient::Pasv( void ) const {
	return	itsCanPasv;
}

// --------------------------------------------------------------------
void				CFtpClient::Pasv( bool aV ) {
	try	{
    	itsMutex.Lock();
        itsCanPasv = aV;
    	itsMutex.Unlock();
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
void				CFtpClient::Clear	( void ) {
	try	{
		if	( itsState != ftp_state_idle ) {
	    	itsMutex.Lock();

	        // Terminate any connections
			if	( itsCtlSock ) delete itsCtlSock;
			itsCtlSock	= NULL;

			// Set the state to idle
			itsState = ftp_state_idle;

            // No more cwd ...
            itsCwd[0] = 0;
            itsDir.clear();

            itsDoAbortion = false;

    		itsMutex.Unlock();
        }
    }
    catch ( ... ) {
    	itsMutex.Unlock();
    }
}

// --------------------------------------------------------------------
void				CFtpClient::Login	( void ) {
	try	{
    	itsMutex.Lock();
		CFtpClient::Clear();
        if	( ! itsHost )	throw CError( "Missing host" );
        if	( ! itsPort )	throw CError( "Missing port" );
        if	( ! itsUser )	throw CError( "Missing user name" );
        if	( ! itsPass )	throw CError( "Missing password" );
		::strcpy( itsLastResponse, "Connecting" );
		itsState = ftp_state_login;
    	itsMutex.Unlock();
    }

    catch	( CError e ) {
    	itsMutex.Unlock();
		throw;
	}

    catch	( ... ) {
    	itsMutex.Unlock();
		throw	CError( "Unknown error" );
    }
}

// --------------------------------------------------------------------
const char *		CFtpClient::Cwd	( void ) const {
	return	itsCwd;
}

// --------------------------------------------------------------------
void				CFtpClient::Cwd	( const char * aPath ) {
	try	{
    	itsMutex.Lock();
		if	( itsState != ftp_state_connect ) {
        	throw CError( "Wrong state" );
        }
		if	( ::strcmp( itsCwd, aPath ) ) {
            ::strcpy( itsCwd, aPath );
			itsState = ftp_state_cwd;
        }
    	itsMutex.Unlock();
    }

    catch	( CError e ) {
    	itsMutex.Unlock();
		throw;
	}

    catch	( ... ) {
    	itsMutex.Unlock();
		throw	CError( "Unknown error" );
    }
}

// --------------------------------------------------------------------
void				CFtpClient::Refresh	( void ) {
	try	{
    	itsMutex.Lock();
		if	( itsState != ftp_state_connect ) {
        	throw CError( "Wrong state" );
        }
		itsState = ftp_state_refresh;
    	itsMutex.Unlock();
    }

    catch	( CError e ) {
    	itsMutex.Unlock();
		throw;
	}

    catch	( ... ) {
    	itsMutex.Unlock();
		throw	CError( "Unknown error" );
    }
}

// --------------------------------------------------------------------
const CFtpFile_l &	CFtpClient::Dir ( void ) const {
	return	itsDir;
}

// --------------------------------------------------------------------
void	CFtpClient::Upload		( const char * aFile ) {
	try	{
    	itsMutex.Lock();
		if	( itsState != ftp_state_connect ) {
        	throw CError( "Wrong state" );
        }
		::strcpy( itsLocalFile, aFile );
		itsState = ftp_state_upload;
    	itsMutex.Unlock();
    }

    catch	( CError e ) {
    	itsMutex.Unlock();
		throw;
	}

    catch	( ... ) {
    	itsMutex.Unlock();
		throw	CError( "Unknown error" );
    }
}

// --------------------------------------------------------------------
void	CFtpClient::Update		( const char * aFile ) {
	try	{
    	itsMutex.Lock();
		if	( itsState != ftp_state_connect ) {
        	throw CError( "Wrong state" );
        }
		::strcpy( itsLocalFile, aFile );
		itsState = ftp_state_update;
    	itsMutex.Unlock();
    }

    catch	( CError e ) {
    	itsMutex.Unlock();
		throw;
	}

    catch	( ... ) {
    	itsMutex.Unlock();
		throw	CError( "Unknown error" );
    }
}

// --------------------------------------------------------------------
void	CFtpClient::Remove		( const char * aFile ) {
	try	{
    	itsMutex.Lock();
		if	( itsState != ftp_state_connect ) {
        	throw CError( "Wrong state" );
        }
		::strcpy( itsLocalFile, aFile );
		itsState = ftp_state_remove;
    	itsMutex.Unlock();
    }

    catch	( CError e ) {
    	itsMutex.Unlock();
		throw;
	}

    catch	( ... ) {
    	itsMutex.Unlock();
		throw	CError( "Unknown error" );
    }
}

// --------------------------------------------------------------------
void	CFtpClient::Abort	( void ) {
	itsDoAbortion	= true;
}

// --------------------------------------------------------------------
void				CFtpClient::StateMachine ( void ) {
	try {
		switch	( itsState ) {
    		case	ftp_state_idle:												break;
			case	ftp_state_connect:	CFtpClient::StateMachineConnect();		break;
			case	ftp_state_login:	CFtpClient::StateMachineLogin();		break;
            case	ftp_state_cwd:		CFtpClient::StateMachineCwd();			break;
            case	ftp_state_refresh:	CFtpClient::StateMachineRefresh();		break;
            case	ftp_state_upload:	CFtpClient::StateMachineUpload();		break;
            case	ftp_state_update:	CFtpClient::StateMachineUpdate();		break;
            case	ftp_state_remove:	CFtpClient::StateMachineRemove();		break;

	        case	ftp_state_error:	CFtpClient::StateMachineError();		break;
    	}
    }
    catch	( CError e ) {
		if	( ::strlen( e.Error() ) > sizeof( itsErrorMessage ) - 1 ) {
        	::memcpy( itsErrorMessage, e.Error(), sizeof( itsErrorMessage ) - 1 );
            itsErrorMessage[sizeof( itsErrorMessage ) - 1] = 0;
        }
        else {
        	::strcpy( itsErrorMessage, e.Error() );
        }
        itsErrorStart = ::time( NULL );
        CFtpClient::Clear();
        itsState = ftp_state_error;
		itsDoAbortion = false;
    }
    catch	( ... ) {
       	::strcpy( itsErrorMessage, "Error" );
        itsErrorStart = ::time( NULL );
        CFtpClient::Clear();
        itsState = ftp_state_error;
		itsDoAbortion = false;
    }
}

// --------------------------------------------------------------------
void				CFtpClient::StateMachineConnect( void ) {
   	if	( itsKeepAlive ) {
		if	( itsLastKeepAlive < ::time( NULL ) ) {
			switch	( CFtpClient::NOOP() ) {
				case	200:	// Ok
				break;

           		case	500:	// Syntax error, command unrecognized.
				itsKeepAlive = false;
                break;

                default:
                throw CError( itsLastResponse );
            }
	 		itsLastKeepAlive = ::time( NULL ) + 15;
        }
    }
}

// --------------------------------------------------------------------
void				CFtpClient::StateMachineLogin( void ) {
	if	( itsCtlSock == NULL ) {
    	itsCtlSock = new CTcpSocket( socket_type_stream );
    }
	if  ( itsCtlSock->Connect( itsHost, itsPort ) ) {
    	if	( CFtpClient::GetReply() == 220 ) {
        	int	code	= CFtpClient::USER( itsUser );
			while	( code != 230 ) {
            	switch	( code ) {
                	case	220:	code = CFtpClient::USER( itsUser );	break;
                	case	331:	code = CFtpClient::PASS( itsPass );	break;
                    case	332:	code = CFtpClient::ACCT( itsAcnt );	break;
                    default:		throw CError( itsLastResponse );
                }
            }
			switch	( CFtpClient::PWD() ) {
            	case	257:	// Response has the current directory
                CFtpClient::GetPwd( itsLastResponse );
                itsCanPwd = true;
                break;

				case	500:	// Not everyone knows PWD ...
				CFtpClient::GetPwd( "\"/\"" );
                itsCanPwd = false;
                break;

             	default:		// Bad things happened ...
                throw CError( itsLastResponse );
            }

			switch	( CFtpClient::SYST() ) {
            	case	215:	// Recorded the system name
                CFtpClient::GetSystem( itsLastResponse );
                break;

				case	500:	// Not everyone knows SYST ...
				CFtpClient::GetSystem	( "UNIX" );
                break;

             	default:		// Bad things happened ...
                throw CError( itsLastResponse );
            }

            // Lastly - change state to cwd ... we take it from there
			itsState = ftp_state_cwd;
        }
    }
}

// --------------------------------------------------------------------
void				CFtpClient::StateMachineCwd( void ) {
	if	( CFtpClient::CWD( itsCwd ) != 250 ) {
		throw	CError( itsLastResponse );
    }

   	if	( itsCanPwd ) {
		if	( CFtpClient::PWD() != 257 ) {
			throw	CError( itsLastResponse );
        }
   	    CFtpClient::GetPwd( itsLastResponse );
    }

	// Turn on ascii mode - just to be sure !
	if	( CFtpClient::TYPE( ftp_type_ASCII ) != 200 ) {
		throw	CError( itsLastResponse );
    }

	// Lastly - get directory entries - first set up port or pasv
	switch	( CFtpClient::LIST() ) {
       	case	226:
        case	250:	// OK !
        break;

        default:	throw	CError( itsLastResponse );
	}
	itsState = ftp_state_connect;
}

// --------------------------------------------------------------------
void				CFtpClient::StateMachineRefresh( void ) {
   	if	( itsCanPwd ) {
		if	( CFtpClient::PWD() != 257 ) {
			throw	CError( itsLastResponse );
        }
   	    CFtpClient::GetPwd( itsLastResponse );
    }

	// Turn on ascii mode - just to be sure !
	if	( CFtpClient::TYPE( ftp_type_ASCII ) != 200 ) {
		throw	CError( itsLastResponse );
    }

	// Lastly - get directory entries - first set up port or pasv
	switch	( CFtpClient::LIST() ) {
       	case	226:
        case	250:	// OK !
        break;

        default:	throw	CError( itsLastResponse );
	}
	itsState = ftp_state_connect;
}

// --------------------------------------------------------------------
void				CFtpClient::StateMachineUpload( void ) {
	struct stat	mystat;
	FILE *		strm = NULL;
	char		remofile	[1024];

    try	{
		// Get file stat
        if	( ::stat( itsLocalFile, &mystat ) ) {
        	throw	CError( ::strerror( errno ) );
        }

		// Open the file in binary mode
    	strm = ::fopen( itsLocalFile, "rb" );
        if	( ! strm ) {
        	throw	CError( ::strerror( errno ) );
        }

		// Turn on binary mode - just to be sure !
		if	( CFtpClient::TYPE( ftp_type_IMAGE ) != 200 ) {
			throw	CError( itsLastResponse );
	    }

        // Set up remote file name
		::strcpy( remofile, itsLocalFile );
        if	( ::strrchr( remofile, '\\' ) ) {
			::strcpy( remofile, ::strrchr( remofile, '\\' ) + 1 );
        }

		// Lastly - upload the file there
		switch	( CFtpClient::STOR( remofile, strm ) ) {
       		case	226:
	        case	250:	// OK !
    	    break;

        	default:	throw	CError( itsLastResponse );
		}
		::fclose( strm );
    }

    catch	( ... ) {
    	if	( strm )	::fclose( strm );
        throw;
    }

	itsState = ftp_state_connect;
}

// --------------------------------------------------------------------
void				CFtpClient::StateMachineUpdate( void ) {
	struct stat	mystat;
	FILE *		strm = NULL;
	char		remofile	[1024];

    try	{
		// Get file stat
        if	( ::stat( itsLocalFile, &mystat ) ) {
        	throw	CError( ::strerror( errno ) );
        }

		// Open the file in binary mode
    	strm = ::fopen( itsLocalFile, "rb" );
        if	( ! strm ) {
        	throw	CError( ::strerror( errno ) );
        }

		// Turn on binary mode - just to be sure !
		if	( CFtpClient::TYPE( ftp_type_IMAGE ) != 200 ) {
			throw	CError( itsLastResponse );
	    }

        // Set up remote file name
		::strcpy( remofile, itsLocalFile );
        if	( ::strrchr( remofile, '\\' ) ) {
			::strcpy( remofile, ::strrchr( remofile, '\\' ) + 1 );
        }

		// Prior uploading remove the file from server
        if	( CFtpClient::DELE( remofile ) != 250 ) {
        	throw	CError( itsLastResponse );
        }

		// Lastly - get directory entries - first set up port or pasv
		switch	( CFtpClient::STOR( remofile, strm ) ) {
       		case	226:
	        case	250:	// OK !
    	    break;

        	default:	throw	CError( itsLastResponse );
		}
		::fclose( strm );
    }

    catch	( ... ) {
    	if	( strm )	::fclose( strm );
        throw;
    }

	itsState = ftp_state_connect;
}

// --------------------------------------------------------------------
void				CFtpClient::StateMachineRemove( void ) {
	// Prior uploading remove the file from server
    if	( CFtpClient::DELE( itsLocalFile ) != 250 ) {
		throw	CError( itsLastResponse );
    }

	itsState = ftp_state_connect;
}

// --------------------------------------------------------------------
void				CFtpClient::StateMachineError( void ) {
	if	( itsErrorStart < ( ::time( NULL ) - 10 ) ) {
        itsState = ftp_state_idle;
    }
}

// --------------------------------------------------------------------
int		CFtpClient::USER	( const char * aUser ) {
	itsCtlSock->Write( "USER " );
    itsCtlSock->Write( aUser );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::PASS	( const char * aPass ) {
	itsCtlSock->Write( "PASS " );
    itsCtlSock->Write( aPass );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::ACCT	( const char * aAcct ) {
	itsCtlSock->Write( "ACCT " );
    itsCtlSock->Write( aAcct );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::CWD		( const char * aPath ) {
	itsCtlSock->Write( "CWD " );
    itsCtlSock->Write( aPath );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::CDUP	( void ) {
	itsCtlSock->Write( "CDUP\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::SMNT	( const char * aPath ) {
	itsCtlSock->Write( "SNMT " );
    itsCtlSock->Write( aPath );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::REIN	( void ) {
	itsCtlSock->Write( "REIN\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::QUIT	( void ) {
	itsCtlSock->Write( "QUIT\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
CTcpSocket *	CFtpClient::PORT	( void ) {
	CTcpSocket *	lisn = NULL;

    try {
		struct sockaddr_in	sin;
    	int					len = sizeof( sin );

        lisn = new CTcpSocket( socket_type_stream );
        lisn->Bind( 0, itsCtlSock );
        lisn->Listen();
	    if	( ::getsockname( lisn->Socket(), (struct sockaddr *)(&sin), &len ) < 0 ) {
    		throw	CError( "Invalid socket" );
	    }

		itsCtlSock->Write( "PORT " );
		itsCtlSock->Write( (unsigned short)sin.sin_addr.S_un.S_un_b.s_b1 );
		itsCtlSock->Write( "," );
		itsCtlSock->Write( (unsigned short)sin.sin_addr.S_un.S_un_b.s_b2 );
		itsCtlSock->Write( "," );
		itsCtlSock->Write( (unsigned short)sin.sin_addr.S_un.S_un_b.s_b3 );
		itsCtlSock->Write( "," );
		itsCtlSock->Write( (unsigned short)sin.sin_addr.S_un.S_un_b.s_b4 );
		itsCtlSock->Write( "," );
		itsCtlSock->Write( (unsigned short)(sin.sin_port & 0xff ) );
		itsCtlSock->Write( "," );
		itsCtlSock->Write( (unsigned short)(sin.sin_port >> 8 ) );
        itsCtlSock->Write( "\r\n" );
	    itsCtlSock->Flush();
		if	( CFtpClient::GetReply() != 200 ) {
        	throw	CError( itsLastResponse );
        }
    }

    catch ( ... ) {
    	if	( lisn ) delete lisn;
    	throw;
    }

    return lisn;
}

// --------------------------------------------------------------------
bool	CFtpClient::PASV	( ftp_sockaddr_t & aAddr ) {
	bool	result = false;
	if	( itsCanPasv ) {
		itsCtlSock->Write( "PASV\r\n" );
    	itsCtlSock->Flush();

	    switch	( CFtpClient::GetReply() ) {
			case	227:	// Aha !
            result = CFtpClient::GetSockAddr( aAddr, itsLastResponse );
        	break;

	        default:
    	    throw	CError( itsLastResponse );
	    }
    }

	return	result;
}

// --------------------------------------------------------------------
int		CFtpClient::TYPE	( ftp_type_t aType ) {
	itsCtlSock->Write( "TYPE " );
	itsCtlSock->Write( (char)aType );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::TYPE	( size_t aSize ) {
	itsCtlSock->Write( "TYPE L " );
	itsCtlSock->Write( (unsigned long)aSize );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::STRU	( ftp_struc_t aStru ) {
	itsCtlSock->Write( "STRU " );
	itsCtlSock->Write( (char)aStru );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::MODE	( ftp_mode_t aMode ) {
	itsCtlSock->Write( "MODE " );
	itsCtlSock->Write( (char)aMode );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::RETR	( const char * aFile ) {
	itsCtlSock->Write( "RETR " );
	itsCtlSock->Write( aFile );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::STOR	( const char * aFile, FILE * aStrm ) {
	time_t			timeout;
	ftp_sockaddr_t	addr;
	CTcpSocket *	lisn	= NULL;
	CTcpSocket *	sock	= NULL;
    int				txch;
    long			count;
    struct stat		mystat;

    try {

	    // Set up passive / port mode
	    if	( CFtpClient::PASV( addr ) ) {
			sock = new CTcpSocket( socket_type_stream );
	    }
        else {
    		lisn = CFtpClient::PORT();
        }

		itsCtlSock->Write( "STOR " );
		itsCtlSock->Write( aFile );
		itsCtlSock->Write( "\r\n" );
    	itsCtlSock->Flush();

        // Open the data socket
		timeout = ::time( NULL ) + 10;
        while	( timeout > ::time( NULL ) ) {
			if	( itsDoTerminate )	break;
            if	( itsDoAbortion )	throw CError( "Aborted" );
        	if	( lisn ) {
            	sock = lisn->Accept();
                if	( sock )	break;
            }
			else {
            	if	( sock->Connect( addr.addr, addr.port ) ) {
   	            	break;
       	        }
            }
        }
		if	( timeout <= ::time( NULL ) ) {
        	throw CError( "No response from host" );
        }

        // Parse the response
        switch	( CFtpClient::GetReply() ) {
        	case	125:
            case	150:	break;
            default:		throw CError( itsLastResponse );
        }

        // Write the shit into the pipe !
        count = 0;
        ::stat( itsLocalFile, &mystat );
        while	( ! feof( aStrm ) ) {
			if	( itsDoTerminate )	break;
            if	( itsDoAbortion )	throw CError( "Aborted" );
        	txch = ::fgetc( aStrm );
            if	( txch >= 0 ) {
            	sock->Write( (char)txch );
                count++;
                if	( (count % 1024) == 0 ) {
                	::sprintf(	itsLastResponse, "Sending %s - %d%%",
                    aFile,
                    (100 * (count + 1)) / ( mystat.st_size) );
                }
            }
		}
		sock->Flush();
		delete sock;
        if	( lisn )	delete lisn;
    }

    catch	( ... ) {
    	if	( sock )	delete sock;
        if	( lisn )	delete lisn;
        throw;
    }

	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::STOU	( void ) {
	itsCtlSock->Write( "STOU\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::APPE	( const char * aFile ) {
	itsCtlSock->Write( "APPE " );
	itsCtlSock->Write( aFile );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::ALLO	( size_t aSize ) {
	itsCtlSock->Write( "APPE " );
	itsCtlSock->Write( aSize );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::ALLO	( size_t aCount, size_t aRecl ) {
	itsCtlSock->Write( "APPE " );
	itsCtlSock->Write( aCount );
	itsCtlSock->Write( " R " );
	itsCtlSock->Write( aRecl );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::REST	( const char * aMarker ) {
	itsCtlSock->Write( "REST " );
	itsCtlSock->Write( aMarker );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::RNFR	( const char * aFile ) {
	itsCtlSock->Write( "RNFR " );
	itsCtlSock->Write( aFile );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::RNTO	( const char * aFile ) {
	itsCtlSock->Write( "RNTO " );
	itsCtlSock->Write( aFile );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::ABOR	( void ) {
	itsCtlSock->Write( "ABOR\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::DELE	( const char * aFile ) {
	itsCtlSock->Write( "DELE " );
	itsCtlSock->Write( aFile );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::RMD		( const char * aDir ) {
	itsCtlSock->Write( "RMD " );
	itsCtlSock->Write( aDir );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::MKD		( const char * aDir ) {
	itsCtlSock->Write( "MKD " );
	itsCtlSock->Write( aDir );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::PWD		( void ) {
	itsCtlSock->Write( "PWD\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::LIST	( const char * aPath ) {
	int				code	= 0;
	time_t			timeout;
	ftp_sockaddr_t	addr;
	CTcpSocket *	lisn	= NULL;
	CTcpSocket *	sock	= NULL;

    try {

	    // Set up passive / port mode
	    if	( CFtpClient::PASV( addr ) ) {
			sock = new CTcpSocket( socket_type_stream );
	    }
        else {
    		lisn = CFtpClient::PORT();
        }

		if	( aPath ) {
			itsCtlSock->Write( "LIST " );
			itsCtlSock->Write( aPath );
	    }
    	else {
			itsCtlSock->Write( "LIST" );
	    }
		itsCtlSock->Write( "\r\n" );
	    itsCtlSock->Flush();

        // Open the data socket
		timeout = ::time( NULL ) + 10;
        while	( timeout > ::time( NULL ) ) {
			if	( itsDoTerminate )	break;
            if	( itsDoAbortion )	throw CError( "Aborted" );
        	if	( lisn ) {
            	sock = lisn->Accept();
                if	( sock )	break;
            }
			else {
            	if	( sock->Connect( addr.addr, addr.port ) ) {
   	            	break;
       	        }
            }
        }
		if	( timeout <= ::time( NULL ) ) {
        	throw CError( "No response from host" );
        }

        // Parse the response
        switch	( CFtpClient::GetReply() ) {
        	case	125:
            case	150:	break;
            default:		throw CError( itsLastResponse );
        }

		code = CFtpClient::GetDirectory( sock );

        if	( lisn )	delete lisn;
    }

    catch	( ... ) {
    	if	( sock )	delete sock;
        if	( lisn )	delete lisn;
        throw;
    }

	return	code;
}

// --------------------------------------------------------------------
int		CFtpClient::NLST	( const char * aPath ) {
	if	( aPath ) {
		itsCtlSock->Write( "NLST " );
		itsCtlSock->Write( aPath );
    }
    else {
		itsCtlSock->Write( "NLST" );
    }
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::SITE  	( const char * aCmnd ) {
	itsCtlSock->Write( "SITE " );
	itsCtlSock->Write( aCmnd );
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::SYST	( void ) {
	itsCtlSock->Write( "SYST\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::STAT	( const char * aObject ) {
	if	( aObject ) {
		itsCtlSock->Write( "STAT " );
		itsCtlSock->Write( aObject );
    }
    else {
		itsCtlSock->Write( "STAT" );
    }
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::HELP	( const char * aObject ) {
	if	( aObject ) {
		itsCtlSock->Write( "HELP " );
		itsCtlSock->Write( aObject );
    }
    else {
		itsCtlSock->Write( "HELP" );
    }
	itsCtlSock->Write( "\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::NOOP	( void ) {
	itsCtlSock->Write( "NOOP\r\n" );
    itsCtlSock->Flush();
	return	CFtpClient::GetReply();
}

// --------------------------------------------------------------------
int		CFtpClient::GetReply( void ) {
	int		code;
	char	repcode	[4]		= { 0 };
    char	repline	[1024]	= { 0 };

    CFtpClient::GetReplyLine( repline, sizeof( repline ) );
	code = CFtpClient::ParseReply( repcode, repline );
	CFtpClient::SkipContinue( repcode, repline, sizeof( repline ) );


	return	code;
}

// --------------------------------------------------------------------
int		CFtpClient::ParseReply( char * aB, const char * aS ) {
	::memcpy( aB, aS, 4 );

	// Let's set the reply line in there
    ::strcpy( itsLastResponse, aS + 4 );

    return 100 * (aS[0] - '0') +
    		10 * (aS[1] - '0') +
           		 (aS[2] - '0');
}

// --------------------------------------------------------------------
void	CFtpClient::SkipContinue( char * aB, char * aL, size_t aS ) {
	// Skip any continuation lines
	if	( aB[3] == '-' ) {
    	aB[3] = ' ';
		while	( ::memcmp( aB, aL, 4 ) ) {
			if	( itsDoTerminate )	break;
			CFtpClient::GetReplyLine( aL, aS );
        }
    }
}

// --------------------------------------------------------------------
void				CFtpClient::GetReplyLine( char * aB, size_t aS ) {
	time_t	timeout	= ::time( NULL ) + 10;
	int		rxch;
    size_t	offs;

	::memset( aB, 0, aS );

	rxch = itsCtlSock->Read();
    offs = 0;
    while	( rxch >= SOCKET_READ_IDLE ) {
        if	( itsDoAbortion )	throw CError( "Aborted" );
		if	( itsDoTerminate )	break;
    	if		( rxch == SOCKET_READ_IDLE ) {
        	if	( timeout < ::time( NULL ) ) {
            	throw CError( "No response from host" );
            }
            SLEEP( 100 );
        }
		else if	( ( rxch == '\r' ) || ( rxch == '\n' ) ) {
        	if		( offs > aS - 2 ) {
            	throw CError( "Invalid response" );
            }
        	else if	( offs > 2 ) {
				break;
            }
            else {
				::memset( aB, 0, aS );
                offs = 0;
            }
        }
    	else if	( rxch > 0 ) {
        	timeout = ::time( NULL ) + 10;
			aB[offs++] = (char)rxch;
        }
        else if	( rxch == SOCKET_READ_DISCONNECT ) {
			if	( offs > 4 ) {
            	break;
            }
            else {
				throw CError( "Premature disconnect" );
            }
        }
        else {
        	throw CError( "Communications error" );
        }
		rxch = itsCtlSock->Read();
    }
}

// --------------------------------------------------------------------
void				CFtpClient::GetSystem	( const char * aS ) {
	itsSystem = -1;
	for	( int s = 0; _official_systems[s].name != NULL; s++ ) {
    	int		len = ::strlen( _official_systems[s].name );
		if	( ! ::strnicmp( aS, _official_systems[s].name, len ) ) {
        	if	( ( aS[len] == ' ' ) || ( aS[len] == 0 ) ) {
            	itsSystem = s;
                break;
            }
        }
    }
    if	( itsSystem == -1 )	CFtpClient::GetSystem	( "UNIX" );
}

// --------------------------------------------------------------------
void				CFtpClient::GetPwd	( const char * aS ) {
	char *	ss = ::strchr( (char *)aS, '"' );
	itsCwd[0] = 0;
    if	( ss ) {
    	::strcpy( itsCwd, ss + 1 );
    	ss = ::strrchr( itsCwd, '"' );
        if	( ss ) {
			*ss = 0;
        }
        else {
			itsCwd[0] = 0;
        }
	}
}

// --------------------------------------------------------------------
bool	CFtpClient::GetSockAddr( ftp_sockaddr_t & aAddr, const char * aS ) {
	bool			result = false;
	const char *	r = ::strchr( aS, '(' );
    char *			w = aAddr.addr;

    if	( r ) {
   	    for	( int dp = 0; dp < 4; dp++ ) {
    		r++;
        	while	( (*r) && (*r != ',' ) )	*(w++) = *(r++);
   	        if	( dp < 3 )	*(w++) = '.';
       	    if	( *r != ',' )	break;
        }
		if	( *r == ',' ) {
			r++;
            aAddr.port = (unsigned short)(256 * ::atoi( r ) );
   	    	while	( (*r) && (*r != ',' ) )	r++;
			if	( *r == ',' ) {
        	    aAddr.port = (unsigned short)(aAddr.port + ::atoi( r + 1 ));
                if	( itsCtlSock->PeerAddr()[0] != 0 ) {
	                if	( ::strcmp( itsCtlSock->PeerAddr(), aAddr.addr ) ) {
                    	::strcpy( aAddr.addr, itsCtlSock->PeerAddr() );
    	            }
                }
                result = true;
            }
        }
    }
    return	result;
}

// --------------------------------------------------------------------
int		CFtpClient::GetDirectory	( CTcpSocket * aSock ) {
	time_t	timeout;
	char	dirline[1024];
    int		rxch, dof, count;

	timeout = ::time( NULL ) + 10;
    dof = 0;
    itsDir.clear();

	try {
	    while	( timeout > ::time( NULL ) ) {
            if	( itsDoAbortion )	throw CError( "Aborted" );
    		rxch = aSock->Read();
        	count = 0;
			while	( ( rxch == SOCKET_READ_IDLE ) && ( count < 10 ) ) {
    	    	SLEEP( 100 );
        	    count++;
				rxch = itsCtlSock->Read();
	        }
    	    if	( ( rxch == SOCKET_READ_IDLE ) && ( dof == 0 ) ) 	break;
			if	( rxch == SOCKET_READ_DISCONNECT )			   		break;
	        if	( ( rxch == '\r' ) || ( rxch == '\n' ) ) {
				timeout = ::time( NULL ) + 10;
        	    if	( dof > 0 ) {
            	    CFtpClient::ParseDirectory( dirline );
	            }
    	        dof = 0;
        	    continue;
	        }
			else if	( rxch >= 0 ) {
				timeout = ::time( NULL ) + 10;
            	dirline[dof++] = (char)rxch;
	            dirline[dof] = 0;
    	        if	( dof > sizeof( dirline ) - 2 ) {
        	    	throw CError( "Invalid response" );
	            }
    	    }
        	else {
				throw CError( "Communications error" );
    	    }
	    }
        delete aSock;
        aSock = NULL;
	    rxch = CFtpClient::GetReply();
    }

    catch	( ... ) {
		if	( aSock ) delete aSock;
		rxch = 0;
    }

	return rxch;
}

// --------------------------------------------------------------------
void	CFtpClient::ParseDirectory( const char * aL ) {
	struct tm	mytm;
    time_t		mytime;
	CFtpFile	myfile;

    // Determine if this is a directory, link or an ordinary file
    if		( *aL == 'd' )	myfile.Dirf( true );
    else if	( *aL == '-' )	myfile.Dirf( false );
    else		  			return;

    // Skip over a few items
    // drwxr-xr-x   2 ftpadmin ftpadmin     4096 Sep 12 19:08 directory
    // -rwxr-xr-x   2 ftpadmin ftpadmin     4096 Sep 12 19:08 file
    for	( int i = 0; i < 4; i++ ) {
    	while	( ( *aL > ' ' ) || ( *aL < 0 ) )	aL++;
    	while	( ( *aL > 0 ) && ( *aL <= ' ' ) )	aL++;
    }

    // We are now pointing to the file size
    if	( ( *aL < '0' ) || ( *aL > '9' ) )	return;
    if	( myfile.Dirf() == false )	myfile.Size( (size_t)::atol( aL ) );
   	while	( ( *aL > ' ' ) || ( *aL < 0 ) )	aL++;
   	while	( ( *aL > 0 ) && ( *aL <= ' ' ) )	aL++;

    // Initialize the time
    mytime = ::time( NULL );
    ::memcpy( &mytm, ::localtime( &mytime ), sizeof( mytm ) );

    // Next one will be the month
	if		( ! ::strnicmp( "jan", aL, 3 ) )	mytm.tm_mon = 0;
	else if	( ! ::strnicmp( "feb", aL, 3 ) )	mytm.tm_mon = 1;
	else if	( ! ::strnicmp( "mar", aL, 3 ) )	mytm.tm_mon = 2;
	else if	( ! ::strnicmp( "apr", aL, 3 ) )	mytm.tm_mon = 3;
	else if	( ! ::strnicmp( "may", aL, 3 ) )	mytm.tm_mon = 4;
	else if	( ! ::strnicmp( "jun", aL, 3 ) )	mytm.tm_mon = 5;
	else if	( ! ::strnicmp( "jul", aL, 3 ) )	mytm.tm_mon = 6;
	else if	( ! ::strnicmp( "aug", aL, 3 ) )	mytm.tm_mon = 7;
	else if	( ! ::strnicmp( "sep", aL, 3 ) )	mytm.tm_mon = 8;
	else if	( ! ::strnicmp( "oct", aL, 3 ) )	mytm.tm_mon = 9;
	else if	( ! ::strnicmp( "nov", aL, 3 ) )	mytm.tm_mon = 10;
	else if	( ! ::strnicmp( "dec", aL, 3 ) )	mytm.tm_mon = 11;
    else										return;
   	while	( ( *aL > ' ' ) || ( *aL < 0 ) )	aL++;
   	while	( ( *aL > 0 ) && ( *aL <= ' ' ) )	aL++;

    // Now we should have the day of the month
    if	( ( *aL < '1' ) || ( *aL > '9' ) )	return;
    mytm.tm_mday = ::atoi( aL );
	if	( ( mytm.tm_mday < 0 ) || ( mytm.tm_mday > 30 ) )	return;
   	while	( ( *aL > ' ' ) || ( *aL < 0 ) )	aL++;
   	while	( ( *aL > 0 ) && ( *aL <= ' ' ) )	aL++;

    // The next one should be either the year or the time
    if	( ( *aL < '0' ) || ( *aL > '9' ) )	return;
   	mytm.tm_sec = 0;
	if	( aL[2] == ':' ) {
	   	mytm.tm_hour = 10 * (aL[0] - '0') + (aL[1] - '0');
   		mytm.tm_min  = 10 * (aL[3] - '0') + (aL[4] - '0');
        if	( ( mytm.tm_hour < 0 ) || ( mytm.tm_hour > 23 ) )	return;
        if	( ( mytm.tm_min  < 0 ) || ( mytm.tm_min  > 59 ) )	return;
    }
    else {
	   	mytm.tm_hour = 0;
   		mytm.tm_min  = 0;
        mytm.tm_year = ::atoi( aL ) - 1900;
        if	( ( mytm.tm_year < 0 ) || ( mytm.tm_year > 400 ) )	return;
    }
    myfile.Time( ::mktime( &mytm ) );
   	while	( ( *aL > ' ' ) || ( *aL < 0 ) )	aL++;
   	while	( ( *aL > 0 ) && ( *aL <= ' ' ) )	aL++;

	// Last but not least - the file name
    char	mypath[1024];
    ::strcpy( mypath, aL );
    if	( ::strstr( mypath, " ->" ) ) {
    	*::strstr( mypath, " ->" ) = 0;
    }
	if	( ! ::strcmp( mypath, "." ) )	return;
	if	( ! ::strcmp( mypath, ".." ) )	return;
	myfile.Name( mypath );
    itsDir.push_back( myfile );
}

// --------------------------------------------------------------------
void	CFtpClient::Cleanup	( void ) {
	itsHost			= NULL;
	itsPort			= NULL;
	itsUser			= NULL;
	itsPass			= NULL;
	itsAcnt			= NULL;

	itsCtlSock		= NULL;

	itsState		= ftp_state_idle;
    itsSystem		= 0;

	itsCanPwd		= true;
    itsCwd[0]		= 0;

    itsCanPasv		= false;

	itsIsRunning	= false;
	itsDoTerminate	= false;
	itsDoAbortion	= false;
    itsThreadId		= 0;

    itsKeepAlive	= false;
    itsLastKeepAlive= ::time( NULL );

	itsErrorMessage[0]	= 0;
    itsLastResponse[0]	= 0;
    itsErrorStart		= ::time( NULL );

}

// --------------------------------------------------------------------
void				CFtpClient::Free	( void ) {
	if	( itsHost )	delete [] itsHost;
	if	( itsUser )	delete [] itsUser;
	if	( itsPass )	delete [] itsPass;
	if	( itsAcnt )	delete [] itsAcnt;
	if	( itsCtlSock ) delete itsCtlSock;

	CFtpClient::Cleanup();
}

// --------------------------------------------------------------------
// EOF:	CFtpClient.cxx
// --------------------------------------------------------------------
