// --------------------------------------------------------------------
// CFtpEngine.cpp
// Whatis:  The FTP engine (RFC 959 - http://www.faqs.org/rfcs/rfc959.html )
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-NOV-2003     Created this source
// --------------------------------------------------------------------
#include	"CFtpEngine.hxx"

// --------------------------------------------------------------------
// local:	RFC943 - http://www.faqs.org/rfcs/rfc943.html
// --------------------------------------------------------------------
static const ftp_system_t	_official_systems[] = {
//	  name
	{ "ASP" },
	{ "AUGUST" },
	{ "BKY" },
	{ "CCP" },
	{ "DOS/360" },
	{ "ELF" },
	{ "EPOS" },
	{ "EXEC-8" },
	{ "GCOS" },
	{ "GPOS" },
	{ "ITS" },
	{ "INTERCOM" },
	{ "INTERLISP" },
	{ "KRONOS" },
	{ "MCP" },
	{ "MOS" },
	{ "MPX-RT" },
	{ "MULTICS" },
	{ "MVT" },
	{ "NOS" },
	{ "NOS/BE" },
	{ "OS/MVS" },
	{ "OS/MVT" },
	{ "RIG" },
	{ "RSX-11M" },
	{ "RT11" },
	{ "SCOPE" },
	{ "SIGNAL" },
	{ "SINTRAN" },
	{ "TAC" },
	{ "TENEX" },
	{ "TOPS-10" },
	{ "TOPS-20" },
	{ "TSS" },
	{ "UNIX" },
	{ "VM/370" },
	{ "VM/CMS" },
	{ "VMS" },
	{ "WAITS" },
	{ "XDE" },
	{ 0 }
};

// --------------------------------------------------------------------
static const ftp_system_t	_default_system = {
	"UNIX"
};

// --------------------------------------------------------------------
CFtpEngine::CFtpEngine	() {
	CFtpEngine::Cleanup();
}

// --------------------------------------------------------------------
CFtpEngine::~CFtpEngine () {
	CFtpEngine::Free();
}

// --------------------------------------------------------------------
void		CFtpEngine::Connect		( const char * aHost, unsigned short aPort ) {
	if	( itsControlSock ) {
		if	( ( itsControlSock->Port() != aPort ) ||
        	  ( ::strcmp( itsControlSock->Host(), aHost ) ) ) {
			delete itsControlSock;
            itsControlSock = NULL;
			itsControlSock = new CFtpSocket( aHost, aPort, &itsControl, &itsControl );
        }
    }
    else {
		itsControlSock = new CFtpSocket( aHost, aPort, &itsControl, &itsControl );
    }
}

// --------------------------------------------------------------------
void		CFtpEngine::Disconnect	( void ) {
	if	( itsListener ) {
    	delete itsListener;
        itsListener = NULL;
    }
	if	( itsControlSock ) {
    	delete itsControlSock;
        itsControlSock = NULL;
    }
    CFtpReply	reply;
	do	{
    	reply = itsControl.Reply();
    }	while	( reply != 0 );
}

// --------------------------------------------------------------------
CTcpSocket *	CFtpEngine::Listener	( void ) {
	return	itsListener;
}

// --------------------------------------------------------------------
CFtpStreamCtl *	CFtpEngine::Control		( void ) {
	return	&itsControl;
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::USER		( const char * aUser ) {
	itsControl.Command( "USER", aUser );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::PASS		( const char * aPass ) {
	itsControl.Command( "PASS", aPass );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::ACCT		( const char * aAcct ) {
	itsControl.Command( "ACCT", aAcct );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::CWD			( const char * aPath ) {
	itsControl.Command( "CWD", aPath );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::CDUP		( void ) {
	itsControl.Command( "CDUP" );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::SMNT		( const char * aPath ) {
	itsControl.Command( "SNMT", aPath );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::REIN		( void ) {
	itsControl.Command( "REIN" );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::QUIT		( void ) {
	itsControl.Command( "QUIT" );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::PORT		( void ) {
	if	( itsListener ) {
    	delete itsListener;
        itsListener = NULL;
    }

   	itsListener = new CTcpSocket( socket_type_stream );
    itsListener->Bind( 0, itsControlSock->Socket() );
    itsListener->Listen();

	char				args [512];
	struct sockaddr_in	sin;
   	int					len = sizeof( sin );
    if	( ::getsockname( itsListener->Socket(), (struct sockaddr *)(&sin), &len ) < 0 ) {
   		throw	CError( "Invalid socket" );
    }
    ::sprintf(	args, "%d,%d,%d,%d,%d,%d",
				(unsigned short)sin.sin_addr.S_un.S_un_b.s_b1,
				(unsigned short)sin.sin_addr.S_un.S_un_b.s_b2,
				(unsigned short)sin.sin_addr.S_un.S_un_b.s_b3,
				(unsigned short)sin.sin_addr.S_un.S_un_b.s_b4,
				(unsigned short)(sin.sin_port & 0xff ),
				(unsigned short)(sin.sin_port >> 8 ) );

    itsControl.Command( "PORT", args );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::PASV		( ftp_sockaddr_t & aAddr ) {
	itsControl.Command( "PASV" );
	return	CFtpEngine::Reply( aAddr );
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::TYPE		( ftp_type_t aType ) {
	itsControl.Command( "TYPE", (char)aType );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::STRU		( ftp_struc_t aStru ) {
	itsControl.Command( "STRU", (char)aStru );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::MODE		( ftp_mode_t aMode ) {
	itsControl.Command( "MODE", (char)aMode );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::RETR		( const char * aFile ) {
	itsControl.Command( "RETR", aFile );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::STOR		( const char * aFile ) {
	itsControl.Command( "STOR", aFile );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::STOU		( char * aFile, size_t aRoom ) {
	itsControl.Command( "STOU" );
	return	CFtpEngine::Reply( aFile, aRoom );
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::APPE		( const char * aFile ) {
	itsControl.Command( "APPE", aFile );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::ALLO		( size_t aSize, size_t aBlkc ) {
	char	args[512];
    if	( aBlkc > 0 ) {
    	::sprintf( args, "%ul R %ul", aSize, aBlkc );
    }
    else {
    	::sprintf( args, "%ul", aSize );
    }
	itsControl.Command( "ALOO", args );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::REST		( const char * aFile ) {
	itsControl.Command( "REST", aFile );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::RNFR		( const char * aFile ) {
	itsControl.Command( "RNFR", aFile );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::RNTO		( const char * aFile ) {
	itsControl.Command( "RNTO", aFile );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::ABOR		( void ) {
	itsControl.Command( "ABOR" );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::DELE		( const char * aFile ) {
	itsControl.Command( "DELE", aFile );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::MKD			( const char * aPath ) {
	itsControl.Command( "MKD", aPath );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::RMD			( const char * aPath ) {
	itsControl.Command( "RMD", aPath );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::PWD			( char * aPath, size_t aRoom ) {
	itsControl.Command( "PWD" );
	return	CFtpEngine::Reply( aPath, aRoom );
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::LIST		( const char * aPath ) {
	itsControl.Command( "LIST", aPath );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::NLST		( const char * aPath ) {
	itsControl.Command( "NLST", aPath );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::SITE		( const char * aPath ) {
	itsControl.Command( "SITE", aPath );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::SYST		( ftp_system_t & aSyst ) {
	itsControl.Command( "SYST" );
	return	CFtpEngine::Reply( aSyst );
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::STAT		( const char * aPath ) {
	itsControl.Command( "STAT", aPath );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::HELP		( const char * aItem ) {
	itsControl.Command( "HELP", aItem );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::NOOP		( void ) {
	itsControl.Command( "NOOP" );
	return	CFtpEngine::Reply();
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::Reply	( void ) {
	CFtpReply	myreply;
	time_t		mytime = ::time( NULL ) + 15;
	do	{
		myreply = itsControl.Reply();
        if	( myreply == 0 ) {
        	if	( mytime < ::time( NULL ) ) {
            	throw	CError( itsControlSock->Socket()->PeerAddr(), "No response" );
            }
        }
    }	while	( myreply == 0 );
    return	myreply;
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::Reply	( ftp_sockaddr_t & aAddr ) {
	CFtpReply		myreply = CFtpEngine::Reply();
	const char *	r		= ::strchr( myreply.Data(), '(' );
    char *			w		= aAddr.addr;
    bool			result	= false;

    if	( r ) {
   	    for	( int dp = 0; dp < 4; dp++ ) {
    		r++;
        	while	( (*r) && (*r != ',' ) )	*(w++) = *(r++);
   	        if	( dp < 3 )	*(w++) = '.';
       	    if	( *r != ',' )	break;
        }
		if	( *r == ',' ) {
			r++;
            aAddr.port = (unsigned short)(256 * ::atoi( r ) );
   	    	while	( (*r) && (*r != ',' ) )	r++;
			if	( *r == ',' ) {
        	    aAddr.port = (unsigned short)(aAddr.port + ::atoi( r + 1 ));
                if	( itsControlSock->Socket()->PeerAddr()[0] != 0 ) {
	                if	( ::strcmp( itsControlSock->Socket()->PeerAddr(), aAddr.addr ) ) {
                    	::strcpy( aAddr.addr, itsControlSock->Socket()->PeerAddr() );
    	            }
                }
                result = true;
            }
        }
    }
    if	( ! result )	throw CError( itsControlSock->Socket()->PeerAddr(), "Invalid reply", myreply.Data() );
	return	myreply;
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::Reply	( char * aPath, size_t aRoom ) {
	CFtpReply		myreply = CFtpEngine::Reply();
    const char *	start	= ::strchr( myreply.Data(), '"' );
	const char *	end		= start ? ::strchr( start + 1, '"' ) : NULL;
    size_t			size	= (size_t)((long)end - (long)start);
    ::memset( aPath, 0, aRoom );
	if	( ! start )			throw CError( itsControlSock->Socket()->PeerAddr(), "Invalid reply", myreply.Data() );
	if	( ! end )			throw CError( itsControlSock->Socket()->PeerAddr(), "Invalid reply", myreply.Data() );
    if	( size > aRoom )	throw CError( itsControlSock->Socket()->PeerAddr(), "Invalid reply", myreply.Data() );
    if	( size < 1 )		throw CError( itsControlSock->Socket()->PeerAddr(), "Invalid reply", myreply.Data() );
	::memcpy( aPath, start + 1, size - 1 );
	return	myreply;
}

// --------------------------------------------------------------------
CFtpReply	CFtpEngine::Reply	( ftp_system_t & aSystem ) {
	CFtpReply	myreply = CFtpEngine::Reply();
    int			system  = 0;

	while	( _official_systems[system].name ) {
		if	( ! ::strnicmp( myreply.Data(), _official_systems[system].name, ::strlen(_official_systems[system].name) ) ) {
			if	( myreply.Data()[::strlen(_official_systems[system].name)] < 0 )	continue;
			if	( myreply.Data()[::strlen(_official_systems[system].name)] > ' ' )	continue;
			aSystem = _official_systems[system];
            break;
        }
    }

    if	( _official_systems[system].name == NULL ) {
    	aSystem = _default_system;
    }

	return	myreply;
}

// --------------------------------------------------------------------
void		CFtpEngine::Cleanup	( void ) {
	itsControlSock	= NULL;
    itsListener		= NULL;
}

// --------------------------------------------------------------------
void		CFtpEngine::Free	( void ) {
	if	( itsListener )		delete itsListener;
	if	( itsControlSock )	delete itsControlSock;
    CFtpEngine::Cleanup();
}

// --------------------------------------------------------------------
// EOF: CFtpEngine.cpp
// --------------------------------------------------------------------

