// --------------------------------------------------------------------
// CFurserSrData.cpp
// Whatis:  Furser Server Data class
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 23-DEC-2003     Created this source
// --------------------------------------------------------------------
#include	"CFurserSrData.h"
#include	"CError.hxx"

// --------------------------------------------------------------------
CFurserSrData::CFurserSrData () : itsError() {
	CFurserSrData::Cleanup();
}

// --------------------------------------------------------------------
CFurserSrData::~CFurserSrData	() {
	CFurserSrData::Free();
}

// --------------------------------------------------------------------
CFurserSrData::CFurserSrData	( const CFurserSrData & aC ) {
	CFurserSrData::Cleanup();
	*this = aC;
}

// --------------------------------------------------------------------
CFurserSrData & CFurserSrData::operator = ( const CFurserSrData & aC ) {
	CFurserSrData::Free();
	itsServerList = aC.ServerList();
	return *this;
}

// --------------------------------------------------------------------
void	CFurserSrData::NetSave		( CFurserClient & aC, bool aClient ) {
	if	( aClient )	aC.SendCommand( 'S' );
    CFurserSrData::Save( aC );
}

// --------------------------------------------------------------------
void   	CFurserSrData::NetLoad		( CFurserClient & aC, bool aClient ) {
	if	( aClient )	aC.SendCommand( 'L' );
    CFurserSrData::Load( aC );
}

// --------------------------------------------------------------------
void	CFurserSrData::NetStart	( CFurserClient & aC, const char * aS, bool aClient ) {
	if	( CFurserSrData::HasServer( aS ) ) {
		CFurserSrSrvr	srvr = CFurserSrData::GetServer( aS );
		if	( aClient ) {
			aC.SendCommand( 'r' );
			aC.TxResponse( srvr.ServerName() );
			aC.RxResponse( "Run Server OK\n" );
	    }
    	else {
			aC.TxResponse( "Run Server OK\n" );
	    }
    	srvr.Running( true );
	    CFurserSrData::SetServer( aS, srvr );
    }
    else {
    	throw CError( "No such server" );
    }
}

// --------------------------------------------------------------------
void	CFurserSrData::NetStop		( CFurserClient & aC, const char * aS, bool aClient ) {
	if	( CFurserSrData::HasServer( aS ) ) {
		CFurserSrSrvr	srvr = CFurserSrData::GetServer( aS );
    	if	( aClient ) {
			aC.SendCommand( 's' );
			aC.TxResponse( srvr.ServerName() );
			aC.RxResponse( "Stop Server OK\n" );
	    }
    	else {
			aC.TxResponse( "Stop Server OK\n" );
	    }
    	srvr.Running( false );
        srvr.GameRunning( false );
	    CFurserSrData::SetServer( aS, srvr );
    }
    else {
    	throw CError( "No such server" );
    }
}

// --------------------------------------------------------------------
void	CFurserSrData::NetImmStart	( CFurserClient & aC, const char * aS, bool aClient ) {
	if	( aClient ) {
		aC.SendCommand( '1' );
		aC.TxResponse( aS );
		aC.RxResponse( "Immediate Run Server OK\n" );
		if	( CFurserSrData::HasServer( aS ) ) {
			CFurserSrSrvr	srvr = CFurserSrData::GetServer( aS );
            srvr.GameRunning( true );
            CFurserSrData::SetServer( aS, srvr );
        }
    }
   	else {
		aC.TxResponse( "Immediate Run Server OK\n" );
    }
}

// --------------------------------------------------------------------
void	CFurserSrData::NetImmStop	( CFurserClient & aC, const char * aS, bool aClient ) {
   	if	( aClient ) {
		aC.SendCommand( '2' );
		aC.TxResponse( aS );
		aC.RxResponse( "Immediate Stop Server OK\n" );
		if	( CFurserSrData::HasServer( aS ) ) {
			CFurserSrSrvr	srvr = CFurserSrData::GetServer( aS );
            srvr.GameRunning( false );
            CFurserSrData::SetServer( aS, srvr );
        }
    }
   	else {
		aC.TxResponse( "Immediate Stop Server OK\n" );
    }
}

// --------------------------------------------------------------------
char *	CFurserSrData::NetLogdata	( CFurserClient & aC, const char * aData, bool aClient ) {
	char *	rxdata	= NULL;
    if	( aClient ) {
		aC.SendCommand( 'g' );
        rxdata = aC.RxData();
		aC.TxResponse( "Log Data OK\n" );
    }
    else {
    	aC.TxResponse( aData );
		aC.RxResponse( "Log Data OK\n" );
    }
    return	rxdata;
}

// --------------------------------------------------------------------
void	CFurserSrData::Load		( CFurserClient & aC ) {
	dword_t			count, i, pos;
    CFurserSrSrvr	srvr;

	CFurserSrData::Free();
	CFurserSrData::ProgressCalc( aC, true );
    count = aC.RxDataDword();
    for ( i = 0; i < count; i++ ) {
    	itsMutex.Lock();
        pos = itsProgressPos;
        itsMutex.Unlock();
    	srvr.Load( aC, pos );
        itsServerList.push_back( srvr );
    }
    aC.TxResponse( "Load OK\n" );
}

// --------------------------------------------------------------------
void	CFurserSrData::Save		( CFurserClient & aC ) {
	CFurserSrSrvr_li	loop;
	dword_t				pos;
	CFurserSrData::ProgressCalc( aC, false );
	aC.TxDataDword( itsServerList.size() );
   	for	(	loop  = itsServerList.begin();
			loop != itsServerList.end();
           	loop++ ) {
    	itsMutex.Lock();
        pos = itsProgressPos;
        itsMutex.Unlock();
    	(*loop).Save( aC, pos );
	}
    aC.RxResponse( "Load OK\n" );
}

// --------------------------------------------------------------------
void	CFurserSrData::Load		( const char * aF ) {
	struct stat		mystat;
	CWinIniFile		winini;

	CFurserSrData::Free();
	if	( ! ::stat( aF, &mystat ) ) {
    	winini.Read( aF );
		CFurserSrData::Load( winini );
	}
}

// --------------------------------------------------------------------
void	CFurserSrData::Load		( const CWinIniFile & aIni ) {
    dword_t			count, i;
    CFurserSrSrvr	srvr;
    char			secname	[64];

	CFurserSrData::Free();
    count = aIni.IntValue( "Servers", "ServerCount" );
	for	( i = 0; i < count; i++ ) {
    	::sprintf( secname, "Server_%d", i );
        srvr.Load( aIni, secname );
        itsServerList.push_back( srvr );
	}
}

// --------------------------------------------------------------------
void	CFurserSrData::Save		( const char * aF ) {
	CWinIniFile			winini;
	CWinIniSection		sec;
	CFurserSrSrvr_li	loop;
    dword_t				count;
    char				secname	[64];

	sec.Name( "Servers" );
	sec.Insert( "ServerCount", (int)itsServerList.size() );
	winini.Insert( sec );

	count = 0;
   	for	(	loop  = itsServerList.begin();
			loop != itsServerList.end();
           	loop++ ) {
    	::sprintf( secname, "Server_%d", count );
    	(*loop).Save( winini, secname );
        count++;
	}
	::unlink( aF );
	winini.Write( aF );
}

// --------------------------------------------------------------------
CError	CFurserSrData::Error		( void ) const {
	return itsError;
}

// --------------------------------------------------------------------
void	CFurserSrData::Error		( const CError & aE ) {
	itsError = aE;
}

// --------------------------------------------------------------------
void	CFurserSrData::Abort		( void ) {
	itsMutex.Lock();
	itsProgressPos = 0;
    itsProgressEnd = 0;
	itsMutex.Unlock();
}

// --------------------------------------------------------------------
void	CFurserSrData::ForceProgress( void ) {
	itsMutex.Lock();
	itsProgressPos = 0;
    itsProgressEnd = 1;
	itsMutex.Unlock();
}

// --------------------------------------------------------------------
void	CFurserSrData::AddServer	( const CFurserSrSrvr & aC ) {
	if	( ! CFurserSrData::HasServer( aC.ServerName() ) ) {
    	itsServerList.push_back( aC );
    }
}

// --------------------------------------------------------------------
void	CFurserSrData::DelServer	( const CFurserSrSrvr & aC ) {
	itsServerList.remove( aC );
}

// --------------------------------------------------------------------
void	CFurserSrData::SetServer	( const char * aN, const CFurserSrSrvr & aC ) {
	if	( ( ! CFurserSrData::HasServer( aC.ServerName() ) ) || ( aC == aN ) ) {
		CFurserSrSrvr_li	loop;
    	for	(	loop  = itsServerList.begin();
				loop != itsServerList.end();
            	loop++ ) {
	    	if	( *loop == aN ) {
    	    	*loop = aC;
        	    break;
            }
        }
    }
}

// --------------------------------------------------------------------
CFurserSrSrvr	CFurserSrData::GetServer	( const char * aN ) const {
	CFurserSrSrvr_lci	loop;
    for	(	loop  = itsServerList.begin();
    		loop != itsServerList.end();
            loop++ ) {
    	if	( *loop == aN )	return *loop;
    }
    CFurserSrSrvr dummy;
    return dummy;
}

// --------------------------------------------------------------------
bool	CFurserSrData::HasServer	( const char * aN ) const {
	CFurserSrSrvr_lci	loop;
    for	(	loop  = itsServerList.begin();
    		loop != itsServerList.end();
            loop++ ) {
    	if	( *loop == aN )	return true;
    }
    return false;
}

// --------------------------------------------------------------------
const CFurserSrSrvr_l &	CFurserSrData::ServerList	( void ) const {
	return itsServerList;
}

// --------------------------------------------------------------------
CFurserSrSrvr_l &		CFurserSrData::ServerMList	( void ) {
	return itsServerList;
}

// --------------------------------------------------------------------
CFurserString_l	CFurserSrData::ServerNames	( void ) const {
	CFurserSrSrvr_lci	loop;
	CFurserString		mydata;
	CFurserString_l		mylist;
    for	(	loop  = itsServerList.begin();
    		loop != itsServerList.end();
            loop++ ) {
		mydata = (*loop).ServerName();
        mydata.Checked( (*loop).Running() );
		mylist.push_back( mydata );
    }
	return mylist;
}

// --------------------------------------------------------------------
void	CFurserSrData::ProgressCalc( CFurserClient & aC, bool aIsLoading ) {
	itsMutex.Lock();
	itsProgressPos = 0;
	itsProgressEnd = 1;
	itsMutex.Unlock();
	if	( aIsLoading ) {
		itsMutex.Lock();
        try {
			itsProgressEnd = aC.RxDataDword();
			itsMutex.Unlock();
        }
        catch ( ... ) {
			itsMutex.Unlock();
			throw;
        }
		aC.TxResponse( "Progress OK\n" );
    }
    else {
		CFurserSrSrvr_li	loop;
		itsMutex.Lock();
    	itsProgressEnd = 1;
		itsMutex.Unlock();
	   	for	(	loop  = itsServerList.begin();
				loop != itsServerList.end();
        	   	loop++ ) {
			itsMutex.Lock();
            try {
	        	itsProgressEnd = itsProgressEnd + (*loop).ProgressCalc();
				itsMutex.Unlock();
            }
            catch ( ... ) {
				itsMutex.Unlock();
                throw;
            }
        }
        dword_t	value;
		itsMutex.Lock();
        itsProgressEnd--;
        value = itsProgressEnd;
		itsMutex.Unlock();
		aC.TxDataDword( value );
        aC.RxResponse( "Progress OK\n" );
    }
}

// --------------------------------------------------------------------
dword_t		CFurserSrData::ProgressEnd	( void ) {
	dword_t	value;
    itsMutex.Lock();
    value = itsProgressEnd;
    itsMutex.Unlock();
    return value;
}

// --------------------------------------------------------------------
dword_t		CFurserSrData::ProgressPos	( void ) {
	dword_t	value;
    itsMutex.Lock();
    value = itsProgressPos;
    itsMutex.Unlock();
    return value;
}

// --------------------------------------------------------------------
void	CFurserSrData::Free		( void ) {
	CFurserSrData::Cleanup();
}

// --------------------------------------------------------------------
void	CFurserSrData::Cleanup	( void ) {
	itsServerList.clear();
    itsProgressEnd	= 0;
    itsProgressPos	= 0;
}

// --------------------------------------------------------------------
// EOF:	CFurserSrData.cpp
// --------------------------------------------------------------------
