// --------------------------------------------------------------------
// CFurserWizQery.cpp
// Whatis:  Furser Document class
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 23-DEC-2003     Created this source
// --------------------------------------------------------------------
#include	"CFurserWizQery.h"
#include	"CError.hxx"

// --------------------------------------------------------------------
CFurserWizQery::CFurserWizQery	() {
	CFurserWizQery::Cleanup();
}

// --------------------------------------------------------------------
CFurserWizQery::~CFurserWizQery	() {
	CFurserWizQery::Free();
}

// --------------------------------------------------------------------
CFurserWizQery::CFurserWizQery ( const CFurserWizQery & aC ) {
	CFurserWizQery::Cleanup();
	*this = aC;
}

// --------------------------------------------------------------------
CFurserWizQery & CFurserWizQery::operator =	( const CFurserWizQery & aC ) {
	CFurserWizQery::Free();
	itsCaption	= ::my_private_strdup( aC.Caption() );
	itsHelp		= ::my_private_strdup( aC.Help() );
	itsType		= aC.Type();
	itsMin		= ::my_private_strdup( aC.Min() );
	itsMax		= ::my_private_strdup( aC.Max() );
	itsDefault	= ::my_private_strdup( aC.Default() );
	itsReplaces	= ::my_private_strdup( aC.Replaces() );
	itsValue	= ::my_private_strdup( aC.Value() );
	return *this;
}

// --------------------------------------------------------------------
void	CFurserWizQery::Load	( const CWinIniFile & aIni, const char * aSecName ) {
	CFurserWizQery::Free();
	itsCaption	= ::my_private_strdup( aIni.Value( aSecName, "Caption" ) );
	itsHelp		= ::my_private_strdup( aIni.Value( aSecName, "Help" ) );
	itsType		= CFurserWizQery::ResolveType( aIni.Value( aSecName, "Type" ) );
	itsMin		= ::my_private_strdup( aIni.Value( aSecName, "Min" ) );
	itsMax		= ::my_private_strdup( aIni.Value( aSecName, "Max" ) );
	itsDefault	= ::my_private_strdup( aIni.Value( aSecName, "Default" ) );
	itsReplaces	= ::my_private_strdup( aIni.Value( aSecName, "Replaces" ) );
	itsValue	= ::my_private_strdup( CFurserWizQery::Default() );
}

// --------------------------------------------------------------------
const char *	CFurserWizQery::Caption	( void ) const {
	return itsCaption ? itsCaption : "";
}

// --------------------------------------------------------------------
const char *	CFurserWizQery::Help	( void ) const {
	return itsHelp ? itsHelp : "";
}

// --------------------------------------------------------------------
qerytype_t		CFurserWizQery::Type	( void ) const {
	return	itsType;
}

// --------------------------------------------------------------------
const char *	CFurserWizQery::Min		( void ) const {
	return itsMin ? itsMin : "";
}

// --------------------------------------------------------------------
const char *	CFurserWizQery::Max		( void ) const {
	return itsMax ? itsMax : "";
}

// --------------------------------------------------------------------
const char *	CFurserWizQery::Default	( void ) const {
	return itsDefault ? itsDefault : "";
}

// --------------------------------------------------------------------
const char *	CFurserWizQery::Replaces( void ) const {
	return itsReplaces ? itsReplaces : "";
}

// --------------------------------------------------------------------
const char *	CFurserWizQery::Value	( void ) const {
	return itsValue ? itsValue : "";
}

// --------------------------------------------------------------------
static const char *	__validFileChars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ.-_1234567890~[]";
void			CFurserWizQery::Check	( const char * aV ) {
	// Check the value
    if		( itsType == qerytype_string )	;	// No checking for strings
    else if	( itsType == qerytype_filename ) {	// Filenames may not contain inv.chars
    	if	( ! aV )					throw CError( "File names are required" );
    	if	( *aV == 0 )				throw CError( "File names are required" );
        if	( ::strlen( aV ) > 128 )	throw CError( "File name may be too long" );
		for	( const char * p = aV; *p; p++ ) {
        	if	( ! ::strchr( __validFileChars, *p ) ) {
				throw CError( "Invalid character for file name" );
            }
        }
    }
    else if	( itsType == qerytype_integer ) {
    	if	( ! aV )	throw CError( "Value required" );
    	long	min = ::atol( CFurserWizQery::Min() );
        long	max = ::atol( CFurserWizQery::Max() );
        if	( min < max ) {
        	char	messbf[64];
            ::sprintf( messbf, "Value out of range (%li-%li)", min, max );
			if	( ::atol( aV ) < min )	throw CError( messbf );
			if	( ::atol( aV ) > max )	throw CError( messbf );
        }
    }
    else if	( itsType == qerytype_boolean ) {
    	if	( ! aV )	throw CError( "Value required" );
		if	( ( ::my_stricmp( aV, "true" ) ) &&
              ( ::my_stricmp( aV, "false" ) ) ) {
        	throw CError( "Value must be either True or False" );
        }
    }
    else if	( itsType == qerytype_float ) {
    	if	( ! aV )	throw CError( "Value required" );
    	double	min = ::atof( CFurserWizQery::Min() );
        double	max = ::atof( CFurserWizQery::Max() );
        if	( min < max ) {
        	char	messbf[64];
            ::sprintf( messbf, "Value out of range (%f-%f)", min, max );
			if	( ::atof( aV ) < min )	throw CError( messbf );
			if	( ::atof( aV ) > max )	throw CError( messbf );
        }
    }
}

// --------------------------------------------------------------------
void			CFurserWizQery::Value	( const char * aV ) {
	if	( itsValue )	delete [] itsValue;
	itsValue = ::my_private_strdup( aV );
}

// --------------------------------------------------------------------
qerytype_t	CFurserWizQery::ResolveType	( const char * aType ) {
	if	( ! ::my_stricmp( aType, "string" ) )	return qerytype_string;
	if	( ! ::my_stricmp( aType, "filename" ) )	return qerytype_filename;
	if	( ! ::my_stricmp( aType, "integer" ) )	return qerytype_integer;
	if	( ! ::my_stricmp( aType, "boolean" ) )	return qerytype_boolean;
	if	( ! ::my_stricmp( aType, "float" ) )	return qerytype_float;
	throw	CError( aType, "Invalid type" );
}

// --------------------------------------------------------------------
void	CFurserWizQery::Free	( void ) {
	if	( itsCaption )	delete [] itsCaption;
	if	( itsHelp )		delete [] itsHelp;
	if	( itsMin )		delete [] itsMin;
	if	( itsMax )		delete [] itsMax;
	if	( itsDefault )	delete [] itsDefault;
	if	( itsReplaces ) delete [] itsReplaces;
	if	( itsValue )	delete [] itsValue;
	CFurserWizQery::Cleanup();
}

// --------------------------------------------------------------------
void	CFurserWizQery::Cleanup	( void ) {
	itsCaption	= NULL;
	itsHelp		= NULL;
	itsType		= qerytype_string;
	itsMin		= NULL;
	itsMax		= NULL;
	itsDefault	= NULL;
	itsReplaces	= NULL;
	itsValue	= NULL;
}

// --------------------------------------------------------------------
// EOF:	CFurserWizQery.cpp
// --------------------------------------------------------------------
