// --------------------------------------------------------------------
// C3DPolygon.cxx
// Whatis:  3D modeling library atom
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 21-OCT-2002     Created this source
// --------------------------------------------------------------------
#include	"C3DPolygon.hxx"

// --------------------------------------------------------------------
// local:	Macro myIsDigit( ... )
// --------------------------------------------------------------------
#define	myIsDigit( d )	(((d)>='0')&&((d)<='9'))

// --------------------------------------------------------------------
// public:	Assignment operator
// --------------------------------------------------------------------
C3DPolygon & C3DPolygon::operator = 	( const C3DPolygon & aPolygon ) {
	itsVertexList 	= aPolygon.VertexList();
	itsOrigin		= aPolygon.Origin();
	itsNormal		= aPolygon.Normal();
	itsTextureU		= aPolygon.TextureU();
	itsTextureV		= aPolygon.TextureV();
	itsPanU			= aPolygon.PanU();
	itsPanV			= aPolygon.PanV();
	itsFlags		= aPolygon.Flags();
	itsLink			= aPolygon.Link();
	C3DPolygon::Item( aPolygon.Item() );
	C3DPolygon::Texture( aPolygon.Texture() );

	return *this;
}

// --------------------------------------------------------------------
// public:	Parse a string into a polygon (multiple vertexes)
// --------------------------------------------------------------------
const char *	C3DPolygon::T3DParse( const char * aS ) {
	C3DVertex		vertex;
	const char *	ref;

	// Default values
	C3DPolygon::Free();

	// Begin Polygon Item=OUTSIDE Texture=horz Flags=1073741824 Link=2

	// Skip blanks
	while	( ( *aS > 0 ) && ( *aS <= ' ' ) )	aS++;
	if	( ::strncmp( aS, "Begin Polygon", 13 ) )	return aS;

	aS = C3DPolygon::T3DParseVal( itsItem,		"Item=",	aS + 13 );
	aS = C3DPolygon::T3DParseVal( itsTexture, 	"Texture=", aS );
	aS = C3DPolygon::T3DParseVal( itsFlags, 	"Flags=",	aS );
	aS = C3DPolygon::T3DParseVal( itsLink,		"Link=",	aS );

	while	( *aS ) {

		vertex = 0.0;

		// Skip blanks
		while	( ( *aS > 0 ) && ( *aS <= ' ' ) )	aS++;

		// Origin   -00075.607231,-00396.498291,+00073.000008
		aS = vertex.T3DParse( "Origin", ref = aS );
		if	( ref != aS ) {
			itsOrigin = vertex;
			continue;
		}

		// Normal   +00000.000000,+00000.000000,+00001.000000
		aS = vertex.T3DParse( "Normal", aS );
		if	( ref != aS ) {
			itsNormal = vertex;
			continue;
		}

		// TextureU -00000.923880,-00000.382683,+00000.000000
		aS = vertex.T3DParse( "TextureU", aS );
		if	( ref != aS ) {
			itsTextureU = vertex;
			continue;
		}

		// TextureV -00000.382683,+00000.923880,+00000.000000
		aS = vertex.T3DParse( "TextureV", aS );
		if	( ref != aS ) {
			itsTextureV = vertex;
			continue;
		}

		// Vertex   -00058.768913,-00437.149536,+00073.000008
		aS = vertex.T3DParse( "Vertex", aS );
		if	( ref != aS ) {
			itsVertexList.push_back( vertex );
			continue;
		}

		// Pan      U=15 V=1
		// Pan      U=-7 V=24
		if	( ! ::strncmp( aS, "Pan", 3 ) ) {
			aS = C3DPolygon::T3DParseVal( itsPanU, "U=", aS + 3 );
			aS = C3DPolygon::T3DParseVal( itsPanV, "V=", aS );
			continue;
		}

		if	( ! ::strncmp( aS, "End Polygon", 11 ) ) {
			aS += 11;
			break;
		}

		break;
	}

	return	aS;
}

// --------------------------------------------------------------------
// public:	Set value for Item
// --------------------------------------------------------------------
void	C3DPolygon::Item	( const char * aV ) {
	if	( itsItem )		delete [] itsItem;
	itsItem = ::my_private_strdup( aV );
}

// --------------------------------------------------------------------
// public:	Set value for Texture
// --------------------------------------------------------------------
void	C3DPolygon::Texture	( const char * aV ) {
	if	( itsTexture )	delete [] itsTexture;
	itsTexture = ::my_private_strdup( aV );
}

// --------------------------------------------------------------------
// private:	T3D parsers
// --------------------------------------------------------------------
const char *	C3DPolygon::T3DParseVal	( int & theInt, const char * aKw, const char * aS ) const {
	bool	negative	= false;
	dword_t	myvalue		= 0;

	// Default value
	theInt = 0;

	// Skip blanks
	while	( ( *aS > 0 ) && ( *aS <= ' ' ) )	aS++;
	if	( ::strncmp( aS, aKw, ::strlen( aKw ) ) )	return aS;
	aS += ::strlen( aKw );

	if		( *aS == '-' )			{ aS++; negative = true; }
	else if	( *aS == '+' )			{ aS++; negative = false; }

	while	( myIsDigit( *aS ) ) {
		myvalue = 10 * myvalue + (dword_t)( ( *(aS++) - '0' ) & 0x0f );
	}

	theInt = negative ? -((int)myvalue) : (int)myvalue;

	return	aS;
}

// --------------------------------------------------------------------
const char *	C3DPolygon::T3DParseVal	( dword_t & theDw, const char * aKw, const char * aS ) const {
	bool	negative	= false;
	dword_t	myvalue		= 0;

	// Default value
	theDw = 0;

	// Skip blanks
	while	( ( *aS > 0 ) && ( *aS <= ' ' ) )	aS++;
	if	( ::strncmp( aS, aKw, ::strlen( aKw ) ) )	return aS;
	aS += ::strlen( aKw );

	if		( *aS == '-' )			{ aS++; negative = true; }
	else if	( *aS == '+' )			{ aS++; negative = false; }

	while	( myIsDigit( *aS ) ) {
		myvalue = 10 * myvalue + (dword_t)( ( *(aS++) - '0' ) & 0x0f );
	}




	theDw = negative ? (dword_t)(-((int32_t)myvalue)) : myvalue;

	return	aS;
}

// --------------------------------------------------------------------
const char *	C3DPolygon::T3DParseVal	( char *& theBuf, const char * aKw, const char * aS ) const {

	// Default value
	if	( theBuf )	delete [] theBuf;
	theBuf = NULL;

	// Skip blanks
	while	( ( *aS > 0 ) && ( *aS <= ' ' ) )	aS++;
	if	( ::strncmp( aS, aKw, ::strlen( aKw ) ) )	return aS;
	aS += ::strlen( aKw );

	const char *	se;
	dword_t			ll;

	for	( se = aS; (*se < 0) || (*se > ' '); se++ );
	ll = (dword_t)( se - aS );

	theBuf = new char [ll + 1];
	::memcpy( theBuf, aS, ll );
	theBuf[ll] = 0;

	return	se;
}

// --------------------------------------------------------------------
// private:	Free up resources
// --------------------------------------------------------------------
void	C3DPolygon::Free	( void ) {
	if	( itsItem )		delete [] itsItem;
	if	( itsTexture )	delete [] itsTexture;
	C3DPolygon::Cleanup();
}

// --------------------------------------------------------------------
// private:	Clean up teh mess
// --------------------------------------------------------------------
void	C3DPolygon::Cleanup	( void ) {
	itsPanV		= 0;
	itsPanU		= 0;
	itsOrigin	= 0.0;
	itsNormal	= 0.0;
	itsTextureU	= 0.0;
	itsTextureV	= 0.0;
	itsVertexList.clear();
	itsItem		= NULL;
	itsTexture	= NULL;
	itsFlags	= 0;
	itsLink		= 0;
}

// --------------------------------------------------------------------
// EOF: C3DPolygon.cxx
// --------------------------------------------------------------------
