// --------------------------------------------------------------------
// CPictureFilterQuantize.cxx
// Whatis:  Quantize the image to limited number of colours
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include	"CError.hxx"
#include	"CPictureFilterQuantize.hxx"

// --------------------------------------------------------------------
// Macros to speed up things
// --------------------------------------------------------------------
#define	UPDATE_MINMAX(Bu,P) \
	v = P.R(); \
	if (v < Bu->min_r) Bu->min_r = v; \
	if (v > Bu->max_r) Bu->max_r = v; \
	v = P.G(); \
	if (v < Bu->min_g) Bu->min_g = v; \
	if (v > Bu->max_g) Bu->max_g = v; \
	v = P.B(); \
	if (v < Bu->min_b) Bu->min_b = v; \
	if (v > Bu->max_b) Bu->max_b = v; \
	Bu->mid_r = (byte_t)((Bu->max_r + Bu->min_r) / 2); \
	Bu->mid_g = (byte_t)((Bu->max_g + Bu->min_g) / 2); \
	Bu->mid_b = (byte_t)((Bu->max_b + Bu->min_b) / 2)

// --------------------------------------------------------------------
// Another needed structure
// --------------------------------------------------------------------
typedef	struct color_bucket_s {
	byte_t	min_r;
	byte_t	min_g;
	byte_t	min_b;
	byte_t	max_r;
	byte_t	max_g;
	byte_t	max_b;
	byte_t	mid_r;
	byte_t	mid_g;
	byte_t	mid_b;
	byte_t	max_extent;
	int		count;
}	color_bucket_t,
*	color_bucket_tp;

// --------------------------------------------------------------------
// public:	Constructor
// --------------------------------------------------------------------
CPictureFilterQuantize::CPictureFilterQuantize ( word_t	aMaxColors ) {
	itsMaxColors= aMaxColors;

	// First check out that the max colors is sane
	if	( aMaxColors == 0 ) {
		throw	CError( "Invalid maximum colors, should not be zero" );
	}

}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CPictureFilterQuantize::~CPictureFilterQuantize () {
}

// --------------------------------------------------------------------
// public:	The data processor
// --------------------------------------------------------------------
void	CPictureFilterQuantize::Process	( CPicturePixmap & aPixmap ) {
	color_bucket_tp		buckets 	= NULL;
	word_tp				picture		= NULL;
	CPicturePixel *		pixmap		= aPixmap.Pixmap();
	dword_t				num_pixels	= aPixmap.W() * aPixmap.H();
	dword_t				num_buckets	= 1;
	dword_t				num_colors	= 0;
	dword_t				i, j, bb_ind;
	word_t				bb_val;
	byte_t				v;

	try {

		// Count number of colors (maybe we can skip the process)
		for	( i = 0; i < num_pixels; i++ ) {
			for	( j = 0; j < i; j++ ) {
				if	( pixmap[i] == pixmap[j] )	break;
			}
			if	( j < i )	continue;
			num_colors++;
			if	( num_colors > (dword_t)itsMaxColors )	break;
		}
		if	( num_colors <= (dword_t)itsMaxColors )	return;

		buckets	= new color_bucket_t [ itsMaxColors ];
		picture = new word_t		 [ num_pixels ];

		::memset( picture, 0, num_pixels * sizeof( word_t ) );

		// First find out minimum and maximum values from the image
		CPictureFilterQuantize::ClearBucket( buckets );
		buckets->count = num_pixels;
		for	( i = 0; i < num_pixels; i++ ) {
			UPDATE_MINMAX( buckets, pixmap[i] );
		}

		// Set the extent as well
		CPictureFilterQuantize::SetExtent( buckets );

		// Do the splitting
		bb_val = buckets->max_extent;

		while	( ( bb_val > 0 ) && ( num_buckets < (dword_t)itsMaxColors ) ) {
			bb_val = 0;
			bb_ind = 0;
			for	( i = 0; i < num_buckets; i++ ) {
				if	( buckets[i].max_extent > bb_val ) {
					bb_ind = i;
					bb_val = buckets[i].max_extent;
				}
			}

			CPictureFilterQuantize::SplitBucket(	pixmap,
													num_pixels,
													picture,
													(word_t)bb_ind,
													(word_t)num_buckets,
													buckets );
			num_buckets++;
		}

		// Last, change the image
		for	( i = 0; i < num_pixels; i++ ) {
			pixmap[i].R( buckets[picture[i]].mid_r );
			pixmap[i].G( buckets[picture[i]].mid_g );
			pixmap[i].B( buckets[picture[i]].mid_b );
		}

		delete []	picture;
		delete []	buckets;
	}

	catch	( ... ) {
		if	( picture ) delete []	picture;
		if	( buckets )	delete []	buckets;
		throw;
	}
}



// --------------------------------------------------------------------
// private:	Clear bucket
// --------------------------------------------------------------------
void	CPictureFilterQuantize::ClearBucket( void * aBucket ) {
	color_bucket_tp	cb	= (color_bucket_tp)aBucket;

	cb->min_r =
	cb->min_g =
	cb->min_b = 255;
	cb->max_r =
	cb->max_g =
	cb->max_b = 0;
	cb->mid_r =
	cb->mid_g =
	cb->mid_b = 127;
	cb->count = 0;
}

// --------------------------------------------------------------------
// private:	Set extent
// --------------------------------------------------------------------
void	CPictureFilterQuantize::SetExtent( void * aBucket ) {
	color_bucket_tp	cb	= (color_bucket_tp)aBucket;
	byte_t			v;

	cb->max_extent = (byte_t)(cb->max_r - cb->min_r);

	v = (byte_t)(cb->max_g - cb->min_g);
	if	( v > cb->max_extent )	cb->max_extent = v;

	v = (byte_t)(cb->max_b - cb->min_b);
	if	( v > cb->max_extent )	cb->max_extent = v;
}

// --------------------------------------------------------------------
// private:	Split a bucket
// --------------------------------------------------------------------
void	CPictureFilterQuantize::SplitBucket	(	CPicturePixel * 	aPixmap,
												dword_t				aSize,
												word_tp				aPicture,
												word_t				aOldIndx,
												word_t				aNewIndx,
												void * 				aBucket ) {
	color_bucket_tp	ob	= &(((color_bucket_tp)aBucket)[aOldIndx]);
	color_bucket_tp	nb	= &(((color_bucket_tp)aBucket)[aNewIndx]);
	dword_t			i;
	byte_t			mid;
	byte_t			v;

	CPictureFilterQuantize::ClearBucket( nb );


	if		( ( ob->max_r - ob->min_r >= ob->max_g - ob->min_g ) &&
			  ( ob->max_r - ob->min_r >= ob->max_b - ob->min_b ) ) {

		// Split in red direction.
		mid = ob->mid_r;
		CPictureFilterQuantize::ClearBucket( ob );

		for	( i = 0; i < aSize; i++ ) {
			if	( aPicture[i] == aOldIndx ) {
				if	( aPixmap[i].R() > mid ) {
					aPicture[i] = aNewIndx;
					UPDATE_MINMAX( nb, aPixmap[i] );
				}
				else {
					UPDATE_MINMAX( ob, aPixmap[i] );
				}
			}
		}
	}

	else if	( ob->max_g - ob->min_g >= ob->max_b - ob->min_b ) {

		// Split in green direction.
		mid = ob->mid_g;
		CPictureFilterQuantize::ClearBucket( ob );

		for	( i = 0; i < aSize; i++ ) {
			if	( aPicture[i] == aOldIndx ) {
				if	( aPixmap[i].G() > mid ) {
					aPicture[i] = aNewIndx;
					UPDATE_MINMAX( nb, aPixmap[i] );
				}
				else {
					UPDATE_MINMAX( ob, aPixmap[i] );
				}
			}
		}
	}

	else {

		// Split in blue direction.
		mid = ob->mid_b;
		CPictureFilterQuantize::ClearBucket( ob );

		for	( i = 0; i < aSize; i++ ) {
			if	( aPicture[i] == aOldIndx ) {
				if	( aPixmap[i].B() > mid ) {
					aPicture[i] = aNewIndx;
					UPDATE_MINMAX( nb, aPixmap[i] );
				}
				else {
					UPDATE_MINMAX( ob, aPixmap[i] );
				}
			}
		}
	}

	CPictureFilterQuantize::SetExtent( ob );
	CPictureFilterQuantize::SetExtent( nb );
}

// --------------------------------------------------------------------
// EOF: CPictureFilterQuantize.cxx
// --------------------------------------------------------------------
