// --------------------------------------------------------------------
// CPictureSourceGIF.cxx
// Whatis:  GIF Picture manipulation class
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include	"CError.hxx"
#include	"CPictureSourceGIF.hxx"
#include	"CPictureFilterQuantize.hxx"
#include	"CPictureFilterIndexed.hxx"

extern "C" {
#include	<gif_lib.h>
}

// --------------------------------------------------------------------
// public:	Constructor
// --------------------------------------------------------------------
CPictureSourceGIF::CPictureSourceGIF ( const char * aGifFile ) {
	CPictureSourceGIF::Cleanup();
	itsFile = ::my_private_strdup( aGifFile );
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CPictureSourceGIF::~CPictureSourceGIF() {
	CPictureSourceGIF::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CPictureSourceGIF::CPictureSourceGIF ( const CPictureSourceGIF & aPicture ) {
	CPictureSourceGIF::Cleanup();
	itsFile = ::my_private_strdup( aPicture.File() );
}

// --------------------------------------------------------------------
CPictureSourceGIF & CPictureSourceGIF::operator =	( const CPictureSourceGIF & aPicture ) {
	CPictureSourceGIF::Free();
	itsFile = ::my_private_strdup( aPicture.File() );
	return *this;
}

// --------------------------------------------------------------------
// public:	Interface
// --------------------------------------------------------------------
CPicturePixmap	CPictureSourceGIF::Read	( void ) const {
	GifFileType *			gs		= NULL;
	bool					ready	= false;
	byte_tp					gifline	= NULL;
	CPicturePixmap			pixmap;
	CPicturePixel			pixel;
	GifRecordType			rectype;
	dword_t					num_pixels, i;
	int						ncolors, c;
	GifColorType *			palette;

	try	{
		gs = ::DGifOpenFileName( itsFile );

		if	( ! gs )	CPictureSourceGIF::Result( GIF_ERROR );

		// ------------------------------------------------------------
		// Read in image descriptor - the next should be image data
		// ------------------------------------------------------------
		while	( ! ready ) {
			CPictureSourceGIF::Result( ::DGifGetRecordType( gs, &rectype ) );

			switch	( rectype ) {
				case	UNDEFINED_RECORD_TYPE:
				throw	CError( itsFile, "Unknown record type" );

				case	SCREEN_DESC_RECORD_TYPE:
				throw	CError( itsFile, "Too many screen descriptors" );

				case	IMAGE_DESC_RECORD_TYPE:
				CPictureSourceGIF::Result( ::DGifGetImageDesc( gs ) );
				ready = true;
				break;

				case	EXTENSION_RECORD_TYPE:
				case	TERMINATE_RECORD_TYPE:
				throw	CError( itsFile, "Can't manage extensions" );
			}
		}

		// Allocate enough data for the image
		num_pixels = gs->Image.Width * gs->Image.Height;

		gifline = new byte_t [num_pixels];

		// Read in teh image
		CPictureSourceGIF::Result( ::DGifGetLine( gs, gifline, num_pixels ) );

		// ------------------------------------------------------------
		// Allocate enuf room to the result and make the conversion
		// ------------------------------------------------------------
		pixmap.Create( gs->Image.Width, gs->Image.Height );
		palette 	= gs->Image.ColorMap->Colors;
		ncolors		= gs->Image.ColorMap->ColorCount;
		pixel.A( 0 );

		for	( i = 0; i < num_pixels; i++ ) {
			c = gifline[i];

			if	( c >= ncolors )	c = ncolors - 1;

			pixel.R( palette[c].Red );
			pixel.G( palette[c].Green );
			pixel.B( palette[c].Blue );
			pixmap.Pixel( i, pixel );
		}

		delete [] gifline;
		::DGifCloseFile( gs );
	}


	catch	( CError e ) {
		if	( gifline ) delete [] gifline;
		if	( gs )	::DGifCloseFile( gs );
		throw	CError( "CPictureSourceGIF::Read", e.Error() );
	}

	catch	( ... ) {
		if	( gifline ) delete [] gifline;
		if	( gs )	::DGifCloseFile( gs );
		throw	CError( "CPictureSourceGIF::Read", "unknown" );
	}

	return	pixmap;
}

// --------------------------------------------------------------------
void			CPictureSourceGIF::Write	( const CPicturePixmap & aPixmap ) {
	CPicturePixmap			pixmap;
	CPictureFilterQuantize	quantize( 256 );
	CPictureFilterIndexed	indexed;
	GifFileType *			gs		= NULL;
	ColorMapObject			palette;
	byte_tp					gifline	= NULL;
	dword_t					i;
	dword_t					num_pixels	= aPixmap.H() * aPixmap.W();

	::memset( &palette, 0, sizeof( palette ) );

	try {
		// Quantize teh image
		pixmap = aPixmap;
		quantize.Process( pixmap );
		indexed.Process( pixmap );

		// Create the palette
		palette.Colors 		= new GifColorType [ 256 ];
		palette.ColorCount	= 256;
		palette.BitsPerPixel= 8;
		for	( i = 0; i < 256; i++ ) {
			if	( i < indexed.Colors() ) {
				palette.Colors[i].Red 	= (indexed.Palette())[i].R();
				palette.Colors[i].Green = (indexed.Palette())[i].G();
				palette.Colors[i].Blue	= (indexed.Palette())[i].B();
			}
			else {
				palette.Colors[i].Red 	= 0;
				palette.Colors[i].Green = 0;
				palette.Colors[i].Blue	= 0;
			}
		}

		// Create the GIF file
		gs = ::EGifOpenFileName( itsFile, FALSE );
		if	( ! gs )	CPictureSourceGIF::Result( GIF_ERROR );

		// Initialize the file
		CPictureSourceGIF::Result( ::EGifPutScreenDesc( gs, aPixmap.W(), aPixmap.H(), 256, 0, &palette ) );
		CPictureSourceGIF::Result( ::EGifPutImageDesc( gs, 0, 0, aPixmap.W(), aPixmap.H(), 0, &palette ) );

		// Write image data
		gifline = new byte_t [num_pixels];
		for	( i = 0; i < num_pixels; i++ ) {
			gifline[i] = (byte_t)(indexed.Image()[i]);
		}
		CPictureSourceGIF::Result( ::EGifPutLine( gs, gifline, num_pixels ) );

		// Close the file (finish)
		delete [] gifline; gifline = NULL;
		int	code = ::EGifCloseFile( gs );
		gs = NULL;
		CPictureSourceGIF::Result( code );
		CPictureSourceGIF::FreePalette( &palette );
	}

	catch ( CError e ) {
		if	( gifline ) delete [] gifline;
		if	( gs )	::EGifCloseFile( gs );
		CPictureSourceGIF::FreePalette( &palette );
		throw CError( "CPictureSourceGIF::Write", e.Error() );
	}

	catch ( ... ) {
		if	( gifline ) delete [] gifline;
		if	( gs )	::EGifCloseFile( gs );
		CPictureSourceGIF::FreePalette( &palette );
		throw CError( "CPictureSourceGIF::Write", "unknown" );
	}
}

// --------------------------------------------------------------------
void			CPictureSourceGIF::File	( const char * aFile ) {
	if	( itsFile )	delete [] itsFile;
	itsFile = ::my_private_strdup( aFile );
}

// --------------------------------------------------------------------
// private:	Helpers
// --------------------------------------------------------------------
void	CPictureSourceGIF::Result		( int aError ) const {
	if	( aError != GIF_OK ) {
		switch	( ::GifLastError() ) {
			case	E_GIF_ERR_OPEN_FAILED:		throw CError( itsFile, "Open failure" );
			case	E_GIF_ERR_WRITE_FAILED:		throw CError( itsFile, "Write failure" );
			case	E_GIF_ERR_HAS_SCRN_DSCR:	throw CError( itsFile, "Has screen descriptor" );
			case	E_GIF_ERR_HAS_IMAG_DSCR:	throw CError( itsFile, "Has image descriptor" );
			case	E_GIF_ERR_NO_COLOR_MAP:		throw CError( itsFile, "No color map" );
			case	E_GIF_ERR_DATA_TOO_BIG:		throw CError( itsFile, "Data too big" );
			case	E_GIF_ERR_NOT_ENOUGH_MEM:	throw CError( itsFile, "Out of memory" );
			case	E_GIF_ERR_DISK_IS_FULL:		throw CError( itsFile, "Disk full" );
			case	E_GIF_ERR_CLOSE_FAILED:		throw CError( itsFile, "Close failed" );
			case	E_GIF_ERR_NOT_WRITEABLE:	throw CError( itsFile, "Not writeable" );
			case	D_GIF_ERR_OPEN_FAILED:		throw CError( itsFile, "Open failure" );
			case	D_GIF_ERR_READ_FAILED:		throw CError( itsFile, "Unable to read" );
			case	D_GIF_ERR_NOT_GIF_FILE:		throw CError( itsFile, "Not a GIF file" );
			case	D_GIF_ERR_NO_SCRN_DSCR:		throw CError( itsFile, "Missing screen descriptor" );
			case	D_GIF_ERR_NO_IMAG_DSCR:		throw CError( itsFile, "Missing image descriptor" );
			case	D_GIF_ERR_NO_COLOR_MAP:		throw CError( itsFile, "Missing color map" );
			case	D_GIF_ERR_WRONG_RECORD:		throw CError( itsFile, "Wrong record" );
			case	D_GIF_ERR_DATA_TOO_BIG:		throw CError( itsFile, "Too much data" );
			case	D_GIF_ERR_NOT_ENOUGH_MEM:	throw CError( itsFile, "Out of memory" );
			case	D_GIF_ERR_CLOSE_FAILED:		throw CError( itsFile, "Close failed" );
			case	D_GIF_ERR_NOT_READABLE:		throw CError( itsFile, "Unable to read" );
			case	D_GIF_ERR_IMAGE_DEFECT:		throw CError( itsFile, "Defective image" );
			case	D_GIF_ERR_EOF_TOO_SOON:		throw CError( itsFile, "Not enough data" );
			default:							throw CError( itsFile, "Unknown error" );
		}
	}
}

// --------------------------------------------------------------------
void	CPictureSourceGIF::FreePalette	( void * aPalette ) {
	ColorMapObject *	p = (ColorMapObject *)aPalette;

	if	( p->Colors )	delete [] p->Colors;
	::memset( p, 0, sizeof( ColorMapObject ) );
}

// --------------------------------------------------------------------
void	CPictureSourceGIF::Cleanup	( void ) {
	itsFile = NULL;
}

// --------------------------------------------------------------------
void	CPictureSourceGIF::Free	( void ) {
	if	( itsFile )	delete [] itsFile;
	CPictureSourceGIF::Cleanup();
}

// --------------------------------------------------------------------
// EOF: CPictureSourceGIF.cxx
// --------------------------------------------------------------------
