// --------------------------------------------------------------------
// CPictureSourceJPG.cxx
// Whatis:  JPG Picture manipulation class
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include	"CError.hxx"
#include	"CPictureSourceJPG.hxx"

// --------------------------------------------------------------------
// NOTE!!!!!
// If linking fails - You have to put the include into an extern "C"
// block !
// --------------------------------------------------------------------
// extern "C" {
#include	<jpeglib.h>
// }

// --------------------------------------------------------------------
// local:       Static jpeg error handlers
// --------------------------------------------------------------------
static void __output_jpeg_error( j_common_ptr cinfo ) {
    char buffer [JMSG_LENGTH_MAX];
    (*cinfo->err->format_message)( cinfo, buffer );
    ::jpeg_destroy( cinfo );
    throw CError( buffer );
}

static void __error_exit( j_common_ptr cinfo ) {
    char buffer [JMSG_LENGTH_MAX];
    (*cinfo->err->format_message)( cinfo, buffer );
    ::jpeg_destroy( cinfo );
    throw CError( buffer );
}

// --------------------------------------------------------------------
// public:	Constructors
// --------------------------------------------------------------------
CPictureSourceJPG::CPictureSourceJPG ( ) {
	CPictureSourceJPG::Cleanup();
}

// --------------------------------------------------------------------
CPictureSourceJPG::CPictureSourceJPG ( const char * aJpgFile, word_t aQuality ) {
	CPictureSourceJPG::Cleanup();
   	itsFile 	= ::my_private_strdup( aJpgFile );
   	itsQuality	= aQuality;
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CPictureSourceJPG::~CPictureSourceJPG() {
	CPictureSourceJPG::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CPictureSourceJPG::CPictureSourceJPG	( const CPictureSourceJPG & aPicture ) {
	Cleanup();
   	itsFile 	= ::my_private_strdup( aPicture.File() );
   	itsQuality	= aPicture.Quality();
}

// --------------------------------------------------------------------
CPictureSourceJPG & CPictureSourceJPG::operator =	( const CPictureSourceJPG & aPicture ) {
	Free();
   	itsFile 	= ::my_private_strdup( aPicture.File() );
   	itsQuality	= aPicture.Quality();
	return *this;
}

// --------------------------------------------------------------------
// public:	Interface
// --------------------------------------------------------------------
CPicturePixmap	CPictureSourceJPG::Read	( void ) const {
	CPicturePixmap					pixmap;
	FILE *							jpgstream	= NULL;
    JSAMPLE *                       imgbuf    	= NULL;
    JSAMPROW						rowbuf[1];
    struct  jpeg_decompress_struct  cinfo;
    struct  jpeg_error_mgr          jerr;
    dword_t							x, y;
    char *							p;
    CPicturePixel					pixel;

    try {
        // Open up teh file for writing - we use stdout if no file present
		if	( itsFile ) {
	        jpgstream = ::fopen( itsFile, "rb" );
	        if	( ! jpgstream ) {
				throw CError( ::strerror( errno ) );
	        }
		}
		else {
			jpgstream = stdin;
		}

        // ------------------------------------------------------------
        //  Set up the error handler and initialize the compression system
        // ------------------------------------------------------------
        cinfo.err                 = ::jpeg_std_error( &jerr );
        cinfo.err->output_message = __output_jpeg_error;
        cinfo.err->error_exit     = __error_exit;

        ::jpeg_create_decompress( &cinfo );

        // ------------------------------------------------------------
        // Set up the source
        // ------------------------------------------------------------
		::jpeg_stdio_src( &cinfo, jpgstream );

        // ------------------------------------------------------------
        // Read in teh header
        // ------------------------------------------------------------
		switch	( ::jpeg_read_header( &cinfo, FALSE ) ) {
			case	JPEG_SUSPENDED:				throw CError( "Suspended due to lack of input data" );
			case	JPEG_HEADER_TABLES_ONLY:	throw CError( "No image data" );
			case	JPEG_HEADER_OK:				break;
			default:							throw CError( ::strerror( errno ) );
		}

        // ------------------------------------------------------------
		// Set up the sample buffer
        // ------------------------------------------------------------
        imgbuf =	new JSAMPLE [	cinfo.image_width
        						  * cinfo.image_height
        						  * cinfo.num_components ];

        // ------------------------------------------------------------
		// Set up our data format
        // ------------------------------------------------------------
		cinfo.out_color_space	= JCS_RGB;

        // ------------------------------------------------------------
		// Start reading the image
        // ------------------------------------------------------------
		::jpeg_start_decompress( &cinfo );

        // ------------------------------------------------------------
		// Read the image in
        // ------------------------------------------------------------
		while ( cinfo.output_scanline < cinfo.output_height ) {
			rowbuf[0] = imgbuf + ( cinfo.output_scanline * cinfo.image_width * cinfo.num_components);
			::jpeg_read_scanlines( &cinfo, rowbuf, 1 );
		}
        ::jpeg_finish_decompress( &cinfo );

        // ------------------------------------------------------------
		// Put image into the pixmap
        // ------------------------------------------------------------
		pixmap.Create( cinfo.image_width, cinfo.image_height );
		p = (char *)imgbuf;
		for	( y = 0; y < pixmap.H(); y++ ) {
			for	( x = 0; x < pixmap.W(); x++ ) {
				pixel.R( *(p++) );
				pixel.G( *(p++) );
				pixel.B( *(p++) );
				pixel.A( 0 );
				pixmap.Pixel( x, y, pixel );
			}
		}

        // ------------------------------------------------------------
		// Free up memory
        // ------------------------------------------------------------
		delete []	imgbuf;
		if	( jpgstream != stdin ) {
			::fclose( jpgstream );
		}
    }

    catch   ( CError e ) {
		if	( imgbuf )	delete []	imgbuf;
		if	( jpgstream ) {
			if	( jpgstream != stdin ) {
				::fclose( jpgstream );
			}
		}
        throw CError( itsFile ? itsFile : "stdin", e.Error() );
    }

    catch   ( ... ) {
		if	( imgbuf )	delete []	imgbuf;
		if	( jpgstream ) {
			if	( jpgstream != stdin ) {
				::fclose( jpgstream );
			}
		}
        throw CError( itsFile ? itsFile : "stdin", "Unknown error" );
    }

	return	pixmap;
}

// --------------------------------------------------------------------
void			CPictureSourceJPG::Write	( const CPicturePixmap & aPixmap ) {
	FILE *							jpgstream	= NULL;
    JSAMPLE *                       imgbuf		= NULL;
    JSAMPROW                        rowbuf[1];
    struct  jpeg_compress_struct    cinfo;
    struct  jpeg_error_mgr          jerr;

    dword_t							x, y;
    char *							p;
    CPicturePixel					pixel;

    try {

        // ------------------------------------------------------------
		// Set up the sample and row buffers
        // ------------------------------------------------------------
        imgbuf = new JSAMPLE 	[ aPixmap.H() * aPixmap.W() * 3 ];
		p = (char *)imgbuf;
		for	( y = 0; y < aPixmap.H(); y++ ) {
			for	( x = 0; x < aPixmap.W(); x++ ) {
				pixel = aPixmap.Pixel( x, y );
				*(p++) = pixel.R();
				*(p++) = pixel.G();
				*(p++) = pixel.B();
			}
		}

        // ------------------------------------------------------------
        //  Set up the error handler and initialize the compression system
        // ------------------------------------------------------------
        cinfo.err                 = ::jpeg_std_error( &jerr );
        cinfo.err->output_message = __output_jpeg_error;
        cinfo.err->error_exit     = __error_exit;
        ::jpeg_create_compress( &cinfo );

        // ------------------------------------------------------------
        // Set up the destination
        // ------------------------------------------------------------
		if	( itsFile ) {
	        jpgstream = ::fopen( itsFile, "wb" );
	        if	( ! jpgstream ) {
				throw CError( ::strerror( errno ) );
	        }
		}
		else {
			jpgstream = stdout;
		}
		::jpeg_stdio_dest( &cinfo, jpgstream );

        // ------------------------------------------------------------
        // Set parameters for compression
        // ------------------------------------------------------------
        cinfo.image_width       = aPixmap.W();  // image width and height, in pixels
        cinfo.image_height      = aPixmap.H();
        cinfo.input_components  = 3;	   		// # of color components per pixel
        cinfo.in_color_space	= JCS_RGB;

        ::jpeg_set_defaults(&cinfo);
        ::jpeg_set_quality( &cinfo, itsQuality, TRUE );

        // ------------------------------------------------------------
        // Do the compression
        // ------------------------------------------------------------
        ::jpeg_start_compress(&cinfo, TRUE );
		while	( cinfo.next_scanline < cinfo.image_height ) {
			rowbuf[0] = imgbuf + ( cinfo.next_scanline * cinfo.image_width * cinfo.input_components );
			::jpeg_write_scanlines( &cinfo, rowbuf, 1 );
		}
        ::jpeg_finish_compress(&cinfo);

        ::jpeg_destroy_compress(&cinfo);
		delete [] imgbuf;
		if	( jpgstream != stdout ) {
			::fclose( jpgstream );
		}
    }

    catch   ( CError e ) {
		if	( imgbuf )	delete [] imgbuf;
		if	( jpgstream ) {
			if	( jpgstream != stdout ) {
				::fclose( jpgstream );
			}
		}
        throw CError( itsFile ? itsFile : "stdout", e.Error() );
	}

    catch   ( ... ) {
		if	( imgbuf )	delete [] imgbuf;
		if	( jpgstream ) {
			if	( jpgstream != stdout ) {
				::fclose( jpgstream );
			}
		}
        throw CError( itsFile ? itsFile : "stdout", "Unknown error" );
    }
}

// --------------------------------------------------------------------
void			CPictureSourceJPG::File	( const char * aFile ) {
	if	( itsFile )	delete [] itsFile;
	itsFile = ::my_private_strdup( aFile );
}

// --------------------------------------------------------------------
// private:	Helpers
// --------------------------------------------------------------------
void	CPictureSourceJPG::Cleanup	( void ) {
	itsFile 	= NULL;
	itsQuality	= 50;
}

// --------------------------------------------------------------------
void	CPictureSourceJPG::Free		( void ) {
	if	( itsFile )	delete [] itsFile;
	CPictureSourceJPG::Cleanup();
}

// --------------------------------------------------------------------
// EOF: CPictureSourceJPG.cxx
// --------------------------------------------------------------------
