// --------------------------------------------------------------------
// CTcpSocket.hxx
// Whatis:  The socket class
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 01-FEB-2003     Created this source
// --------------------------------------------------------------------
#ifndef CTcpSocket_hxx
#define CTcpSocket_hxx
#include    "Platform.hxx"

// -----------------------------------------------------------------------
// Including system files
// -----------------------------------------------------------------------
#ifdef  WIN32               // ##### Windows platform inclusions
#include	<winsock.h>
typedef		int	socklen_t;

#else
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<netdb.h>
#include	<netinet/in.h>
#include	<arpa/inet.h>

#endif

// -----------------------------------------------------------------------
// Data type declarations and/or definitions
// -----------------------------------------------------------------------
#ifdef  WIN32
typedef SOCKET          SOCKET_t;
#else
typedef int				SOCKET_t;
#endif

// -----------------------------------------------------------------------
// Define the socket data type and the invalid socket
// -----------------------------------------------------------------------
#define SOCKET_NONE         ((SOCKET_t)(-1))

// -----------------------------------------------------------------------
// Socket states
// -----------------------------------------------------------------------
#define SOCKET_STATE_IDLE       (0)     // Just inited, idling
#define SOCKET_STATE_PROGRESS   (1)     // Connection under progress
#define SOCKET_STATE_CONNECT    (2)     // Connection ready
#define SOCKET_STATE_ERROR      (3)     // Entered an error state

// -----------------------------------------------------------------------
// Socket read results
// -----------------------------------------------------------------------
#define SOCKET_READ_ERROR       (-3)    // Read caused an error
#define SOCKET_READ_DISCONNECT  (-2)    // Socket was disconnected
#define SOCKET_READ_IDLE        (-1)    // Socket OK but nothing was read

// -----------------------------------------------------------------------
// This enumeration is just to ensure proper calling of constructor
// -----------------------------------------------------------------------
typedef enum {
    socket_type_stream      = SOCK_STREAM,
    socket_type_datagram    = SOCK_DGRAM
}   socket_type_t;

// -----------------------------------------------------------------------
// The CTcpSocket Class
// -----------------------------------------------------------------------
class   CTcpSocket {

    // -------------------------------------------------------------------
    public:     // Constructor and destructor
    // -------------------------------------------------------------------
    CTcpSocket( socket_type_t );
    CTcpSocket( SOCKET_t );
    ~CTcpSocket( void );

    // -------------------------------------------------------------------
    private:    // Copy constructor and assignment operator overload
    // -------------------------------------------------------------------
    CTcpSocket  ( const CTcpSocket& );
    CTcpSocket& operator = ( const CTcpSocket & );

    // -------------------------------------------------------------------
    public:     // Bind the socket to:
    // - select port automatically
    // - use a spesific named service
    // - use a spesific port
    // -------------------------------------------------------------------
    void            Bind        ( void );
    void            Bind        ( const char * );
    void            Bind        ( unsigned short, CTcpSocket * = NULL );

    // -------------------------------------------------------------------
    public:     // Set listen mode on this socket
    // -------------------------------------------------------------------
    void            Listen      ( void );

    // -------------------------------------------------------------------
    public:     // Accept a connection from a listening socket
    // The function returns NULL if no connection accepted
    // -------------------------------------------------------------------
    CTcpSocket *    Accept      ( void );

    // -------------------------------------------------------------------
    public:     // Connect the socket to a server
    // -------------------------------------------------------------------
    bool            Connect     ( const char *, unsigned short );

    // -------------------------------------------------------------------
    public:     // Read one byte from a socket. The return values are:
    // 0 ... 255    A valid byte
    // -1           Nothing was read
    // -2           Connection was closed
    // -------------------------------------------------------------------
    int             Read        ( void );

    // -------------------------------------------------------------------
    public:     // Diverse write methods
    // -------------------------------------------------------------------
    void            Write       ( const char *, int );
    void            Write       ( const char * );
    void            Write       ( char );
    void            Write       ( short aS );
    void            Write       ( dword_t aUs );
    void            Write       ( unsigned short aUs );
    void            Write       ( unsigned long aUl );
    void            Write       ( long );

    // -------------------------------------------------------------------
	public:     // Flush output buffer into line
    // -------------------------------------------------------------------
	void			Flush		( void );

    // -------------------------------------------------------------------
    public:     // Accessing instance data
    // -------------------------------------------------------------------
    const char *    HostName    ( void );
    const char *    HostAddr    ( void );
    const char *    PeerName    ( void );
    const char *    PeerAddr    ( void );
    unsigned short  HostPort    ( void );
    unsigned short  PeerPort    ( void );
    const char *    HostService ( void );
	SOCKET_t		Socket		( void );
	socket_type_t	Type		( void );

    // ------------------------------------------------------------
    private:    // Private helpers
    // ------------------------------------------------------------
    void            ChkError    ( int );
    void            SetHost     ( void );
    void            SetPeer     ( void );
#ifdef WIN32
    void            EnterWinSock( void );
    void            ExitWinSock ( void );
#endif

    // ------------------------------------------------------------
    private:    // Instance data
    // ------------------------------------------------------------
#ifdef  WIN32
    int             itsWsaFound;
#endif
    int             itsState;
    SOCKET_t        itsSocket;
	socket_type_t	itsType;
    char *          itsHostName;
    char *          itsHostAddr;
    char *          itsPeerName;
    char *          itsPeerAddr;
    unsigned short  itsHostPort;
    unsigned short  itsPeerPort;
    char *          itsHostService;
	char			itsWriteBuff[16384];
	char			itsReadBuff	[32768];
	dword_t			itsWriteOffs;
	dword_t			itsReadOffs;
	dword_t			itsReadSize;
    bool            itsConnect;
};

#endif  // #ifndef CTcpSocket_hxx

// ----------------------------------------------------------------
// EOF: CTcpSocket.hxx
// ----------------------------------------------------------------
