// --------------------------------------------------------------------
// PlatDir.cxx
// Whatis:  Getting directory lists
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "Platform.hxx"
#include    "CError.hxx"

// --------------------------------------------------------------------
// Get directory listing
// --------------------------------------------------------------------
extern  dir_file_tl my_private_dir  ( const char * aDir, bool aAll ) {
    char    cmdbuf  [2048];
    FILE *  mypipe  = NULL;
#ifdef	WIN32
    struct  _stat   mystat;
#else
    struct  stat    mystat;
#endif
    dir_file_tl     filelist;

    // ----------------------------------------------------------------
    // use the ls command to retrieve a list of files
    // ----------------------------------------------------------------
    ::sprintf( cmdbuf, "ls -1 \"%s\"", aDir );
#ifdef	WIN32
    mypipe = ::_popen( cmdbuf, "r" );
#else
    mypipe = ::popen( cmdbuf, "r" );
#endif
    if  ( ! mypipe )    return  filelist;

    while   ( ! feof( mypipe ) ) {
        cmdbuf[0] = 0;
        if  ( ::fgets( cmdbuf, sizeof( cmdbuf ), mypipe ) ) {
            ::my_strtail( ::my_strhead( cmdbuf) );
            if  ( cmdbuf[0] != 0 ) {
                dir_file_t  item;
                ::memset( &item, 0, sizeof( item ) );
                ::sprintf( item.name, "%s/%s", aDir, cmdbuf );
#ifdef	WIN32
                if  ( ! ::_stat( item.name, &mystat ) ) {
                	if		( ( mystat.st_mode & _S_IFREG ) != 0 ) {
                        filelist.push_back( item );
                    }
                    else if	( ( mystat.st_mode & _S_IFDIR ) != 0 ) {
                    	if	( aAll ) {
	                        filelist.push_back( item );
                    	}
                    }
#else
                if  ( ! ::stat( item.name, &mystat ) ) {
                	if		( ( mystat.st_mode & __S_IFMT ) == __S_IFREG ) {
                        filelist.push_back( item );
                    }
                    else if	( ( mystat.st_mode & __S_IFMT ) == __S_IFDIR ) {
                    	if	( aAll ) {
	                        filelist.push_back( item );
                    	}
                    }
#endif
                }
            }
        }
        else if  ( ferror( mypipe ) ) {
            ::fflush( mypipe );
#ifdef	WIN32
            ::_pclose( mypipe );
#else
            ::pclose( mypipe );
#endif
            filelist.clear();
            return  filelist;
        }
    }
    ::fflush( mypipe );
#ifdef	WIN32
    ::_pclose( mypipe );
#else
    ::pclose( mypipe );
#endif
    return  filelist;
}


// --------------------------------------------------------------------
// Extract the filename (strip the path away)
// --------------------------------------------------------------------
extern  void    my_extract_file	( char * aFname, size_t aLen, const char * aPathname ) {
    const char *    ps = ::strrchr( aPathname, '/' );
    ::my_strfit( aFname, aLen, ps ? ps + 1 : aPathname );
}

// --------------------------------------------------------------------
// Strip suffix from file name
// --------------------------------------------------------------------
extern  void    my_strip_suffix	( char * aFilename ) {
	if	( aFilename ) {
		if	( ::strrchr( aFilename, '.' ) ) {
			*::strrchr( aFilename, '.' ) = 0;
		}
	}
}

// --------------------------------------------------------------------
// public:      Load a file into memory
// --------------------------------------------------------------------
extern  char *  my_load_file        ( const char *  aFileName ) {
    size_t  size    = 0;
    FILE *  strm    = NULL;
    char *  data    = NULL;

    try {
        strm = ::fopen( aFileName, "r" );
        if  ( ! strm )  throw CError( aFileName, ::strerror( errno ) );

        ::fseek( strm, 0, SEEK_END );
        if  ( ferror( strm ) )  throw CError( aFileName, ::strerror( errno ) );

        size = ::ftell( strm );
        ::fseek( strm, 0, SEEK_SET );

        data = new char [size + 1];

        ::fread( data, size, 1, strm );
        data[size] = 0;

        if  ( ferror( strm ) )  throw CError( aFileName, ::strerror( errno ) );
        ::fclose( strm );
        strm = NULL;
    }
    
    catch   ( ... ) {
        if  ( data )    delete [] data;
        if  ( strm )    ::fclose( strm );
        throw;
    }

    return  data;
}

// --------------------------------------------------------------------
// Make a zip
// --------------------------------------------------------------------
extern	void	my_make_zip	(	const char *	aZipName,
								const char *	aSource ) {
	struct stat	mystat;
	char		command	[2048];

	// Probe the name
	if		( ::stat( aSource, &mystat ) ) {
		throw CError( aSource, ::strerror( errno ) );
	}

	// Is it a file ?
#ifdef	WIN32
	else if	( ( mystat.st_mode & _S_IFREG ) != 0 ) {
#else
	else if	( ( mystat.st_mode & __S_IFMT ) == __S_IFREG ) {
#endif
		::sprintf( 	command, "zip -qqjn .Z:.zip:.tiff:.gif:.snd:.jpg:.jpeg \"%s\" \"%s\"",
					aZipName, aSource );
    }

	// Is it a directory ?
#ifdef	WIN32
    else if	( ( mystat.st_mode & _S_IFDIR ) != 0 ) {
#else
    else if	( ( mystat.st_mode & __S_IFMT ) == __S_IFDIR ) {
#endif
		::sprintf( 	command, "cd \"%s\"; zip -qqRn .Z:.zip:.tiff:.gif:.snd:.jpg:.jpeg \"%s\" *.*",
					aSource, aZipName );
    }

	// Needs either a directory or a file name
    else {
		throw CError( aSource, "Not a regular file or a directory" );
    }

	::system( command );
}

// --------------------------------------------------------------------
// EOF: PlatDir.cxx
// --------------------------------------------------------------------
