// --------------------------------------------------------------------
// CUnSound.cxx
// Whatis:  Unreal Texture
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CUnProperty.hxx"
#include    "CUnSound.hxx"

// --------------------------------------------------------------------
// public:      Constructors
// --------------------------------------------------------------------
CUnSound::CUnSound (	CUnFile_t &					aUf,
						const CUnNameTable_t &		aUn,
						const CUnExportTable_t &	aUe,
						const CUnImportTable_t &	aUi,
						const char *				aSound,
						bool						aLoad ) {
    CUnSound::Cleanup();

	// No sound classes in the file ?
	if	( ! aUn.Exist( "sound" ) )	throw CError( aSound, "sound class not found" );
	if	( ! aUn.Exist( aSound ) )	throw CError( aSound, "sound object not found" );

	// Adjust the file into place
	dword_t		classnmix	= aUn.Find( "sound" );
	dword_t		objecnmix	= aUn.Find( aSound );
	dword_t		eix;
	CUnExport	exp;

	for	( eix = 0; eix < aUe.Count(); eix++ ) {
		exp = aUe.Export( eix );

		// Skip those whose name does not match
		if	( objecnmix != exp.ObjectName() )	continue;

		// Skip those that are not of this class
		if	( classnmix != aUi.Import( NAMEIMPORT( exp.Class() ) ).ObjectName() )	continue;

		break;	// Found it !
	}
	if	( eix >= aUe.Count() )	throw CError( aSound, "sound object not found" );

	// Seek teh file into place
	aUf.Seek( exp.SerialOffset() );

	// Read the properties (this is an object after all)
    CUnProperty property;
    do  {
        property = CUnProperty( aUf, aUn, aUi, aUe );
    }   while   ( ::strcmp( "none", aUn.LcName( property.Name() ) ) );

	// UT2k3 packages
	if		( aUf.PackageVersion() >= 120 ) {
		::my_strfit( itsFormat, sizeof( itsFormat ), aUn.Name( aUf.ReadIndex() ) );
		aUf.ReadDword();	// Flags
		aUf.ReadDword();	// Aux dword ?
		itsSize = aUf.ReadIndex();
	}

	// AAO packages
	else if	( aUf.PackageVersion() >= 100 ) {
		aUf.ReadDword();
		::my_strfit( itsFormat, sizeof( itsFormat ), aUn.Name( aUf.ReadIndex() ) );
		aUf.ReadDword();
		itsSize = aUf.ReadIndex();
	}

	// UT Packages
	else if	( aUf.PackageVersion() >= 63 ) {
		::my_strfit( itsFormat, sizeof( itsFormat ), aUn.Name( aUf.ReadIndex() ) );
		aUf.ReadDword();
		itsSize = aUf.ReadIndex();
	}

	// Ancient packages
	else {
		::my_strfit( itsFormat, sizeof( itsFormat ), aUn.Name( aUf.ReadIndex() ) );
		itsSize = aUf.ReadIndex();
	}

	if	( ( itsSize < 20 ) || ( itsSize > 50000000 ) ) {
		throw CError( aSound, "Invalid data size" );
	}

	// If this is a WAV formatted data we fill in the WAV header
	if	( ! ::my_stricmp( "WAV", itsFormat ) ) {
		dword_t	rmboff = aUf.Offset();

		aUf.ReadRawData( itsWavHead.riff, 4 );
		itsWavHead.totalbytes = aUf.ReadDword();
		aUf.ReadRawData( itsWavHead.wave, 4 );
		aUf.ReadRawData( itsWavHead.fmt_, 4 );
		itsWavHead.formbytes	= aUf.ReadDword();
		itsWavHead.filler 		= aUf.ReadWord();
		itsWavHead.channels		= aUf.ReadWord();
		itsWavHead.samplerate	= aUf.ReadDword();
		itsWavHead.byterate		= aUf.ReadDword();
		itsWavHead.samplebytes	= aUf.ReadWord();
		itsWavHead.samplebits	= aUf.ReadWord();

		if	( ::memcmp( itsWavHead.riff, "RIFF", 4 ) ) {
			throw CError( aSound, "Invalif WAV format" );
		}

		aUf.Seek( rmboff );
	}


	// Get the data itself
	if	( aLoad ) {
		itsData = new char [itsSize];
		aUf.ReadRawData( itsData, itsSize );
	}
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnSound::~CUnSound () {
    CUnSound::Free();
}

// --------------------------------------------------------------------
// public:      Copy constructor
// --------------------------------------------------------------------
CUnSound::CUnSound( const CUnSound & aS ) {
    CUnSound::Cleanup();
	*this = aS;
}

// --------------------------------------------------------------------
// public:		Assignment operator
// --------------------------------------------------------------------
CUnSound & CUnSound::operator = ( const CUnSound & aS ) {
    CUnSound::Free();
	itsSize = aS.Size();
	if	( aS.Data() ) {
		itsData = new char [itsSize];
		::memcpy( itsData, aS.Data(), itsSize );
	}
	::my_strfit( itsFormat, sizeof( itsFormat ), aS.Format() );
	return	*this;
}

// --------------------------------------------------------------------
// private: Clean up the mess
// --------------------------------------------------------------------
void    CUnSound::Cleanup	( void ) {
	itsSize = 0;
	itsData = NULL;
	::memset( itsFormat, 0, sizeof( itsFormat ) );
	::memset( &itsWavHead, 0, sizeof( itsWavHead ) );
}

// --------------------------------------------------------------------
// private:	Free allocated resource
// --------------------------------------------------------------------
void    CUnSound::Free		( void ) {
	if	( itsData )	delete [] itsData;
    CUnSound::Cleanup();
}

// --------------------------------------------------------------------
const char *		CUnSound::Data		( void )	const { return itsData; }
const char *		CUnSound::Format	( void )	const { return itsFormat; }
dword_t				CUnSound::Size		( void )	const { return itsSize; }
WavFileHeader_t	&	CUnSound::WavHead	( void )		  { return itsWavHead; }

// --------------------------------------------------------------------
// EOF: CUnSound.cxx
// --------------------------------------------------------------------
