// --------------------------------------------------------------------
// CUnTexture.cxx
// Whatis:  Unreal Texture
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CUnProperty.hxx"
#include    "CUnTexture.hxx"

// --------------------------------------------------------------------
// public:      Constructors
// --------------------------------------------------------------------
CUnTexture::CUnTexture (	CUnFile_t &					aUf,
							const CUnNameTable_t &		aUn,
							const CUnExportTable_t &	aUe,
							const CUnImportTable_t &	aUi ) {
	try	{
	    CUnTexture::Cleanup();
	    CUnTexture::ReadImageData ( aUf, aUn, aUe, aUi );
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::CUnTexture", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::CUnTexture", "unknown" );
	}
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnTexture::~CUnTexture () {
}

// --------------------------------------------------------------------
// private:     Read in image data
// --------------------------------------------------------------------
void    CUnTexture::ReadImageData(	CUnFile_t & 				aUf,
									const CUnNameTable_t &		aUn,
									const CUnExportTable_t &	aUe,
									const CUnImportTable_t &	aUi ) {

	try {
	    // ------------------------------------------------------------
	    // First we read the properties for the texture
	    // ------------------------------------------------------------
	    dword_t     n_format        = aUn.Find( "format" );
	    dword_t     n_compformat    = aUn.Find( "compformat" );
	    dword_t     n_palette       = aUn.Find( "palette" );
	    dword_t     n_usize         = aUn.Find( "usize" );
	    dword_t     n_vsize         = aUn.Find( "vsize" );
	    dword_t     n_uclamp        = aUn.Find( "uclamp" );
	    dword_t     n_vclamp        = aUn.Find( "vclamp" );
	    dword_t     n_alphatexture  = aUn.Find( "balphatexture" );
	    dword_t     n_maxcolor      = aUn.Find( "maxcolor" );
	    dword_t     n_mipmaps       = aUn.Find( "mipmaps" );
	    dword_t     n_normallod     = aUn.Find( "normallod" );
	    dword_t     n_minlod        = aUn.Find( "minlod" );
	    CUnProperty property;
	    byte_t      format          = 0;
	    byte_t      compformat      = 0;
	    word_t      mipmaps;
	    dword_t     palette         = 0;
	    dword_t     blocksize       = 0;
	    bool        alphatexture    = false;

	    do  {
	        property = CUnProperty( aUf, aUn, aUi, aUe );
	        property.Rewind();

	        if      ( property.Name() == n_format )         format          = property.GetByte();
	        else if ( property.Name() == n_compformat )     compformat      = property.GetByte();
	        else if ( property.Name() == n_palette )        palette         = property.GetDword();
	        else if ( property.Name() == n_usize )          itsUsize        = property.GetWord();
	        else if ( property.Name() == n_vsize )          itsVsize        = property.GetWord();
	        else if ( property.Name() == n_uclamp )         itsUclamp       = property.GetWord();
	        else if ( property.Name() == n_vclamp )         itsVclamp       = property.GetWord();
	        else if ( property.Name() == n_alphatexture )   alphatexture    = property.GetBool();
	        else if ( property.Name() == n_maxcolor )       property.GetWord();
	        else if ( property.Name() == n_mipmaps )        property.GetWord();
	        else if ( property.Name() == n_normallod )      property.GetByte();
	        else if ( property.Name() == n_minlod )         property.GetByte();
		}   while   ( ::strcmp( "none", aUn.LcName( property.Name() ) ) );

	    if  ( ! format  )   format = compformat;
	    // ------------------------------------------------------------
	    // Now, read data before the image data itself
	    // ------------------------------------------------------------
		if		( aUf.PackageVersion() < 63 ) {	// Oldest ones have only the mipmaps
		    mipmaps = aUf.ReadByte();
		}

		else if	( aUf.PackageVersion() < 100 ) {	// Most UT textures fall here
		    mipmaps = aUf.ReadByte();
			aUf.ReadDword();
		    blocksize = aUf.ReadIndex();
		}

		else if	( aUf.PackageVersion() < 120 ) {	// Pre UT2k3 engine
		    mipmaps = aUf.ReadByte();
			aUf.ReadDword();
			aUf.ReadDword();
		    blocksize = aUf.ReadIndex();
		}
	
		else {	// UT2k3 engine
		    mipmaps = aUf.ReadByte();
			aUf.ReadDword();
		    blocksize = aUf.ReadIndex();
		}
	
		if		( mipmaps == 0 ) {
			throw CError( "The image is empty" );
		}
	
		else if	( blocksize == 0 ) {
			throw CError( "The image is empty" );
		}
	
	    else if	( blocksize > 2048 * 2048 ) {
	        throw CError( "Possibly invalid data or too big image" );
	    }
	
	    // ------------------------------------------------------------
	    // Reading differently formatted data
	    // ------------------------------------------------------------
	    switch  ( format ) {
	        case    0:  // TEXF_P8 - 8 bits, paletted
	            CUnTexture::ReadImageP8( aUf, aUn, aUe, aUi, blocksize, palette );
	            break;
	
	        case    3:  // DXT1
	            if	( palette )	CUnTexture::ReadImageP8( aUf, aUn, aUe, aUi, blocksize, palette );
				else			CUnTexture::ReadImageDXT( aUf, itsUsize, itsVsize, 1 );
	            break;
	
	        case    5:  // TEXF_RGBA8 - 4 bytes per pixel
				CUnTexture::ReadImageRGBA( aUf, blocksize, 4, alphatexture );
				break;
	
	        case    7:  // DXT3
	            CUnTexture::ReadImageDXT( aUf, itsUsize, itsVsize, 3 );
	            break;
	
	        case    8:  // DXT5
	            CUnTexture::ReadImageDXT( aUf, itsUsize, itsVsize, 5 );
	            break;
	
	
	        case    10: // TEXF_G16 - 2 bytes / pixel Grayscale
	            CUnTexture::ReadImageGRAY( aUf, blocksize, 2, alphatexture );
	            break;
	
			case	1:	// TEXF_RGBA7 - 7 bits RGB format w.t.f.
			case	2:	// TEXF_RGB16 - 16 bits RGB format 565
			case	4:	// TEXF_RGB8  - 8 bits RGB format w.t.f.
	        case    6:  // TEXF_NODATA - No image data
	        case    9:  // TEXF_L8 - ????
	        case    11: // TEXF_RRRGGGBBB - ????
	        default:
	            throw CError( "Unsupported image format" );
	    }
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::ReadImageData", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::ReadImageData", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:     Read image - 8 bits, paletted
// --------------------------------------------------------------------
void    CUnTexture::ReadImageP8 (   CUnFile_t &                 aUf,
                                    const CUnNameTable_t &      aUn,
                                    const CUnExportTable_t &    aUe,
									const CUnImportTable_t &	aUi,
                                    dword_t                     aBlockSize,
                                    dword_t                     aPalette ) {

	try {
	    // ------------------------------------------------------------
	    // Remember current position as we read the palette first
	    // ------------------------------------------------------------
	    dword_t dataoffset  = aUf.Offset();
	
	    // ------------------------------------------------------------
	    // Seek to position where we can fetch the dimensions
	    // And then just read them in
	    // ------------------------------------------------------------
	    aUf.Seek( dataoffset + aBlockSize );
	    dword_t w = aUf.ReadDword();
	    dword_t h = aUf.ReadDword();
	
	    // ------------------------------------------------------------
	    // The blocksize and dimensions should match
	    // ------------------------------------------------------------
	    if  ( w * h > aBlockSize ) {
	        throw CError( "Blocksize and dimensions out of sync" );
	    }
	    itsPixmap.Create( w, h );
	
	    // ------------------------------------------------------------
	    // Seek to the palette and read it in
	    // ------------------------------------------------------------
	    aUf.Seek( aUe.Export( NAMEEXPORT( aPalette ) ).SerialOffset() );
	
	    // ------------------------------------------------------------
	    // Skip palette properties
	    // ------------------------------------------------------------
	    CUnProperty property;
	    do  {
	        property = CUnProperty( aUf, aUn, aUi, aUe );
	    }   while   ( ::strcmp( "none", aUn.LcName( property.Name() ) ) );
	
	    // ------------------------------------------------------------
	    // Number of colors in the palette
	    // ------------------------------------------------------------
	    dword_t palsize = aUf.ReadIndex();
	
	    // ------------------------------------------------------------
	    // Check sanity of the number of colors 1 ... 255
	    // ------------------------------------------------------------
	    if  ( ( palsize < 1 ) || ( palsize > 256 ) ) {
	        throw CError( "Invalid number of colours" );
	    }
	
	    // ------------------------------------------------------------
	    // Create a local data for reading the image in
	    // ------------------------------------------------------------
	    CPicturePixel * mypalette = new CPicturePixel[palsize];
	    byte_t *        mydata    = new byte_t [ aBlockSize ];
	    dword_t         loop, x, y;
	
	    try {
	
	        // ------------------------------------------------------------
	        // Read the palette in
	        // ------------------------------------------------------------
	        for ( loop = 0; loop < palsize; loop++ ) {
	            mypalette[loop].R( aUf.ReadByte() );
	            mypalette[loop].G( aUf.ReadByte() );
	            mypalette[loop].B( aUf.ReadByte() );
	            mypalette[loop].A( aUf.ReadByte() );
	        }
	
	        // ------------------------------------------------------------
	        // Read the image in
	        // ------------------------------------------------------------
	        aUf.Seek( dataoffset );
	        aUf.ReadRawData( mydata, aBlockSize );
	
	        // ------------------------------------------------------------
	        // Finally, construct the image
	        // ------------------------------------------------------------
	        for ( y = 0, loop = 0; y < h; y++ ) {
	            for ( x = 0; x < w; x++, loop++ ) {
	                itsPixmap.Pixel( x, y, mypalette[mydata[loop]] );
	            }
	        }
	
	        // ------------------------------------------------------------
	        // Release some resources
	        // ------------------------------------------------------------
	        delete [] mypalette;
	        delete [] mydata;
	    }
	
	    catch   ( ... ) {
	        if  ( mypalette )   delete [] mypalette;
	        if  ( mydata )      delete [] mydata;
	        throw;
	    }
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::ReadImageP8", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::ReadImageP8", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:     Read some DXT variant
// --------------------------------------------------------------------
void    CUnTexture::ReadImageDXT    (   CUnFile_t &                 aUf,
										dword_t						aW,
										dword_t						aH,
                                        int                         aFormat ) {
	try {
	    // ------------------------------------------------------------
	    // Derive blocksize from image dimensions
	    // ------------------------------------------------------------
	    dword_t	blocksize;
	    switch  ( aFormat ) {
	        case    1:
			blocksize = (aW/4) * (aH/4) * 8;
	        break;
	        
			default:
			blocksize = (aW/4) * (aH/4) * 16;
	        break;
	
	    }
	    itsPixmap.Create( aW, aH );
	
	    byte_t *        mydata    = new byte_t [ blocksize ];
	
	    try {
	        // ------------------------------------------------------------
	        // Read the image data in
	        // ------------------------------------------------------------
	        aUf.ReadRawData( mydata, blocksize );
	
			if	( aUf.ReadDword() != aW )	throw CError( "Image read error" );
			if	( aUf.ReadDword() != aH )	throw CError( "Image read error" );
	
	        // ------------------------------------------------------------
	        // What compression do we use to set up the image ?
	        // ------------------------------------------------------------
	        switch  ( aFormat ) {
	            case    1:  CUnTexture::DecompressDXT1( mydata );   break;
	            case    3:  CUnTexture::DecompressDXT3( mydata );   break;
	            case    5:  CUnTexture::DecompressDXT5( mydata );   break;
	        }
	
	        // ------------------------------------------------------------
	        // Release some resources
	        // ------------------------------------------------------------
	        delete [] mydata;
	    }
	
	    catch   ( ... ) {
	        if  ( mydata )      delete [] mydata;
	        throw;
	    }
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::ReadImageDXT", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::ReadImageDXT", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:     Read RGBA
// --------------------------------------------------------------------
void    CUnTexture::ReadImageRGBA   (   CUnFile_t &                 aUf,
                                        dword_t                     aBlockSize,
                                        dword_t                     aBytesPerPixel,
                                        bool						aUseAlpha ) {
	try {
	    // ------------------------------------------------------------
	    // Remember current position as we read the palette first
	    // ------------------------------------------------------------
	    dword_t dataoffset  = aUf.Offset();
	
	    // ------------------------------------------------------------
	    // Seek to position where we can fetch the dimensions
	    // And then just read them in
	    // ------------------------------------------------------------
	    aUf.Seek( dataoffset + aBlockSize );
	    dword_t w = aUf.ReadDword();
	    dword_t h = aUf.ReadDword();
	
	    // ------------------------------------------------------------
	    // The blocksize and dimensions should match
	    // ------------------------------------------------------------
	    if  ( w * h * aBytesPerPixel > aBlockSize ) {
	        throw CError( "Blocksize and dimensions out of sync" );
	    }
	
	    itsPixmap.Create( w, h );
	
	    byte_t *        mydata    = new byte_t [ aBlockSize ];
	    byte_t *        myptr;
	    dword_t         x, y;
	    CPicturePixel   pixel;
	
	    try {
	        // ------------------------------------------------------------
	        // Read the image data in
	        // ------------------------------------------------------------
	        aUf.Seek( dataoffset );
	        aUf.ReadRawData( mydata, aBlockSize );
	
	        // ------------------------------------------------------------
	        // Encode the image
	        // ------------------------------------------------------------
	        myptr = mydata;
	        for ( y = 0; y < h; y++ ) {
	            for ( x = 0; x < w; x++ ) {
	                switch  ( aBytesPerPixel ) {
	                    case    1:
	                    break;
	
	                    case    2:
	                    pixel = CUnTexture::GetTexelW( myptr );
	                    pixel.A( 0xff );
	                    break;
	
	                    case    4:
	                    if	( aUseAlpha )	pixel.RGBA( myptr[2], myptr[1], myptr[0], myptr[3] );
	                    else				pixel.RGBA( myptr[2], myptr[1], myptr[0], 0xff );
	                    break;
	                }
	                itsPixmap.Pixel( x, y, pixel );
	                myptr = myptr + aBytesPerPixel;
	            }
	        }
	
	        // ------------------------------------------------------------
	        // Release some resources
	        // ------------------------------------------------------------
	        delete [] mydata;
	    }
	
	    catch   ( ... ) {
	        if  ( mydata )      delete [] mydata;
	        throw;
	    }
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::ReadImageRGBA", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::ReadImageRGBA", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:     Read GRAY
// --------------------------------------------------------------------
void    CUnTexture::ReadImageGRAY   (   CUnFile_t &                 aUf,
                                        dword_t                     aBlockSize,
                                        dword_t                     aBytesPerPixel,
                                        bool                        aIsAlpha ) {
	try {
	    // ------------------------------------------------------------
	    // Remember current position as we read the palette first
	    // ------------------------------------------------------------
	    dword_t dataoffset  = aUf.Offset();
	
	    // ------------------------------------------------------------
	    // Seek to position where we can fetch the dimensions
	    // And then just read them in
	    // ------------------------------------------------------------
	    aUf.Seek( dataoffset + aBlockSize );
	    dword_t w = aUf.ReadDword();
	    dword_t h = aUf.ReadDword();
	
	    // ------------------------------------------------------------
	    // The blocksize and dimensions should match
	    // ------------------------------------------------------------
	    if  ( w * h * aBytesPerPixel > aBlockSize ) {
	        throw CError( "Blocksize and dimensions out of sync" );
	    }
	
	    itsPixmap.Create( w, h );
	
	    byte_t *        mydata    = new byte_t [ aBlockSize ];
	    byte_t *        myptr;
	    dword_t         x, y;
	    CPicturePixel   pixel;
	
	    try {
	        // ------------------------------------------------------------
	        // Read the image data in
	        // ------------------------------------------------------------
	        aUf.Seek( dataoffset );
	        aUf.ReadRawData( mydata, aBlockSize );
	
	        // ------------------------------------------------------------
	        // Encode the image
	        // ------------------------------------------------------------
	        myptr = mydata;
	        for ( y = 0; y < h; y++ ) {
	            for ( x = 0; x < w; x++ ) {
	                switch  ( aBytesPerPixel ) {
	                    case    1:
	                    if  ( aIsAlpha )    pixel.RGBA( 0xff, 0xff, 0xff, myptr[0] );
	                    else                pixel.RGBA( myptr[0], myptr[0], myptr[0], 0xff );
	                    break;
	
	                    case    2:
						if	( aIsAlpha )	pixel.RGBA( myptr[0], myptr[0], myptr[0], myptr[1] );
						else {
							byte_t	gw = (byte_t)(CUnTexture::GetTexelW( myptr ) / 256);
							pixel.RGBA( gw, gw, gw, 0xff );
						}
	                    break;
	                }
	                itsPixmap.Pixel( x, y, pixel );
	                myptr = myptr + aBytesPerPixel;
	            }
	        }
	
	        // ------------------------------------------------------------
	        // Release some resources
	        // ------------------------------------------------------------
	        delete [] mydata;
	    }
	
	    catch   ( ... ) {
	        if  ( mydata )      delete [] mydata;
	        throw;
	    }
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::ReadImageGRAY", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::ReadImageGRAY", "unknown" );
	}
}

// --------------------------------------------------------------------
// private: Decompressing DXT1
// --------------------------------------------------------------------
void    CUnTexture::DecompressDXT1  ( const byte_t * aData ) {
	try {
	    dword_t         x, y;
	    CPicturePixel   texel[16];
	    dword_t         tw = itsPixmap.W() / 4;
	    dword_t         th = itsPixmap.H() / 4;
	
	    for ( y = 0; y < th; y++ ) {
	        for ( x = 0; x < tw; x++ ) {
	            CUnTexture::GetTexelDXT1( texel, aData );
	            CUnTexture::PutTexel( 4 * x, 4 * y, texel );
	            aData = aData + 8;
	        }
	    }
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::DecompressDXT1", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::DecompressDXT1", "unknown" );
	}
}

// --------------------------------------------------------------------
// private: Decompressing DXT3
// --------------------------------------------------------------------
void    CUnTexture::DecompressDXT3  ( const byte_t * aData ) {
	try {
	    dword_t         x, y;
	    CPicturePixel   texel[16];
	    dword_t         tw = itsPixmap.W() / 4;
	    dword_t         th = itsPixmap.H() / 4;
	
	    for ( y = 0; y < th; y++ ) {
	        for ( x = 0; x < tw; x++ ) {
	            CUnTexture::GetTexelDXT3( texel, aData );
	            CUnTexture::PutTexel( 4 * x, 4 * y, texel );
	            aData = aData + 16;
	        }
	    }
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::DecompressDXT3", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::DecompressDXT3", "unknown" );
	}
}

// --------------------------------------------------------------------
// private: Decompressing DXT5
// --------------------------------------------------------------------
void    CUnTexture::DecompressDXT5  ( const byte_t * aData ) {
	try {
	    dword_t         x, y;
	    CPicturePixel   texel[16];
	    dword_t         tw = itsPixmap.W() / 4;
	    dword_t         th = itsPixmap.H() / 4;
	
	    for ( y = 0; y < th; y++ ) {
	        for ( x = 0; x < tw; x++ ) {
	            CUnTexture::GetTexelDXT5( texel, aData );
	            CUnTexture::PutTexel( 4 * x, 4 * y, texel );
	            aData = aData + 16;
	        }
	    }
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::DecompressDXT5", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::DecompressDXT5", "unknown" );
	}
}

// --------------------------------------------------------------------
// private: Write one 4 x 4 texel into the image
// --------------------------------------------------------------------
void    CUnTexture::PutTexel (  dword_t                 aX,
                                dword_t                 aY,
                                const CPicturePixel *   aTexel ) {
	try {
	    dword_t x, y, r;
	    for ( y = 0, r = 0; y < 4; y++ ) {
	        for ( x = 0; x < 4; x++, r++ ) {
	            itsPixmap.Pixel( x + aX, y + aY, aTexel[r] );
	        }
	    }
	}
	catch	( CError e ) {
		throw	CError( "CUnTexture::PutTexel", e.Error() );
	}
	catch	( ... ) {
		throw	CError( "CUnTexture::PutTexel", "unknown" );
	}
}

// --------------------------------------------------------------------
// private: Read in one 4 x 4 texel (DXT1)
// --------------------------------------------------------------------
void    CUnTexture::GetTexelDXT1 (  CPicturePixel * aTexel,
                                    const byte_t *  aData ) const {
    CUnTexture::GetTexelA1( aTexel, aData );
    CUnTexture::GetTexelC1( aTexel, aData );
}

// --------------------------------------------------------------------
// private: Read in one 4 x 4 texel (DXT3)
// --------------------------------------------------------------------
void    CUnTexture::GetTexelDXT3 (  CPicturePixel * aTexel,
                                    const byte_t *  aData ) const {
    CUnTexture::GetTexelA3( aTexel, aData );
    CUnTexture::GetTexelC3( aTexel, aData + 8 );
}

// --------------------------------------------------------------------
// private: Read in one 4 x 4 texel (DXT5)
// --------------------------------------------------------------------
void    CUnTexture::GetTexelDXT5 (  CPicturePixel * aTexel,
                                    const byte_t *  aData ) const {
	CUnTexture::GetTexelA5( aTexel, aData );
	CUnTexture::GetTexelC3( aTexel, aData + 8 );
}

// --------------------------------------------------------------------
// private: Read in alpha channel (DXT1)
// --------------------------------------------------------------------
void    CUnTexture::GetTexelA1  (   CPicturePixel * aTexel,
                                    const byte_t *  aData ) const {
    // The image is all opaque !
    for ( int i = 0; i < 16; i++ )  aTexel[i].A( 0xff );
}

// --------------------------------------------------------------------
// private: Read in alpha channel (DXT3)
// --------------------------------------------------------------------
void    CUnTexture::GetTexelA3  (   CPicturePixel * aTexel,
                                    const byte_t *  aData ) const {
    int     i;
    byte_t  a;
    dword_t alpha = CUnTexture::GetTexelDW( aData );

    for ( i = 0; i < 8; i++ ) {
        a = (byte_t)( alpha & 0x0f );
        aTexel[i].A( (byte_t)(a | ( a << 4 )) );
        alpha >>= 4;
    }

    alpha = CUnTexture::GetTexelDW( aData + 4 );

    for ( i = 8; i < 16; i++ ) {
        a = (byte_t)( alpha & 0x0f );
        aTexel[i].A( (byte_t)(a | ( a << 4 )) );
        alpha >>= 4;
    }
}

// --------------------------------------------------------------------
// private: Read in alpha channel (DXT5)
// --------------------------------------------------------------------
void    CUnTexture::GetTexelA5  (   CPicturePixel * aTexel,
                                    const byte_t *  aData ) const {
    byte_t  aa  [8];
    word_t  alpha;
    word_t  bitnow  = 0;
    word_t  roffs   = 2;

    aa[0] = aData[0];
    aa[1] = aData[1];

    if  ( aa[0] > aa[1] ) {
        aa[2] = (byte_t)((6 * (word_t)aa[0] + 1 * (word_t)aa[1]) / 7);
        aa[3] = (byte_t)((5 * (word_t)aa[0] + 2 * (word_t)aa[1]) / 7);
        aa[4] = (byte_t)((4 * (word_t)aa[0] + 3 * (word_t)aa[1]) / 7);
        aa[5] = (byte_t)((3 * (word_t)aa[0] + 4 * (word_t)aa[1]) / 7);
        aa[6] = (byte_t)((2 * (word_t)aa[0] + 5 * (word_t)aa[1]) / 7);
        aa[7] = (byte_t)((1 * (word_t)aa[0] + 6 * (word_t)aa[1]) / 7);
    }
    else {
        aa[2] = (byte_t)((4 * (word_t)aa[0] + 1 * (word_t)aa[1]) / 5);
        aa[3] = (byte_t)((3 * (word_t)aa[0] + 2 * (word_t)aa[1]) / 5);
        aa[4] = (byte_t)((2 * (word_t)aa[0] + 3 * (word_t)aa[1]) / 5);
        aa[5] = (byte_t)((1 * (word_t)aa[0] + 4 * (word_t)aa[1]) / 5);
        aa[6] = 0;
        aa[7] = 0xff;
    }

    alpha =  (word_t)aData[roffs++];

    for ( int i = 0; i < 16; i++ ) {
        aTexel[i].A( aa[ alpha & 0x07 ] );

        if      ( ( bitnow % 8 ) == 0 ) {
            alpha |= (word_t)(((word_t)aData[roffs++] << 8));
            alpha >>= 3;
        }
        else if ( ( bitnow % 8 ) > 5 ) {
            alpha >>= (8 - ( bitnow % 8 ));
            alpha |= (word_t)(((word_t)aData[roffs++] << 8));
            alpha >>= (3 - 8 + ( bitnow % 8 ));
        }
        else {
            alpha >>= 3;
        }
        bitnow = (word_t)(bitnow + 3);
    }

}

// --------------------------------------------------------------------
// private: Read in RGB info (DXT1)
// --------------------------------------------------------------------
void    CUnTexture::GetTexelC1  (   CPicturePixel * aTexel,
                                    const byte_t *  aData ) const {
    CPicturePixel   tp[4];
    byte_t          tx[16];
    CUnTexture::GetTexelTP( tp, aData, true );
    CUnTexture::GetTexelTX( tx, aData + 4 );
    for ( int i = 0; i < 16; i++ ) aTexel[i].RGB( tp[tx[i]] );
}

// --------------------------------------------------------------------
// private: Read in RGB info (DXT3 and DXT5)
// --------------------------------------------------------------------
void    CUnTexture::GetTexelC3  (   CPicturePixel * aTexel,
                                    const byte_t *  aData ) const {
    CPicturePixel   tp[4];
    byte_t          tx[16];
    CUnTexture::GetTexelTP( tp, aData, false );
    CUnTexture::GetTexelTX( tx, aData + 4 );
    for ( int i = 0; i < 16; i++ )  aTexel[i].RGB( tp[tx[i]] );
}

// --------------------------------------------------------------------
// private: Read in the four color palette
// --------------------------------------------------------------------
void    CUnTexture::GetTexelTP  (   CPicturePixel * aTp,
                                    const byte_t *  aData,
                                    bool            aAlpha ) const {
    word_t  color0  = GetTexelW( aData );
    word_t  color1  = GetTexelW( aData + 2 );
    word_t  r, g, b;

    aTp[0] = color0;
    aTp[1] = color1;

    if  ( ( aAlpha ) && ( color0 <= color1 ) ) {
        r = (word_t)(((word_t)aTp[0].R() + (word_t)aTp[1].R()) / 2);
        g = (word_t)(((word_t)aTp[0].G() + (word_t)aTp[1].G()) / 2);
        b = (word_t)(((word_t)aTp[0].B() + (word_t)aTp[1].B()) / 2);

        aTp[2].RGB( (byte_t)r, (byte_t)g, (byte_t)b );

        aTp[3].RGB( 0, 0, 0 );
    }
    else {
        r = (word_t)(((word_t)aTp[0].R() + (word_t)aTp[0].R() + (word_t)aTp[1].R()) / 3);
        g = (word_t)(((word_t)aTp[0].G() + (word_t)aTp[0].G() + (word_t)aTp[1].G()) / 3);
        b = (word_t)(((word_t)aTp[0].B() + (word_t)aTp[0].B() + (word_t)aTp[1].B()) / 3);

        aTp[2].RGB( (byte_t)r, (byte_t)g, (byte_t)b );

        r = (word_t)(((word_t)aTp[0].R() + (word_t)aTp[1].R() + (word_t)aTp[1].R()) / 3);
        g = (word_t)(((word_t)aTp[0].G() + (word_t)aTp[1].G() + (word_t)aTp[1].G()) / 3);
        b = (word_t)(((word_t)aTp[0].B() + (word_t)aTp[1].B() + (word_t)aTp[1].B()) / 3);

        aTp[3].RGB( (byte_t)r, (byte_t)g, (byte_t)b );
    }

}

// --------------------------------------------------------------------
// private: Read in the texel image
// --------------------------------------------------------------------
void    CUnTexture::GetTexelTX  (   byte_t *        aTx,
                                    const byte_t *  aData ) const {
    dword_t texim = CUnTexture::GetTexelDW( aData );
    for ( int i = 0; i < 16; i++, texim >>= 2 ) aTx[i] = (byte_t)( texim & 0x03 );
}

// --------------------------------------------------------------------
// private: Read in four bytes as a dword
// --------------------------------------------------------------------
word_t  CUnTexture::GetTexelW   (   const byte_t *  aData ) const {
    return  (word_t)(	  (word_t)aData[0]
						+ ((word_t)aData[1] << 8));
}

// --------------------------------------------------------------------
// private: Read in four bytes as a dword
// --------------------------------------------------------------------
dword_t CUnTexture::GetTexelDW  (   const byte_t *  aData ) const {
    return     (dword_t)aData[0]
            + ((dword_t)aData[1] << 8)
            + ((dword_t)aData[2] << 16)
            + ((dword_t)aData[3] << 24);
}

// --------------------------------------------------------------------
// private: Clean up the mess
// --------------------------------------------------------------------
void    CUnTexture::Cleanup ( void ) {
    itsUsize    = 0;
    itsVsize    = 0;
    itsUclamp   = 0;
    itsVclamp   = 0;
}

// --------------------------------------------------------------------
const CPicturePixmap &	CUnTexture::Pixmap		( void ) const { return itsPixmap; }
dword_t					CUnTexture::W			( void ) const { return itsPixmap.W(); }
dword_t					CUnTexture::H			( void ) const { return itsPixmap.H(); }

// --------------------------------------------------------------------
// EOF: CUnTexture.cxx
// --------------------------------------------------------------------
