// --------------------------------------------------------------------
// CUpFileType.cpp
// Whatis:  File types
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-FEB-2003     Created this source
// --------------------------------------------------------------------
#include	"CUpFileType.h"
#include	"CUnFile.hxx"
#include	"CUnNameTable.hxx"

// --------------------------------------------------------------------
// public:	Constructor and destructor
// --------------------------------------------------------------------
CUpFileType::CUpFileType	() {
	CUpFileType::Cleanup();
}

// --------------------------------------------------------------------
CUpFileType::~CUpFileType	() {
	CUpFileType::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CUpFileType::CUpFileType				( const CUpFileType & aC ) {
	CUpFileType::Cleanup();
    *this = aC;
}

// --------------------------------------------------------------------
CUpFileType & CUpFileType::operator =	( const CUpFileType & aC ) {
	CUpFileType::Free();
	itsName				= aC.Name();
	itsRules			= aC.Rules();
	itsTargetDir		= aC.TargetDir();
	itsAutoSearch		= aC.AutoSearch();
	itsSuffixes			= aC.Suffixes();
	itsPaneUnreal		= aC.PaneUnreal();
	itsPaneLevel		= aC.PaneLevel();
	itsPaneHexdump		= aC.PaneHexdump();
	itsPaneScreenshot	= aC.PaneScreenshot();
	itsPaneTextures		= aC.PaneTextures();
	itsPanePicture		= aC.PanePicture();
	itsPaneText			= aC.PaneText();
	itsPaneHtml			= aC.PaneHtml();
	itsPaneFile			= aC.PaneFile();
	itsSelect			= aC.Select();
	itsUz2Only			= aC.Uz2Only();
    return *this;
}

// --------------------------------------------------------------------
// public:	Comparison operators
// --------------------------------------------------------------------
bool CUpFileType::operator == ( const CUpFileType & aC ) const {
	return	::stricmp( CUpFileType::Name(), aC.Name() ) == 0;
}

// --------------------------------------------------------------------
bool CUpFileType::operator != ( const CUpFileType & aC ) const {
	return	::stricmp( CUpFileType::Name(), aC.Name() ) != 0;
}

// --------------------------------------------------------------------
// public:	Interface for the list management
// --------------------------------------------------------------------
bool			CUpFileType::ListSelect	( void ) const {
	return	itsSelect;
}

// --------------------------------------------------------------------
void			CUpFileType::ListSelect	( bool aV ) {
	itsSelect = aV;
}

// --------------------------------------------------------------------
const char *	CUpFileType::ListName	( void ) const {
	return	CUpFileType::Name();
}

// --------------------------------------------------------------------
// public:	Type detection
// --------------------------------------------------------------------
bool	CUpFileType::Detect	( const char * aPath ) const {
	bool			result	= false;
    const char *	rules	= itsRules.Value();
	const char *	nl;
    dword_t			rulelen;
    char *			onerule = NULL;
	struct stat		mystat;

    // The file MUST exist
    if	( ::stat( aPath, &mystat ) == 0 ) {
	    while	( ( *rules ) && ( ! result ) ) {
			nl = ::strchr( rules, '\n' );
        	if	( ! nl )	nl = rules + ::strlen( rules );

			rulelen = (dword_t)nl - (dword_t)rules;
    	    if	( onerule ) delete [] onerule;
        	onerule = new char [ rulelen + 1 ];
			::memcpy( onerule, rules, rulelen );
    	    onerule[rulelen] = 0;

	        result = CUpFileType::TryRuleLine( onerule, aPath );

			if	( *nl )	rules = nl + 1;
        	else		rules = nl;
	    }
    	if	( onerule ) delete [] onerule;
    }
	return	result;
}

// --------------------------------------------------------------------
// public:	Streaming
// --------------------------------------------------------------------
void			CUpFileType::Write		( CWinIniSection & aSec ) const {
	itsName.Write		( aSec, "Name" );
	itsRules.Write		( aSec, "Rules" );
	itsTargetDir.Write	( aSec, "TargetDir" );
	itsSuffixes.Write	( aSec, "Suffixes" );
	aSec.Insert( "AutoSearch",		itsAutoSearch );
	aSec.Insert( "PaneUnreal",		itsPaneUnreal );
	aSec.Insert( "PaneLevel", 		itsPaneLevel );
	aSec.Insert( "PaneHexdump", 	itsPaneHexdump );
	aSec.Insert( "PaneScreenshot",	itsPaneScreenshot );
	aSec.Insert( "PaneTextures", 	itsPaneTextures );
	aSec.Insert( "PanePicture", 	itsPanePicture );
	aSec.Insert( "PaneText", 		itsPaneText );
	aSec.Insert( "PaneHtml", 		itsPaneHtml );
	aSec.Insert( "PaneFile", 		itsPaneFile );
	aSec.Insert( "Select", 			itsSelect );
	aSec.Insert( "Uz2Only", 		itsUz2Only );
}

// --------------------------------------------------------------------
void			CUpFileType::Read		( const CWinIniFile & aIni, const char * aSecName ) {
	CUpFileType::Free();
	itsName.Read		( aIni, aSecName, "Name" );
	itsRules.Read	 	( aIni, aSecName, "Rules" );
	itsTargetDir.Read	( aIni, aSecName, "TargetDir" );
	itsAutoSearch		= aIni.BoolValue( aSecName, "AutoSearch" );
	itsSuffixes.Read	( aIni, aSecName, "Suffixes" );
	itsPaneUnreal		= aIni.BoolValue( aSecName, "PaneUnreal" );
	itsPaneLevel		= aIni.BoolValue( aSecName, "PaneLevel" );
	itsPaneHexdump		= aIni.BoolValue( aSecName, "PaneHexdump" );
	itsPaneScreenshot	= aIni.BoolValue( aSecName, "PaneScreenshot" );
	itsPaneTextures		= aIni.BoolValue( aSecName, "PaneTextures" );
	itsPanePicture		= aIni.BoolValue( aSecName, "PanePicture" );
	itsPaneText			= aIni.BoolValue( aSecName, "PaneText" );
	itsPaneHtml			= aIni.BoolValue( aSecName, "PaneHtml" );
	itsPaneFile			= aIni.BoolValue( aSecName, "PaneFile" );
    itsSelect			= aIni.BoolValue( aSecName, "Select" );
    itsUz2Only			= aIni.BoolValue( aSecName, "Uz2Only" );
}

// --------------------------------------------------------------------
// public:	Accessing instance data
// --------------------------------------------------------------------
const char *	CUpFileType::Name			( void ) const {
	return itsName.Value();
}

// --------------------------------------------------------------------
const char *	CUpFileType::Rules			( void ) const {
	return	itsRules.Value();
}

// --------------------------------------------------------------------
const char *	CUpFileType::TargetDir		( void ) const {
	return	itsTargetDir.Value();
}

// --------------------------------------------------------------------
bool			CUpFileType::AutoSearch		( void ) const {
	return	itsAutoSearch;
}

// --------------------------------------------------------------------
const char *	CUpFileType::Suffixes		( void ) const {
	return	itsSuffixes.Value();
}

// --------------------------------------------------------------------
bool			CUpFileType::PaneUnreal		( void ) const {
	return	itsPaneUnreal;
}

// --------------------------------------------------------------------
bool			CUpFileType::PaneLevel		( void ) const {
	return	itsPaneLevel;
}

// --------------------------------------------------------------------
bool			CUpFileType::PaneHexdump	( void ) const {
	return	itsPaneHexdump;
}

// --------------------------------------------------------------------
bool			CUpFileType::PaneScreenshot	( void ) const {
	return	itsPaneScreenshot;
}

// --------------------------------------------------------------------
bool			CUpFileType::PaneTextures	( void ) const {
	return	itsPaneTextures;
}

// --------------------------------------------------------------------
bool			CUpFileType::PanePicture	( void ) const {
	return	itsPanePicture;
}

// --------------------------------------------------------------------
bool			CUpFileType::PaneText		( void ) const {
	return	itsPaneText;
}

// --------------------------------------------------------------------
bool			CUpFileType::PaneHtml		( void ) const {
	return	itsPaneHtml;
}

// --------------------------------------------------------------------
bool			CUpFileType::PaneFile		( void ) const {
	return	itsPaneFile;
}

// --------------------------------------------------------------------
bool			CUpFileType::Select			( void ) const {
	return	itsSelect;
}

// --------------------------------------------------------------------
bool			CUpFileType::Uz2Only		( void ) const {
	return	itsUz2Only;
}

// --------------------------------------------------------------------
void			CUpFileType::Name			( const char * aV ) {
	itsName = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::Rules			( const char * aV ) {
	itsRules = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::TargetDir		( const char * aV ) {
	itsTargetDir = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::AutoSearch		( bool aV ) {
	itsAutoSearch = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::Suffixes		( const char * aV ) {
	itsSuffixes = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::PaneUnreal		( bool aV ) {
	itsPaneUnreal = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::PaneLevel		( bool aV ) {
	itsPaneLevel = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::PaneHexdump	( bool aV ) {
	itsPaneHexdump = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::PaneScreenshot	( bool aV ) {
	itsPaneScreenshot = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::PaneTextures	( bool aV ) {
	itsPaneTextures = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::PanePicture	( bool aV ) {
	itsPanePicture = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::PaneText		( bool aV ) {
	itsPaneText = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::PaneHtml		( bool aV ) {
	itsPaneHtml = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::PaneFile		( bool aV ) {
	itsPaneFile = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::Select			( bool aV ) {
	itsSelect = aV;
}

// --------------------------------------------------------------------
void			CUpFileType::Uz2Only		( bool aV ) {
	itsUz2Only = aV;
}

// --------------------------------------------------------------------
// private:	Private helpers
// --------------------------------------------------------------------
void	CUpFileType::Free	( void ) {
	CUpFileType::Cleanup();
}

// --------------------------------------------------------------------
void	CUpFileType::Cleanup	( void ) {
	itsName				= (const char *)NULL;
	itsRules			= (const char *)NULL;
	itsTargetDir		= (const char *)NULL;
	itsAutoSearch		= false;
	itsSuffixes			= (const char *)NULL;
	itsPaneUnreal		= false;
	itsPaneLevel		= false;
	itsPaneHexdump		= false;
	itsPaneScreenshot	= false;
	itsPaneTextures		= false;
	itsPanePicture		= false;
	itsPaneText			= false;
	itsPaneHtml			= false;
	itsPaneFile			= false;
    itsSelect			= false;
    itsUz2Only			= false;
}

// --------------------------------------------------------------------
// private:	Try one rule line
// --------------------------------------------------------------------
bool	CUpFileType::TryRuleLine	( char * aRule, const char * aPath ) const {
	bool	result = true;
    char *	token;
    char *	arg;

    while	( ( *aRule ) && ( result ) ) {

		// Skip over rule whitespace
        while	( ( *aRule > 0 ) && ( *aRule <= ' ' ) )	aRule++;

		// If there is a '&' skip over it as well
        if	( *aRule == '&' ) {
        	aRule++;
            continue;
        }
        if	( *aRule == 0 )	break;

        // Memorize the token and skip over non-whitespace
        token = aRule;
		while	( ( *aRule < 0 ) || ( *aRule > ' ' ) )	aRule++;

		// Skip over rule whitespace
		if	( *aRule ) {
        	*aRule = 0;
            aRule++;
	        while	( ( *aRule > 0 ) && ( *aRule <= ' ' ) )	aRule++;
        }

		// If we have an argument, memorize it as well
        if	( ( *aRule != '&' ) && ( *aRule ) ) {
            arg = aRule;
			while	( ( *aRule < 0 ) || ( *aRule > ' ' ) )	aRule++;
			if	( *aRule ) {
    	    	*aRule = 0;
        	    aRule++;
	        	while	( ( *aRule > 0 ) && ( *aRule <= ' ' ) )	aRule++;
	        }
        }
        else {
        	arg = "";
        }

		// Now, let's see what we got
        if		( ! ::stricmp( token, "IsExe" ) )		result = CUpFileType::TryExe( arg, aPath );
        else if	( ! ::stricmp( token, "IsUnreal" ) )	result = CUpFileType::TryUnreal( arg, aPath );
        else if	( ! ::stricmp( token, "IsU1" ) )		result = CUpFileType::TryU1( arg, aPath );
        else if	( ! ::stricmp( token, "IsUT" ) )		result = CUpFileType::TryUt( arg, aPath );
        else if	( ! ::stricmp( token, "IsUT2003" ) )	result = CUpFileType::TryUt2003( arg, aPath );
        else if	( ! ::stricmp( token, "IsU2" ) )		result = CUpFileType::TryU2( arg, aPath );
        else if	( ! ::stricmp( token, "NotExe" ) )		result = ! CUpFileType::TryExe( arg, aPath );
        else if	( ! ::stricmp( token, "NotUnreal" ) )	result = ! CUpFileType::TryUnreal( arg, aPath );
        else if	( ! ::stricmp( token, "NotU1" ) )		result = ! CUpFileType::TryU1( arg, aPath );
        else if	( ! ::stricmp( token, "NotUT" ) )		result = ! CUpFileType::TryUt( arg, aPath );
        else if	( ! ::stricmp( token, "NotUT2003" ) )	result = ! CUpFileType::TryUt2003( arg, aPath );
        else if	( ! ::stricmp( token, "NotU2" ) )		result = ! CUpFileType::TryU2( arg, aPath );
        else if	( ! ::stricmp( token, "Suffix" ) )		result = CUpFileType::TrySuffix( arg, aPath );
        else if	( ! ::stricmp( token, "Prefix" ) )		result = CUpFileType::TryPrefix( arg, aPath );
        else if	( ! ::stricmp( token, "Name" ) )		result = CUpFileType::TryName( arg, aPath );
        else	result = false;
    }
	return	result;
}

// --------------------------------------------------------------------
bool	CUpFileType::TryExe		( const char * aArg, const char * aFile ) const {
	char	mzbuf[2];
	FILE *	mystm = ::fopen( aFile, "rb" );
	if	( ! mystm )	return false;

	mzbuf[0] = 0;
    mzbuf[1] = 0;
    ::fread( mzbuf, 2, 1, mystm );
    ::fclose( mystm );
    return	( mzbuf[0] == 'M' ) && ( mzbuf[1] == 'Z' );
}

// --------------------------------------------------------------------
bool	CUpFileType::TryUnreal	( const char * aArg, const char * aFile ) const {
	bool	result	= true;
    try	{
    	CUnFile			uf	( aFile );
        CUnNameTable	nt	( uf );
    }
    catch	( ... ) {
    	result = false;
    }
    return	result;
}

// --------------------------------------------------------------------
bool	CUpFileType::TryU1		( const char * aArg, const char * aFile ) const {
	bool	result	= true;
    try	{
    	CUnFile			uf	( aFile );
        CUnNameTable	nt	( uf );
        if	( uf.PackageVersion() >= 69 )	result = false;
    }
    catch	( ... ) {
    	result = false;
    }
    return	result;
}

// --------------------------------------------------------------------
bool	CUpFileType::TryUt		( const char * aArg, const char * aFile ) const {
	bool	result	= true;
    try	{
    	CUnFile			uf	( aFile );
        CUnNameTable	nt	( uf );
		if	( uf.PackageVersion() >= 100 )	result = false;
		if	( uf.PackageVersion() < 69 )		result = false;
    }
    catch	( ... ) {
    	result = false;
    }
    return	result;
}

// --------------------------------------------------------------------
bool	CUpFileType::TryUt2003	( const char * aArg, const char * aFile ) const {
	bool	result	= true;
    try	{
    	CUnFile			uf	( aFile );
        CUnNameTable	nt	( uf );
       	if	( uf.PackageVersion() < 119 )	result = false;
    }
    catch	( ... ) {
    	result = false;
    }
    return	result;
}

// --------------------------------------------------------------------
bool	CUpFileType::TryU2		( const char * aArg, const char * aFile ) const {
	bool	result	= true;
    try	{
    	CUnFile			uf	( aFile );
        CUnNameTable	nt	( uf );
       	if	( uf.PackageVersion() < 119 )	result = false;
    }
    catch	( ... ) {
    	result = false;
    }
    return	result;
}

// --------------------------------------------------------------------
bool	CUpFileType::TrySuffix	( const char * aArg, const char * aFile ) const {
	char	suffix[64];
    ::fnsplit( aFile, NULL, NULL, NULL, suffix );
    return	::stricmp( aArg, suffix ) == 0;
}

// --------------------------------------------------------------------
bool	CUpFileType::TryPrefix	( const char * aArg, const char * aFile ) const {
	return	::strnicmp( aArg, aFile, ::strlen( aArg ) ) == 0;
}

// --------------------------------------------------------------------
bool	CUpFileType::TryName		( const char * aArg, const char * aFile ) const {
	return	::stricmp( aArg, aFile ) == 0;
}

// --------------------------------------------------------------------
// EOF:	CUpFileType.cpp
// --------------------------------------------------------------------
