// --------------------------------------------------------------------
// CUtxDoc.cpp
// Whatis:  File details
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-FEB-2003     Created this source
// --------------------------------------------------------------------
#include	<vcl.h>
#include 	<dir.h>
#include	"CUtxDoc.h"
#include	<Registry.hpp>
#include	<process.h>

// --------------------------------------------------------------------
CUtxDoc::CUtxDoc	() {
	CUtxDoc::Cleanup();
}

// --------------------------------------------------------------------
CUtxDoc::~CUtxDoc	() {
	CUtxDoc::Free();
}

// --------------------------------------------------------------------
void	CUtxDoc::Init	( const char * aAppExe ) {
	char		drive	[16];
    char		path	[512];
    struct stat	mystat;

	::fnsplit( aAppExe, drive, path, NULL, NULL );
    ::fnmerge( itsNvExe, drive, path, "nvdxt", ".exe" );

 	itsDrive = (char)::getdisk();
    CUtxDoc::FillFileList();
    CUtxDoc::DetectSystem();

	// Check the system
    if	( ::stat( itsNvExe, &mystat ) )	throw 1;
    ::strcpy( path, itsUtRoot );
    ::strcat( path, "\\System\\ucc.exe" );
    if	( ::stat( path, &mystat ) )		throw 1;
}

// --------------------------------------------------------------------
CUtxFile &	CUtxDoc::Item	( int aIx ) {
	CUtxFile_li	loop;

    loop = itsFileList.begin();
	while	( aIx > 0 ) {
    	loop++;
        aIx--;
    }
    return	*loop;
}

// --------------------------------------------------------------------
bool	CUtxDoc::CanCompress	( void ) {
	char			path [512];
	struct stat		mystat;
    CUtxFile_lci	loop;
    int				i;

    if	( ! itsTextureFile ) {
    	itsWarning = "No pack name";
    	return false;
    }

    if	( *itsTextureFile == 0 ) {
    	itsWarning = "No pack name";
    	return false;
    }

	// Check out the texture file name
    for	( i = 0; itsTextureFile[i]; i++ ) {
    	if	( ::strchr( "0123456789", itsTextureFile[i] ) )	continue;
    	if	( ::strchr( "abcdefghijklmnopqrstuvwxyz", itsTextureFile[i] ) )	continue;
    	if	( ::strchr( "ABCDEFGHIJKLMNOPQRSTUVWXYZ", itsTextureFile[i] ) )	continue;
    	itsWarning = "Invalid pack name";
		return false;
    }

	// Check out for already existing package with the name
	::sprintf( path, "%s\\Textures\\%s.utx", itsUtRoot, itsTextureFile );
    if	( ! ::stat( path, &mystat ) ) {
    	itsWarning = "Pack exists";
    	return false;
    }

	::sprintf( path, "%s\\System\\%s.u", itsUtRoot, itsTextureFile );
    if	( ! ::stat( path, &mystat ) ) {
    	itsWarning = "Pack exists";
    	return false;
    }

	::sprintf( path, "%s\\Sounds\\%s.uax", itsUtRoot, itsTextureFile );
    if	( ! ::stat( path, &mystat ) ) {
    	itsWarning = "Pack exists";
    	return false;
    }

	::sprintf( path, "%s\\StaticMeshes\\%s.usx", itsUtRoot, itsTextureFile );
    if	( ! ::stat( path, &mystat ) ) {
    	itsWarning = "Pack exists";
    	return false;
    }

	// Last check that at least one file is selected
    for	( loop = itsFileList.begin(); loop != itsFileList.end(); loop++ ) {
    	if	( ! (*loop).Select() )		continue;
        if	( (*loop).Directory() )		continue;
	    itsWarning = NULL;
		return	true;
    }
   	itsWarning = "No files to pack";
    return	false;
}

// --------------------------------------------------------------------
void	CUtxDoc::Propagate	( const CUtxFile & aItem ) {
    CUtxFile_li	loop;
    for	( loop = itsFileList.begin(); loop != itsFileList.end(); loop++ ) {
    	(*loop).Propagate( aItem );
    }
}

// --------------------------------------------------------------------
void	CUtxDoc::Lock		( void ) {
	itsMutex.Lock();
}

// --------------------------------------------------------------------
void	CUtxDoc::Unlock		( void ) {
	itsMutex.Unlock();
}

// --------------------------------------------------------------------
char	CUtxDoc::Drive		( void ) const {
	return	itsDrive;
}

// --------------------------------------------------------------------
void	CUtxDoc::Drive		( char aV ) {
	itsDrive = aV;
    CUtxDoc::FillFileList();
}

// --------------------------------------------------------------------
const char *	CUtxDoc::Warning	( void ) const {
	return itsWarning;
}

// --------------------------------------------------------------------
void	CUtxDoc::TextureFile( const char * aV ) {
	if	( itsTextureFile )	delete [] itsTextureFile;
    itsTextureFile = ::my_private_strdup( aV );
}

// --------------------------------------------------------------------
const CUtxFile_l &	CUtxDoc::FileList	( void ) const {
	return	itsFileList;
}

// --------------------------------------------------------------------
bool				CUtxDoc::CompressDon	( void ) const {
    return itsCompressDon;
}

// --------------------------------------------------------------------
const char *		CUtxDoc::CompressMsg	( void ) const {
    return itsCompressMsg;
}

// --------------------------------------------------------------------
const char *		CUtxDoc::CompressFil	( void ) const {
    return itsCompressFil;
}

// --------------------------------------------------------------------
int					CUtxDoc::CompressMax	( void ) const {
    return itsCompressMax;
}

// --------------------------------------------------------------------
int					CUtxDoc::CompressPos	( void ) const {
    return itsCompressPos;
}

// --------------------------------------------------------------------
void				CUtxDoc::CompressCan	( void ) {
	itsMutex.Lock();
    itsCompressDon = true;
    itsCompressMsg = "";
    itsCompressFil = "";
    itsMutex.Unlock();
}

// --------------------------------------------------------------------
void				CUtxDoc::CompressAll	( void ) {
	int				count;
    CUtxFile_lci	loop;

	itsMutex.Lock();
    itsCompressDon = false;
	itsCompressMax = 0;
    itsCompressPos = 0;
    itsCompressMsg = "";
    itsCompressFil = "";
	itsMutex.Unlock();

	// First, for the statistics, count number of files to process
    count = 0;
    for	( loop = itsFileList.begin(); loop != itsFileList.end(); loop++ ) {
    	if	( ! (*loop).Select() )		continue;
        if	( (*loop).Directory() )		continue;
        count++;
    }
	itsCompressMax = count;

	// Nothing to do ?
    if	( count == 0 )	return;

    // Let's create all directories needed for this
	CUtxDoc::CreateCompressDirs();

	// Compress all selected image files
    CUtxDoc::CreateCompressImgs();

    // Create class file for texture compilation
	CUtxDoc::CreateCompressClaf();

	// Create ini file for texture compilation
	CUtxDoc::CreateCompressInif();

    // Compile the package with UCC
	CUtxDoc::CreateCompressTexf();

	// Rename the compiled file as texture package
	CUtxDoc::CreateCompressMove();

    // Inform the user
	itsMutex.Lock();
    itsCompressMsg = "Cleaning up the mess";
    itsCompressFil = "";
	itsMutex.Unlock();

    // Remove the ini file
	CUtxDoc::RemoveCompressInif();

    // Remove the class file
	CUtxDoc::RemoveCompressClaf();

    // Remove the compressed images
    CUtxDoc::RemoveCompressImgs();

    // Finally remove all compress directories
	CUtxDoc::RemoveCompressDirs();
}

// --------------------------------------------------------------------
void	CUtxDoc::CreateCompressMove	( void ) {
	char		oldfile	[512];
    char		newfile	[512];

    ::sprintf( oldfile, "%s\\System\\%sUtexPack.u", itsUtRoot, itsTextureFile );
    ::sprintf( newfile, "%s\\Textures\\%s.utx", itsUtRoot, itsTextureFile );
	::rename( oldfile, newfile );
}

// --------------------------------------------------------------------
void	CUtxDoc::CreateCompressTexf	( void ) {
	char				uccfile	[512];
    char				uccpath	[512];
    char				inifile	[512];
    char				cmdline	[1024];
	BOOL				result;
    STARTUPINFO			mystartup;
	PROCESS_INFORMATION	myinfo;
    time_t				reftime;

	if	( itsCompressDon )	return;

	itsMutex.Lock();
    itsCompressMsg = "Compiling into texture file";
    itsCompressFil = itsTextureFile;
	itsMutex.Unlock();

	// Set up needed file names
    ::sprintf( uccfile, "%s\\System\\ucc.exe", itsUtRoot );
    ::sprintf( uccpath, "%s\\System", itsUtRoot );
    ::sprintf( inifile, "%s\\System\\%sUtexPack.uccini", itsUtRoot, itsTextureFile );

    ::sprintf( cmdline, "\"%s\" make -ini=%s", uccfile, inifile );

    GetStartupInfo( &mystartup );
    mystartup.lpReserved		= NULL;
    mystartup.lpDesktop			= NULL;
    mystartup.lpTitle			= NULL;
    mystartup.dwFlags			= STARTF_USESHOWWINDOW;
    mystartup.wShowWindow		= SW_HIDE;

	result = CreateProcess(	uccfile,
    						cmdline,
                            NULL, NULL,
                            FALSE,
                            CREATE_DEFAULT_ERROR_MODE |
							DETACHED_PROCESS |
                            NORMAL_PRIORITY_CLASS,
                            NULL,
                            uccpath,
                            &mystartup,
                            &myinfo );
	if	( result != 0 ) {
    	DWORD	mycode;
		reftime = ::time( NULL ) + 300;	// Must be ready in 5 minutes
        while	( GetExitCodeProcess( myinfo.hProcess, &mycode ) != 0 ) {
			SLEEP( 100 );
			if	( mycode != STILL_ACTIVE )	break;
			if	( ::time( NULL ) > reftime ) {
				TerminateProcess( myinfo.hProcess, 0 );
            }
        }
    }
    else {
    	CUtxDoc::CompressCan();
    }
}

// --------------------------------------------------------------------
void	CUtxDoc::CreateCompressInif	( void ) {
	char		inifile	[1024];
    FILE *		is;

	if	( itsCompressDon )	return;

    ::sprintf( inifile, "%s\\System\\%sUtexPack.uccini", itsUtRoot, itsTextureFile );
	::unlink( inifile );

    is = ::fopen( inifile, "wt" );

	if	( ! is ) {
        CUtxDoc::CompressCan();
    	return;
    }

	::fprintf( is, "[Engine.Engine]\n" );
	::fprintf( is, "EditorEngine=Editor.EditorEngine\n" );
	::fprintf( is, "[Core.System]\n" );
	::fprintf( is, "Paths=../System/*.u\n" );
	::fprintf( is, "[Editor.EditorEngine]\n" );
	::fprintf( is, "CacheSizeMegs=32\n" );
	::fprintf( is, "EditPackages=%sUtexPack\n", itsTextureFile );

    ::fclose( is );
}

// --------------------------------------------------------------------
void	CUtxDoc::RemoveCompressInif	( void ) {
	char		inifile	[1024];
    ::sprintf( inifile, "%s\\System\\%sUtexPack.uccini", itsUtRoot, itsTextureFile );
	::unlink( inifile );
}

// --------------------------------------------------------------------
void	CUtxDoc::CreateCompressClaf	( void ) {
	char		classfile	[1024];
    FILE *		cs;
    CUtxFile_li	loop;

	if	( itsCompressDon )	return;

    ::sprintf( classfile, "%s\\%sUtexPack\\Classes\\UtexPackDummyClass.uc", itsUtRoot, itsTextureFile );
	::unlink( classfile );
    cs = ::fopen( classfile, "wt" );
	if	( ! cs ) {
        CUtxDoc::CompressCan();
    	return;
    }

	itsMutex.Lock();
    itsCompressMsg = "Creating class file";
    itsCompressFil = "UtexPackDummyClass.uc";
	itsMutex.Unlock();

    ::fprintf( cs, "class UtexPackDummyClass extends object;\n" );
    for	( loop = itsFileList.begin(); loop != itsFileList.end(); loop++ ) {
    	if	( ! (*loop).Select() )		continue;
        if	( (*loop).Directory() )		continue;

		::fprintf( cs, "#exec TEXTURE IMPORT " );
        ::fprintf( cs, "NAME=%s ", 	(*loop).Name() );
        ::fprintf( cs, "GROUP=%s ", (*loop).Group() );
        ::fprintf( cs, "Mips=%s ", 	(*loop).MipMaps() ? "true" : "false" );
        ::fprintf( cs, "Masked=%s ",(*loop).Masked()  ? "true" : "false" );
        ::fprintf( cs, "Alpha=%s ",	(*loop).Alpha()   ? "true" : "false" );
		::fprintf( cs, "FILE=Textures\\%s%s\n", (*loop).File(), (*loop).TextureType() > 0 ? ".dds" : "" );
	}
    ::fclose( cs );
}

// --------------------------------------------------------------------
void	CUtxDoc::RemoveCompressClaf	( void ) {
	char	classfile	[1024];
    ::sprintf( classfile, "%s\\%sUtexPack\\Classes\\UtexPackDummyClass.uc", itsUtRoot, itsTextureFile );
	::unlink( classfile );
}

// --------------------------------------------------------------------
void	CUtxDoc::CreateCompressDirs	( void ) {
	char	path	[1024];

	if	( itsCompressDon )	return;

    ::sprintf( path, "%s\\%sUtexPack", itsUtRoot, itsTextureFile );
    ::mkdir( path );

    ::sprintf( path, "%s\\%sUtexPack\\Textures", itsUtRoot, itsTextureFile );
    ::mkdir( path );

    ::sprintf( path, "%s\\%sUtexPack\\Classes", itsUtRoot, itsTextureFile );
    ::mkdir( path );
}

// --------------------------------------------------------------------
void	CUtxDoc::RemoveCompressDirs	( void ) {
	char	path	[1024];

    ::sprintf( path, "%s\\%sUtexPack\\Textures", itsUtRoot, itsTextureFile );
    ::rmdir( path );

    ::sprintf( path, "%s\\%sUtexPack\\Classes", itsUtRoot, itsTextureFile );
    ::rmdir( path );

    ::sprintf( path, "%s\\%sUtexPack", itsUtRoot, itsTextureFile );
    ::rmdir( path );
}

// --------------------------------------------------------------------
void	CUtxDoc::CreateCompressImgs	( void ) {
	char		destination	[1024];
    CUtxFile_li loop;

    ::sprintf( destination, "%s\\%sUtexPack\\Textures", itsUtRoot, itsTextureFile );

	itsMutex.Lock();
    itsCompressMsg = "Compressing images";
    itsCompressPos = 0;
    itsMutex.Unlock();

    for	( loop = itsFileList.begin(); loop != itsFileList.end(); loop++ ) {
    	if	( ! (*loop).Select() )		continue;
        if	( (*loop).Directory() )		continue;

        itsMutex.Lock();
        itsCompressFil = (*loop).File();
        itsMutex.Unlock();

		if	( (*loop).TextureType() > 0 ) {
	        CUtxDoc::DxtCompress( *loop, destination );
        }
        else {
			CUtxDoc::NonCompress( *loop, destination );
        }

        itsMutex.Lock();
        itsCompressPos++;
        itsMutex.Unlock();
        if	( itsCompressDon )	break;
	}
}

// --------------------------------------------------------------------
void	CUtxDoc::RemoveCompressImgs	( void ) {
    char		target	[1024];
    CUtxFile_li loop;
    for	( loop = itsFileList.begin(); loop != itsFileList.end(); loop++ ) {
    	if	( ! (*loop).Select() )		continue;
        if	( (*loop).Directory() )		continue;
	    ::sprintf( target, "%s\\%sUtexPack\\Textures\\%s", itsUtRoot, itsTextureFile, (*loop).File() );
        if	( (*loop).TextureType() > 0 ) {
        	::strcat( target, ".dds" );
        }
		::unlink( target );
    }
}

// --------------------------------------------------------------------
void	CUtxDoc::NonCompress( const CUtxFile & aFile, const char * aPath ) {
	char		source	[1024];
    char		target	[1024];
    FILE *		ss;
    FILE *		ts;
    int			rc;

	if	( itsCompressDon )	return;

    ::getcwd( source, sizeof( source ) - 1 );
    ::sprintf( source + ::strlen( source ), "\\%s", aFile.File() );
    ::sprintf( target, "%s\\%s", aPath, aFile.File() );

	if	( ! ::stricmp( source, target ) )	return;
	::unlink( target );
	ss = ::fopen( source, "rb" );
    ts = ::fopen( target, "wb" );

	if	( ( ss ) && ( ts ) ) {
	    while	( ! feof( ss ) ) {
			rc = ::fgetc( ss );
        	if	( rc >= 0 )	::fputc( rc, ts );
	    }

    	::fclose( ss );
	    ::fclose( ts );
    }
    else {
    	CUtxDoc::CompressCan();
    }
}

// --------------------------------------------------------------------
void	CUtxDoc::DxtCompress( const CUtxFile & aFile, const char * aPath ) {
	char				mysuff	[64];
    char				cmdline	[2048];
	char				source	[1024];
	BOOL				result;
    STARTUPINFO			mystartup;
	PROCESS_INFORMATION	myinfo;
    time_t				reftime;

	if	( itsCompressDon )	return;

    ::sprintf( cmdline, "\"%s\" ", itsNvExe );
    ::getcwd( source, sizeof( source ) - 1 );
	::strcat( source, "\\" );
    ::strcat( source, aFile.File() );
	::sprintf( cmdline + ::strlen( cmdline ), "-file \"%s\" ", source );
	::sprintf( cmdline + ::strlen( cmdline ), "-outdir \"%s\" ", aPath );
	::sprintf( cmdline + ::strlen( cmdline ), "-output \"%s.dds\" ", aFile.File() );

	switch	( aFile.TextureType() ) {
    	case	1:	// DXT1
        if	( aFile.Alpha() )	::strcat( cmdline, "-dxt1a " );
        else					::strcat( cmdline, "-dxt1c " );
        break;

        case	2:	// DXT3
		::strcat( cmdline, "-dxt3 " );
        break;

        case	3:	// DXT5
		::strcat( cmdline, "-dxt5 " );
        break;
    }

    // If the source image is a JPG - one must use -swap
    ::fnsplit( source, NULL, NULL, NULL, mysuff );
    ::strlwr( mysuff );
    if	( ::strstr( ".jpg.jpeg", mysuff ) ) {
	    ::strcat( cmdline, "-swap " );
    }

    if	( ! aFile.MipMaps() )	::strcat( cmdline, "-nomipmap " );
    ::strcat( cmdline, "-rescale hi -overwrite" );

    GetStartupInfo( &mystartup );
    mystartup.lpReserved		= NULL;
    mystartup.lpDesktop			= NULL;
    mystartup.lpTitle			= NULL;
    mystartup.dwFlags			= STARTF_USESHOWWINDOW;
    mystartup.wShowWindow		= SW_HIDE;

    ::getcwd( source, sizeof( source ) - 1 );

	result = CreateProcess(	itsNvExe,
    						cmdline,
                            NULL, NULL,
                            FALSE,
                            CREATE_DEFAULT_ERROR_MODE |
							DETACHED_PROCESS |
                            NORMAL_PRIORITY_CLASS,
                            NULL,
                            source,
                            &mystartup,
                            &myinfo );
	if	( result != 0 ) {
    	DWORD	mycode;
		reftime = ::time( NULL ) + 300;	// Must be ready in 5 minutes
        while	( GetExitCodeProcess( myinfo.hProcess, &mycode ) != 0 ) {
			SLEEP( 100 );
			if	( mycode != STILL_ACTIVE )	break;
			if	( ::time( NULL ) > reftime ) {
				TerminateProcess( myinfo.hProcess, 0 );
            }
        }
    }
    else {
    	CUtxDoc::CompressCan();
    }
}

// --------------------------------------------------------------------
void	CUtxDoc::FillFileList( void ) {
	char		myworkdir	[1024];
    char		mywild		[1024];
    char		myname		[1024];
    char		mysuff		[64];
    struct stat	mystat;
	TSearchRec	sr;

    ::setdisk( itsDrive );
    if	( ::getdisk() != itsDrive ) {
    	itsDrive = (char)::getdisk();
    }
    ::getcwd( myworkdir, sizeof( myworkdir ) - 1 );
    if	( myworkdir[ ::strlen( myworkdir ) - 1 ] != '\\' ) {
    	::strcat( myworkdir, "\\" );
    }
	::strcpy( mywild, myworkdir );
    ::strcat( mywild, "*.*" );

	itsFileList.clear();
    if	( FindFirst( mywild, faAnyFile, sr ) == 0 ) {
		do {
			if	( sr.Name == "." )	continue;
			if		( ::stat( sr.Name.c_str(), &mystat ) != 0 )	;
			else if	( (mystat.st_mode & S_IFREG) == S_IFREG ) {
				::fnsplit( sr.Name.c_str(), NULL, NULL, myname, mysuff );
                ::strlwr( mysuff );
				if		( ::strstr( ".jpg.jpeg.dds.bmp.tga.gif.ppm.tif.cel.pcx", mysuff ) ) {
	            	CUtxFile	myitem;
    	            myitem.File( sr.Name.c_str() );
                    myitem.Name( myname );
                    myitem.Group( "MyGroup" );
            	    itsFileList.push_back( myitem );
                }
            }
			else if	( (mystat.st_mode & S_IFDIR) == S_IFDIR ) {
            	CUtxFile	myitem;
                myitem.File( sr.Name.c_str() );
                myitem.Directory( true );
                itsFileList.push_back( myitem );
            }
		}	while ( FindNext( sr ) == 0 );
		FindClose( sr );
    }
}

// --------------------------------------------------------------------
void	CUtxDoc::DetectSystem( void ) {
	// From registry check out for copies of UT and UT2003
    TRegistry *	reg;
	AnsiString	mypath;

	reg = new TRegistry;
	try {
	    reg->RootKey = HKEY_LOCAL_MACHINE;

		if	( reg->OpenKeyReadOnly( "\\SOFTWARE\\Unreal Technology\\Installed Apps\\UT2003" ) ) {
            mypath = reg->ReadString( "Folder" );
            reg->CloseKey();
			::strcpy( itsUtRoot, mypath.c_str() );
        }
    }

	catch	( ... ) {
    }
	delete reg;
}

// --------------------------------------------------------------------
void	CUtxDoc::Free		( void ) {
	if	( itsTextureFile )	delete [] itsTextureFile;
	CUtxDoc::Cleanup();
}

// --------------------------------------------------------------------
void	CUtxDoc::Cleanup	( void ) {
	::memset( itsNvExe,  0, sizeof( itsNvExe ) );
	::memset( itsUtRoot, 0, sizeof( itsUtRoot ) );
	itsDrive	= 0;
    itsCompressMsg	= "";
    itsCompressFil	= "";
    itsCompressPos	= 0;
    itsCompressMax	= 0;
    itsCompressDon	= false;
	itsTextureFile	= NULL;
    itsWarning		= NULL;
}

// --------------------------------------------------------------------
// EOF:	CUtxDoc.cpp
// --------------------------------------------------------------------
