// --------------------------------------------------------------------
// CCgiArgs.cxx
// Whatis:  Utility class to read in the CGI arguments
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "CCgiArgs.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// local:       Structure of holding the variables in a linked list
// --------------------------------------------------------------------
typedef struct  cgi_variable_s {
    struct cgi_variable_s * next;
    char *                  name;
    char *                  value;
}   cgi_variable_t,
*   cgi_variable_tp;

// --------------------------------------------------------------------
// public:      Constructor
// --------------------------------------------------------------------
CCgiArgs::CCgiArgs () {
    itsArgs = NULL;

    // ----------------------------------------------------------------
	// Default stuff from apache server
    // ----------------------------------------------------------------
    CCgiArgs::AddArg( CGI_SERVER_SOFTWARE,          ::getenv( CGI_SERVER_SOFTWARE ) );
    CCgiArgs::AddArg( CGI_SERVER_NAME,              ::getenv( CGI_SERVER_NAME ) );
    CCgiArgs::AddArg( CGI_GATEWAY_INTERFACE,        ::getenv( CGI_GATEWAY_INTERFACE ) );
    CCgiArgs::AddArg( CGI_SERVER_PROTOCOL,          ::getenv( CGI_SERVER_PROTOCOL ) );
    CCgiArgs::AddArg( CGI_SERVER_PORT,              ::getenv( CGI_SERVER_PORT ) );
    CCgiArgs::AddArg( CGI_REQUEST_METHOD,           ::getenv( CGI_REQUEST_METHOD ) );
    CCgiArgs::AddArg( CGI_HTTP_ACCEPT,              ::getenv( CGI_HTTP_ACCEPT ) );
    CCgiArgs::AddArg( CGI_PATH_INFO,                ::getenv( CGI_PATH_INFO ) );
    CCgiArgs::AddArg( CGI_PATH_TRANSLATED,          ::getenv( CGI_PATH_TRANSLATED ) );
    CCgiArgs::AddArg( CGI_SCRIPT_NAME,              ::getenv( CGI_SCRIPT_NAME ) );
    CCgiArgs::AddArg( CGI_SCRIPT_FILENAME,          ::getenv( CGI_SCRIPT_FILENAME ) );
    CCgiArgs::AddArg( CGI_QUERY_STRING,             ::getenv( CGI_QUERY_STRING ) );
    CCgiArgs::AddArg( CGI_REMOTE_HOST,              ::getenv( CGI_REMOTE_HOST ) );
    CCgiArgs::AddArg( CGI_REMOTE_ADDR,              ::getenv( CGI_REMOTE_ADDR ) );
    CCgiArgs::AddArg( CGI_REMOTE_USER,              ::getenv( CGI_REMOTE_USER ) );
    CCgiArgs::AddArg( CGI_AUTH_TYPE,                ::getenv( CGI_AUTH_TYPE ) );
    CCgiArgs::AddArg( CGI_CONTENT_TYPE,             ::getenv( CGI_CONTENT_TYPE ) );
    CCgiArgs::AddArg( CGI_CONTENT_LENGTH,           ::getenv( CGI_CONTENT_LENGTH ) );
    CCgiArgs::AddArg( CGI_DOCUMENT_ROOT,            ::getenv( CGI_DOCUMENT_ROOT ) );
    CCgiArgs::AddArg( CGI_GATEWAY_INTERFACE,        ::getenv( CGI_GATEWAY_INTERFACE ) );
    CCgiArgs::AddArg( CGI_HTTP_ACCEPT_LANGUAGE,     ::getenv( CGI_HTTP_ACCEPT_LANGUAGE ) );
    CCgiArgs::AddArg( CGI_HTTP_CONNECTION,          ::getenv( CGI_HTTP_CONNECTION ) );
    CCgiArgs::AddArg( CGI_HTTP_EXTENSION,           ::getenv( CGI_HTTP_EXTENSION ) );
    CCgiArgs::AddArg( CGI_HTTP_HOST,                ::getenv( CGI_HTTP_HOST ) );
    CCgiArgs::AddArg( CGI_HTTP_IF_MODIFIED_SINCE,   ::getenv( CGI_HTTP_IF_MODIFIED_SINCE ) );
    CCgiArgs::AddArg( CGI_HTTP_UA_CPU,              ::getenv( CGI_HTTP_UA_CPU ) );
    CCgiArgs::AddArg( CGI_HTTP_UA_OS,               ::getenv( CGI_HTTP_UA_OS ) );
    CCgiArgs::AddArg( CGI_HTTP_USER_AGENT,          ::getenv( CGI_HTTP_USER_AGENT ) );

    // ----------------------------------------------------------------
    // Which method, get or post ?
    // ----------------------------------------------------------------
    if		( ! ::my_stricmp( ::getenv( CGI_REQUEST_METHOD ), "get" ) ) {
		CCgiArgs::GetUrlLine();
    }

    else if	( ! ::my_stricmp( ::getenv( CGI_REQUEST_METHOD ), "post" ) ) {

		// What sort of encoding is used here ???
		const char *	contype	= ::getenv( CGI_CONTENT_TYPE );

		if		( contype == NULL )										CCgiArgs::GetUrlPost();
		else if	( ! ::strncmp( contype, "multipart/form-data;", 20 ) )	CCgiArgs::GetFrmPost();
		else															CCgiArgs::GetUrlPost();

    }

    // ----------------------------------------------------------------
	// Cookie processing
	// This decodes the HTTP_COOKIE environment variable and places
	// the retrieved name=value pairs into the CGI data - ready for use
    // ----------------------------------------------------------------
	const char *	cookie = ::getenv( CGI_HTTP_COOKIE );
	if	( cookie ) {
	    char    	argnam  [1024];
    	char    	argval  [1024];
	    int			state, buf, chr, w;
		while	( *cookie ) {
			::memset( argnam, 0, sizeof( argnam ) );
			::memset( argval, 0, sizeof( argval ) );
	
			w = state = 0;
			while	( ( state >= 0 ) && ( *cookie ) ) {
				chr = CCgiArgs::Decode( state, buf, *(cookie++), '=' );
				if	( w < (int)sizeof( argnam ) - 1 ) {
					if	( state == 0 )	argnam[w++] = chr;
				}
			}
	
			w = state = 0;
			while	( ( state >= 0 ) && ( *cookie ) ) {
				chr = CCgiArgs::Decode( state, buf, *(cookie++), ';' );
				if	( w < (int)sizeof( argval ) - 1 ) {
					if	( state == 0 )	argval[w++] = chr;
				}
			}
	
			::my_strfix( argnam );
			CCgiArgs::AddArg( argnam, argval );
		}
	}
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CCgiArgs::~CCgiArgs () {
    while   ( itsArgs ) {
        cgi_variable_tp item = (cgi_variable_tp)itsArgs;
        itsArgs = (void *)item->next;
        delete [] item->name;
        if  ( item->value ) delete [] item->value;
        delete item;
    }
}

// --------------------------------------------------------------------
// public:      Get an arg by name
// --------------------------------------------------------------------
const char *    CCgiArgs::Arg ( const char * aName ) const {
	if	( ! aName )		return	"";
	if	( *aName == 0 )	return	"";

    cgi_variable_tp item = (cgi_variable_tp)itsArgs;

    while   ( item ) {
        if  ( ! ::my_stricmp( aName, item->name ) ) break;
        item = item->next;
    }

    return  item ? (item->value ? item->value : "") : "";
}


// --------------------------------------------------------------------
// public:      Set an argument a new value. 
// --------------------------------------------------------------------
void            CCgiArgs::Arg ( const char * aName, dword_t aVal ) {
	char	mybuf[32];
	::sprintf( mybuf, "%d", aVal );
	CCgiArgs::Arg( aName, mybuf );
}

// --------------------------------------------------------------------
// public:      Set an argument a new value. Add argument if not there
// --------------------------------------------------------------------
void            CCgiArgs::Arg ( const char * aName, const char * aVal ) {
	if	( ! aName )		return;
	if	( *aName == 0 )	return;

    cgi_variable_tp item = (cgi_variable_tp)itsArgs;

    while   ( item ) {
        if  ( ! ::my_stricmp( aName, item->name ) ) break;
        item = item->next;
    }
	
	if	( ! aVal )	aVal = "";
	if	( *aVal ) {
	    if  ( item ) {
			if  ( item->value ) delete [] item->value;
			item->value = ::my_private_strdup( aVal );
	    }
	    else {
	        CCgiArgs::AddArg( aName, aVal );
	    }
	}
	else {
		if  ( item->value ) delete [] item->value;
		if ( itsArgs == item ) {
		    itsArgs = item->next;
		    delete item;
		}
		else {
		    cgi_variable_tp prev = (cgi_variable_tp)itsArgs;
		    while   ( prev ) {
		        if  ( prev->next == item )  break;
		        prev = prev->next;
		    }
		    prev->next = item->next;
		    delete item;
		}
	}
}

// --------------------------------------------------------------------
// public:      Does an arg exist ?
// --------------------------------------------------------------------
bool            CCgiArgs::Exist ( const char * aName ) const {
	if	( ! aName )		return	false;
	if	( *aName == 0 )	return	false;

    cgi_variable_tp item = (cgi_variable_tp)itsArgs;

    while   ( item ) {
        if  ( ! ::my_stricmp( aName, item->name ) ) break;
        item = item->next;
    }

    return  item != NULL;
}

// --------------------------------------------------------------------
// public:		Encode a value
// --------------------------------------------------------------------
void    CCgiArgs::Encode( const char * aVal ) const {
	::CGI_quote( aVal );
}

// --------------------------------------------------------------------
// public:		Save a list of arguments into a file
// --------------------------------------------------------------------
dword_t	CCgiArgs::Save	(	const char *	aSaveDir,
							const char **	aArgList ) {
	char		savepath	[1024];
	dword_t		saveid		= (dword_t)::rand() % 100;
	struct stat	mystat;
	FILE *		stg			= NULL;

	do	{
		saveid++;
		::sprintf( savepath, "%s/%d", aSaveDir, saveid );

		// Lucky !
		if		( ::stat( savepath, &mystat ) ) {
			stg = ::fopen( savepath, "wb" );
		}

		// Overwrite any file older than 1 hour
		else if	( mystat.st_mtime < (::time( NULL ) - 3600) ) {
			stg = ::fopen( savepath, "wb" );
		}

		// Try another identifier
		else {
			saveid = (dword_t)::rand() % 100;
		}

	}	while	( stg == NULL );

	dword_t	mylen;
	
	mylen = ::strlen( CCgiArgs::Arg( CGI_REMOTE_ADDR ) ) + 1;
	::fwrite( &mylen, sizeof( mylen ), 1, stg );
	if	( mylen ) {
		::fwrite( CCgiArgs::Arg( CGI_REMOTE_ADDR ), mylen, 1, stg );
	}

	while	( *aArgList ) {
		mylen = ::strlen( CCgiArgs::Arg( *(aArgList) ) ) + 1;
		::fwrite( &mylen, sizeof( mylen ), 1, stg );
		::fwrite( CCgiArgs::Arg( *(aArgList) ), mylen, 1, stg );
		aArgList++;
	}

	::fclose( stg );
	return	saveid;
}

// --------------------------------------------------------------------
// public:		Load a list of arguments from a file
// --------------------------------------------------------------------
void	CCgiArgs::Load ( const char * aSaveDir, const char ** aArgNames, dword_t aIdnt ) {
	char		savepath	[1024];
	FILE *		stg			= NULL;
	char *		linebuf		= NULL;
	dword_t		mylen;

	try {
		::sprintf( savepath, "%s/%d", aSaveDir, aIdnt );
		stg = ::fopen( savepath, "rb" );
		if	( ! stg )		throw CError( savepath, ::strerror( errno ) );

		mylen = 0;
		::fread( &mylen, sizeof( mylen ), 1, stg );
		if	( mylen == 0 )	throw CError( savepath, "Invalid data" );

		linebuf = new char [mylen];
		::fread( linebuf, mylen, 1, stg );

		if	( ! ::strcmp( linebuf, CCgiArgs::Arg( CGI_REMOTE_ADDR ) ) ) {
			while	( *aArgNames ) {
				if	( ! ::fread( &mylen, sizeof( mylen ), 1, stg ) )	break;
				if	( linebuf )	{
					delete [] linebuf;
					linebuf = NULL;
				}
				linebuf = new char [mylen];
				if	( ! ::fread( linebuf, mylen, 1, stg ) )				break;
				if	( *linebuf ) CCgiArgs::Arg( *aArgNames, linebuf );
				aArgNames++;
			}
		}
		::fclose( stg );
	
		if	( linebuf )	{
			delete [] linebuf;
			linebuf = NULL;
		}
		::unlink( savepath );
	}
	catch	( CError e ) {
		if	( stg )		::fclose( stg );
		if	( linebuf )	delete [] linebuf;
		throw	CError( "CCgiArgs::Load", e.Error() );
	}
	catch	( ... ) {
		if	( stg )		::fclose( stg );
		if	( linebuf )	delete [] linebuf;
		throw	CError( "CCgiArgs::Load", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:		Decode a single char
// --------------------------------------------------------------------
int			CCgiArgs::Decode( int & aState, int & aBuf, char aChar, char aTerm ) {
	int		chr = aChar;

	switch	( aState ) {
		case	-1:	// Coming here after termination
		break;

		case	0:	// Reading/accepting a character
		if	( chr == aTerm )	{ aState = -1; break; }
		if	( chr == 0 )		{ aState = -1; break; }
		if	( chr == '+' )		{ chr = ' ';   break; }
		if	( chr == '%' )		{ aState = 1;  break; }
		break;

		case	1:	// Reading the first nibble after %
		if	( chr == 0 )		{ aState = -1; break; }
		if	( chr == aTerm )	{ aState = -1; break; }
		aBuf = 16 * CCgiArgs::DecodeHex( aChar );
		aState = 2;
		break;

		case	2:	// Reading the second nibble after %
		if	( chr == 0 )		{ aState = -1; break; }
		if	( chr == aTerm )	{ aState = -1; break; }
		chr = aBuf + CCgiArgs::DecodeHex( aChar );
		aState = 0;
		break;
	}

	return	chr;
}

// --------------------------------------------------------------------
// private:     Decode hex digit
// --------------------------------------------------------------------
int		CCgiArgs::DecodeHex( int aC ) const {
    switch  ( aC ) {
        case    '1':    return  1;
        case    '2':    return  2;
        case    '3':    return  3;
        case    '4':    return  4;
        case    '5':    return  5;
        case    '6':    return  6;
        case    '7':    return  7;
        case    '8':    return  8;
        case    '9':    return  9;
        case    'A':
        case    'a':    return  10;
        case    'B':
        case    'b':    return  11;
        case    'C':
        case    'c':    return  12;
        case    'D':
        case    'd':    return  13;
        case    'E':
        case    'e':    return  14;
        case    'F':
        case    'f':    return  15;
    }
    return  0;
}

// --------------------------------------------------------------------
// private:     Add an argument
// --------------------------------------------------------------------
void		CCgiArgs::AddArg( const char * aName, const char * aValue ) {
	if	( ( aName ) && ( aValue ) ) {
		if	( ( *aName ) && ( *aValue ) ) {
			cgi_variable_tp item = new cgi_variable_t;
			item->next  = (cgi_variable_tp)itsArgs;
			item->name  = ::my_private_strdup( aName );
			item->value = ::my_private_strdup( aValue );
			itsArgs     = item;
		}
	}
}

// --------------------------------------------------------------------
// private:		Get CGI (GET) data from the "command line"
// --------------------------------------------------------------------
void	CCgiArgs::GetUrlLine	( void ) {
	const char *	qs = ::getenv( CGI_QUERY_STRING );
    char    		argnam  [1024];
    char    		argval  [1024];
    int				state, buf, chr, w;
	while	( *qs ) {
		::memset( argnam, 0, sizeof( argnam ) );
		::memset( argval, 0, sizeof( argval ) );

		w = state = 0;
		while	( ( state >= 0 ) && ( *qs ) ) {
			chr = CCgiArgs::Decode( state, buf, *(qs++), '=' );
			if	( w < (int)sizeof( argnam ) - 1 ) {
				if	( state == 0 )	argnam[w++] = chr;
			}
		}

		w = state = 0;
		while	( ( state >= 0 ) && ( *qs ) ) {
			chr = CCgiArgs::Decode( state, buf, *(qs++), '&' );
			if	( w < (int)sizeof( argval ) - 1 ) {
				if	( state == 0 )	argval[w++] = chr;
			}
		}

		CCgiArgs::AddArg( argnam, argval );
	}
}

// --------------------------------------------------------------------
// private:		Get CGI (POST) data URL encoded
// --------------------------------------------------------------------
void	CCgiArgs::GetUrlPost	( void ) {

	dword_t		len = ::atol( ::getenv( CGI_CONTENT_LENGTH ) );
	dword_t		now = 0;
    char		argnam  [1024];
    char		argval  [16384];
    int			state, buf, chr, w;
	while	( now < len ) {
		::memset( argnam, 0, sizeof( argnam ) );
		::memset( argval, 0, sizeof( argval ) );

		w = state = 0;
		while	( ( state >= 0 ) && ( now < len ) ) {
			chr = getchar();	now++;
			chr = CCgiArgs::Decode( state, buf, chr, '=' );
			if	( w < (int)sizeof( argnam ) - 1 ) {
				if	( state == 0 )	argnam[w++] = chr;
			}
		}

		w = state = 0;
		while	( ( state >= 0 ) && ( now < len ) ) {
			chr = getchar();	now++;
			chr = CCgiArgs::Decode( state, buf, chr, '&' );
			if	( w < (int)sizeof( argval ) - 1 ) {
				if	( state == 0 )	argval[w++] = chr;
			}
		}

		CCgiArgs::AddArg( argnam, argval );
	}
}

// --------------------------------------------------------------------
// private:		Get CGI (POST) data multipart/form-data encoded
// --------------------------------------------------------------------
void	CCgiArgs::GetFrmPost	( void ) {
    char		argnam  [1024];
    char		argval  [16384];

	// Initialize the boundary detection system
	CCgiArgs::InitBoundary();

	// Do reading until everything is done
	while	( CCgiArgs::ReadPart( argnam, sizeof(argnam), argval, sizeof(argval) ) ) {
		CCgiArgs::AddArg( argnam, argval );
	}	
}

// --------------------------------------------------------------------
// private:		Initialize the boundary detection system
// --------------------------------------------------------------------
void	CCgiArgs::InitBoundary	( void ) {
	::memset( itsBoundary, 0, sizeof( itsBoundary ) );
	::memset( itsBoundBuf, 0, sizeof( itsBoundBuf ) );
	itsContLen	= ::atol( ::getenv( CGI_CONTENT_LENGTH ) );
	itsContOff	= 0;

	const char *	boundary= ::getenv( CGI_CONTENT_TYPE );

	if	( ! boundary ) {
		::HTTP_text( "en" );
		::printf( "CGI Error: No boundary found\n" );
		::exit( - 1 );
	}

	boundary = ::strstr( boundary, "boundary=" );
	if	( ! boundary ) {
		::HTTP_text( "en" );
		::printf( "CGI Error: No boundary found\n" );
		::exit( - 1 );
	}
	boundary = boundary + 9;

	if	( ::strlen( boundary ) > (sizeof( itsBoundary ) - 5) ) {
		::HTTP_text( "en" );
		::printf( "CGI Error: Boundary string is too long\n" );
		::exit( - 1 );
	}

	::sprintf( itsBoundary, "\x0d\x0a--%s\x0d\x0a", boundary );
	itsBoundLen = ::strlen( itsBoundary );

	// Fill in the buffer
	itsBuffLen = 2;
	itsBoundBuf[0] = '\x0d';
	itsBoundBuf[1] = '\x0a';
	while	( itsBuffLen < itsBoundLen ) {
		itsBoundBuf[itsBuffLen++] = getchar();

	}
	itsContOff = itsBoundLen - 2;

	// Keep on reading until we get the leading boundary
	while	( ::memcmp( itsBoundary, itsBoundBuf, itsBoundLen ) != 0 ) {
		itsBoundBuf[itsBoundLen] = getchar();
		::memcpy( itsBoundBuf, itsBoundBuf + 1, itsBoundLen );
		itsContOff++;
		if	( itsContOff >= itsContLen )	break;
	}

	if	( itsContOff >= itsContLen ) {
		::HTTP_text( "en" );
		::printf( "CGI Error: No boundary found on the stream\n" );
		::exit( - 1 );
	}

	itsSkipLen	= itsBoundLen;
}

// --------------------------------------------------------------------
// private:		Read one part of the argument stream
//				This assumes that the boundary has just been detected
//				the line feed / carriage return are still unread
// --------------------------------------------------------------------
bool	CCgiArgs::ReadPart	( 	char * aNam, dword_t aNamLen,
								char * aVal, dword_t aValLen ) {
	// Clean up the previous mess
	::memset( aNam, 0, aNamLen );
	::memset( aVal, 0, aValLen );

	// Read the header information
	char	linebuf	[1024];
	char	filename[1024];
	char	name	[1024];
	char	type	[1024];


	::memset( name,     0, sizeof( name ) );
	::memset( filename, 0, sizeof( filename ) );
	::memset( type,     0, sizeof( type ) );

	do	{
		if	( ! CCgiArgs::ReadLine( linebuf, sizeof( linebuf ) ) ) {
			return false;
		}
		if	( linebuf[0] == '\x0d' )	break;

		// Decoding the line
		if		( ! ::strncmp( linebuf, "Content-Disposition: form-data", 30 ) ) {
			char *	p;
			p = ::strstr( linebuf + 30, "; name=\"" );
			if	( p ) {
				::strcpy( name, p + 8 );
				p = ::strchr( name, '"' );
				if	( p )	*p = 0;
			}
			p = ::strstr( linebuf + 30, "; filename=\"" );
			if	( p ) {
				::strcpy( filename, p + 12 );
				p = ::strchr( filename, '"' );
				if	( p )	*p = 0;
			}
		}
		else if	( ! ::strncmp( linebuf, "Content-Type: ", 14 ) ) {
			::strcpy( type, linebuf + 14 );
			::my_strtail( ::my_strhead( type ) );
		}

		else {
			::HTTP_text( "en" );
			::printf( "CGI Error: Unknown header: %s\n", linebuf );
			::exit( - 1 );
		}

	}	while	( linebuf[0] != '\x0d' );

	// The argument name
	if	( ::strlen( name ) > (aNamLen - 1) ) {
		::HTTP_text( "en" );
		::printf( "CGI Error: Argument name is too long\n" );
		::exit( - 1 );
	}

	if	( name[0] == 0 ) {
		::HTTP_text( "en" );
		::printf( "CGI Error: Missing argument name\n" );
		::exit( - 1 );
	}

	::strcpy( aNam, name );

	// If not a file, we will copy the data from stream into the value itself
	dword_t	off = 0;
	int		chr;

	if	( filename[0] == 0 ) {

		chr = CCgiArgs::ReadByte();
		while	( chr >= 0 ) {
			if	( off < (aValLen - 1) ) {
				aVal[off++] = chr;
			}
			chr = CCgiArgs::ReadByte();
		}
		return true;
	}

	// Since we are dealing with a file, we store it into the upload
	// directory and set up the value pointing into it
	// Value is constructed as follows:
	// type; filepath; originalname
	char		filepath[1300];
	dword_t		filenum	= 1;
	struct stat	mystat;

	// Get rid of the path components in the file name
	if	( ::strrchr( filename, '/' ) ) {
		::strcpy( filename, ::strrchr( filename, '/' ) + 1 );
	}
	if	( ::strrchr( filename, '\\' ) ) {
		::strcpy( filename, ::strrchr( filename, '\\' ) + 1 );
	}
	if	( ::strrchr( filename, ':' ) ) {
		::strcpy( filename, ::strrchr( filename, ':' ) + 1 );
	}

	do	{
		::sprintf( filepath, "./upload/upl-%d-%s", filenum++, filename );
	}	while	( ! ::stat( filepath, &mystat ) );

	if	( ::strlen( filepath ) > (aValLen-1) ) {
		::HTTP_text( "en" );
		::printf( "CGI Error: Argument name is too long\n" );
		::exit( - 1 );
	}

	::strcpy( aVal, filepath );

	// Now, store the data
	FILE *	stg = ::fopen( filepath, "wb" );

	chr = CCgiArgs::ReadByte();
	while	( chr >= 0 ) {
		if	( stg )	fputc( chr, stg );
		chr = CCgiArgs::ReadByte();
	}
	
	if	( stg )	::fclose( stg );

	return true;
}

// --------------------------------------------------------------------
// private:		Read one byte from the incoming stream
//				returns:	-1 	-> No more data
//							-2 	-> Boundary detected
//							>= 0-> Actual data
// --------------------------------------------------------------------
int		CCgiArgs::ReadByte	( void ) {

	// Should we do some skipping ?
	if	( itsSkipLen > 0 ) {
		while	( itsSkipLen > 0 ) {
			if	( itsContOff < itsContLen ) {
				itsBoundBuf[itsBoundLen] = getchar();
				itsContOff++;		
			}
			else {
				if	( itsBuffLen > 0 )	itsBuffLen--;
				else					break;
			}
			::memcpy( itsBoundBuf, itsBoundBuf + 1, itsBoundLen );
			itsSkipLen--;
		}
	}

	// Already at the end of data ???
	else if	( ( itsContOff >= itsContLen ) && ( itsBuffLen < 1 ) ) {
		return -1;
	}

	// Read in one byte
	else {
		if	( itsContOff < itsContLen ) {
			itsBoundBuf[itsBoundLen] = getchar();
			itsContOff++;		
		}
		else {
			itsBuffLen--;
		}
		::memcpy( itsBoundBuf, itsBoundBuf + 1, itsBoundLen );
	}

	// Is this a boundary ???
	if	( ! ::memcmp( itsBoundary, itsBoundBuf, itsBoundLen - 2 ) ) {
		if	( ( itsBoundBuf[itsBoundLen-2] == '\x0d' ) &&
			  ( itsBoundBuf[itsBoundLen-1] == '\x0a' ) ) {
			itsSkipLen = itsBoundLen;
			return -2;
		}
		if	( ( itsBoundBuf[itsBoundLen-2] == '-' ) &&
			  ( itsBoundBuf[itsBoundLen-1] == '-' ) ) {
			while	( itsContOff < itsContLen ) {
				getchar();
				itsContOff++;
			}
			itsBuffLen = 0;
			return -1;
		}
	}

	return	(int)(((word_t)itsBoundBuf[0]) & 0xff);
}

// --------------------------------------------------------------------
// private:		Read one line from the input stream - include the eol
// --------------------------------------------------------------------
bool	CCgiArgs::ReadLine (	char * aBuf, dword_t aLen ) {
	dword_t	ix = 0;
	int		pch;
	int		ch;
	

	::memset( aBuf, 0, aLen );
	pch = ch = 0;

	do {
		pch = ch;
		ch = CCgiArgs::ReadByte();
		if	( ch < 0 )				return false;
		aBuf[ix++] = ch;
		if	( ix >= (aLen - 1) )	return false;
	}	while	( ( pch != '\x0d' ) && ( ch != '\x0a' ) );

	return	true;
}

// --------------------------------------------------------------------
// EOF: CCgiArgs.cxx
// --------------------------------------------------------------------
